/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/alloca.c
 * Summary:
 *	Implementation of `alloca' function. If you compile (as recommended)
 *	with -fno-builtin, you will get this implementation of a checked
 *	alloca, with optional checks for stale pointers and so forth.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		31/3/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

#if defined(alloca)
#undef alloca
#endif

#if defined(__alloca)
#undef __alloca
#endif

inline void *
__alloca (size_t size)
{
  alloca_item *item;
  object *obj;
  char *function_name;

  item = __bounds_malloc (sizeof (alloca_item));
  if (!item)
    __bounds_internal_error ("out of space allocating alloca chain",
			     __FILE__, __LINE__);

  /* Get `functions.c' to add the current alloca item to the current function
   * context. When the current function exits, we will be called (in
   * __bounds_alloca_free) with a chain of these allocations.
   */
  function_name = __bounds_add_alloca_chain (item);

  /* Try to allocate the space as a checked memory allocation. Note:
   * (1) alloca never fails,
   * (2) `free' will be called for this object, but according to the
   * current heap mode, it may not actually be freed, and
   * (3) we allocate from the heap, not the stack.
   */
  item->pointer = malloc (size);
  if (!item->pointer)
    __bounds_internal_error ("alloca failed (out of memory)", NULL, 0);
  item->size = size;

  /* Search for the corresponding object and add a note saying what function
   * it came from. This debugging information gets printed in `error.c'.
   */
  if (function_name &&
      (obj = __bounds_find_object_by_base (item->pointer)) != NULL)
    obj->alloca_function = function_name;

  return item->pointer;
}

void *
alloca (size_t size)
{
  return __alloca (size);
}

/*----------------------------------------------------------------------
 *	Free up the alloca stack for the current function.
 *----------------------------------------------------------------------*/

void
__bounds_alloca_free (alloca_item *alloca_stack)
{
  alloca_item *next;

  while (alloca_stack)
    {
      next = alloca_stack->next;
      free (alloca_stack->pointer);
      __bounds_free (alloca_stack);
      alloca_stack = next;
    }
}
