/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes.events.engine;

/** The class <code>IdeEvensList</code> supports a dual linked list of IdeEventsListItems.
 *  All the Listeners (attached to corresponding IdeEventsListItem) will accept
 *  and handle the Events of the same Class (kind).
 *  SomeListener.getEventsClass() for all the listeners return the same Class.
 *  Some of the Listeners can handle also derivated from this Event Class new Events.
 *  if there are (is) such a Listener(s) then SomeListener.isInheritedEvents() have to return true.
 *
 *  Events are Dispatched in this scenario
 *  if (CrntEvent.getClass == EventsBaseClass)
 *    this event will be delivered to all the Listeners in the list.
 *  else
 *   if (isInheritedEvents()) and (CrntEvent.getClass is subclass of EventsBaseClass)
 *     this event will be delivered to all the Listeners which method isChildEventsListener() returns true.
 *
 * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
 * @version 0.7, 02-12-97 */

class IdeEvensList {
  IdeEvensList      nexEventsList;     /** which is the next EventsClass to manage */
  IdeEvensList      prevEventsList;    /** which is the previous one               */
  IdeEventsListItem EventsList;        /** the first item of the list, or null     */
  private Class     EventsBaseClass;   /** Which Events will handle the listeners in this list */
  boolean           isInheritedEvents; /** true - if there is at least one listener which will handle inherited events */

  IdeEvensList(Class AEventsBaseClass) {
    nexEventsList     = null;
    prevEventsList    = null;
    EventsList        = null;
    isInheritedEvents = false;
    EventsBaseClass   = AEventsBaseClass;
  }

  /** true if some of the listeners in the list will procceed child events of this Event class too */
  boolean isChildEventsListener()        { return isInheritedEvents; }
  IdeEvensList      getNext()            { return nexEventsList;  }
  IdeEvensList      getPrev()            { return prevEventsList; }
  Class             getEventsBaseClass() { return EventsBaseClass; }

  /** find listener's item  in the list and returns it's reference or null */
  IdeEventsListItem findListItem(IdeEventListenerI EventListener) {
    IdeEventsListItem CrntListItem;

    CrntListItem = EventsList;
    while (CrntListItem != null) {
      if (CrntListItem.TheListener == EventListener)
        return CrntListItem;

      CrntListItem = CrntListItem.getNext();
    }
    return null;
  }

  /** find listener in the list and returns it's reference or null */
  IdeEventListenerI findListener(IdeEventListenerI EventListener) {
    IdeEventsListItem CrntListItem;

    CrntListItem = findListItem(EventListener);
    if (CrntListItem == null) return null;
    return CrntListItem.TheListener;
  }

  private void setInheritedEvents(boolean Value) {
    isInheritedEvents = Value;
  }

  /** Checks if there is at least one listener which will handle inherited events
    * If so, then setInheritedEvents(true) */
  private void CheckForInheritedEvents() {
    IdeEventsListItem CrntListItem;

    CrntListItem = EventsList;
    while (CrntListItem != null) {
      if (CrntListItem.TheListener != null) {
        if (CrntListItem.TheListener.isChildEventsListener()) {
          setInheritedEvents(true);
          return;
        }
      }
      CrntListItem = CrntListItem.getNext();
    }

    setInheritedEvents(false);
  }

  /** Puts this IdeEvensList before NewItem.
    * this method shall be called only from synchronized function */
  void PutBefore(IdeEvensList ELsList) {
    if (ELsList == null) return;

    nexEventsList  = ELsList;
    prevEventsList = ELsList.prevEventsList;
    ELsList.prevEventsList = this;
  }

  /** Removes the List Item from the list.
    * this method shall be called only from synchronized function */
  void Remove() {
    if (nexEventsList != null)
      nexEventsList.prevEventsList = prevEventsList;

    if (prevEventsList != null)
      prevEventsList.nexEventsList = nexEventsList;
  }

  /** Adds another Listener to the list. This method shall be called only from synchronized function */
  void AddEventListener(IdeEventListenerI EventListener) {
    IdeEventsListItem CrntListItem;

    if (EventListener == null) return;
    if (findListener(EventListener) != null) return; // we will not add the same listener twise

    CrntListItem = new IdeEventsListItem(EventListener);
    CrntListItem.PutBefore(EventsList); // Make this new item first in the chain
    EventsList   = CrntListItem;        // Said to chain to begin from this one.

    if (CrntListItem.TheListener.isChildEventsListener())
      setInheritedEvents(true);
    // CheckForInheritedEvents();
  }

  /** Removes this event listener (if find) This method shall be called only from synchronized function */
  boolean RemoveEventListener(IdeEventListenerI EventListener) { /** true if found and removed */
    IdeEventsListItem SearchedListItem;

    SearchedListItem = findListItem(EventListener);
    if (SearchedListItem == null) return false;

    if (EventsList == SearchedListItem)
      EventsList = SearchedListItem.getNext();

    SearchedListItem.Remove();
    if (SearchedListItem.TheListener.isChildEventsListener())
      CheckForInheritedEvents();
      
    return true;
  }

  /** Dispatches the events to the listeners in scenario written above
    * @param  AEventClass = TheEvent.getClass(). For more quick checking
    * @param  TheEvent    The event we want to deliver to the listeners.
    * @param  WhichOne    Which method will be called
    * WhichOne < 0 - BeforeAction, 0 - OnAction, > 0 AfterAction
    * @see    <code>IdeEventsEngine.CALL_BEFORE_METHOD</code>
    * @param  JustCheck   if true, we will only check if there is any listener wich will accept
    * the event. and if so, TheEvent.setHandled(); to indicate this. 
  */
  void DispatchEvent(Class AEventClass, IdeSuperEvent TheEvent, int WhichOne, boolean JustCheck) {
    IdeEventsListItem CrntListItem;

    if (EventsBaseClass == AEventClass) { // if this event is for us
      CrntListItem = EventsList;
      while (CrntListItem != null) {
        CrntListItem.DispatchEvent(TheEvent, WhichOne, JustCheck);
        CrntListItem = CrntListItem.getNext();
      }
      return;
    }

    if (isChildEventsListener() != true) return;

    if (EventsBaseClass.isAssignableFrom(AEventClass)) { // if this Event is derivated from EventsBaseClass
      CrntListItem = EventsList;
      while (CrntListItem != null) {
        if (CrntListItem.isChildEventsListener())
          CrntListItem.DispatchEvent(TheEvent, WhichOne, JustCheck);

        CrntListItem = CrntListItem.getNext();
      }
    }
  }
}



