/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.beans;

import java.awt.event.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;
import org.freebuilder.gui.beans.*;

// ------------------------------------------------------------------------------
// Begin of XTrackLayout
// ------------------------------------------------------------------------------

/**
  This class is a layout manager that keeps track of the
  parent's width and resizes the container to fit int it's parent
  horizontally.
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
   @version 0.7, 07-01-98
*/
class XTrackLayout implements LayoutManager {
  /** Does nothing*/
  public void addLayoutComponent(String name, Component comp) {}

  /** Does nothing*/
  public void removeLayoutComponent(Component comp) {}

  /** Does nothing. Returns new Dimension.*/
  public Dimension preferredLayoutSize(Container parent) {
    return new Dimension();
  }

  /** Does nothing. Returns new Dimension.*/
  public Dimension minimumLayoutSize(Container parent) {
    return new Dimension();
  }

  /**
    Resizes the parent to fit in it's parent horizontally.
    All the parent's components, which are not containers
    are resized to fit the parent's width, but only if their
    preferred width is less than the parent's width.
    All the parent's components, which are containers are
    invalidated, so their layout manager does it's job.
  */
  public void layoutContainer(Container parent) {
    int CompCount = parent.getComponentCount();
    int i;
    int ParentWidth = parent.getParent().getSize().width;
    int XLocation = parent.getLocation().x;
    int PrefWidth = parent.getPreferredSize().width;
    Class PC = (new Panel ()).getClass();

    if (PrefWidth > ParentWidth - XLocation)
      parent.setSize (PrefWidth, parent.getSize().height);
    else
      parent.setSize (ParentWidth - XLocation, parent.getSize().height);

    for (i = 0; i < CompCount; i ++) {
      Component C = parent.getComponent(i);
      Class CompClass = C.getClass();
      if (PC.isInstance (C))
        C.invalidate();
      else {
        int CompX = C.getLocation().x;
        C.setSize (ParentWidth - CompX - XLocation, C.getSize().height);
      }
    }
  }
}

// ------------------------------------------------------------------------------
// End of XTrackLayout
// ------------------------------------------------------------------------------


// ------------------------------------------------------------------------------
// Begin of LabelsPanel
// ------------------------------------------------------------------------------

/**
  This class represents a panel, which contains labels.
  Objects of this type MUST be added to a NamedIconList.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class LabelsPanel extends Panel {
  /** The preferred width of this panel.*/
  private int PrefWidth = 0;
  private int Height = 36;

  /**
    Notifies the owner NamedIconList, that the specified
    label must be selected.
  */
  public void select (int No) {
    ((NamedIconList) getParent()).doSelect (No);
  }

  /** Returns the preferred size of this panel.*/
  public Dimension getPreferredSize () {
    return new Dimension (PrefWidth, getComponentCount() * Height);
  }

  /**
    Sets the preferred width of this panel. This method is
    called by the owner NamedIconList, after it calculates
    all the widths of the labels, contained in this panel.
  */
  public void setHeight (int Height) {
    int i;
    int ComponentCount = getComponentCount();
    this.Height = Height;
    for (i = 0; i < ComponentCount; i++ ) {
      IconLabel IL = (IconLabel) getComponent(i);
      IL.setIconSize (Height);
      IL.setLocation (0, i*Height);
      IL.setSize (IL.getPreferredSize());
    }
    repaint();
  }

  public void setPrefWidth (int Width) {
    PrefWidth = Width;
  }
}

// ------------------------------------------------------------------------------
// End of LabelsPanel
// ------------------------------------------------------------------------------

// ------------------------------------------------------------------------------
// Begin of IconsPanel
// ------------------------------------------------------------------------------

/**
  This class represents a panel, which contains icons.
  Objects of this type MUST be added to a NamedIconList.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class IconsPanel extends Panel implements IconCanvasContainer, IconCanvasListener  {
  private int IconSize = 36;

  /**
    Notifies the owner NamedIconList, that the specified
    icon must be selected.
  */
  public void select (int No) {
    ((NamedIconList) getParent()).doSelect (No);
  }

  /** Returns the preferred size of this object. */
  public Dimension getPreferredSize () {
    return new Dimension (IconSize, getComponentCount() * IconSize);
  }

  public void setIconSize (int NewSize) {
    if (NewSize < 16)
      return;

    int ComponentCount = getComponentCount();
    int i;
    IconSize = NewSize;
    for (i = 0; i < ComponentCount; i++) {
      IconCanvas IC = ((IconCanvas) getComponent(i));
      IC.setLocation (0, i * NewSize);
      IC.setSize(NewSize, NewSize);
    }
    validate();
  }
}

// ------------------------------------------------------------------------------
// End of IconsPanel
// ------------------------------------------------------------------------------

// ------------------------------------------------------------------------------
// Begin of IconLabel
// ------------------------------------------------------------------------------

/**
  This class represins a single line of text int the
  NamedIconList class. Objects of this class MUST be
  added in a LabelsPanel.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
class IconLabel extends Component implements MouseListener {
  /** If this label is selected. */
  private boolean Selected = false;

  /**
    If this label is focused. The label may be selected,
    but not focused.
  */
  private boolean Focused = false;

  private int IconSize = 36;

  /** The text of the label.*/
  private String Text;

  /** The preferred width of the label.*/
  private int PrefWidth;

  /** The number of the label int the LabelsPanel.*/
  private int No;

  /** Initializes the object. */
  public IconLabel (String Text, int No) {
    addMouseListener (this);
    this.No = No;
    this.Text = Text;
    PrefWidth = Text.length()*8 + IconSize;
  }

  /**
    Paints the label, depending on if it is selected
    and focused.
  */
  public void paint (Graphics g) {
    if (g == null)
      return;
    int Height = g.getFontMetrics().getHeight();

    if (Selected) {
      g.setColor (Color.lightGray);
      g.fillRect (1, 1, getSize().width - 2, getSize().height - 2);
      if (Focused) {
        g.setColor (Color.black);
        g.drawRect (0,0,getSize().width - 1, getSize().height - 1);
      }
      g.setColor (Color.white);
    } else {
      g.setColor (Color.gray);
      g.fillRect (0, 0, getSize().width, getSize().height);
      g.setColor (Color.black);
    }
    g.drawString (Text, IconSize, (IconSize + Height)/2  );
    PrefWidth = g.getFontMetrics().stringWidth(Text) + IconSize;
  }

  /** Selects this label. */
  public void select (boolean Focused) {
    Selected = true;
    this.Focused = Focused;
    repaint ();
  }

  /** Deselects this label. */
  public void deselect () {
    Selected = false;
    repaint ();
  }

  /** Does nothing.*/
  public void mouseClicked (MouseEvent e) {}

  /** Does nothing.*/
  public void mouseReleased (MouseEvent e) {}

  /** Does nothing.*/
  public void mouseEntered (MouseEvent e) {}

  /** Does nothing.*/
  public void mouseExited (MouseEvent e) {}

  /**
    Notifies the owner LabelsPanel, that this label should
    be selected.
  */
  public void mousePressed (MouseEvent e) {
    ((LabelsPanel) getParent()).select (No);
  }

  /** Returns the preferred size of this label. */
  public Dimension getPreferredSize () {
    return new Dimension (PrefWidth + IconSize, IconSize);
  }

  public void setIconSize (int NewSize) {
    IconSize = NewSize;
    repaint();
  }
  public void setPrefWidth (int Size) {
    PrefWidth = Size;
  }
}

// ------------------------------------------------------------------------------
// End of IconLabel
// ------------------------------------------------------------------------------

// ------------------------------------------------------------------------------
// Begin of NamedIconList
// ------------------------------------------------------------------------------

/**
  This class represents a list of items, that implement
  NamedIcon interface. Objects of this class should be added
  to a ScrollPane.
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
public class NamedIconList extends Panel implements ItemSelectable, FocusListener {

  /** The panel, that contains the icons. */
  private IconsPanel IP = new IconsPanel ();

  /** The panel, that contains the icons. */
  private LabelsPanel TP = new LabelsPanel ();

  /** The source of NamedIcon-s. */
  private NamedIconsCollection Source;

  private int IconSize = 36;

  /**
    The listeners, which will be notified, when a new item from
    the list is selected, are stored here.
  */
  private Vector ItemListeners = new Vector();

  /** The position of the currently selected item. */
  private int Selected = -1;

  /**
    A magic digit to fix the inclorrect dimensions, returned
    by the ScrollPane's <code>getSize()</code> method.
    It is ofcourse 4 (four).
  */
  private static final int MagicDigitFour = 4;


  /** Notifies the listeners, when a new item is selected. */
  public void notifySelection () {
    int i;
    Enumeration e = ItemListeners.elements();

    for (;e.hasMoreElements();) {
      ItemListener L = (ItemListener) e.nextElement();
      L.itemStateChanged (new ItemEvent (this, Selected, null, ItemEvent.SELECTED));
    }
  }

  /**
    Adds an ItemListener, to be notified, when a new item
    is selected.
  */
  public void addItemListener (ItemListener L) {
    ItemListeners.addElement (L);
  }

  /** Removes the specified ItemListener. */
  public void removeItemListener (ItemListener L) {
    ItemListeners.removeElement (L);
  }

  /** Returns <code>null</code>. */
  public Object[] getSelectedObjects () {
    return null;
  }

  /**
    Updates the panels, when a new source of NamedIcon-s
    is set.
  */
  private void updatePanels () {
    setCursor  (new Cursor (Cursor.WAIT_CURSOR));

    int IconsCount = Source.getCount();
    int i;
    int MaxWidth = 0;
    int CurrentWidth = 0;

    IP.removeAll();
    TP.removeAll();

    for (i = 0; i < IconsCount; i++) {
      IconCanvas IC = new IconCanvas (Source.getNamedIcon(i).getIcon(), i);
      String LS = Source.getNamedIcon (i).getName();
      IconLabel L = new IconLabel (LS, i);
      CurrentWidth = L.getPreferredSize().width;
      if (CurrentWidth > MaxWidth)
        MaxWidth = CurrentWidth;
      IP.add (IC);
      IC.addIconCanvasListener (IP);
      TP.add (L);
    }

    for (i = 0; i < IconsCount; i++) {
      ((IconLabel)TP.getComponent(i)).setPrefWidth (MaxWidth);
    }

    IP.setSize (IconSize, IconsCount*IconSize);
    TP.setSize (MaxWidth, IconsCount*IconSize);

    IP.setIconSize (IconSize);
    TP.setPrefWidth (MaxWidth);
    TP.setHeight (IconSize);

    invalidate();
    TP.invalidate();
    IP.invalidate();
    getParent().invalidate();
    getParent().validate();

    setCursor  (new Cursor (Cursor.DEFAULT_CURSOR));
  }

  /** Sets a new source of NamedIcon-s.*/
  public void setSource (NamedIconsCollection Source) {
    int PrevSelected = getSelectedIndex();
    int ToSelect = -1;

    if ((Source != null) && (Source.getCount() > 0))
      if ((!Source.equals(this.Source)) || (PrevSelected < 0))
        ToSelect = 0;
      else
        ToSelect = PrevSelected;

    this.Source = Source;
    if (Source != null) {
      updatePanels();
      select (ToSelect);
    }
  }

  /** Initializes the object. */
  private void init () {
    setLayout (new XYLayout());
    IP.setLayout (new XYLayout());
    add (IP);
    IP.setLocation (0, 0);
    TP.setLayout (new XTrackLayout());
    TP.setLocation (IconSize, 0);
    add (TP);
    addFocusListener (this);
  }

  /** Construcs the object with no source. */
  public NamedIconList () {
    super();
    init();
    setSource (null);
  }

  /**
    Construcs the object and sets the source to be the passed
    NamedIconsCollection.
  */
  public NamedIconList (NamedIconsCollection Source) {
    super();
    init();
    setSource(Source);
  }

  /** Returns the index of the selected item. */
  public int getSelectedIndex() {
    if ((Source == null) || (Source.getCount() == 0))
      return -1;
    return Selected;
  }

  /** Selects the specified item and focuses the list.*/
  public void doSelect (int NewPos) {
    select (NewPos);
    requestFocus();
    softSelect (true);
  }

  /**
    Repaints the slelected item, depending on if the list is
    focused or not.
  */
  private void softSelect (boolean Focused) {
    if (Selected != -1) {
      ((IconCanvas)IP.getComponent (Selected)).select();
      ((IconLabel)TP.getComponent (Selected)).select(Focused);
    }
  }

  /** Selects the specified item without focusing the list.*/
  public void select (int NewPos) {
    if ((Selected != -1) && (Selected < Source.getCount())) {
      ((IconCanvas)IP.getComponent (Selected)).deselect();
      ((IconLabel)TP.getComponent (Selected)).deselect();
    }
    if (NewPos >= Source.getCount())
      Selected = Source.getCount()-1;
    else
      Selected = NewPos;

    softSelect (false);
    notifySelection();
  }

  /** Called when the focus is lost. */
  public void focusLost (FocusEvent e) {
    softSelect (false);
  }

  /** Does nothing. */
  public void focusGained (FocusEvent e) {}

  /** Returns the number of items in the list. */
  public int getItemCount() {
    if (Source == null)
      return 0;
    return Source.getCount();
  }

  /** Returns the preffered size ofthe list. */
  public Dimension getPreferredSize () {
    if (Source == null)
      return new Dimension ();

    int RealHeight = IP.getSize().height;
    int ParentHeight = getParent().getSize().height;

    int TheWidth = TP.getPreferredSize().width + IP.getSize().width;

    int ParentWidth = getParent().getSize().width;

    if (ParentHeight < RealHeight) {
      ParentWidth -= ((ScrollPane)getParent()).getVScrollbarWidth() + MagicDigitFour;
    } else {
      ParentWidth -= MagicDigitFour;
    }

    int RealWidth = TheWidth < ParentWidth ? ParentWidth : TheWidth;
    return new Dimension (RealWidth, RealHeight);
  }

  public void setRowSize (int NewSize) {
    IconSize = NewSize;
    IP.setIconSize (NewSize);
    TP.setHeight (NewSize);
    TP.setLocation (NewSize, 0);
    invalidate();
    validate();
  }
}

// ------------------------------------------------------------------------------
// End of NamedIconList
// ------------------------------------------------------------------------------

