/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */


package org.freebuilder.tools.editor;

import org.freebuilder.gui.idemenu.*;
import org.freebuilder.gui.beans.*;
import org.freebuilder.system.ideengine.*;

import org.freebuilder.system.classes.options.*;
import org.freebuilder.system.classes.options.manager.*;
import org.freebuilder.system.classes.events.engine.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.system.debug.*;

import java.io.*;
import java.awt.*;
import java.awt.event.*;

import com.sun.java.swing.*;
import com.sun.java.swing.text.*;
import com.sun.java.swing.event.*;

/** This class is a JTextArea enchansement. It implements the Shift+DownArrow
    functionality, allowing the user to select a row from the keyboard.
  * @author	<a href=mailto:wreck@nat.bg> Peter Radkov </a>
*/
class ShiftDownAction extends AbstractAction {
  public void actionPerformed (ActionEvent e) {
    JTextArea TheSource = (JTextArea) e.getSource();
    String Selection = TheSource.getSelectedText();
    boolean isSthSelected = ((Selection != null) && (Selection.length() > 0));
    int PrevSelStart = 0;
    int PrevSelEnd   = 0;
    int PrevCaretPos = 0;

    if (isSthSelected) {
      PrevSelStart = TheSource.getSelectionStart();
      PrevSelEnd   = TheSource.getSelectionEnd();
    } else {
      PrevCaretPos = TheSource.getCaretPosition();
    }

    Keymap TheMap = TheSource.getKeymap();
    Action DownAction = TheMap.getAction (KeyStroke.getKeyStroke (KeyEvent.VK_DOWN, 0));
    if (DownAction != null) {
      DownAction.actionPerformed (e);
      int NewCaretPos = TheSource.getCaretPosition();
      if (isSthSelected) {
        if (NewCaretPos > PrevSelEnd)
          TheSource.setCaretPosition (PrevSelStart);
        else
          TheSource.setCaretPosition (PrevSelEnd);
      } else
        TheSource.setCaretPosition (PrevCaretPos);
      TheSource.moveCaretPosition (NewCaretPos);
    }
  }
}

/** This class is a JTextArea enchansement. It implements the Shift+UpArrow
    functionality, allowing the user to select a row from the keyboard.
*/
class ShiftUpAction extends AbstractAction {
  public void actionPerformed (ActionEvent e) {
    JTextArea TheSource = (JTextArea) e.getSource();
    String Selection = TheSource.getSelectedText();
    boolean isSthSelected = ((Selection != null) && (Selection.length() > 0));
    int PrevSelStart = 0;
    int PrevSelEnd   = 0;
    int PrevCaretPos = 0;
    if (isSthSelected) {
      PrevSelStart = TheSource.getSelectionStart();
      PrevSelEnd   = TheSource.getSelectionEnd();
    } else {
      PrevCaretPos = TheSource.getCaretPosition();
    }

    Keymap TheMap = TheSource.getKeymap();
    Action UpAction = TheMap.getAction (KeyStroke.getKeyStroke (KeyEvent.VK_UP, 0));
    if (UpAction != null) {
      UpAction.actionPerformed (e);
      int NewCaretPos = TheSource.getCaretPosition();
      if (isSthSelected) {
        if (NewCaretPos < PrevSelStart)
          TheSource.setCaretPosition (PrevSelEnd);
        else
          TheSource.setCaretPosition (PrevSelStart);
      } else
        TheSource.setCaretPosition (PrevCaretPos);
      TheSource.moveCaretPosition (NewCaretPos);
    }
  }
}

/** This class is a JTextArea enchansement. It implements the (Shift)+Home
    functionality, allowing the user to select a part of a row from the
    keyboard, or just to move the cursor at the beginning of th row.
*/
class HomeAction extends AbstractAction {
  public void   actionPerformed (ActionEvent e) {
    try {
      JTextArea TheSource = (JTextArea) e.getSource();
      int CaretPos = TheSource.getCaretPosition();
      int LineNo    = TheSource.getLineOfOffset   (CaretPos);
      int LineStart = TheSource.getLineStartOffset(LineNo);
      if ((e.getModifiers() & Event.SHIFT_MASK) != 0)
        TheSource.moveCaretPosition (LineStart);
      else
        TheSource.setCaretPosition (LineStart);
    } catch (Exception ex) {
      System.out.println (ex);
    }
  }
}

/** This class is a JTextArea enchansement. It implements the (Shift)+End
    functionality, allowing the user to select a part of a row from the
    keyboard, or just to move the cursor at the end of th row.
*/
class EndAction extends AbstractAction {
  public void   actionPerformed (ActionEvent e) {
    try {
      JTextArea TheSource = (JTextArea) e.getSource();
      int CaretPos = TheSource.getCaretPosition();
      int LineNo    = TheSource.getLineOfOffset   (CaretPos);
      int LineEnd   = TheSource.getLineEndOffset(LineNo);
      if ((e.getModifiers() & Event.SHIFT_MASK) != 0)
        TheSource.moveCaretPosition (LineEnd-1);
      else
        TheSource.setCaretPosition (LineEnd-1);
    } catch (Exception ex) {
      System.out.println (ex);
    }
  }
}

/** This class is a JTextArea enchansement. It implements the (Shift)+PageUp
    functionality, allowing the user to select a large part od the document
    from the keyboard, or just to move the cursor a page up in the document.
*/
class PageUpAction extends AbstractAction {
  public void   actionPerformed (ActionEvent e) {
    try {
      JTextArea TheSource = (JTextArea) e.getSource();
      JScrollPane TextAreaScroll = (JScrollPane) TheSource.getParent().getParent();
      JScrollBar VScrollBar = TextAreaScroll.getVerticalScrollBar();
      if (VScrollBar == null)
        return;
      int CaretPos = TheSource.getCaretPosition();
      Rectangle VCaretPos = TheSource.modelToView (CaretPos);
      int Increment = VScrollBar.getBlockIncrement(0);
      Point NewLoc = new Point (VCaretPos.x, VCaretPos.y - Increment);
      if ((e.getModifiers() & Event.SHIFT_MASK) != 0)
        TheSource.moveCaretPosition (TheSource.viewToModel(NewLoc));
      else
        TheSource.setCaretPosition (TheSource.viewToModel(NewLoc));
    } catch (Exception ex) {
      System.out.println (ex);
    }
  }
}

/** This class is a JTextArea enchansement. It implements the (Shift)+PageDown
    functionality, allowing the user to select a large part od the document
    from the keyboard, or just to move the cursor a page down in the document.
*/
class PageDownAction extends AbstractAction {
  public void   actionPerformed (ActionEvent e) {
    try {
      JTextArea TheSource = (JTextArea) e.getSource();
      JScrollPane TextAreaScroll = (JScrollPane) TheSource.getParent().getParent();
      JScrollBar VScrollBar = TextAreaScroll.getVerticalScrollBar();
      if (VScrollBar == null)
        return;
      int CaretPos = TheSource.getCaretPosition();
      Rectangle VCaretPos = TheSource.modelToView (CaretPos);
      int Increment = VScrollBar.getBlockIncrement(0);
      Point NewLoc = new Point (VCaretPos.x, VCaretPos.y + Increment);
      if ((e.getModifiers() & Event.SHIFT_MASK) != 0)
        TheSource.moveCaretPosition (TheSource.viewToModel(NewLoc));
      else
        TheSource.setCaretPosition (TheSource.viewToModel(NewLoc));
    } catch (Exception ex) {
      System.out.println (ex);
    }
  }
}

/**
   @author	Peter Radkov <wreck@nat.bg>
   @version 0.7, 07-01-98
*/
/** This is the simple text editor, where the user edits (or just views) his/her source files.*/
public class SourceTextArea extends JPanel implements CaretListener, DocumentListener {
    private JTextArea textArea = null;
    private StatusPanel statusLine = null;
    private boolean modified = false;
    private JScrollPane TheScrollPane = new JScrollPane();
    private boolean DocListenerAdded = false;

    public SourceTextArea() {
      super();
      setLayout(new BorderLayout());
      textArea = new JTextArea();

      Keymap TheKeymap = textArea.getKeymap();
      HomeAction Home = new HomeAction();
      EndAction  End  = new EndAction ();
      PageUpAction PgUp = new PageUpAction();
      PageDownAction  PgDn  = new PageDownAction ();

      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0), Home);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_END, 0), End);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_HOME, Event.SHIFT_MASK), Home);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_END, Event.SHIFT_MASK), End);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, 0),  PgDn);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, Event.SHIFT_MASK),  PgUp);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, Event.SHIFT_MASK),  PgDn);
      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, 0),  PgUp);

      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, Event.SHIFT_MASK), new ShiftDownAction ());

      TheKeymap.addActionForKeyStroke (KeyStroke.getKeyStroke(KeyEvent.VK_UP, Event.SHIFT_MASK), new ShiftUpAction ());

      OptionsEventAdapter OptionsAdapter = null;
      try {
        OptionsAdapter = new OptionsEventAdapter (
		          EditorOptions.EditorOptionsName,  // Name of the options
			  OptionsEvent.msgOptionsRefresh,    // Options have been loaded
			  IdeEventListenerI.CALL_MASK,       // When you will be notified (there are 3 passes)
			  this,                              // Which object will handle the event
			  "OnLoadEditorOptions");            // Name of the function which will be called.
	
        Globals.EventsEngine.AddEventListener(OptionsAdapter);     // Register listener
	
        OptionsAdapter = new OptionsEventAdapter (
			  EditorOptions.EditorOptionsName,  // Name of the options
			  OptionsEvent.msgOptionsChanged,    // Options have been changed
			  IdeEventListenerI.CALL_MASK,       // When you will be notified (there are 3 passes)
			  this,                              // Which object will handle the event
			  "OnChangeEditorOptions");          // Name of the function which will be called.

        Globals.EventsEngine.AddEventListener(OptionsAdapter);
	
      } catch (NoSuchMethodException e) {
	Debug.Assert(e, this, "SomeEditorClass: No such Method.");
      }
      
      OptionsMngr AOptions = Globals.IEngine.getOptions();
      EditorOptions   EdOptions     = (EditorOptions) (AOptions.getOptions(EditorOptions.EditorOptionsName));
      
      textArea.setFont(new Font(EdOptions.getFontName(), EdOptions.getFontStyle(), EdOptions.getFontSize()));
      textArea.setBackground(EdOptions.getBGColor());
      textArea.setForeground(EdOptions.getFGColor());

      TheScrollPane.getViewport().setView(textArea);
      add(TheScrollPane, BorderLayout.CENTER);
      textArea.addCaretListener(this);
      statusLine = new StatusPanel();
      add(statusLine, BorderLayout.SOUTH);
      textArea.setCaretPosition(0);
    }

    public JTextArea getTextArea () {
      return textArea;
    }

    public void setText(String txt) {
      textArea.setText(txt);
      if (!DocListenerAdded)
        textArea.getDocument().addDocumentListener (this);
    }

    public String getText() {
      return textArea.getText();
    }

    public boolean isModified() {
      return modified;
    }

    public void setModified(boolean mod) {
      if (modified != mod) {
        modified = mod;
        statusLine.setMod(mod);
      }
    }

   public void changedUpdate(DocumentEvent e) {
     setModified (true);
   }
   public void insertUpdate(DocumentEvent e) {
     setModified (true);
   }
   public void removeUpdate(DocumentEvent e) {
     setModified (true);
   }

    public void caretUpdate (CaretEvent e) {
      try {
        int LineNo          = textArea.getLineOfOffset (textArea.getCaretPosition());
        int LineStartOffset = textArea.getLineStartOffset (LineNo);
        statusLine.setLine(LineNo);
        statusLine.setCol(textArea.getCaretPosition() - LineStartOffset);
      } catch (Exception ee) {
        System.out.println (ee);
      }
    }

    public void setStatusMessage(String msg) {
      statusLine.setMessage(msg);
    }

    public boolean isEditable() {
      return textArea.isEditable();
    }

    public void setEditable(boolean ed) {
      textArea.setEditable(ed);
      if (ed)
        setStatusMessage ("");
      else
        setStatusMessage ("Read Only");
    }

    public void requestFocus() {
      textArea.requestFocus();
    }

    public Dimension getMinimumSize () {
      return new Dimension (20, 20);
    }

    // This method will be called after Editor Options have been loaded.
    public void OnLoadEditorOptions(IdeSuperEvent AEvent) {
      OptionsEvent  Event    = (OptionsEvent)AEvent;
      EditorOptions FOptions = (EditorOptions)(Event.getOptions());
      
      if (textArea != null) {
	textArea.setFont(new Font(FOptions.getFontName(), FOptions.getFontStyle(), FOptions.getFontSize()));
	textArea.setBackground(FOptions.getBGColor());
	textArea.setForeground(FOptions.getFGColor());
      }
    }
  
    // This method will be called after Editor Options have been changed.
    public void OnChangeEditorOptions(IdeSuperEvent AEvent) {
      OptionsEvent  Event    = (OptionsEvent)AEvent;
      EditorOptions FOptions = (EditorOptions)(Event.getOptions());
      
      if (textArea != null) {
	textArea.setFont(new Font(FOptions.getFontName(), FOptions.getFontStyle(), FOptions.getFontSize()));
	textArea.setBackground(FOptions.getBGColor());
	textArea.setForeground(FOptions.getFGColor());
      }
    }

    /**
     *  Inner class managing status line info.
     */
    class StatusPanel extends BPanel {
      private String messageLbl = "Status: Ready";
      private String lineNoLbl  = "Line:";
      private String colNoLbl   = "Col:";
      private String modLbl     = "";
      private int StrHeight;

      StatusPanel() {
        super();
        StrHeight = Misc.calcStringHeight();
        Insets I = getInsets();
        setPreferredSize (new Dimension(20, StrHeight + I.top + I.bottom));
      }

      public void setLine(int lineno) {
        lineNoLbl  = "Line: "+lineno;
      }

      public void setCol(int colno) {
        colNoLbl = "Col: " + colno;
        repaint();
      }

      public void setMod(boolean mod) {
        if (mod) {
          modLbl = "Mod  ";
          Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiFileSave, MainMenuEvent.msubcRefresh, this));
        } else {
          modLbl = "";
          Globals.EventsEngine.PostEvent (new MainMenuEvent (IdeMainMenu.msgmiFileSave, MainMenuEvent.msubcRefresh, this));
        }
        repaint();
      }

      public void setMessage(String msg) {
        messageLbl = msg;
        repaint();
      }

      public void paint (Graphics g) {
        super.paint(g);
        String Gap = "  ";
        Insets I = getInsets();
        g.drawString (messageLbl + Gap + lineNoLbl + Gap + colNoLbl + Gap + modLbl, 20, StrHeight + I.top - 2);
      }
    } 
}


