/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.classes.events.engine;

import java.awt.EventQueue;
import java.awt.peer.*; // Sorry !!! in version 1.1 ActiveEvent was in packet java.awt.peer
import java.awt.*;      // Sorry !!! in version 1.2 ActiveEvent was in packet java.awt


/** The class <code>IdeEventsEngine</code> is the one which manage message dispatching to all the listeners
  * Events are grouped by their kinds. Kind of event is it's Class ( SomeEvent.getClass() )
  * Every listener handle some Kind of events.
  *   if the listener wants to handle also all deriveted from this class events,
  *   SomeListener.isChildEventsListener() have to return true.
  * see IdeEventListenerI
  * Events can be Posed (to the queue) function exits before dispatching it
  * Sent function exits after dispatching it
  * Checked if there is any listener for some event IsListenerFor()
  * Enabled or Disabled 
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 02-12-97 */

public class IdeEventsEngine {
  final public static int CALL_BEFORE_METHOD = -1;
  final public static int CALL_DO_METHOD     = 0;
  final public static int CALL_AFTER_METHOD  = 1;

  final public static int EVENT_PASS1 = 1;
  final public static int EVENT_PASS2 = 2;
  final public static int EVENT_PASS3 = 3;

  private boolean IsEnabled = false;

  private IdeEvensListColl BeforeList;  // All the listeners which will call BeforeAction
  private IdeEvensListColl DoList;      // All the listeners which will call DoAction
  private IdeEvensListColl AfterList;   // All the listeners which will call AfterAction
  private EventQueue       Queue;       // This Queue will handle all the messages

  public IdeEventsEngine() {
    IsEnabled  = false;

    BeforeList = new IdeEvensListColl(CALL_BEFORE_METHOD);
    DoList     = new IdeEvensListColl(CALL_DO_METHOD);
    AfterList  = new IdeEvensListColl(CALL_AFTER_METHOD);

    Queue      = new EventQueue();
  }

  /** Events are dispatched in 3 passes. Preview pass, Action pass, After Action pass.
      IdeEventsEngine automatically set it before deliver the event to the listeners. */
  private void SetEventPass(IdeSuperEvent Event, int Pass) {
    if (Event == null) return;
    Event.setEventPass(Pass);
  }

  /** Deliver the event to the listeners if all the following rules are true.
      1. (if event != null),
      2. JustCheck = false,
      3. events are enabled, and individual listenre's. AcceptEvent() returns true

      if JustCheck == false, we are cheking if there are listener(s) handling the event.
  */
  void Dispatch(IdeSuperEvent Event, boolean JustCheck) {
    if (! IsEnabled) return;    // the new one.

    if (! JustCheck)            // if have to deliver a event
      if (! IsEnabled) return;  // and not enabled then return.

    Class AEventClass = null;
    if (Event != null) AEventClass = Event.getClass();

    SetEventPass(Event,EVENT_PASS1);
    BeforeList.DispatchEvent(AEventClass, Event, JustCheck);

    SetEventPass(Event,EVENT_PASS2);
    DoList    .DispatchEvent(AEventClass, Event, JustCheck);

    SetEventPass(Event,EVENT_PASS3);
    AfterList .DispatchEvent(AEventClass, Event, JustCheck);
  }

  /** Is Events enabled */
  public boolean IsEnabled() { return IsEnabled; }

  /** Add This Listener. */
  public synchronized void AddEventListener(IdeEventListenerI EventListener) {
    if (EventListener == null) return;

    if (EventListener.ShallCallBefore()) BeforeList.AddEventListener(EventListener);
    if (EventListener.ShallCall      ()) DoList    .AddEventListener(EventListener);
    if (EventListener.ShallCallAfter ()) AfterList .AddEventListener(EventListener);
  }

  /** Remove this listener from everywhere if found
     (before removing disables the events for this listener)*/
  public synchronized void RemoveEventListener(IdeEventListenerI EventListener) {
    if (EventListener == null) return;

    BeforeList.RemoveEventListener(EventListener);
    DoList    .RemoveEventListener(EventListener);
    AfterList .RemoveEventListener(EventListener);
  }

  /** Enables / Disables dispatching the events */
  public synchronized void EnableEvents(boolean AEnable) {
    if (IsEnabled == AEnable) return;
    IsEnabled = AEnable;
  }

  /** event is posted to the queue, and execution of the function terminates */
  public synchronized void PostEvent(IdeSuperEvent Event) {
    if (! IsEnabled) return;

    IdeEngineEvent EEvent = new IdeEngineEvent(this, Event);
    Queue.postEvent(EEvent);
  }

  /** sends this event to all the listeners wich are waiting for it.
      function execution terminates after dispatching of the event terminates. */
  public synchronized void SendEvent(IdeSuperEvent Event) {
    Dispatch(Event, false);
  }

  /** return true, if there is(are) listener(s) wich handles this events and they are enabled */
  public synchronized boolean IsListenerFor(IdeSuperEvent Event) {
    if (Event == null) return false;
    Dispatch(Event, true);
    return Event.isHandled();
  }
}
