/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.system.projectman;

import org.freebuilder.system.debug.*;
import org.freebuilder.gui.ideoptions.*;
import org.freebuilder.system.classes.*;
import org.freebuilder.system.packages.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.classes.events.*;
import org.freebuilder.system.classes.options.*;
import org.freebuilder.system.classes.events.engine.*;

/** Class <code>ProjectItems</code> is a collection of all the Project items. Every time you
  * add some item to the project (java source file, packet, root packet directory .. ) and instance
  * of ProjectItem class is created and added to and object of class ProjectItem.
  * So an object of this class will hold your project.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 12/12/97
*/


class ProjectItem {
  PacketItemI PacketItemRef;
  int         PacketItemKind;

  public ProjectItem (PacketItemI APacketItemRef, int APacketItemKind) {
    PacketItemRef  = APacketItemRef;
    PacketItemKind = APacketItemKind;
  }
}

public class ProjectItems extends StringList implements StringsListener {
  public ProjectItems () {
    OptionsEventAdapter OptionsCAdapter = null;
    OptionsEventAdapter OptionsRAdapter = null;
    OptionsEventAdapter PathsRAdapter   = null;
    OptionsEventAdapter PathsCAdapter   = null;
    try {
      OptionsRAdapter = new OptionsEventAdapter(ProjectOptions.cDefaultName, OptionsEvent.msgOptionsRefresh,
                                             IdeEventListenerI.AFTER_CALL_MASK, this, "update");
      OptionsCAdapter = new OptionsEventAdapter(ProjectOptions.cDefaultName, OptionsEvent.msgOptionsChanged,
                                             IdeEventListenerI.AFTER_CALL_MASK, this, "update");

      PathsRAdapter   = new OptionsEventAdapter(PathsOptions.PathsOptionsName, OptionsEvent.msgOptionsRefresh,
                                             IdeEventListenerI.AFTER_CALL_MASK, this, "OnChangePaths");
      PathsCAdapter   = new OptionsEventAdapter(PathsOptions.PathsOptionsName, OptionsEvent.msgOptionsChanged,
                                             IdeEventListenerI.AFTER_CALL_MASK, this, "OnChangePaths");

    } catch (NoSuchMethodException e) {
      Debug.Assert (e, this, "ProjectItems : No such Method.");
    }
    Globals.EventsEngine.AddEventListener(OptionsRAdapter);
    Globals.EventsEngine.AddEventListener(OptionsCAdapter);
    Globals.EventsEngine.AddEventListener(PathsRAdapter);
    Globals.EventsEngine.AddEventListener(PathsCAdapter);
    AddChangeListener(this);
  }

  private void SendMsg(int MsgNumber) {
    Globals.EventsEngine.PostEvent (new SystemEvent (MsgNumber, this, ""));
  }

  // Begin of the methods from StringsListener interface
  public void BeforeChange(Strings Sender) {}
  public void AfterChange (Strings Sender) {
    System.gc();
    SendMsg(SystemEvent.msgPItemsChanged);
  }
  // End   of the methods from StringsListener interface


  public void rebuild (IdeSuperEvent AEvent) {
    OnRefreshProjectItems();
  }

  public void update (IdeSuperEvent AEvent) {
    OnChangeProjectItems();
  }

  private void DoOnChangePaths() {
    String      PrOptsName;
    ProjectItem PrItem = null;
    int         Cntr, To = getCount();
    int         PrItemKind;
    for (Cntr = 0; Cntr < To; Cntr++) {
      PrItem  = getProjectItem(Cntr); Debug.Assert(PrItem != null, this, "PrItem == null");
      PrItemKind = getPacketItemKind (Cntr);
      PrOptsName = getString(Cntr);
      if ((PrItemKind == BrowseCollItem.ikPackage) || (PrItemKind == BrowseCollItem.ikPackagesRoot))
        PrItem.PacketItemRef = Globals.RootPaths.getPacketItemByDotName (PrOptsName);
    }
    SendMsg(SystemEvent.msgPItemsRefresh);
  }

  public void OnChangePaths(IdeSuperEvent AEvent) {
    DoOnChangePaths();
  }

  private void OnRefreshProjectItems() {
    BeginUpdate();
    try {
      Clear();
      OnChangeProjectItems();
    }
    finally {
      EndUpdate();
    }
  }

  private void AddPacketItemI(PacketItemI NewPacketItem, String AName, int PrOptItemKind) {
    ProjectItem NewItem = new ProjectItem (NewPacketItem, PrOptItemKind);
    Add (AName, NewItem);
  }

  private void AddNewItem(BrowseCollItem APrOptsItem, String APrOptsName) {
    PacketItemI NewPacketItem;
    int         PrItemKind;

    if (APrOptsItem == null) return;
    PrItemKind = APrOptsItem.getItemKind();

    if ((PrItemKind == BrowseCollItem.ikPackage) ||(PrItemKind == BrowseCollItem.ikPackagesRoot)) {
      NewPacketItem = Globals.RootPaths.getPacketItemByDotName (APrOptsName);
      AddPacketItemI(NewPacketItem, APrOptsName, PrItemKind);
      return;
    }

    if ((PrItemKind == BrowseCollItem.ikDirectory) ||(PrItemKind == BrowseCollItem.ikDirectoryRoot)) {
      NewPacketItem = PCreateEntry.CreateDirEntry(APrOptsName);
      AddPacketItemI(NewPacketItem, APrOptsName, PrItemKind);
      return;
    }

    NewPacketItem = Globals.RootPaths.findPacketItemI (APrOptsName);
    if (NewPacketItem == null)
      NewPacketItem = PCreateEntry.CreateEntry (APrOptsName);
    AddPacketItemI(NewPacketItem, APrOptsName, PrItemKind);
  }

  private void AddNewItems(ProjectOptions POptions) {
    BrowseCollItem CrntPrOptsItem  = null;
    String         CrntPrOptsName;
    if (POptions == null) return;
    int Cntr, To = POptions.getCount(); if (To == 0) return;

    for (Cntr = 0; Cntr < To; Cntr++) {
      try {
        CrntPrOptsItem = POptions.getBrowseItem(Cntr);
        CrntPrOptsName = POptions.getString(Cntr);
        AddNewItem(CrntPrOptsItem, CrntPrOptsName);
      }
      catch(ArrayIndexOutOfBoundsException e) {}
    }
  }

  public void OnChangeProjectItems() {
    ProjectOptions POptions = (ProjectOptions) Globals.Options.getOptions (ProjectOptions.cDefaultName);
    if (POptions == null) return;

    ProjectItem    CrntProjectItem = null;
    String         CrntProjectName;
    BrowseCollItem CrntPrOptsItem  = null;

    int Cntr             = 0;
    int PrOptIndex       = 0;

    BeginUpdate();
    try {
      while (Cntr < getCount()){
        CrntProjectItem = getProjectItem(Cntr);
        CrntProjectName = getString(Cntr);
        if ((CrntProjectItem != null) && (CrntProjectName != null)) {
          PrOptIndex = POptions.isItem (getPacketItemKind(Cntr), CrntProjectName);

          try {
            if (PrOptIndex >= 0) {
              CrntPrOptsItem = POptions.getBrowseItem(PrOptIndex);
              if (CrntPrOptsItem != null)  // if this item already exist here, just delete it from new items (POptions)
                POptions.Delete(PrOptIndex);
            }
            else { // this item have been deleted from new Options, so delete it here
              Delete(Cntr); continue;
            }
          }
          catch(ArrayIndexOutOfBoundsException e) {}
        }
        Cntr++;
      }
      AddNewItems(POptions);
    }
    finally {
      EndUpdate();
    }
  }

  public ProjectItem getProjectItem(int AIndex) {
    ProjectItem Result = (ProjectItem)(getObject(AIndex));
    Debug.Assert(Result != null, this, "getProjectItem() == null");
    return Result;
  }

  public PacketItemI getPacketItem (int AIndex) {
    ProjectItem PrItem = getProjectItem(AIndex); if (PrItem == null) return null;
    PacketItemI Result = PrItem.PacketItemRef;
    return Result;
  }

  public int getPacketItemKind (int AIndex) {
    ProjectItem PrItem = getProjectItem(AIndex); if (PrItem == null) return -1;
    int         Result = PrItem.PacketItemKind;
    return Result;
  }
}
