/*
 * Copyright (c) 1998 Vanand Ltd.
 * <vanand@mail.techno-link.com>, <vanand@iname.com>, <a-hristov@usa.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * If you modify this file, please send us a copy.
 */

package org.freebuilder.gui.idemenu;

import com.sun.java.swing.*;
import com.sun.java.swing.event.*;

import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import org.freebuilder.system.classes.events.engine.IdeEventsEngine;
import org.freebuilder.system.classes.events.engine.*;
import org.freebuilder.system.ideengine.*;
import org.freebuilder.system.debug.Debug;

/** The <code>IdeMenuItem</code> class is Ide version of MenuItem class.
  * Every IdeMenuItem is constructed via EventsEngine & CmndNumber = number of events which
  * this menu item will post.
  * It is listener for MainMenuEvent class events.
  * When receive MainMenuEvent.mmsgWhoAreYou sets MainMenuEvent.SenderMenuItem to this.
  * In this way you can query all the menu items from anywhere.
  * When the received sub command msubcTryToSendDoIt, if it isEnabled
  * the MenuItem sends event with number FCmndNumber and sub command msubcDoIt.
  * In such a way disable/enable of MenuItem will enable/disable a command corresponding to it.
  * With setSpeedButton() method a SpeedButton can be attached/deattached to the MenuItem.
  * With setCheckStatusListener() can set a object which is responsible for discovering if
  * that menu item have to be disabled, or enabled. The class implements CheckEnDisI, so
  * it is possible to say SomeMenuItem.setCheckStatusListener(SomeMenuItem); In this way
  * the MenuItem is enabled if EventsEngine.IsListenerFor(FCmndNumber) returns true.
  *
  * @author	George Petkov <a href=mailto:pgeorge@mail.techno-link.com> pgeorge@mail.techno-link.com </a>,  <a href=mailto:gpetkov@usa.net> gpetkov@usa.net</a>
  * @version 0.7, 07-01-98 */


public class IdeMenuItem extends JMenuItem
  implements EnDisableI, ActionListener, IdeEventListenerI, CheckEnDisI {

  private int             FCmndNumber;   // Event number for this menu item
  private CheckEnDisI     CheckListener;
  private EnDisableI      FSpeedButton;  // Speed button attached to this menu

  Class   EventsClass;
  boolean IsIdeMenuEventsEnabled;

  int     CalcStatusCntr  = 0;
  int     SendDoItCntr    = 0;

  public IdeMenuItem(int CmndNumber) {
    Init( CmndNumber);
  }

  public IdeMenuItem(int CmndNumber, String label) {
    super(label);
    Init(CmndNumber);
  }

  public IdeMenuItem(int CmndNumber, String label, MenuShortcut s) {
    super(label);
    Init(CmndNumber);
  }

  private void Init(int CmndNumber) {
    MainMenuEvent  AEvent;
    FCmndNumber  = CmndNumber;

    AEvent       = new MainMenuEvent(CmndNumber, this);
    EventsClass  = AEvent.getClass();
  }

  protected boolean CalcStatus() {
    try {
      synchronized(this) {
        if (CalcStatusCntr != 0) {
          Debug.Assert(this, "IdeMenuItem.CalcStatus() have been recursivelly called from the same thread !!!!!!! " + this);
          return false;
        }
        CalcStatusCntr++;
      }

      if (CheckListener != null)
        return CheckListener.HaveToEnableMenuItem(this);
      return isEnabled();
    }
    finally {
      CalcStatusCntr = 0;
    }
  }

  /** Every time IdemMenuItem change status from Disabled to Enabled, this method is called */
  protected void Enabled() {
  }

  /** Every time IdemMenuItem change status from Enabled to Disabled, this method is called */
  protected void Disabled() {
  }

  private void SendDoItCmnd(MainMenuEvent Event) {
    try {
      synchronized(this) {
        if (SendDoItCntr != 0) {
          Debug.Assert(this, "IdeMenuItem.SendDoItCmnd() have been recursivelly called from the same thread !!!!!!! " + this);
          return;
        }
        SendDoItCntr++;
      }

      if (! isEnabled()) return;
      Event.MenuSubCommand = MainMenuEvent.msubcDoIt;
      Globals.EventsEngine.SendEvent(Event);
    }
    finally {
      SendDoItCntr = 0;
    }
  }

  protected void PostCmnd() {
    if (! isEnabled()) return;
    Globals.EventsEngine.PostEvent(new MainMenuEvent(FCmndNumber, MainMenuEvent.msubcDoIt, this, this));
  }

  // From EnDisableI interface ----------------------------------------------------
  public void    setEnabled(boolean Enable) {
    super.setEnabled(Enable);
    if (FSpeedButton != null)
      FSpeedButton.setEnabled(Enable);
  }

  public boolean isEnabled() {
    return super.isEnabled();
  }

  public boolean RefreshIsEnabled() {
    boolean newStatus, oldStatus;

    oldStatus = isEnabled();
    newStatus = CalcStatus();
    if (oldStatus == newStatus) return oldStatus;

    setEnabled(newStatus);
    if (newStatus) Enabled ();
    else           Disabled();
    return newStatus;
  }

  public void    RemoveFromMenu() {
    Globals.EventsEngine.RemoveEventListener(this);
    removeActionListener(this);

    if (FSpeedButton != null)
      FSpeedButton.RemoveFromMenu();

    setIdeEventLEnable(false);
  }

  public void    AddToMenu() {
    addActionListener(this);
    Globals.EventsEngine.AddEventListener(this);
    if (FSpeedButton != null)
      FSpeedButton.AddToMenu();
    setIdeEventLEnable(true);
  }

  public int     getCmndNumber() {
    return FCmndNumber;
  }
  //-------------------------------------------------------------------------------

  // From ActionListener interface ------------------------------------------------
  public void actionPerformed(ActionEvent e) {
    PostCmnd();
  }
  //-------------------------------------------------------------------------------

  // From IdeEventListenerI interface ---------------------------------------------
  public boolean ShallCallBefore()       { return false; }
  public boolean ShallCall()             { return true;  }
  public boolean ShallCallAfter()        { return false; }
  public boolean isChildEventsListener() { return false; }

  public void    BeforeAction(IdeSuperEvent AEvent) {}
  public void    AfterAction (IdeSuperEvent AEvent) {}

  public Class getEventsClass() {
    return EventsClass;
  }

  public void setIdeEventLEnable(boolean AEnable) {
    if (AEnable == IsIdeMenuEventsEnabled) return;
    IsIdeMenuEventsEnabled = AEnable;
  }

  public boolean isIdeEventLEnabled() {
    return IsIdeMenuEventsEnabled;
  }

  public boolean AcceptEvent (IdeSuperEvent AEvent) {
    int EventNumber;
    int SubCommand;
    if (AEvent == null) return false;

    MainMenuEvent Event = (MainMenuEvent) AEvent;
    EventNumber = Event.EventID;
    SubCommand  = Event.MenuSubCommand;

    if (SubCommand == MainMenuEvent.msubcWhoAreYou)
      return true;

    if (SubCommand == MainMenuEvent.msubcRefreshForAll)
      return true;

    if (EventNumber == FCmndNumber)
      if (SubCommand == MainMenuEvent.msubcDoIt)
        return false;
      else
        return true;
    return false;
  }

  public void OnAction    (IdeSuperEvent AEvent) {
    int EventNumber;
    int SubCommand;

    if (AEvent == null) return;

    MainMenuEvent Event = (MainMenuEvent)AEvent;
    EventNumber = Event.EventID;
    SubCommand  = Event.MenuSubCommand;

    if (EventNumber == FCmndNumber) {
      if (SubCommand == MainMenuEvent.msubcTryToSendDoIt)
      { SendDoItCmnd(Event); return; }

      if (SubCommand == MainMenuEvent.msubcRefresh)
      { setEnabled (CalcStatus()); return; }
    }

    if (SubCommand == MainMenuEvent.msubcWhoAreYou)
    { Event.SenderMenuItem = this; return; }

    if (SubCommand == MainMenuEvent.msubcRefreshForAll)
    { setEnabled (CalcStatus()); return; }
  }
  //-------------------------------------------------------------------------------


  // From CheckEnDisI interface ---------------------------------------------
  public boolean HaveToEnableMenuItem(IdeMenuItem menuItem) {
    if (menuItem == null) return false;
    MainMenuEvent MEvent = new MainMenuEvent(menuItem.getCmndNumber(), menuItem);
    return Globals.EventsEngine.IsListenerFor(MEvent);
  }
  //-------------------------------------------------------------------------------

  public void setCheckStatusListener(CheckEnDisI ACheckListener) {
    CheckListener = ACheckListener;
  }

  public void    setSpeedButton(EnDisableI ASpeedButton) {
    FSpeedButton = ASpeedButton;
  }

}

