{$G+,F+,O+}

{$DEFINE Copy}
{$DEFINE Mode256}								{ Possible modes: Mode256,Mode64,Mode16 }

{ TABSize = 3 !!!! }

UNIT Mouse;

{ Written by:
									wct@po.CWRU.Edu (William C. Thompson)
  ReWritten for SVGA by :
									Mark Stehr (mkstehr@cip.informatik.uni-erlangen.de )
									Alpo Vrri (varri@cs.tut.fi)
}

INTERFACE

CONST
	LeftButton 		= 1;
  	RightButton 	= 2;
  	MiddleButton 	= 4;
  	AnyButton 		= 7;

   TEXT				= TRUE;
   GRAPHICS			= FALSE;

FUNCTION MouseDriverInstalled : BOOLEAN;

PROCEDURE InitDriver(VAR Buttons : WORD; Modus : BOOLEAN; VAR Err : BOOLEAN);
{ Initializes the mouse driver.
  Call:    Mode = Text or graphics mode
  Returns: Buttons = number of mouse buttons
  Note:    * After a call to this function the driver is initialized to the
	     following state:
	     - Mouse pointer at screen center and hidden.
	     - Display page is set to zero.
	     - Mouse pointer shape set to default arrow shape in graphics modes,
	       or reverse block in text modes.
	     - User mouse event handlers are disabled.
	     - Light pen emulation enabled.
	     - Mouse sensitivity set to default vales (see SetMouseSense)
	     - Pointer limits set to entire screen.
}
PROCEDURE ShowPointer;
{ Displays the mouse pointer, and cancels any pointer exclusion area.
  Call:    nothing
  Returns: nothing
  Note:    * A counter is maintained which is decremented by HidePointer and is
	     incremented (if nonzero) by this function.  The mouse pointer is
	     displayed any time the counter is zero.  The counter is set to -1
	     when the mouse driver is reset.
}
PROCEDURE HidePointer;
{ Removes the mouse pointer from the screen, but continues to track the
  position of the mouse.
  Call:    nothing
  Returns: nothing
  Note:    * see ShowPointer
}
FUNCTION ButtonPressed: BYTE;
{ Returns a non zero value if a button is pressed.
  Call:    Mask = bit mask of desired button(s)
		  bit(s)  Significance(if set)
		  0       left button
		  1       right button
		  2       center button
		  3-15    reserved(0)
  Returns: True is button is pressed, false otherwise.
  Note:    * The constants LeftButton, RightButton, CenterButton, and
	     AnyButton can be used for the bit masking.  They equal 1, 2,
	     4, and 7 respectivly.
}
PROCEDURE GetMousePosition(VAR Horiz, Vert : Integer);
{ Returns the current mouse button status and pointer position.
  Call:    nothing
  Returns: Buttons = mouse button status
	   Horiz   = horizontal (X) coordinate
	   Vert    = vertical (Y) coordinate
  Note:    * Coordinates are in pixels regardless of the current display mode.
	     Position (0,0) is the upper left corner of the screen.
}

FUNCTION MouseIn(x1,y1,x2,y2: INTEGER):BOOLEAN;
{ Returns true if mouse is within rectangle with upper-left
  corner (x1,y1) and lower-right corner (x2,y2).
}

PROCEDURE SetMickeyPixelRatio(Horiz,Vert : WORD);
{ Defines the mickey/pixel ratio.
  Use smaller values to speed up and bigger to slow down.
  Call:	Horiz = number of mickeys per 8 pixels horizontally (default 8)
			Vert	= number of mickeys per 8 pixels vertically (default 16)
  Returns:	-
}

PROCEDURE SetPointerPosition(Horiz, Vert : INTEGER);
{ Set the position of the pointer.  The pointer is displayed in the new
  position unless it has been hidden using HidePointer or it is an exclusion
  area defined by SetPointerExcl.
  Call:    Horiz = horizontal (X) coordinate
	   Vert  = vertical (Y) coordinate
  Returns: nothing
  Notes:   * Coordinates are in pixels regardless of the current display mode.
	     Position (0,0) is the upper left corner of the screen.
	   * The position is adjusted if necessary to lie within the pointer
	     limits set by SetLimits.
}
PROCEDURE GetPressInfo(Button : WORD;
		       VAR Stat, Count : WORD; VAR Horiz, Vert : INTEGER);
{ Returns the current status of all mouse buttons, and the number of presses
  and position of the last press for a specifed mouse button since the last
  call to this procedure for that button.  The press counter for the button
  is reset to zero.
  Call:    Button = button identifier
		    0 = left button
		    1 = right button
		    2 = center button
  Returns: Stat   = button status
		    bit(s)  Significance(if set)
		    0       left button is down
		    1       right button is down
		    2       center button is down
		    3-15    reserved(0)
	   Count  = button press counter
	   Horiz  = horizontal (X) coordinate of last button press
	   Vert   = vertical (Y) coordinate of last button press
}
PROCEDURE GetReleaseInfo(Button : WORD;
			 VAR Stat, Count : WORD; VAR Horiz, Vert : INTEGER);
{ Returns the current status of all mouse buttons, and the number of releases
  and position of the last release for a specifed mouse button since the last
  call to this procedure for that button.  The release counter for the button
  is reset to zero.
  Call:    Button = button identifier
		    0 = left button
		    1 = right button
		    2 = center button
  Returns: Stat   = button status
		    bit(s)  Significance(if set)
		    0       left button is down
		    1       right button is down
		    2       center button is down
		    3-15    reserved(0)
	   Count  = button release counter
	   Horiz  = horizontal (X) coordinate of last button release
	   Vert   = vertical (Y) coordinate of last button release
}
PROCEDURE SetLimits(HorMin, VerMin, HorMax, VerMax : INTEGER);
{ Limits the mouse pointer to stay within a certian area.
  Call:    HorMin = Minimum horizontal (X) coordinate
	   HorMax = Maximum horizontal (X) coordinate
	   VerMin = Minimum vertical (Y) coordinate
	   VerMax = Maximum vertical (Y) coordinate
  Returns: nothing
  Note:    * If both HorMin and HorMax are zero then then the previous
	     horizontal limits remain unchanged; the same is true for
	     VerMin and VerMax.
}
PROCEDURE SetPointerShape(Horiz, Vert : INTEGER; Buffer : pointer);
{ Defines the shape, color, and hot spot of the pointer in graphics modes.
  Call:    Horiz  = hot spot offset from the left
			  Vert   = hot spot offset from the top
			  Buffer = pointer to mouse pointer image buffer
  Returns: nothing
  Note:
		* The pointer image buffer is 64 bytes long.  The first 32 bytes
		  contain a bit mask which is ANDed with the screen image, and the
		  remaining 32 bytes are then XORed with the screen image.
		* The hot spot is relative to the upper left corner of the pointer
		  image, and each offset must be in the range -16 to 16.  In display
		  modes 4 and 5, the horizontal offset must be an even number.
}

PROCEDURE SetTextPointer(PtrTyp, AND_Str, XOR_End : WORD);
{ Defines the shape and attributes of the mouse pointer in text modes.
  Call:    PtrTyp  = pointer type
		     0 = software cursor
		     1 = hardware cursor
	   AND_Str = AND mask value (if PtrTyp = 0) or starting line for
		     cursor (if PtrTyp = 1)
	   XOR_End = XOR mask value (if PtrTyp = 0) or ending line for
		     cursor (if PtrTyp = 1)
  Returns: nothing
  Notes:   * If the software text cursor is selected, the masks in AND_Str and
	     XOR_End are mapped as follows:
	     Bit(s)   Significance
	     0-7      character code
	     8-10     foreground color
	     11       intensity
	     12-14    background color
	     15       blink
	     For Example, the following call would yeild a software cursor
	     that inverts the foreground and background colors:
	     SetTextPointer(0, $77FF, $7700);
	   * When the hardware text cursor is selected, the values in AND_Str
	     and XOR_End are the starting and ending lines for the blinking
	     cursor generated by the video adapter.  The maximum scan line
	     depends on the type of adapter and the current display mode.
}
PROCEDURE GetMotionCount(VAR Horiz, Vert : Integer);
{ Returns the net mouse displacement since the last call to this procedure.
  The returned value is in MICKEYS; a positive number indicates travel to the
  right or downwards, a negative number indicates travel to the left or
  upwards.  One mickey represents approximately 1/200 of an inch of mouse
  movement.
  Call:    nothing
  Returns: Horiz = horizontal (X) mickey count
	   Vert  = vertical (Y) mickey count
}
PROCEDURE SetEventHandler(EventMask : WORD; Handler : pointer);
{ Sets the address and event mask for an application program's mouse event
  handler.  The handler is called by the mouse drvier whenever the specifed
  mouse events occur.
  Call:    EventMask = event mask
		       Bit(s)  Significance(if set)
		       0       mouse movement
		       1       left button pressed
		       2       left button released
		       3       right button pressed
		       4       right button released
		       5       center button pressed
		       6       center button released
		       7-15    reserved(0)
				Handler   = Pointer to the handler procedure
  Returns: 	nothing
  Notes:
		* The user-defined handler is entered from the mouse driver by a
		  far call with the registers set up as follows:
		  AX       mouse event flags (see event mask)
		  BX       button state
				Bit(s)  Significance(if set)
				0       left button is down
				1       right button is down
				2       center button is down
				3-15    reserved(0)
		  CX       horizontal (X) pointer coordinate
		  DX       vertical (Y) pointer coordinate
		  SI       last raw vertical mickey count
		  DI       last raw horizontal mickey count
		  DS       mouse driver data segment
		* If an event does not generate a call to the user-defined handler
		  because its bit is not set in the event mask, it is still reported
		  in the event falgs during calls to the handler for events which
		  are enabled.
}

PROCEDURE SetLightPen(On_Off : WORD);
{ Turns the light pen emulation by the mouse driver for IBM BASIC on or off.
  A "pen down" condition is created by pressing the left and right mouse
  buttons simultaneosly.
  Call:    On_Off = true to enable or false to disable emulation.
  Returns: nothing
}
PROCEDURE SetPointerExcl(HorMin, VerMin, HorMax, VerMax : WORD);
{ Defines an exclusion area for the mouse pointer.  When the mouse pointer
  lies within the specified area, it is not displayed.
  Call:    HorMin = upper left X coordinate
	   VerMin = upper left Y coordinate
	   HorMax = lower right X coordinate
	   VerMax = lower right Y coordinate
  Returns: nothing
  Note:    *  The exclusion area is replaced by another call to this
	      procdure or cancelled by InitMouse and ShowPointer.
}
PROCEDURE SwapEventHandlers(VAR Mask : WORD; VAR Buffer : pointer);
{ Set the address and event mask for an application program's mouse event
  handler and returns the address and event mask for the previous handler.
  The newly installed handler is called by the mouse driver whenever the
  specified mouse events occur.
  Call:    Mask    = event mask
		     Bit(s)  Significance(if set)
		     0       mouse movement
		     1       left button pressed
		     2       left button released
		     3       right button pressed
		     4       right button released
		     5       center button pressed
		     6       center button released
		     7-15    reserved(0)
				Handler = Pointer to the handler procedure
  Returns: 	Mask    = previous event mask
				Handler = pointer to previous handler
  Notes:
		* The notes for SetEventHandler describe the information passed to
		  the user-defined event handler.  Also see SetAltEventHandler.
		* Calls to the event handler are disabled with InitMouse or by
		  setting an event mask of zero.
}
FUNCTION GetSaveStateSize : WORD;
{ Returns the size of the buffer required to store the current state of the
  mouse driver.
  Note:    * also see SaveDriverState and RestoreDriverState.
}
PROCEDURE SaveDriverState(Buffer : pointer; Size : WORD);
{ Saves the mouse driver state in a user buffer.  THe minimum size for the
  buffer must be determined by GetSaveStateSize.
  Call:    Buffer = pointer to the user defined buffer.
  Returns: nothing
  Note:    * Use this procedure before executing a child program (Exec), in
	     case the child aslo uses the mouse. After the Exec call, restore
	     the previous mouse driver state using RestoreDriverState.
}
PROCEDURE RestoreDriverState(Buffer : pointer; Size : WORD);
{ Restores the mouse driver state from a user buffer.
  Call:    Buffer = pointer to the user defined buffer.
  Returns: nothing
  Note:    * The mouse driver state must have been previously saved into the
	     same buffer with SaveDriverState.  The format of the data in the
	     buffer in undocumented and subject to change.
}
PROCEDURE SetAltEventHandler(Mask : WORD; Handler : pointer; VAR Err: BOOLEAN);
{ Sets the address and event mask for an application program's mouse event
  handler.  As many as three handlers with distinct event masks can be
  registered with this function.  When an event occurs that matches one of the
  masks, the corresponding handler is called by the mouse driver.
  Call:    Mask    = event mask
		     Bit(s)  Significance(if set)
		     0       mouse movement
		     1       left button pressed
		     2       left button released
		     3       right button pressed
		     4       right button released
		     5       Shift key pressed during button press or release
		     6       Ctrl key pressed during button press or release
		     7       Alt key pressed during button press or release
		     8-15    reserved(0)
	   Handler = Pointer to the handler procedure
  Returns: Err     = false if successful, true otherwise
  Notes:   * When this procedure is called, at least one of the bits 5, 6, and
	     7 must be set in Mask.
	   * The user-defined handler is entered from the mouse driver by a
	     far call with the registers set up as follows:
	     AX       mouse event flags (see event mask)
	     BX       button state
		      Bit(s)  Significance(if set)
		      0       left button is down
		      1       right button is down
		      2       center button is down
		      3-15    reserved(0)
	     CX       horizontal (X) pointer coordinate
	     DX       vertical (Y) pointer coordinate
	     SI       last raw vertical mickey count
	     DI       last raw horizontal mickey count
	     DS       mouse driver data segment
	   * If an event does not generate a call to the user-defined handler
	     because its bit is not set in the event mask, it is still reported
	     in the event falgs during calls to the handler for events which
	     are enabled.
	   * Calls to the handler are disabled with InitMouse.
	   * Also see SetEventHandler and SwapEventHandlers.
}
PROCEDURE GetAltEventAdrs(VAR Mask : WORD; VAR Handler : pointer;
			  VAR Err : BOOLEAN);
{ Returns the address for the mouse event handler matching the specified
  event mask.
  Call:    Mask    = event mask
		     (see SetAltEventHandler)
  Returns: Mask    = event mask
	   Handler = pointer to the alternate event handler
	   Err     = false if successful, true if not successful (no handler
		     installed or event mask does not match any installed
		     handler.
  Note:    * SetAltEventHandler allows as many as three event handler with
	     distinct event masks to be installed.  This procedure can be
	     called to search for a handler that matches a specific event, so
	     that it can be replaced or disabled.
}
PROCEDURE SetMouseSense(Horiz, Vert, Double : WORD);
{ Set the number of mickeys per 8 pixels for horizontal and vertical mouse
  motion and the threshold speed for doubleing pointer motion on the screen.
  One mickey represents approximately 1/200 of an inch of mouse travel.
  Call:    	Horiz  = horizontal mickeys (1-32,767; default=8)
				Vert   = vertical mickeys (1-32,767; default=16)
				Double = double speed threshold in mickeys/second (default=64);
  Returns: 	nothing
}
PROCEDURE GetMouseSense(VAR Horiz, Vert, Double : WORD);
{ Return the current mickeys to pixels ratios for vertical and horizontal
  screen movement and the threshold speed for doubling of pointer motion.
  Call:    	nothing
  Returns: 	Horiz  = horizontal mickeys (1-32,767; default=8)
				Vert   = vertical mickeys (1-32,767; default=16)
				Double = double speed threshold in mickeys/second (default=64);
}

PROCEDURE SetPointerPage(Page : WORD);
{ Selects the display page for the mouse pointer.
  Call:    Page = display page
  Returns: nothing
  Note:    * The valid page numbers depend on the current display mode.
}

FUNCTION GetPointerPage : WORD;
{ Returns the current display page for the mouse pointer.
}
PROCEDURE DisableMouseDriver(VAR Handler : pointer; VAR Err : BOOLEAN);
{ Disables the mouse driver and returns the address of the previous Int 33H
  handler.
  Call:    nothing
  Returns: Handler = pointer to previous Int 33H handler
	   Err     = false if successful, true otherwise
  Notes:   * When this procedure is called, the mouse driver releases any
	     intErrupt vectors it hase captured OTHER than Int 33H (which may
	     be Int 10H, Int 71H, and/or Int 74H).  The application program
	     can COMPLETE the process of logically removing the mouse driver
	     by restoring the original contents of the Int 33H vector with
	     SetIntVec using the pointer returned by the procedure.
	   * Also see EnableMouseDriver.
}
PROCEDURE EnableMouseDriver;
{ Enables the mouse driver and the servicing fo mouse intErrupts.
  Call:    nothing
  Returns: nothing
  Note:    * Also see DisableMouseDriver
}
PROCEDURE ResetMouseDriver(VAR Err : BOOLEAN);
{ Resets the mouse driver and returns driver status.  If the mouse pointer was
  previously visible, is is removed trom the screen, and any presoiusly
  installed user event handlers for mouse events are disabled.
  Call:    nothing
  Returns: nothing
  Note:    * This procedure differ from InitMouse in that there is no
	     initialization of the mouse hardware.
}
PROCEDURE SetMouseLang(LangNumber : WORD);
{ Selects the language that will be used by the mouse driver for prompts and
  Error messages.
  Call:    LangNumber = language number
			0 = English
			1 = French
			2 = Dutch
			3 = German
			4 = Swedish
			5 = Finnish
			6 = Spanish
			7 = Portuguese
			8 = Italian
  Returns: nothing
  Note:    * This procedure is only functional in international versions of
	     the Microsoft Mouse drive.
}
FUNCTION GetMouseLang : WORD;
{ Returns the number of the language that is used by the mouse driver for
  prompts and Error messages.
  Call:    nothing
  Returns: language number (see above)
  Note:    * This procedure is only functional in international versions of
	     the Microsoft Mouse drive.
}
PROCEDURE GetMouseInfo(VAR MajVer, MinVer, MouseType, IRQ : WORD);
{ Returns the mouse driver version number, mouse type, and the IRQ number of
  the intErrupt used by the mouse adapter.
  Call:    nothing
  Returns: MajVer    = major version number (6 for version 6.10, etc.)
	   MinVer    = minor version number (10 for version 6.10, etc.)
	   MouseType = mouse type
		       1 = bus mouse
		       2 = serial mouse
		       3 = InPort mouse
		       4 = PS/2 mouse
		       5 = HP mouse
	   IRQ       = IRQ number
		       0                = PS/2
		       2, 3, 4, 5, or 7 = IRQ number
}

{--------------------------------------------------------------------------}

IMPLEMENTATION

USES
	VGraph,Dos,Crt;

CONST
	MouseInt = $33;

TYPE
{$IFDEF Mode256}
	MousePointerType = ARRAY[0..15,0..15] OF BYTE;
{$ENDIF}
{$IFDEF Mode64}
	MousePointerType = ARRAY[0..15,0..15] OF WORD;
{$ENDIF}
{$IFDEF Mode16}
	MousePointerType = ARRAY[0..16*16*3-1] OF BYTE;
{$ENDIF}

CONST
	OldX			: INTEGER = 0;
	OldY			: INTEGER = 0;
	OldHandler 	: POINTER = NIL;
	OldMask	  	: WORD = 1;
	Counter		: INTEGER = 0;

VAR
	BackGround			:	MousePointerType;
	AndPointer,XorPointer : MousePointerType;
  	Reg 					: Registers;
   Mode					: BOOLEAN;

{$F+}
PROCEDURE DrawMouse(x,y : WORD);
BEGIN
{$ifdef COPY}
	{ Variante 1: CopyPut }
	{ Restore old background }
	PutImage(oldx,oldy,oldx+15,oldy+15,BackGround);
	{ Get new background }
	GetImage(x,y,x+15,y+15,BackGround);
	{ Draw the pointer }
	PutSprite(x,y,x+15,y+15,XorPointer);
{$else}
	{ Variante 2: XorPut }
	{ Restore old background }
	PutImage(oldx,oldy,oldx+15,oldy+15,XorPointer);
	{ Draw the pointer }
	PutImage(x,y,x+15,y+15,XorPointer);
{$endif}
	oldx := x;
	oldy := y;
END;

PROCEDURE SVGAHandler;ASSEMBLER;
asm
	cli
	push	ds

	mov	ax,seg @DATA
	mov	ds,ax

	cmp	[counter],0
	jle	@@Ende

	push	cx
	push	dx
	call	[DrawMouse]

@@Ende:
	pop	ds
	sti
END;
{$F-}


FUNCTION MouseDriverInstalled : BOOLEAN;
{ Maus-Driver installiert ? }
CONST
	iret = $cf;
VAR
	Int33Ptr : Pointer;
BEGIN
	GetIntVec(MouseInt,Int33Ptr);
   IF (Byte(Int33Ptr^) = iret) OR (Int33Ptr = NIL) THEN
		MouseDriverInstalled := False
	ELSE BEGIN
   { Reset Driver and Read Status }
		reg.ax := 0;
		Intr(MouseInt,reg);
		MouseDriverInstalled := (reg.ax = $ffff);
	END;
END;

PROCEDURE InitDriver(VAR Buttons : WORD; Modus : BOOLEAN;VAR Err : BOOLEAN);
{ Iniialize the mouse driver }
BEGIN
	{ Reset Driver and Read Status }
  	Reg.AX := 0;
  	intr(MouseInt, Reg);
   Err := (reg.ax = $ffff);
   Buttons := reg.bx;
   IF Buttons = $FFFF THEN
   	Buttons := 2;
   Mode := Modus;
   IF Mode = GRAPHICS THEN BEGIN
		SetPointerShape(0,0,NIL);
		OldHandler := @SVGAHandler;
		SwapEventHandlers(OldMask,OldHandler);
		GetMousePosition(oldx,oldy);
   END;
END;

PROCEDURE ShowPointer;
BEGIN
	{ Show Mouse Cursor }
	IF mode = TEXT THEN BEGIN
		reg.AX := 1;
		intr(MouseInt, reg);
	END ELSE BEGIN
		Inc(counter);
		IF counter > 0 THEN BEGIN
{$ifdef COPY}
			GetImage(oldx,oldy,oldx+15,oldy+15,BackGround);
			PutSprite(oldx,oldy,oldx+15,oldy+15,XorPointer);
{$else}
			PutImage(oldx,oldy,oldx+15,oldy+15,XorPointer);
{$endif}
		END;
	END;
END;

PROCEDURE HidePointer;
BEGIN
	{ Hide Mouse Cursor }
	IF mode = TEXT THEN BEGIN
		reg.ax := 2;
		Intr(MouseInt,reg);
	END
	ELSE BEGIN
		Dec(counter);
		IF counter>=0 THEN BEGIN
{$ifdef COPY}
			PutImage(oldx,oldy,oldx+15,oldy+15,BackGround);
{$else}
			PutImage(oldx,oldy,oldx+15,oldy+15,XorPointer);
{$endif}
		END;
   END;
END;

FUNCTION ButtonPressed : BYTE;
BEGIN
  Reg.AX := 3;
  intr(MouseInt, Reg);
  ButtonPressed := Reg.BX;
END;

PROCEDURE GetMousePosition(VAR Horiz, Vert : INTEGER);
BEGIN
  Reg.AX := 3;
  intr(MouseInt, Reg);
  Horiz := Reg.CX;
  Vert  := Reg.DX;
END;

FUNCTION MouseIn(x1,y1,x2,y2: INTEGER):BOOLEAN;
VAR
   x,y	: INTEGER;
BEGIN
  GetMousePosition(x,y);
  MouseIn := (x>=x1) AND (x<=x2) AND (y>=y1) AND (y<=y2)
END;

PROCEDURE SetPointerPosition(Horiz, Vert : INTEGER);
BEGIN
   { Maus positionieren }
	reg.ax := 4;
	reg.cx := Horiz;
	reg.dx := Vert;
	Intr(MouseInt,reg);
END;

PROCEDURE GetPressInfo(Button : WORD; VAR Stat, Count :WORD; VAR Horiz, Vert : INTEGER);
BEGIN
  Reg.AX := 5;
  Reg.BX := Button SHR 1;
  intr(MouseInt, Reg);
  Stat := Reg.AX;
  Count:= Reg.BX;
  Horiz:= Reg.CX;
  Vert := Reg.DX;
END;

PROCEDURE GetReleaseInfo(Button : WORD; VAR Stat, Count : WORD; VAR Horiz, Vert : INTEGER);
BEGIN
  Reg.AX := 6;
  Reg.BX := Button SHR 1;
  intr(MouseInt, Reg);
  Stat := Reg.AX;
  Count:= Reg.BX;
  Horiz:= Reg.CX;
  Vert := Reg.DX;
END;

PROCEDURE SetLimits(HorMin, VerMin, HorMax, VerMax : INTEGER);
BEGIN
	{ Define Horizontal Cursor Range }
	reg.ax := 7;
	reg.cx := HorMin;
	reg.dx := HorMax;
	Intr(MouseInt,reg);

{ Define Vertical Cursor Range }
	reg.ax := 8;
	reg.cx := VerMin;
	reg.dx := VerMax;
	Intr(MouseInt,reg);
END;

PROCEDURE SetPointerShape(Horiz, Vert : INTEGER; Buffer : pointer);
CONST
	AndMask		: ARRAY[0..15] OF WORD = (
	65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535 );

	XorMask		: ARRAY[0..15] OF WORD = (
	65472,	65408,	65280,	65024,	64512,	65024,	63232,	58240,
	49600,	32992,	112,	56,	28,	14,	7,	3);
VAR
	x,y	: BYTE;
BEGIN
	IF mode = TEXT THEN BEGIN
		Reg.AX := 9;
		Reg.BX := Horiz;
		Reg.CX := Vert;
		IF Buffer = NIL THEN BEGIN
			Reg.ES := Seg(AndMask);
			Reg.DX := Ofs(AndMask);
		END ELSE BEGIN
			Reg.ES := Seg(Buffer);
			Reg.DX := Ofs(Buffer);
		END;
		intr(MouseInt, Reg);
	END ELSE BEGIN
		FOR y := 0 TO 15 DO BEGIN
			FOR x := 0 TO 15 DO BEGIN

{$IFDEF Mode16}
				IF AndMask[y] AND (32768 SHR x) > 0 THEN BEGIN
					AndPointer[y*16*3+x*3+0] := 255;
					AndPointer[y*16*3+x*3+1] := 255;
					AndPointer[y*16*3+x*3+2] := 255;
				END ELSE BEGIN
					AndPointer[y*16*3+x*3+0] := 0;
					AndPointer[y*16*3+x*3+1] := 0;
					AndPointer[y*16*3+x*3+2] := 0;
				END;

				IF XorMask[y] AND (32768 SHR x) > 0 THEN BEGIN
					XorPointer[y*16*3+x*3+0] := 255;
					XorPointer[y*16*3+x*3+1] := 255;
					XorPointer[y*16*3+x*3+2] := 255;
				END ELSE BEGIN
					XorPointer[y*16*3+x*3+0] := 0;
					XorPointer[y*16*3+x*3+1] := 0;
					XorPointer[y*16*3+x*3+2] := 0;
				END;
{$ELSE}
				IF AndMask[y] AND (32768 SHR x) > 0 THEN
					AndPointer[x,y] := GetMaxColor
				ELSE
					AndPointer[x,y] := 0;

				IF XorMask[y] AND (32768 SHR x) > 0 THEN
					XorPointer[x,y] := GetMaxColor
				ELSE
					XorPointer[x,y] := 0;
{$ENDIF}
			END;
		END;
	END;
END;

PROCEDURE SetTextPointer(PtrTyp, AND_Str, XOR_End : WORD);
BEGIN
  Reg.AX := 10;
  Reg.BX := PtrTyp;
  Reg.CX := AND_Str;
  Reg.DX := XOR_End;
  intr(MouseInt, Reg);
END;

PROCEDURE GetMotionCount(VAR Horiz, Vert : Integer);
BEGIN
	Reg.AX := 11;
	intr(MouseInt, Reg);
	Horiz:= Reg.CX;
	Vert := Reg.DX;
	{ MICKEYS !!! }
END;

PROCEDURE SetEventHandler(EventMask : WORD; Handler : pointer);
BEGIN
  Reg.AX := 12;
  Reg.CX := EventMask;
  Reg.ES := seg(Handler^);
  Reg.DX := ofs(Handler^);
  intr(MouseInt, Reg);
END;

PROCEDURE SetLightPen(On_Off : WORD);
BEGIN
  IF (On_Off = 0) THEN
    Reg.AX := 14
  ELSE
    Reg.AX := 13;
  intr(MouseInt, Reg);
END;

PROCEDURE SetMickeyPixelRatio(Horiz,Vert : WORD);
BEGIN
	Reg.AX := 15;
	Reg.CX := Horiz;
	Reg.DX := Vert;
	Intr(MouseInt,Reg);
END;

PROCEDURE SetPointerExcl(HorMin, VerMin, HorMax, VerMax : WORD);
BEGIN
  Reg.AX := 16;
  Reg.CX := HorMin;
  Reg.DX := VerMin;
  Reg.SI := HorMax;
  Reg.DI := VerMax;
  intr(MouseInt, Reg);
END;

PROCEDURE SwapEventHandlers(VAR Mask : WORD; VAR Buffer : pointer);
BEGIN
  	Reg.AX := 20;
  	Reg.CX := Mask;
  	Reg.ES := Seg(Buffer^);
  	Reg.DX := Ofs(Buffer^);
  	intr(MouseInt, Reg);
	Mask 	 := Reg.CX;
	Buffer := Ptr(Reg.ES, Reg.DX);
END;

FUNCTION GetSaveStateSize : WORD;
BEGIN
  Reg.AX := 21;
  intr(MouseInt, Reg);
  GetSaveStateSize := Reg.BX;
END;

PROCEDURE SaveDriverState(Buffer : pointer; Size : WORD);
BEGIN
  Reg.AX := 22;
  Reg.BX := Size;
  Reg.ES := Seg(Buffer^);
  Reg.DX := Ofs(Buffer^);
  intr(MouseInt, Reg);
END;

PROCEDURE RestoreDriverState(Buffer : pointer; Size : WORD);
BEGIN
  Reg.AX := 23;
  Reg.BX := Size;
  Reg.ES := Seg(Buffer^);
  Reg.DX := Ofs(Buffer^);
  intr(MouseInt, Reg);
END;

PROCEDURE SetAltEventHandler(Mask : WORD; Handler : pointer; VAR Err: BOOLEAN);
BEGIN
  Reg.AX := 24;
  Reg.CX := Mask;
  Reg.ES := Seg(Handler);
  Reg.DX := Ofs(Handler);
  intr(MouseInt, Reg);
  Err := (Reg.AX <> $18);
END;

PROCEDURE GetAltEventAdrs(VAR Mask : WORD; VAR Handler : pointer; VAR Err : BOOLEAN);
BEGIN
  Reg.AX := 25;
  Reg.CX := Mask;
  intr(MouseInt, Reg);
  IF (Reg.CX > 0) THEN BEGIN
      Mask := Reg.CX;
      Handler := Ptr(Reg.ES, Reg.DX);
      Err := false;
	END ELSE
    Err := true;
END;

PROCEDURE SetMouseSense(Horiz, Vert, Double : WORD);
BEGIN
  Reg.AX := 26;
  Reg.BX := Horiz;
  Reg.CX := Vert;
  Reg.DX := Double;
  intr(MouseInt, Reg);
END;

PROCEDURE GetMouseSense(VAR Horiz, Vert, Double : WORD);
BEGIN
  Reg.AX := 27;
  intr(MouseInt, Reg);
  Horiz 	:= Reg.BX;
  Vert 	:= Reg.CX;
  Double := Reg.DX;
END;

PROCEDURE SetPointerPage(Page : WORD);
BEGIN
  Reg.AX := 29;
  Reg.BX := Page;
  intr(MouseInt, Reg);
END;

FUNCTION GetPointerPage : WORD;
BEGIN
  Reg.AX := 30;
  intr(MouseInt, Reg);
  GetPointerPage := Reg.BX;
END;

PROCEDURE DisableMouseDriver(VAR Handler : pointer; VAR Err : BOOLEAN);
BEGIN
  Reg.AX := 31;
  intr(MouseInt, Reg);
  IF (Reg.AX = 31) THEN BEGIN
      Handler := ptr(Reg.ES, Reg.DX);
      Err := false;
	END ELSE
    Err := true;
END;

PROCEDURE EnableMouseDriver;
BEGIN
  Reg.AX := 32;
  intr(MouseInt, Reg);
END;

PROCEDURE ResetMouseDriver(VAR Err : BOOLEAN);
BEGIN
	SwapEventHandlers(OldMask,OldHandler);
  	Reg.AX := 33;
  	intr(MouseInt, Reg);
   Err := Reg.BX <> $21;
END;

PROCEDURE SetMouseLang(LangNumber : WORD);
BEGIN
  Reg.AX := 34;
  Reg.BX := LangNumber;
  intr(MouseInt, Reg);
END;

FUNCTION GetMouseLang : WORD;
BEGIN
  Reg.AX := 35;
  intr(MouseInt, Reg);
  GetMouseLang := Reg.BX;
END;

PROCEDURE GetMouseInfo(VAR MajVer, MinVer, MouseType, IRQ : WORD);
BEGIN
  Reg.AX := 36;
  intr(MouseInt, Reg);
  MajVer := Reg.BH;
  MinVer := Reg.BL;
  MouseType := Reg.CH;
  IRQ := Reg.CL;
END;

END.
