UNIT GroupMgr;
  {Group manager for Fumiko}


INTERFACE


  USES
    CRT, GloblCTV, Arcana, Butler;


  PROCEDURE Make_New_Group (var x, y: byte; var board: board_type);
    {Builds a new group containing one stone, in terms of size and members}
    {only, and adds it to the group list}


  PROCEDURE Merge_Groups (a, b: byte; var board: board_type);
    {Merges two groups, in terms of size and members only,}
    {and removes group b from the group list}


  FUNCTION Maneuvering_Room (var x, y: byte; var owner: color_type; var board: board_type): boolean;
    {Returns true if the point is not controlled by the other player}


  PROCEDURE Update_Interpretation (var group: byte; var board: board_type; depth: byte);
    {Updates a group's interpretation of its surroundings}


  PROCEDURE Capture  (group: byte; var board: board_type; var changes: boolean_board; depth: byte);
    {Removes the offending stones from the board and their group from the list}
    {Also updates the liberties of nearby groups}


IMPLEMENTATION


  USES
    Sequence;


  PROCEDURE Make_New_Group (var x, y: byte; var board: board_type);
    {Builds a new group containing one stone, in terms of size and members}
    {only, and adds it to the group list}
    Var
      a, b: byte;
    Begin  {Make New Group}
      board.data [x, y].group := 1;
      While board.groups [board.data [x, y].group]^.size <> 0 do
        Inc (board.data [x, y].group);
      With board.groups [board.data [x, y].group]^ do
        Begin
          size := 1;
          aliveness := 4;
          room := 0;
          last_update := 0;
          owner := board.data [x, y].color;
          to_kill.x := 0;
          to_kill.y := 0;
          For a := 1 to board_size do  {List members in minimal interpretation}
            For b := 1 to board_size do
              interpretation [a, b].member := false;
          interpretation [x, y].member := true
        End;
      board.groups [board.data [x, y].group + 1]^.size := 0
    End;  {Make New Group}


  PROCEDURE Merge_Groups (a, b: byte; var board: board_type);
    {Merges two groups, in terms of size and members only,}
    {and removes group b from the group list}
    Var
      x, y: byte;
    Begin  {Merge Groups}
      board.groups [a]^.size := board.groups [a]^.size + board.groups [b]^.size;
      If board.groups [b]^.aliveness > board.groups [a]^.aliveness  {The new group has the stronger of the two lives}
        Then board.groups [a]^.aliveness := board.groups [b]^.aliveness;
      If board.groups [b]^.last_update < board.groups [a]^.last_update  {The new group has the least recent update}
        Then board.groups [a]^.last_update := board.groups [b]^.last_update;
      For x := 1 to board_size do
        For y := 1 to board_size do
          If board.groups [a]^.interpretation [x, y].member or board.groups [b]^.interpretation [x, y].member
            Then Begin
                   board.groups [a]^.interpretation [x, y].member := true;
                   board.data [x, y].group := a
                 End;
      Compress (b, board)
    End;  {Merge Groups}


  FUNCTION Maneuvering_Room (var x, y: byte; var owner: color_type; var board: board_type): boolean;
    {Returns true if the point is not controlled by the other player}
    Var
      direction: direction_type;
      theirs: byte;
      neighbor: location_type;
    Begin  {Maneuvering Room}
      If (board.data [x, y].color = empty) or
           ((board.data [x, y].color = opposite (owner)) and (board.groups [board.data [x, y].group]^.aliveness < 3))
        Then Begin
               theirs := 0;
               For direction := north to ne do
                 Begin
                   neighbor := neighbors [x, y, direction];
                   If on_board (neighbor.x, neighbor.y)
                     Then With board.data [neighbors [x, y, direction].x, neighbors [x, y, direction].y] do
                            If (color = opposite (owner)) and (board.groups [group]^.aliveness > 2)
                              Then If direction in [north..west]
                                     Then Inc (theirs, 2)
                                     Else Inc (theirs, 1)
                 End;
               maneuvering_room := (theirs = 0) or ((theirs = 1) and (not on_edge (x)) and (not on_edge (y)))
             End
        Else maneuvering_room := false
    End;  {Maneuvering Room}


  PROCEDURE Consider_Point (var x, y: byte; var group: group_type; var interpretation: point_interpretation;
                             var board: board_type; var distance: byte);
    {The ugly part of look_around.  Updates the interpretation of a}
    {particular point.  LOCAL}
    Begin  {Consider Point}
      If interpretation.too_distant
        Then Begin  {The group has influence here--analyze it}
               interpretation.too_distant := false;
               interpretation.distance := distance;
               If board.data [x, y].color = empty
                 Then Begin
                        If maneuvering_room (x, y, group.owner, board)
                          Then Begin
                                 interpretation.connected := true;
                                 Inc (group.room);
                               End
                      End
                 Else If board.data [x, y].color = group.owner
                        Then Begin
                               interpretation.connected := true;
                               If board.groups [board.data [x, y].group]^.aliveness > group.aliveness
                                 Then group.aliveness := board.groups [board.data [x, y].group]^.aliveness
                                 Else board.groups [board.data [x, y].group]^.aliveness := group.aliveness
                             End
                        Else If maneuvering_room (x, y, group.owner, board)
                               Then Begin
                                      interpretation.connected := true;
                                      Inc (group.room)
                                    End
               End
    End;  {Consider Point}


  PROCEDURE Look_Around (var group: group_type; distance, previous_differences: byte; var board: board_type);
    {The recursive part of update_interpretation.  Looks at points}
    {around those not labelled too_distant.  LOCAL}
    Var
      x, y, differences: byte;
      new_interpretation: ^board_interpretation;
      direction: direction_type;
      point: location_type;
    Begin  {Look Around}
      New (new_interpretation);
      new_interpretation^ := group.interpretation;
      For x := 1 to board_size do
        For y := 1 to board_size do
          If (group.interpretation [x, y].connected) and not
               ((board.data [x, y].group <> 0) and (board.groups [board.data [x, y].group]^.aliveness > 3)
                  and (board.data [x, y].color = opposite (group.owner)))
            Then For direction := north to west do
                   Begin
                     point := neighbors [x, y, direction];
                     If on_board (point.x, point.y)
                       Then Consider_Point (point.x, point.y, group, new_interpretation^ [point.x, point.y], board, distance);
                   End;
      differences := 0;
      For x := 1 to board_size do
        For y := 1 to board_size do
          Begin
            If not equal (group.interpretation [x, y], new_interpretation^ [x, y], point_interpretation_size)
              Then Inc (differences);
            group.interpretation [x, y] := new_interpretation^ [x, y]
          End;
      Dispose (new_interpretation);
      If (distance < group.aliveness) and (differences > previous_differences)
        Then Look_Around (group, distance + 1, differences, board)
    End;  {Look Around}


  PROCEDURE Clear_Out (var interpretation: board_interpretation);
    {Sets all interpretation bits to false, except member (which is}
    {unaffected) and too_distant (which is := not member).  LOCAL}
    Var
      x, y: byte;
    Begin  {Clear Out}
      For x := 1 to board_size do
        For y := 1 to board_size do
          With interpretation [x, y] do
            Begin
              too_distant := not member;
              adjacent := false;
              connected := false;
              distance := 255
            End
    End;  {Clear Out}


  PROCEDURE Consider_Adjacent_Point (var x, y: byte; var group: group_type; var board: board_type);
    {The ugly part of update_interpretation.  Updates the interpretation of}
    {a particular adjacent point.  LOCAL}
    Begin  {Consider Adjacent Point}
      If (on_board (x, y)) and (group.interpretation [x, y].too_distant)
        Then With group.interpretation [x, y] do
               Begin
                 too_distant := false;
                 adjacent := true;
                 group.interpretation [x, y].distance := 1;
                 If board.data [x, y].color = empty
                   Then Begin
                          If maneuvering_room (x, y, group.owner, board)
                            Then Begin
                                   Inc (group.room);
                                   connected := true
                                 End;
                          Inc (group.liberties)
                        End
                   Else If board.data [x, y].color in [black_stone, white_stone]
                             {It must be occupied by the opponent}
                          Then Begin
                                 If maneuvering_room (x, y, group.owner, board)
                                   Then Begin
                                          connected := true;
                                          Inc (group.room)
                                        End
                               End
               End
    End;  {Consider Adjacent Point}


  PROCEDURE Update_Interpretation (var group: byte; var board: board_type; depth: byte);
    {Updates a group's interpretation of its surroundings}
    Var
      x, y: byte;
      direction: direction_type;
      neighbor: location_type;
    Begin  {Update Interpretation}
      With board.groups [group]^ do
        Begin
          liberties := 0;
          room := 0;
          last_update := board.turn;
          Clear_Out (interpretation);
          For x := 1 to board_size do
            For y := 1 to board_size do
              If interpretation [x, y].member
                Then Begin
                       interpretation [x, y].connected := true;
                       interpretation [x, y].distance := 0;
                       For direction := north to west do
                         Begin
                           neighbor := neighbors [x, y, direction];
                           Consider_Adjacent_Point (neighbor.x, neighbor.y, board.groups [group]^, board);
                         End
                     End;
          If depth <= maximum_search_depth
            Then Begin
                   Look_Around (board.groups [group]^, 2, 0, board);
                   Update_Aliveness (board.groups [group]^, board, depth)
                 End
        End
    End;  {Update Interpretation}


  PROCEDURE Capture  (group: byte; var board: board_type; var changes: boolean_board; depth: byte);
    {Removes the offending stones from the board and their group from the list}
    {Also updates the liberties of nearby groups}
    Var
      x, y: byte;
    Begin  {Capture}
      For x := 1 to board_size do
        For y := 1 to board_size do
          If board.groups [group]^.interpretation [x, y].member
            Then Begin
                   changes [x, y] := true;
                   board.data [x, y].color := empty;
                   board.data [x, y].group := 0
                 End;
      Compress (group, board)
    End;  {Capture}


END.  {GroupMgr}

