UNIT TIFF256;
{
	TIFF256
	- by Bjarke Viksoe

	Converts TIFF image files in format 320x200 in 256 colours
	to a raw image.
	Does a little IO checking. Little picturesize checking.
	Picture must be uncompressed or PackBits-compressed!
}

INTERFACE

USES
	DEMOINIT, PICTURE;


Procedure LoadPix(buffer : pScreen; filename : string);
Procedure ConvertTIFF(dst : pScreen; src : pBuffer);


(*--------------------------------------*)

IMPLEMENTATION

{$I-}

TYPE
	pHeader = ^HeaderType;
	HeaderType = RECORD
		ID : word;
		version : word;
		offset : longint;
	end;
	pStartIFDType = ^IFDStartType;
	IFDStartType = RECORD
		entries : word;
	end;
	pEndIFDType = ^IFDEndType;
	IFDEndType = RECORD
		next : longint;
	end;
	pFieldType = ^FieldType;
	FieldType = RECORD
		tag : word;
		fieldtype : word;
		length : longint;
		value : longint;
	end;

Var
	Config : RECORD
		width, height : word;    {image sizes are in pixels}
		depth         : word;    {image depth: 8 for 256 coloured images}
		bitsprpixel   : word;    {bits/plane}
		sampprpixel   : word;    {bytes/pixel; 1=gray, 3=RGB}
		compression   : word;    {compression scheme; 1=none, 5=lzw + others}
		colorscheme   : word;    {color scheme; 0/1 = gray, 2=RGB, 3=ColorMap}
		orientation   : word;    {should be 1 for top/left image}
		fileoffset    : longint; {offset to bitmap data}
		coloroffset   : longint; {offset to colormap}
	end;


(*---------------------------------------------------------*)
(*                    BASIC TIFF ROUTINES                  *)
(*---------------------------------------------------------*)

Procedure ParseTIFF(src : pBuffer; pos : longint );
Type
	pByte = ^byte;
	pShort = ^word;
	pLong = ^longint;
Var
	pIFD : pStartIFDType;
	pNextIFD : pEndIFDType;
	pField : pFieldType;
	entries : word;
Begin
	{parse Image Field Directory and TIFF Fields too}
	while (pos<>1) AND (pos<65535) do begin
		pIFD := @src^[pos];
		Inc(pos,2);
		entries := pIFD^.entries;
		while entries > 0 do begin
			pField := @src^[pos];
			case (pField^.Tag) of
			 256 : Config.width       := pField^.value;
			 257 : Config.height      := pField^.value;
			 258 : Config.depth       := pField^.value;
			 259 : Config.compression := pField^.value;
			 262 : Config.colorscheme := pField^.value;
			 273 : Config.fileoffset  := pLong( @src^[pField^.value+1] )^ + 1;
			 274 : Config.orientation := pField^.value;
			 277 : Config.sampprpixel := pField^.value;
			 284 : Config.bitsprpixel := pField^.value;
			 320 : Config.coloroffset := pField^.value + 1;
			end;
			Dec( entries );
			Inc( pos, SizeOf( FieldType ) );
		end;
		pNextIFD := @src^[pos];
		pos := pNextIFD^.next + 1;
	end;
End;

Procedure ExtractCMAP(src : pBuffer);
Type
	pShortArray = ^tShortArray;
	tShortArray = array[0..(256*3)-1] of word;
Var
	i,j : word;
	pColors : pShortArray;
Begin
	case (Config.colorscheme) of
	 0: begin {gray-scaled: (63,63,63) - (0,0,0)}
		for i:=0 to 255 do begin
			CMAP[(i*3)+1]:=(255-i) SHR 2;
			CMAP[(i*3)+2]:=(255-i) SHR 2;
			CMAP[(i*3)+3]:=(255-i) SHR 2;
		end;
	 end;
	 1: begin {gray-scaled: (0,0,0) - (63,63,63)}
		for i:=0 to 255 do begin
			CMAP[(i*3)+1]:=i SHR 2;
			CMAP[(i*3)+2]:=i SHR 2;
			CMAP[(i*3)+3]:=i SHR 2;
		end;
	 end;
	  3: begin
		{if a colormap was in there somewhere, copy it...}
		if Config.ColorOffset<>0 then	begin
			{hmm, colorvalues are stored as longs in a large colormap array}
			pColors:=@src^[Config.ColorOffset];
			for i:=0 to 255 do begin
				CMAP[(i*3)+1]:=pColors^[i] SHR 10;
				CMAP[(i*3)+2]:=pColors^[i+256] SHR 10;
				CMAP[(i*3)+3]:=pColors^[i+512] SHR 10;
			end;
		end;
	 end;
	end;
End;

Procedure ExtractImage_Raw(src : pBuffer; dst : pScreen);
Var
	i,srcpos,dstpos : word;
Begin
	case (Config.colorscheme) of
	 0,1,3 : begin
			  srcpos:=Config.FileOffset;
			  dstpos:=0;
			  for i:=1 to Config.Height do begin
				  Move( src^[srcpos], dst^[dstpos], Config.width );
				  Inc( srcpos, Config.Width );
				  Inc( dstpos, 320 );
			  end;
			end;
	end;
End;

Procedure ExtractImage_PackBits(src : pBuffer; dst : pScreen);
Var
	ImageSize : word;
	s : pointer;
Begin
	case (Config.colorscheme) of
	 0,1,3 : begin
			  ImageSize:=Config.Width * Config.Height;
			  s:=@src^[Config.FileOffset];
			  asm
				push	ds
				les	di,[dst]
				lds	si,[s]
				xor	dx,dx
				cld
@TIFF_loop:	xor	ah,ah
				lodsb
				or		al,al
				js		@TIFF_lower
				inc	ax
				add	dx,ax
				mov	cx,ax
				rep movsb
				cmp	dx,[ImageSize]
				jb		@TIFF_loop
				jmp	NEAR PTR @TIFF_done
@TIFF_lower:cmp	al,$80
				je		@TIFF_loop
				neg	al
				inc	ax
				add	dx,ax
				mov	cx,ax
				lodsb
				rep stosb
				cmp	dx,[ImageSize]
				jb		@TIFF_loop
@TIFF_done:	pop	ds
			  end;
			end;
	end;
End;


(*---------------------------------------------------------*)
(*                 HIGH LEVEL TIFF ROUTINES                *)
(*---------------------------------------------------------*)

Procedure ConvertTIFF(dst : pScreen; src : pBuffer);
{Parse tga file}
Var
	i,j,k : word;
	pos : word;
	h : pHeader;
Begin
	h := pHeader(src);
	with h^ do begin
		{safety check for correct TIFF file}
		if ((ID<>$4949) AND (ID<>$4D4D)) then exit; {ID must be "II".
																	We also include "MM",
																	though it's not supported}
		if (version < 42) then exit;                {support at least v42}
	end;

	FillChar( Config, SizeOf( Config ), 0 );
	ParseTIFF( src, h^.offset + 1 );

	{more safety checks...}
	with Config do begin
		if (width>320) OR (height>240) OR (depth<>8) then exit;
	end;

	ExtractCMAP(src);
	case (Config.Compression) of
	 1     : ExtractImage_raw(src,dst);
	 $8005 : ExtractImage_packbits(src,dst);
	end;
End;


Procedure LoadPix(buffer : pScreen; filename : string);
Var
	pFileMem: pBuffer;
	FileHandle : file;
	size : longint;
Begin
	Assign(FileHandle, filename);
	Reset(FileHandle, 1);
	size := FileSize(FileHandle);
	if (size > 65535) then exit;
	if (size > MaxAvail) then exit;
	GetMem(pFileMem, size);
	BlockRead(FileHandle, pFileMem^, size);
	Close(FileHandle);
	if (IOResult<>0) then exit;
	ConvertTIFF(buffer, pFileMem);
	FreeMem(pFileMem, size);
End;

{$I+}

End.
