/* __________________________________________________________________________
  | FireMod 1.05 CopyRight (c) 1995 by FireLight                             |
  | Source Code Released for use with FireLight's mod coders howto document. |
  | Contact at firelght@yoyo.cc.monash.edu.au                                |
  | Didnt comment the interface code, who wants a heap of printf comments.   |
  |--------------------------------------------------------------------------|
  | Legal Notice:  This source is public domain.  This code cannot be        |
  | redistributed commercially, in any form whether source code or binary    |
  | without the prior consent of the author.  Use of this product does not   |
  | cost anything, but on using this you are expected to adhere to the       |
  | following conditions:                                                    |
  | The author is not liable in any way for any loss of data, property,      |
  | brains, or hearing due to the use of this product.  Upon using this      |
  | product you have agreed that the author, Brett Paterson shall not be held|
  | responsible for any misfortune encountered its use.                      |
  |__________________________________________________________________________|

  tabstops = 4.
*/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <string.h>
#include <alloc.h>

#include "fmod.h"

#define version 05
#define GUSfreq(x)              83117/x

/****************************************************************************
 *    Name : Error                                                          *
 * Purpose : General error handler for different error codes                *
 *  Passed : int code - the type of error                                   *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void Error(int code) {
	asm mov ax, 0x3
	asm int 0x10

	textcolor(15);
	textbackground(1);
	cprintf("FireMOD 1.%02da - CopyRight (c) FireLight, 1995                                   ", version);
	textcolor(7);
	textbackground(0);
	cprintf("ERROR #%d : ", code);

	switch (code) {
		case 1 : cprintf("ULTRASND Environment variable not found..\r\n");
				cprintf("Most likely cause GUS not present..\r\n");
				break;
		case 2 : cprintf("Module not found.");
				break;
		case 3 : cprintf("Unknown format.");
				break;
		case 4 : cprintf("Out of GUS DRAM! go spend some more money..");
				break;
		case 5 : cprintf("Not enough memory!..");
				break;
		case 6 : cprintf("Usage FMOD <module name>\r\n");
				break;
		default : ;
	};
	cprintf("\r\n");
	exit(1);
}


/****************************************************************************
 *    Name : SetTimerSpeed                                                  *
 * Purpose : To change the rate of the system timer.  If the bpm is 0 then  *
 *           we are stopping the mod and resetting the timer.               *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : int Speed - a temporary variable holding the value of the      *
 *           calculated rate using divisor / (bpm * 2 / 5)                  *
 ****************************************************************************/
void SetTimerSpeed() {
	int Speed;
	if (bpm == 0) Speed = 0;
	else Speed = int(1193180L/(bpm*2/5));

	asm {
		mov dx, 0x43
		mov al, 0x36
		out dx, al
		mov dx, 0x40
		mov ax, Speed
		out dx, al
		shr ax, 8
		out dx, al
	}
}


/****************************************************************************
 *    Name : UpdateNote                                                     *
 * Purpose : To actually update a row of notes and play them out on tick 0. *
 *  Passed : char &pbreak (by reference) - This gets the value of a pattern *
 *             break if any and returns it to our main handler for use there*
 *           char &pjump (by reference) - This gets the value of a pattern  *
 *             jump if any and returns it to our main handler for use there *
 *           The reason we do this is because we want the whole row to play *
 *           out before we start any pattern jumping.                       *
 * Returns : -                                                              *
 *  Locals : byte track - the number of the column or track we are in       *
 *           byte sample - a temp variable to hold the sample number called *
 *           byte effect - a temp variable to hold the effect number called *
 *           byte eparmx - a temp variable to hold the effect parameter x   *
 *           byte eparmy - a temp variable to hold the effect parameter y   *
 *           word soff - a value holding the # of bytes to sample offset if *
 *                       the effect is used                                 *
 *           int perdiod - a temp signed variable to hold the period value  *
 *           Note *current - a temporary note structure to hold the pointed *
 *                           to note in the pattern buffer                  *
 ****************************************************************************/
void UpdateNote() {
	byte track, sample, eparmx, eparmy;
	word soff;
	int period;

	// calculate where in the pattern buffer we should be according to
	// the pattern number and row.
	offset = long(channels)*256L*MOD.order[ord]+(4L*long(row)*channels);

	// new row? now we loop through each channel until we have finished
	for (track=0; track<channels; track++) {
		current = (Note *)(patbuff+offset);	// sit our note struct at the
											// right offset in pattn buffer
		period = current -> period;        	// get period
		sample = current -> number;         // get instrument #
		eparmx = current -> eparm >> 4;     // get effect param x
		eparmy = current -> eparm & 0xF;    // get effect param y

		// for the interface, the next 3 lines just store the effect # to
		// remember for later
		geffect[track] = current -> effect;
		if (geffect[track] == 0xE) geffect[track]+=2+eparmx;
		if (current->effect == 0 && current -> eparm > 0) geffect[track] = 32;

		if (sample > 0) {                   // if instrument > 0 set volume
			lastins[track] = sample - 1;    // remember the sample #
			volume[track] = MOD.inst[sample-1].volume;  // set chan's vol
			GUSSetVolume(track, volume[track]*mastervol/64);
		}
		if (period >= 0) {                  // if period >= 0 set freq
			midival[track] = period;        // remember the note
			// if not a porta effect, then set the channels frequency to the
			// looked up amiga value + or - any finetune
			if (current->effect != 0x3 && current->effect != 0x5) freq[track]=
				freqtab[midival[track]+MOD.inst[lastins[track]].finetune];
		}
		soff = 0;                           // sample offset = nothing now

		switch (current -> effect) {
			case 0x0: break;  // dont waste my time in here!!!
			//   0x1: not processed on tick 0
			//   0x2: not processed on tick 0
			case 0x3:
			case 0x5: porto[track] = freqtab[midival[track]+MOD.inst[lastins[track]].finetune];
					  if (current -> eparm > 0 && current->effect == 0x3)
					  portsp[track] = current -> eparm;
					  period = -1;
					  break;
			case 0x4: if (eparmx > 0) vibspe[track] = eparmx;
					  if (eparmy > 0) vibdep[track] = eparmy;
					  break;
			//   0x6: not processed on tick 0
			case 0x7: if (eparmx > 0) tremspe[track] = eparmx;
					  if (eparmy > 0) tremdep[track] = eparmy;
					  break;
			case 0x8: if (current -> eparm == 0xa4) panval[track]=7;
					  else panval[track] = (current -> eparm >> 3)-1;
					  if (panval[track] < 0) panval[track] = 0;
					  GUSSetBalance(track, panval[track]);
					  break;
			case 0x9: soff = current -> eparm << 8;
					  if (soff > MOD.inst[lastins[track]].length) soff =
								 MOD.inst[lastins[track]].length;
					  break;
			//   0xA: processed in UpdateEffect() (tick based)
			case 0xB: ord = current->eparm;
					  row = 0;
					  if (ord >= MOD.songLength) ord=0;
					  break;
			case 0xC: volume[track] = current -> eparm;
					  if (volume[track] < 0)  volume[track] = 0;
					  if (volume[track] > 64) volume[track] = 64;
					  GUSSetVolume(track, volume[track]*mastervol/64);
					  break;
			case 0xD: row = eparmx*10 + eparmy -1;
					  if (row > 63) row =0;
					  ord++;
					  if (ord >= MOD.songLength) ord=0;
					  break;
			case 0xF: if (current->eparm < 0x20) speed = current->eparm;
					  else { bpm = current->eparm; SetTimerSpeed(); }
					  break;
			case 0xE: switch (eparmx) {
						  case 0x1: freq[track] -= eparmy;
									break;
						  case 0x2: freq[track] += eparmy;
									break;
						  //   0x3: not supported (glissando)
						  //   0x4: not supported (set vibrato waveform)
						  case 0x5: MOD.inst[sample-1].finetune = eparmy;
									if (MOD.inst[sample-1].finetune > 7)
									MOD.inst[sample-1].finetune -= 16;
									break;
						  case 0x6: if (eparmy == 0) patlooprow = row;
									else {
										if (patloopno == 0) patloopno=eparmy;
										else patloopno--;
										if (patloopno > 0) row = patlooprow-1;
									}
									break;
						  //   0x6: not supported (set tremolo waveform)
						  case 0x8: panval[track] = eparmy;
									GUSSetBalance(track, eparmy);
									break;
						  //   0x9: not processed on tick 0
						  case 0xA: volume[track] += eparmy;
									if (volume[track] > 64) volume[track]=64;
									GUSSetVolume(track, volume[track]*mastervol/64);
									break;
						  case 0xB: volume[track] -= eparmy;
									if (volume[track] < 0)  volume[track]=0;
									GUSSetVolume(track, volume[track]*mastervol/64);
									break;
						  //   0xC: not processed on tick 0
						  case 0xD: period = -1;
									break;
						  case 0xE: patdelay = eparmy;
									break;
						  //   0xF: not supported (Invert loop)
					  };
					  break;
		};

		// set the frequency on the GUS for the voice, as long as its > 0
		if (freq[track] > 0) GUSSetFreq(track, GUSfreq(freq[track]));
		// only play the note if there is a note or there is an effect 9xy
		if (period >= 0 || soff > 0x00FF) {
			sinepos[track]=0;                  // retrig waveform
			sineneg[track]=0;
			// if sample has a loop then loop it.
			if (MOD.inst[lastins[track]].loopend > 2) GUSPlayVoice(track, 8,
				MOD.inst[lastins[track]].offset+soff,
				MOD.inst[lastins[track]].offset+MOD.inst[lastins[track]].loopstart,
				MOD.inst[lastins[track]].offset+MOD.inst[lastins[track]].loopend);
			// else just play the sample straight and no loop.
			else GUSPlayVoice(track, 0,
				MOD.inst[lastins[track]].offset+soff,
				MOD.inst[lastins[track]].offset,
				MOD.inst[lastins[track]].offset+MOD.inst[lastins[track]].length);
		}
		offset += 4;                         // increment our note pointer.
	}
}


/****************************************************************************
 *    Name : doporta                                           				*
 * Purpose : to carry out a tone portamento to a certain note               *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void doporta(byte track) {
	if (freq[track] < porto[track]) {
		freq[track] += portsp[track];
		if (freq[track] > porto[track]) freq[track]=porto[track];
	}
	if (freq[track] > porto[track]) {
		freq[track] -= portsp[track];
		if (freq[track] < 1) freq[track]=1;
		if (freq[track] < porto[track]) freq[track]=porto[track];
	 }
	 GUSSetFreq(track, GUSfreq(freq[track]));
}


/****************************************************************************
 *    Name : dovibrato                                         			    *
 * Purpose : to carry out a vibrato at a certain depth and speed            *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : int vib - size of delta to add or subtract from period         *
 ****************************************************************************/
void dovibrato(byte track) {
	int vib;

	vib = vibdep[track]*sintab[sinepos[track]] >> 7; // div 128
	if (sineneg[track] == 0) GUSSetFreq(track, GUSfreq(freq[track]+vib));
	else                     GUSSetFreq(track, GUSfreq(freq[track]-vib));

	sinepos[track]+=vibspe[track];
	if (sinepos[track] > 31) {
		sinepos[track] -=32;
		sineneg[track] = ~sineneg[track];		// flip pos/neg flag
	}
}


/****************************************************************************
 *    Name : dotremolo                                         			    *
 * Purpose : to carry out a tremolo at a certain depth and speed            *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : int vib - size of delta to add or subtract from volume         *
 ****************************************************************************/
void dotremolo(byte track) {
	int vib;

	vib = tremdep[track]*sintab[sinepos[track]] >> 6; // div64
	if (sineneg[track] == 0) {
		if (volume[track]+vib > 64) vib = 64-volume[track];
		GUSSetVolume(track, (volume[track]+vib)*mastervol/64);
	}
	else {
		if (volume[track]-vib < 0) vib = volume[track];
		GUSSetVolume(track, (volume[track]-vib)*mastervol/64);
	}

	sinepos[track]+= tremspe[track];
	if (sinepos[track] > 31) {
		sinepos[track] -=32;
		sineneg[track] = ~sineneg[track];			// flip pos/neg flag
	}
}


/****************************************************************************
 *    Name : UpdateEffect                                                   *
 * Purpose : To update any tick based effects after tick 0                  *
 *  Passed : int tick - the actual tick number.                             *
 * Returns : -                                                              *
 *  Locals : int track - the number of the column/track or channel we are in*
 *           int vib - the delta which to vibrate from the frequency with   *
 *           byte effect - a temp variable to get the effect number wanted  *
 *           byte eparmx - a term variable to get the effect parameter x    *
 *           byte eparmy - a term variable to get the effect parameter y    *
 *           static byte arpcount - a counter to tell us what to do with    *
 *               arpeggio (we dont want to forget it too, so its static)    *
 *           Note *current - a temporary note structure to hold the pointed *
 *                           to note in the pattern buffer                  *
 *    Note : To see explanations of effects check out the howto document    *
 ****************************************************************************/
void UpdateEffect(int tick) {
	byte track, effect, eparmx, eparmy;

	if (row < 1) return;                    // if at row 0 nothing to update
	offset -= (4L * channels);              // go back 4 bytes in buffer
											// becuase update note did +4
	for (track=0; track<channels; track++) {// start counting through tracks
		current = (Note *)(patbuff+offset); // point our Note to right spot

		effect = current -> effect;         // grab the effect number
		eparmx = current -> eparm >> 4;     // grab the effect parameter x
		eparmy = current -> eparm & 0xF;    // grab the effect parameter y

		if (freq[track] == 0) goto skip;

		switch(effect) {
			case 0x0: if (current -> eparm > 0) {
						  switch (tick%3) {
							  case 0: GUSSetFreq(track, GUSfreq(freq[track]));
									  break;
							  case 1: GUSSetFreq(track, GUSfreq(freqtab[midival[track]+(8*eparmx)+MOD.inst[lastins[track]].finetune]));
									  break;
							  case 2: GUSSetFreq(track, GUSfreq(freqtab[midival[track]+(8*eparmy)+MOD.inst[lastins[track]].finetune]));
									  break;
						  };
					  }
					  break;

			case 0x3: doporta(track);
					  break;

			case 0x4: dovibrato(track);
					  break;

			case 0x1: freq[track]-= current -> eparm;      	// subtract freq
					  GUSSetFreq(track, GUSfreq(freq[track]));
					  if (freq[track] < 54) freq[track]=54;	// stop at C-5
					  break;

			case 0x2: freq[track]+= current -> eparm;
					  GUSSetFreq(track, GUSfreq(freq[track]));
					  break;


			case 0x5: doporta(track);
					  volume[track] += eparmx - eparmy;
					  if (volume[track] < 0)  volume[track] = 0;
					  if (volume[track] > 64) volume[track] = 64;
					  GUSSetVolume(track, volume[track]*mastervol/64);
					  break;

			case 0x6: dovibrato(track);
					  volume[track] += eparmx - eparmy;
					  if (volume[track] < 0)  volume[track] = 0;
					  if (volume[track] > 64) volume[track] = 64;
					  GUSSetVolume(track, volume[track]*mastervol/64);
					  break;

			case 0x7: dotremolo(track);
					  break;

			case 0xA: volume[track] += eparmx - eparmy;
					  if (volume[track] < 0)  volume[track] = 0;
					  if (volume[track] > 64) volume[track] = 64;
					  GUSSetVolume(track, volume[track]*mastervol/64);
					  break;

			case 0xE: switch(eparmx) {
						  case 0xC: if (eparmy == tick) {
										volume[track] = 0;
										GUSSetVolume(track, volume[track]);
									}
									break;

						  case 0x9: if (tick % eparmy == 0) {
									if (MOD.inst[lastins[track]].loopend > 2)
										 GUSPlayVoice(track, 8, MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].loopstart,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].loopend);
									else GUSPlayVoice(track, 0, MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].length);
									}
									break;
						  case 0xD:	if (eparmy==tick) {
									if (MOD.inst[lastins[track]].loopend > 2)
										 GUSPlayVoice(track, 8, MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].loopstart,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].loopend);
									else GUSPlayVoice(track, 0, MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset,
										   MOD.inst[lastins[track]].offset+
										   MOD.inst[lastins[track]].length);
									}
									break;
					  };
					  break;
		};
skip:
		offset+=4;      // increment our buffer to next note
	}
}


void interrupt modhandler(...) {
	static int tick=speed-1;

	tick++;                  					// increment the tick value
	if (tick >= speed) {
		clock+= (1.0/(bpm * 2 / 5))*speed;      // increment the clock
		tick = 0;                               // set the tick to nothing
		if (row == 64) {                        // if end of pattn (64)
			ord++;                              // next order
			if (ord >= MOD.songLength) ord=0;   // if end goto 1st order
			row = 0;                            // start at top of pattn
		}

		if (patdelay == 0) {                    // if there is no pat delay
			UpdateNote();          				// Update and play the note
			row++;                              // increment the row
		}
		else patdelay --;                       // else decrement pat delay
	}
	else UpdateEffect(tick);                    // Else update the effects

	// return the computer to what it was doing before
	outportb(0x20, 0x20);
}


/****************************************************************************
 *    Name : interrupt keyhandler                                           *
 * Purpose : to sit on the keyboard interrupt and test for F12 to be pressed*
 *  Passed :                                                                *
 * Returns : -                                                              *
 *  Locals : byte key - the key pressed.                                    *
 ****************************************************************************/
void interrupt keyhandler(...) {
	byte key;

	asm  In      Al,0x060              // record key stroke
	asm  mov     key, al               // store it in 'key'

	if (key==88) SetTimerSpeed();      // if F12 is pressed reset timer

	oldkeyhandler();
}


/****************************************************************************
 *    Name : PlayMOD                                                        *
 * Purpose : Start the action and lets hear some music                      *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : int count - a counter variable for resetting GUS voices.       *
 ****************************************************************************/
void PlayMOD() {
	int count;

	for (count=0; count<channels; count++) {
		GUSSetVolume(count, 0);       			// turn the volume right down
		GUSSetBalance(count, panval[count]); 	// set the default panning
		GUSPlayVoice(count, 0,0,0,0); 			// set all voices at 0
		GUSStopVoice(count);          			// stop all the voices.

		volume[count]=0;						// clear all variables of
		freq[count]=0;							// rubbish
		midival[count]=0;
		lastins[count]=0;
		porto[count]=0;
		portsp[count]=0;
		vibspe[count]=0;
		vibdep[count]=0;
		tremspe[count]=0;
		tremdep[count]=0;
		sinepos[count]=0;
		sineneg[count]=0;
		geffect[count]=0;

	}
	speed = 6;                         // start using default speed 6
	bpm = 125;                         // start using default BPM of 125
	row = 0;                           // start at row 0
	ord = 0;                           // start at order 0
	SetTimerSpeed();                   // set the timer to 125BPM
	oldmodhandler = _dos_getvect(0x8); // save old timer's handler vector
	setvect(0x8, modhandler);          // sit our modplayer on the timer!

	oldkeyhandler = _dos_getvect(0x9); // get int vect for old keyboard handler
	setvect(0x9, keyhandler);          // sit our new handler on the keyboard
}


/****************************************************************************
 *    Name : StopMOD                                                        *
 * Purpose : Shut everything down and stop the music.                       *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : int count - blah you know                                      *
 ****************************************************************************/
void StopMOD() {
	int count=0;

	setvect(0x8,oldmodhandler);       // put the old timer back on its vector
	setvect(0x9,oldkeyhandler);   	  // put the old keyboard handler back
	bpm=0;                        	  // set bpm =0 (code for settimerspeed)
	SetTimerSpeed();              	  // tell settimerspeed to reset to normal
									  // stop all voices
	for (count=0; count<channels; count++) GUSStopVoice(count);
	farfree(patbuff);                 // free the pattern data memory.
}

/****************************************************************************
 *    Name : LoadMOD                                                        *
 * Purpose : The loader                                                     *
 *  Passed : char *filename - the name of the tune to load                  *
 * Returns : TRUE or FALSE - if the mod could be loaded or not              *
 *  Locals : FILE *handle - the file pointer                                *
 *
 ****************************************************************************/
int LoadMOD(char *filename) {
	FILE *handle;
	int count=0, count2=0, pattcount, period;
	dword gusoff, size;
	byte part[4];
	fpos_t pos;

	// if there is no filename extension, then add one
	for (count=0; count<strlen(filename); count++) if (filename[count]=='.') count2=1;
	if (count2==0) strcat(filename,".mod");

	// open the file
	if ((handle = fopen(filename, "rb")) == NULL) return FALSE;

	// get the size of the file for the interface
	fseek(handle, 0, SEEK_END);
	fgetpos(handle, &pos);
	filelen = pos;

	/**********************/
	/***  VERIFICATION  ***/
	/**********************/
	fseek(handle, 1080, SEEK_SET);				// skip to offset 1080
	for (count=0; count<4; count++) part[count]=fgetc(handle);
												// check what MOD format
	if      (strncmp(part, "M.K.",4) == 0) channels = 4;
	else if (strncmp(part, "6CHN",4) == 0) channels = 6;
	else if (strncmp(part, "8CHN",4) == 0) channels = 8;
	else Error(3);                              // not a recognized format
	cprintf("Loading %s...\r\n", filename);

	fseek(handle, 0, SEEK_SET);                 // start at the beginning.
	fread(MOD.name, 20, 1, handle);       		// read in module name.
	for (count2=0; count2<20; count2++) if (MOD.name[count2]<32) MOD.inst[count].name[count2] = 32;

	/*********************************/
	/***  LOAD SAMPLE INFORMATION  ***/
	/*********************************/
	for (count=0; count<31; count++) {          // go through 31 instruments.
		fread(MOD.inst[count].name, 22, 1, handle);
		for (count2=0; count2<22; count2++) if (MOD.inst[count].name[count2]<32) MOD.inst[count].name[count2] = 32;

		part[0] = fgetc(handle);                // get samples length
		part[1] = fgetc(handle);
		MOD.inst[count].length = ((part[0] * 0x100) + part[1]) * 2;

		MOD.inst[count].finetune = fgetc(handle);// get finetune
		if (MOD.inst[count].finetune > 7)
			MOD.inst[count].finetune -= 16;

		MOD.inst[count].volume = fgetc(handle); // get sample default volume

		// get sample loop start
		MOD.inst[count].loopstart = ((fgetc(handle) << 8)+fgetc(handle))*2;
		// get sample loop end
		MOD.inst[count].loopend = ((fgetc(handle) << 8)+fgetc(handle))*2 +
			MOD.inst[count].loopstart;
		// incase the loopend is past the end of the sample fix it
		if (MOD.inst[count].loopend > MOD.inst[count].length)
			MOD.inst[count].loopend = MOD.inst[count].length;
	}

	/**************************************/
	/***  LOAD ORDER AND SIGNATURE DATA ***/
	/**************************************/
	MOD.songLength = fgetc(handle);             // get number of orders.
	fgetc(handle);                              // unused byte, skip it

	MOD.numpats = 0;                            // highest pattern is now 0
	for (count=0; count<128; count++) {
		MOD.order[count] = fgetc(handle);       // get 128 orders.
		if (MOD.order[count] > MOD.numpats)
			MOD.numpats = MOD.order[count];     // get highest pattern.
	}
	size = long(channels)*256L*(MOD.numpats+1); // calculate buffer size
												// try and allocate memory
	if ((patbuff = (char huge *)farmalloc(size)) == NULL) Error(5);
	_fmemset(patbuff, 0, size);                 // wipes buffer clean

	for (count=0; count<4; count++) fgetc(handle);// at 1080 again, skip it

	offset =0;                                  // set buffer offset to 0
	/**************************/
	/***  LOAD PATTERN DATA ***/
	/**************************/
	for (pattcount=0; pattcount <= MOD.numpats; pattcount++) {
		for (row=0; row<64; row++) {   			// loop down through 64 notes.
			for (count=0; count<channels; count++) {
				// point our little note structure to patbuff
				current = (Note *)(patbuff + offset);

				// load up 4 bytes of note information from file
				for (count2=0; count2<4; count2++) part[count2]=fgetc(handle);

				// store sample number
				current -> number = ((part[0] & 0xF0) + (part[2] >> 4));

				// get period
				period = ((part[0] & 0xF) << 8) + part[1];

				// do the look up in the table against what is read in.
				// store note (in midi style format)
				current -> period = -1;
				for (count2=1;count2<37; count2++) {
					if (period > freqtab[count2*8]-3 &&
						period < freqtab[count2*8]+3 )
						current -> period = (count2)*8;
				}
				// store effects and arguments
				current -> effect = part[2] & 0xF; 	// Effect
				current -> eparm = part[3];        	// parameter

				offset += 4;                  		// increment buffer pos
			}
		}
	}

	/*************************/
	/***  LOAD SAMPLE DATA ***/
	/*************************/
	gusoff =0;
	textcolor(BLUE);
	for (count =0; count< 31; count++) putch(178);
	cprintf("\r");
	textcolor(LIGHTBLUE);
	for (count = 0; count <31; count++) {
		putch(178);                             // print little blue bar graph
		MOD.inst[count].offset = gusoff;        // get offsets in GUS memory

		// upload each sample, reading and poking them to GUS byte by byte.
		for (count2=0; count2<MOD.inst[count].length; count2++) {
			GUSPoke(gusoff, fgetc(handle));		// read one byte and poke it.
			gusoff++;                           // increment offset
//             if (gusoff>=gusdram) Error(4); 	// <- seems buggy.
		}
		GUSPoke(gusoff, 0);						// remove ultraclick bugfix
		gusoff++;
	}
	fclose(handle);                             // close mod file
	return TRUE;                                // successful load
}


/***************************************************************************
 ***************************************************************************

					INTERFACE STARTS FROM HERE DOWN

 ***************************************************************************
 ***************************************************************************/
void help(byte panflag, byte pancycle, byte panrow) {
	int count;
	byte min, sec;
	char *buff = (char *)MK_FP(0xb800,0);

	textbackground(0);
	clrscr();
	setrgbpalette(1,0,0,14);
	textcolor(1);
	textsquare(1,1,80,50,1);

	textbox(1,1,79,3,2,"FireMOD 1.05. CopyRight (c) FireLight 1995");
	textcolor(LIGHTBLUE);
	gotoxy(3, 2); cprintf("Name : ");
	gotoxy(35,2); cprintf("Time :");
	gotoxy(49,2); cprintf("Size : ");
	gotoxy(63,2); cprintf("Mem used : ");
	gotoxy(10,2); for (count=0; count<20; count++) eputch(buff,MOD.name[count],10+count,2,1,7);
	textcolor(1);
	gotoxy(56,2); cprintf("%dkb", filelen/1024);
	gotoxy(74,2); cprintf("%dkb", (256L*channels*MOD.numpats)/1024);

	textbox(5,7,75,23,2,"While Playing");
	gotoxy(7,8); cprintf(" +,-         - Change Master volume.");
	gotoxy(7,9); cprintf(" [,]         - Change speed faster, slower.");
	gotoxy(7,10); cprintf(" {,}         - Change BPM slower, faster.");
	gotoxy(7,11); cprintf(" (,)         - Change Defaul Panning left, right.");
	gotoxy(7,12); cprintf(" ESC         - Ummm:)");
	gotoxy(7,13); cprintf(" c           - Pan Cycle - (cycles at speed of song)");
	gotoxy(7,14); cprintf(" d           - DOS shell.");
	gotoxy(7,15); cprintf(" h           - Help.");
	gotoxy(7,16); cprintf(" right arrow - Move forward a pattern.");
	gotoxy(7,17); cprintf(" left arrow  - Move back a pattern.");
	gotoxy(7,18); cprintf(" SPACE       - (Play screen) shuttle scrolly forward faster.");
	gotoxy(7,19); cprintf("             - (MDP style screen) Toggle between channel numbers and");
	gotoxy(7,20); cprintf("               blobs.");
	gotoxy(7,21); cprintf(" ENTER       - Toggle between Player and MDP style screens.");
	gotoxy(7,22); cprintf(" 1 thru 8    - Mute a channel.");

	textbox(5,26,75,28,2,"Anywhere (even dos shell)");
	gotoxy(7,27); cprintf(" F12         - Resets the mod (the timer) to its proper speed.\r\n");

	textbox(5,31,75,38,2,"Contact Information");
	gotoxy(7,32); cprintf(" email       - firelght@yoyo.cc.monash.edu.au");
	gotoxy(7,33); cprintf(" Post        - Brett Paterson,");
	gotoxy(7,34); cprintf("               48/A Parr st,");
	gotoxy(7,35); cprintf("               Leongatha, 3953.");
	gotoxy(7,36); cprintf("               Victoria, Australia.");
	gotoxy(7,37); cprintf(" IRC         - FireLight, on #aussies, #trax or #coders");

	textbackground(7);
	textcolor(1);
	do {
		waitvrt();
		sec = int(clock) % 60;
		min = int(clock) / 60;
		if (min > 59) {
			clock=0;
			sec =0;
			min =0;
		}
		gotoxy(42, 2); cprintf("%02d:%02d", min,sec);
		if (panflag!=0 && row!=panrow) {
			panrow = row;
			if (panval[0] == 0 && pancycle == 1) pancycle=0;
			if (panval[0] == 0xf && pancycle == 0) pancycle=1;
			for (count=0; count<channels; count++) {
				switch(count) {
					case 0:
					case 3:
					case 4:
					case 7: if (pancycle == 0) panval[count] ++;
							else panval[count] --;
							break;
					default: if (pancycle == 0) panval[count] --;
							else panval[count] ++;
				};
				GUSSetBalance(count, panval[count]);
			}
		}
	} while (!kbhit());
	getch();
}


void guidisplay(struct text_info ti, byte &panflag, byte &pancycle, byte &panrow) {
	char key =0, flag =0;
	int x[8] = { 2,2,2,2,2,2,2,2 }, y[8] = { 5,5,5,5,5,5,5,5 };
	int a,b,c,count,count2;
	byte col[8] = { 59,2,3,4,5,60,56,58 };
	byte bar[8] = { 0,0,0,0,0,0,0,0};
	char *buff = (char *)MK_FP(0xb800,0);
	byte min,sec;

start:
	setrgbpalette(0,8,16,24);
	setrgbpalette(1,0,0,14);
	_setcursortype(_NOCURSOR);
	_wscroll = 0;
	directvideo = 1;

	textbackground(0);
	clrscr();
	textcolor(1);
	textsquare(1,1,80,50,1);

	textbox(1,1,79,3,2,"FireMOD 1.05. CopyRight (c) FireLight 1995");
	textcolor(LIGHTBLUE);
	gotoxy(3, 2); cprintf("Name : ");
	gotoxy(35,2); cprintf("Time :");
	gotoxy(49,2); cprintf("Size : ");
	gotoxy(63,2); cprintf("Mem used : ");
	textcolor(1);
	gotoxy(10,2); for (count=0; count<20; count++) eputch(buff,MOD.name[count],10+count,2,1,7);
	gotoxy(56,2); cprintf("%dkb", filelen/1024);
	gotoxy(74,2); cprintf("%dkb", (256L*channels*MOD.numpats)/1024);

	textbackground(0);
	textcolor(WHITE);
	for (count=0; count<31; count++) {
		gotoxy(2,count+6);
		textcolor(7);
		cprintf("%-22s", MOD.inst[count].name);
		textcolor(1);
		cprintf("");
	}
	textcolor(15);
	gotoxy(1,39);
	textbox(1,39,59,48,1,"Bop Bars");
	for (count=0; count<8; count++) { gotoxy(3,count+40); cprintf("%d",count+1); }

	textbox(62,39,79,48,1,"Info");
	textcolor(LIGHTBLUE);
	gotoxy(63,41); cprintf("  Speed :");
	gotoxy(63,42); cprintf("  Order :");
	gotoxy(63,43); cprintf("Pattern :");
	gotoxy(63,44); cprintf("    Row :");
	gotoxy(63,45); cprintf(" Volume :");
	textcolor(1);
	gotoxy(63,47); cprintf("Press H for help");

	for (count=0; count<channels; count++)
		for (count2=0; count2<53; count2++)
			eputch(buff, 254, count2+5, 40+count,1,7);

	do {
		sec = int(clock) % 60;
		min = int(clock) / 60;
		if (min > 59) {
			clock=0;
			sec =0;
			min =0;
		}
		waitvrt();
		textbackground(7);
		textcolor(1);
		gotoxy(42, 2); cprintf("%02d:%02d", min,sec);

		for (count=0; count<channels; count++) {
			a = 22+((908-freq[count])/14);
			if (a<25) a=25;
			if (a>79) a=79;
			b = lastins[count]+6;
			c = volume[count]-1;
			if (c<0) c=0;
			setrgbpalette(col[count], c,16, 24);
			if (x[count]!= a || y[count] != b || key == 32) {
				eputch(buff, 32, x[count], y[count], 0 ,0);
				x[count]=a;
				y[count]=b;
				if (flag == 0) eputch(buff, 219, x[count], y[count], col[count], 0);
				else {
					textbackground(0);
					textcolor(col[count]);
					gotoxy(x[count], y[count]); putch(count+49);
				}
				if (key != 32) bar[count]=volume[count];
			}
			if (bar[count] > 0) bar[count]--;
			if (geffect[count] == 0xA || geffect[count]== 0xC ||
				geffect[count]== 5 || geffect[count]==6 ||
				geffect[count]== 26 || geffect[count]==27)
				bar[count]=volume[count];

			// Blaster volume bar technique (tm) ;)
			if (mute[count] == 0)
				for (count2=1; count2<54; count2++) {
					if (count2 > bar[count]*.875) eputch(buff, 254, count2+4, count+40, 1, 7);
					else eputch(buff, 254, count2+4, count+40, LIGHTBLUE, 7);
			}
			else for (count2=1; count2<54; count2++)
					eputch(buff, 254, count2+4, count+40, 7, 7);
		}
		key =0;
		if (kbhit()) key=getch();
		if (key == char(32)) flag = ~flag;		// flip flag
		if (key == 'h' || key == 'H') {
			help(panflag, pancycle, panrow);
			goto start;
		}
		if (key == '-') mastervol--;
		if (key == '+') mastervol++;
		if (mastervol < 0) mastervol = 0;
		if (mastervol > 64) mastervol = 64;
		if (key == '[') speed--;
		if (speed < 1) speed = 1;
		if (key == ']') speed++;
		if (key == '{') {bpm--; SetTimerSpeed();}
		if (bpm < 32) bpm=32;
		if (key == '}') {bpm++; SetTimerSpeed();}
		if (bpm > 254) bpm=254;
		if (key == char(333)) {
			ord++;
			row=0;
			if (ord >= MOD.songLength) ord = 0;
		}
		if (key == char(331)) {
			ord--;
			row=0;
			if (ord < 0) ord = 0;
		}
		if (key == 'c') {
			panflag = ~panflag;
			if (panflag!=0) {
				panval[0]=0;
				panval[1]=0xf;
				panval[2]=0xf;
				panval[3]=0;
				panval[4]=0;
				panval[5]=0xf;
				panval[6]=0xf;
				panval[7]=0;
			}
		}
		if (key == 'd' || key == 'D') {
			textmode(ti.currmode);
			textbackground(0);
			clrscr();
			setrgbpalette(0,  0, 0, 0);
			setrgbpalette(1,  0, 0,42);
			setrgbpalette(2,  0,42, 0);
			setrgbpalette(7, 42,42,42);
			textcolor(LIGHTBLUE);
			cprintf("Type 'EXIT' to return to FireMOD.\r\n");
			textcolor(15);
			cprintf("(Hit F12 at any time to reset speed if song slows down for no reason)");
			system("COMMAND");
			textmode(64);
			goto start;
		}
		if (key >= char(49) && key <= char(56)) {
			mute[key-49] = ~mute[key-49];
			GUSSetVolume(key-49, volume[key-49]*mastervol/64);
		}
		if (panflag!=0 && row!=panrow) {
			panrow = row;
			if (panval[0] == 0 && pancycle == 1) pancycle=0;
			if (panval[0] == 0xf && pancycle == 0) pancycle=1;
			for (count=0; count<channels; count++) {
				switch(count) {
					case 0:
					case 3:
					case 4:
					case 7: if (pancycle == 0) panval[count] ++;
							else panval[count] --;
							break;
					default: if (pancycle == 0) panval[count] --;
							else panval[count] ++;
				};
				GUSSetBalance(count, panval[count]);
			}
		}

		textcolor(1);
		textbackground(7);
		gotoxy(73,41); cprintf("%02d/%03d",speed,bpm);
		gotoxy(73,42); cprintf("%02d/%02d",ord,MOD.songLength);
		gotoxy(73,43); cprintf("%02d/%02d",MOD.order[ord],MOD.numpats);
		gotoxy(73,44); cprintf("%02d/64",row);
		gotoxy(73,45); cprintf("%02d/64", mastervol);
	} while (key!=char(13) && key!=char(27));
}


void ViewMOD() {
	int count;

	textcolor(15);
	textbackground(5);
	gotoxy(2,18);
	cprintf("S# C# Sample name           Vol FT Length LoopB LoopE  GUSBegn GUSStrt GUSEnd");
	textbackground(0);
	textcolor(7);
	for (count=0; count<31; count++) {
		gotoxy(2,count+19);
		cprintf("%02d    ", count+1);
		cprintf("%-22s", MOD.inst[count].name);
		cprintf(" %2d ", MOD.inst[count].volume);
		cprintf("%2d  ", MOD.inst[count].finetune);
		cprintf("%5u ", MOD.inst[count].length);
		cprintf("%5u ", MOD.inst[count].loopstart);
		cprintf("%5u ", MOD.inst[count].loopend);
		cprintf("%7ld ", MOD.inst[count].offset);
		cprintf("%7ld ", MOD.inst[count].offset+MOD.inst[count].loopstart);
		cprintf("%7ld\r\n", MOD.inst[count].offset+MOD.inst[count].length);
	}
}


void Interface() {
	byte r=32,g=32,b=32, glow=0, min,sec, pancycle=0,panrow=0, panflag=0;
	int count;
	float messcount=0;
	char key, fullchan[8];
	struct text_info ti;
	char *buff = (char *)MK_FP(0xb800, 0), *message, *effstr[33] = {
			"                  ",
			"Porta Up          ",
			"Porta Down        ",
			"Porta to Note     ",
			"Vibrato           ",
			"Porta+Vol Slide   ",
			"Vibrato+Vol Slide ",
			"Tremolo           ",
			"Set Pan           ",
			"Sample Offset     ",
			"Volume Slide      ",
			"Pattern Jump      ",
			"Set Volume        ",
			"Pattern Break     ",
			"                  ",
			"Set Speed         ",
			"(Set Amiga Filter)",
			"Fine Porta Up     ",
			"Fine Porta Down   ",
			"(Glissando Ctrl)  ",
			"(Set Vib Waveform)",
			"Set Fine Tune     ",
			"Pattern Loop      ",
			"(Set Tremolo Wave)",
			"Set 16 Pos Panning",
			"Retrig Note       ",
			"Fine VolSlide Up  ",
			"Fine VolSlide Down",
			"Cut Note          ",
			"Delay Note        ",
			"Pattern Delay     ",
			"(Invert Loop)     ",
			"Arpeggio          ",};

	if ((message = (char *)malloc(2400))==NULL) Error(5);
	for (count=0; count<8; count++) fullchan[count] =0;
	gettextinfo(&ti);

start:
	sprintf(message,"                                                                                 FireMOD 1.%02d.,", version);
	strcat(message," CopyRight (c) FireLight, 1995..........  (Press space");
	strcat(message," to shuttle this text forward/pause will pause interface)  Moo this player is kool");
	strcat(message," huh hehehe.      Dig those GUS clicks....       Hey I");
	strcat(message," like this tune.             I bet this is the only");
	strcat(message," player you've seen with a scrolly in it heheh..   Hey");
	strcat(message," anyone recognize glow here? :) (Referring to this ");
	strcat(message,"epileptic cycling text.)   I am now reasonably happy with");
	strcat(message," the players' performance now, it uses a lot less memory");
	strcat(message," and cpu time than previous versions of fmod did, and it ");
	strcat(message,"plays mods perfectly, apart from not supporting about ");
	strcat(message,"2 minor effects..(set waveforms and glissando).. ");
	strcat(message," Who knows i might implement them.  Anyway, before I run");
	strcat(message," out of space, I better mention the 'OFFICIAL MOD CODERS");
	strcat(message," DOCUMENT'  This is a document I have written over the ");
	strcat(message,"last few months, and is a very comprehensive document ");
	strcat(message,"on HOW to actually code a mod player from scratch, not");
	strcat(message," leaving out any detail.  34 MS Word printed pages of ");
	strcat(message,"text! It covers everything from the loader to interrupt");
	strcat(message," handlers, and the effects section actually explains HOW");
	strcat(message," to code the effects, not just some vague reference on ");
	strcat(message,"what it does.  For example, the description for vibrato,");
	strcat(message," or effect 4xy runs to about 4 to 5 pages of text!  Get");
	strcat(message," it from wherever you got this player as FMODDOC.ZIP...");
	strcat(message,"This is the last version of fmod of this type, the new");
	strcat(message," version I am working on at the moment will have 32 ");
	strcat(message,"channel 9 octave support, including the formats up to");
	strcat(message," 32CH .MOD, .MTM and .S3M.  MTM is already implemented");
	strcat(message," and is working perfectly.  Also the interface will have");
	strcat(message," a radical change to it, as some people have said this");
	strcat(message," interface crashes on their computer.. The new interface");
	strcat(message," will have COWS :) hehe.. look out for FMOD 1.1..    ");
	strcat(message,"  If you want to contact me, about my music or this ");
	strcat(message,"player, then do so by the means displayed in the help or");
	strcat(message," when you press escape to quit.. bye!");
	strcat(message,"                                                                                                  ");

	// INITIALIZE SCREEN
	textmode(64);
	_setcursortype(_NOCURSOR);
	_wscroll = 0;
	directvideo = 1;
	clrscr();

	// SET TYPEMATIC DELAY RATE TO FASTEST (for keyboard)
	asm {
		mov ah, 0x3
		mov al, 0x5
		mov bh, 0
		mov bl, 0
		int 0x16
	}

	// SET UP SOME PALETTE ENTRIES
	setrgbpalette(0,8,16,24);
	setrgbpalette(1,0,0,14);
	setrgbpalette(2,10,10,63);
	setrgbpalette(3,21,42,63);
	setrgbpalette(4,32,5,5);
	setrgbpalette(5,5,10,20);

	// SAMPLE WINDOW
	textcolor(1);
	textsquare(1,1,80,50,1);
	ViewMOD();
	textbox(1,5,60,6+channels,1,"P # Sample  Vol Note Effect ");
	for (count=0; count<channels; count++) {
		textcolor(15);
		gotoxy(3,count+6);
		if (count >= channels) textcolor(8);
		cprintf(" %d", count+1);
		textcolor(15);
	}

	// TITLE BAR
	textbox(1,1,79,3,2,"FireMOD 1.05. CopyRight (c) FireLight 1995");
	textcolor(LIGHTBLUE);
	gotoxy(3, 2); cprintf("Name : ");
	gotoxy(35,2); cprintf("Time :");
	gotoxy(49,2); cprintf("Size : ");
	gotoxy(63,2); cprintf("Mem used : ");
	textcolor(1);
	gotoxy(10,2); for (count=0; count<20; count++) eputch(buff,MOD.name[count],10+count,2,1,7);
	gotoxy(56,2); cprintf("%dkb", filelen/1024);
	gotoxy(74,2); cprintf("%dkb", (256L*channels*MOD.numpats)/1024);

	// INFO BOX
	textbox(62,5,79,14,1,"Info");
	textcolor(LIGHTBLUE);
	gotoxy(63, 7); cprintf("  Speed :");
	gotoxy(63, 8); cprintf("  Order :");
	gotoxy(63, 9); cprintf("Pattern :");
	gotoxy(63,10); cprintf("    Row :");
	gotoxy(63,11); cprintf(" Volume :");
	textcolor(4);
	gotoxy(63,13); cprintf("Press H for help");

	// DRAW LITTLE SQUARES NEXT TO SAMPLE NAMES (to be lit up later)
	for (count=0; count < 31; count++) eputch(buff, 254, 5,count+19,1,0);

	// START MAIN LOOP
	do {
		// THE GLOWING SCROLLER
		if (glow == 0) { g--; b++; if (b>62) glow = 1; }
		if (glow == 1) { b--; r++; if (r>62) glow = 2; }
		if (glow == 2) { r--; g++; if (g>62) glow = 0; }
		setrgbpalette(CYAN, r,g,b);
		waitvrt();
		for (count=0; count<78; count++) eputch(buff, message[count+messcount], count+2, 16, CYAN, 0);
		messcount+= .1;

		// READ A KEY
		key = 0;
		if (kbhit()) key = getch();
		if (key == 'h' || key == 'H') {
			help(panflag, pancycle, panrow);
			goto start;
		}
		if (key == ' ') messcount++;
		if (messcount>strlen(message)-80) messcount=0;

		// MASTER VOLUME CONTROLS
		if (key == '-') mastervol--;
		if (key == '+') mastervol++;
		if (mastervol < 0) mastervol = 0;
		if (mastervol > 64) mastervol = 64;

		// SPEED CONTROLS
		if (key == '[') speed--;
		if (speed < 1) speed = 1;
		if (key == ']') speed++;
		if (key == '{') {bpm--; SetTimerSpeed();}
		if (bpm < 32) bpm=32;
		if (key == '}') {bpm++; SetTimerSpeed();}
		if (bpm > 254) bpm = 254;

		// PAN CYCLE KEY
		if (key == 'c') {
			panflag = ~panflag;
			if (panflag!=0) {
				panval[0]=0;
				panval[1]=0xf;
				panval[2]=0xf;
				panval[3]=0;
				panval[4]=0;
				panval[5]=0xf;
				panval[6]=0xf;
				panval[7]=0;
			}
		}

		// THE ACTUAL PANCYCLE
		if (panflag!=0 && row!=panrow) {
			panrow = row;
			if (panval[0] == 0 && pancycle == 1) pancycle=0;
			if (panval[0] == 0xf && pancycle == 0) pancycle=1;
			for (count=0; count<channels; count++) {
				switch(count) {
					case 0:
					case 3:
					case 4:
					case 7: if (pancycle == 0) panval[count] ++;
							else panval[count] --;
							break;
					default: if (pancycle == 0) panval[count] --;
							else panval[count] ++;
				};
				GUSSetBalance(count, panval[count]);
			}
		}

		// SET PANNING
		if (key == '(') {
			for (count=0; count<channels; count++) {
				switch(count) {
					case 0:
					case 3:
					case 4:
					case 7: panval[count] --;
							break;
					default: panval[count] ++;
				};
				if (panval[count] <0) panval[count]=0;
				if (panval[count] >0xF) panval[count]=0xF;
				GUSSetBalance(count, panval[count]);
			}
		}
		if (key == ')') {
			for (count=0; count<channels; count++) {
				switch(count) {
					case 0:
					case 3:
					case 4:
					case 7: panval[count] ++;
							break;
					default: panval[count] --;
				};
				if (panval[count] <0) panval[count]=0;
				if (panval[count] >0xF) panval[count]=0xF;
				GUSSetBalance(count, panval[count]);
			}
		}

		// LEFT RIGHT ARROW KEYS INCREMENT DECREMENT ORDER
		if (key == char(333)) {
			ord++;
			row=0;
			if (ord >= MOD.songLength) ord = 0;
		}
		if (key == char(331)) {
			ord--;
			row=0;
			if (ord < 0) ord = 0;
		}

		// DOS SHELL
		if (key == 'd' || key == 'D') {
			textmode(ti.currmode);
			textbackground(0);
			setrgbpalette(0,  0, 0, 0);
			setrgbpalette(1,  0, 0,42);
			setrgbpalette(2,  0,42, 0);
			setrgbpalette(7, 42,42,42);
			clrscr();
			textcolor(LIGHTBLUE);
			cprintf("Type 'EXIT' to return to FireMOD.\r\n");
			textcolor(15);
			cprintf("(Hit F12 at any time to reset speed if song slows down for no reason)");
			system("COMMAND");
			goto start;
		}

		// IF RETURN IS PRESSED GO TO GUI (MDP) TYPE SCREEN
		if (key == char(13)) {
			guidisplay(ti, panflag, pancycle, panrow);
			goto start;
		}

		// IF 1 TO 8 IS PRESSED MUTE/UNMUTE CHANNEL
		if (key >= char(49) && key <= char(56)) {
			mute[key-49] = ~mute[key-49];
			GUSSetVolume(key-49, volume[key-49]*mastervol/64);
		}

		// WORK OUT CLOCK
		sec = int(clock) % 60;
		min = int(clock) / 60;
		if (min > 59) {
			clock=0;
			sec =0;
			min =0;
		}

		// UPDATE INFO BOX
		textbackground(7);
		textcolor(1);
		gotoxy(42, 2); cprintf("%02d:%02d", min,sec);
		gotoxy(73, 7); cprintf("%02d/%03d",speed,bpm);
		gotoxy(73, 8); cprintf("%02d/%02d",ord,MOD.songLength);
		gotoxy(73, 9); cprintf("%02d/%02d",MOD.order[ord],MOD.numpats);
		gotoxy(73,10); cprintf("%02d/64",row);
		gotoxy(73,11); cprintf("%02d/64", mastervol);


		// UPDATE CHANNEL SCREEN
		for (count=0; count<channels; count++) {
			if (mute[count] == 0) {
				if (freq[count] > 0) {
					textcolor(1);
					textbackground(7);
					gotoxy(2,6+count);
					cprintf("%X", panval[count]);
					gotoxy(6,6+count);
					cprintf("%02d %-22s  %02d  %3s  %s",
						lastins[count]+1, MOD.inst[lastins[count]].name,
						volume[count], notetab[midival[count]/8],
						effstr[geffect[count]]);
					eputch(buff, 254, 5, fullchan[count]+19, 4,0);
					fullchan[count]=lastins[count];
					eputch(buff, count+49, 5,fullchan[count]+19, YELLOW, 0);
				}
				else {
					textcolor(RED);
					eputch(buff, ' ', 2, 6+count, RED, 7);
					gotoxy(6,6+count);
					cprintf("                                                      ");
				}
			}
			else {
				textcolor(RED);
				eputch(buff, '-', 2, 6+count, RED, 7);
				gotoxy(6,6+count);
				cprintf("-- --------[MUTE]--------  --  ---  ------------------");
			}
		}
	} while(key!=char(27));

	// RESET SCREEN ETC
	_setcursortype(_NORMALCURSOR);
	_wscroll = 1;
	textmode(ti.currmode);
	setrgbpalette(0,  0, 0, 0);
	setrgbpalette(1,  0, 0,42);
	setrgbpalette(2,  0,42, 0);
	setrgbpalette(7, 42,42,42);

	// DRAW FINAL MESSAGE
	clrscr();
	textcolor(15);
	textbackground(1);
	cprintf("FireMOD 1.%02d - CopyRight (c) FireLight, 1995                                    ", version);
	textcolor(7);
	textbackground(0);
	cprintf("You have just used FireMod by FireLight.  I can be contacted below.\r\n");
	cprintf("Post : Brett Paterson,                 Email : firelght@yoyo.cc.monash.edu.au\r\n");
	cprintf("       48/A Parr st,                   IRC : FireLight\r\n");
	cprintf("       Leongatha, 3953.\r\n");
	cprintf("       Victoria, Australia\r\n");

	free(message);
}


/****************************************************************************
 *    Name :                                                                *
 * Purpose :                                                                *
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals :                                                                *
 ****************************************************************************/
void main(int argc, char *argv[]) {
	char *ptr;

	ptr = getenv("ULTRASND");               // grab ULTRASND env variable
	if (ptr == NULL) Error(1);              // if it doesnt exist spit it
	else Base = ((ptr[1]-48)*0x10)+0x200;   // else grab the hex base address

	textcolor(WHITE);
	cprintf("Found GUS at Base Port %x, ", Base);
	gusdram = GUSFindMem();                         // find the amount of gus dram?
	cprintf("with %d kb DRAM\r\n", gusdram/1024+1);
	if (argc > 1) {
		textcolor(WHITE);
		_setcursortype(_NOCURSOR);
		if (LoadMOD(argv[1]) == FALSE) Error(2);  // load in file.
		GUSReset(14);             // initialize GUS with 14 voices (44.1khz)
		PlayMOD();                                // GO!
		Interface();                              // Start the interface.
	}
	else Error(6);
	StopMOD();        // Unhook everything and stop.
}
