/*
 * This library provides the basic functions for initializing and using
 * unchained (planar) 256-color VGA modes.  Currently supported are:
 *
 *    - 256x256
 *    - 320x200
 *    - 320x240
 *    - 360x270
 *    - 400x600
 *
 * Functions are provided for:
 *
 *    - initializing one of the available modes
 *    - setting the start address of the VGA refresh data
 *    - setting active and visible display pages
 *    - writing and reading a single pixel to/from video memory
 *    - writing and reading horisontal lines of pixel data
 *
 * Thanks to Robert Smith for details on the VGA register programming.
 * Written by Arve Holmbo
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <graph.h>
#include <dos.h>
#include <conio.h>

#include "pragmas.h"
#include "vgapage.h"

/*
 * Define the port addresses of some VGA registers.
 */
#define CRTC_ADDR    0x3d4    /* Base port of the CRT Controller (color) */
#define SEQU_ADDR    0x3c4    /* Base port of the Sequencer */
#define GRAC_ADDR    0x3ce    /* Base port of the Graphics Controller */

/*
 * Make a pointer to the VGA graphics buffer segment.
 */
static uchar *vga = (uchar *)(0xA0000);

/*
 * width and height should specify the mode dimensions.  widthBytes
 * specify the width of a line in addressable bytes.
 */
static int width, height, widthBytes;
static int physWidth;  /* pixel resolution width */
/*
 * actStart specifies the start of the page being accessed by
 * drawing operations.  visStart specifies the contents of the Screen
 * Start register, i.e. the start of the visible page.
 */
static int actStart, visStart;
static void ScreenInit( int w, int h, int flag );


/* Use to determine if transparent drawing is on or off
*/
static int  Transparent = 0;




/*
 * VGAPAGEinit: Set up desired VGA mode-x mode
 */
int  VGAPAGEinit( short mode )
{
  int    i, iStat = EXIT_SUCCESS;
  uchar *buf;

  if ( mode == VGAPAGEmode256x256 )        VGAPAGEset256x256();
  else if ( mode == VGAPAGEmode320x200 )   VGAPAGEset320x200();
  else if ( mode == VGAPAGEmode320x240 )   VGAPAGEset320x240();
  else if ( mode == VGAPAGEmode360x270 )   VGAPAGEset360x270();
  else if ( mode == VGAPAGEmode400x300 )   VGAPAGEset400x300();
  else if ( mode == VGAPAGEmode400x600 )   VGAPAGEset400x600();  

  /* Clear the display */
  if (( buf = (uchar *)malloc( width )) != NULL )  
  {
    memset( buf, 0, width );
    for ( i=0; i < height; i++ )  VGAPAGEdrawLine( 0, i, width, buf );
    free( buf );
  }
  else
    iStat = EXIT_FAILURE;
  return  iStat;
}




/* Get paged mode data
 */
VGAPAGEdata  *VGAPAGEgetData( void )
{
  static  VGAPAGEdata  data;

  data.Xres = width;
  data.Yres = height;

  return &data;
}




/*
 * VGAPAGEset320x200()
 *    Set mode 13h, then turn it into an unchained (planar), 4-page
 *    320x200x256 mode.
 */
void VGAPAGEset320x200( void )
{
  _setvideomode( _MRES256COLOR );  /* Set standard VGA lowres 256 color mode*/

  /* Turn off the Chain-4 bit (bit 3 at index 4, port 0x3c4): */
  outpw(SEQU_ADDR, 0x0604);

  /* Turn off word mode, by setting the Mode Control register
   of the CRT Controller (index 0x17, port 0x3d4):
  */
  outpw(CRTC_ADDR, 0xE317);

  /* Turn off doubleword mode, by setting the Underline Location
   register (index 0x14, port 0x3d4):
  */
  outpw(CRTC_ADDR, 0x0014);

  /* Clear entire video memory, by selecting all four planes, then
   writing 0 to entire segment.
  */
  outpw(SEQU_ADDR, 0x0F02);

  /* Update the global variables to reflect dimensions of this
   mode.  This is needed by most future drawing operations.
  */
  ScreenInit( 320, 200, 1 );
}





/*
 * VGAPAGEsetActiveStart() tells our graphics operations which address in
 * video memory should be considered the top left corner.
 */
void VGAPAGEsetActiveStart( int offset )
{
  actStart = offset;
}

/*
 * setXXXPage() sets the specified page by multiplying the page number
 * with the size of one page at the current resolution, then handing the
 * resulting offset value over to the corresponding setXXXStart()
 * function.  The first page is number 0.
 */
void VGAPAGEsetActivePage( int iPage )
{
  VGAPAGEsetActiveStart( iPage * widthBytes * height );
}



/*
 * setVisibleStart() tells the VGA from which byte to fetch the first
 * pixel when starting refresh at the top of the screen.  This version
 * won't look very well in time critical situations (games for
 * instance) as the register outputs are not synchronized with the
 * screen refresh.  This refresh might start when the high byte is
 * set, but before the low byte is set, which produces a bad flicker.
 */
void  VGAPAGEsetVisibleStart( unsigned offset )
{
/*  VGAwaitVertRetrace(); */

  visStart = offset;
  outpw(CRTC_ADDR, 0x0C);      /* set high byte */
  outpw(CRTC_ADDR+1, visStart >> 8);
  outpw(CRTC_ADDR, 0x0D);      /* set low byte */
  outpw(CRTC_ADDR+1, visStart & 0xff);
}
void  VGAPAGEsetVisiblePage( int page )
{
  VGAPAGEsetVisibleStart( page * widthBytes * height );
}




extern  void  _setup_plane( uchar );

#pragma aux _setup_plane =\
  "mov ax, 102h"\
  "mov dx, 3C4h"\
  "shl ah, cl"\
  "out dx, ax"\
  parm [cl] \
  modify [ax dx];




void VGAPAGEputPixel( int x, int y, uchar color )
{
  /* Each address accesses four neighboring pixels, so set
   Write Plane Enable according to which pixel we want
   to modify.  The plane is determined by the two least
   significant bits of the x-coordinate:
  */

  _setup_plane((uchar )( x & 3 ));


  /* The offset of the pixel into the video segment is
   offset = (width * y + x) / 4, and write the given
   color to the plane we selected above.  Heed the active
   page start selection.
  */
  vga[(unsigned )( widthBytes * y ) + (x / 4) + actStart ] = color;
}



/* Copy each 4th byte from src to dest
*/
extern void  _FastCopy( uchar *dest, uchar *src, int cnt );
#pragma inline _FastCopy;
#pragma aux    _FastCopy =\
  "mov  edx, 3"\
  "cld"\
"over:"\
  "movsb"\
  "add  esi, edx"\
  "dec  ecx"\
  "jg   over"\
  parm [edi][esi][ecx]\
  modify [eax ecx edx esi edi];

extern void  _FastCopyTransp( uchar *dest, uchar *src, int cnt );
#pragma inline _FastCopyTransp;
#pragma aux    _FastCopyTransp =\
  "mov  edx, 4"\
"over:"\
  "mov  al, [esi]"\
  "or   al, al"\
  "jz   dont"\
    "mov  [edi], al"\
"dont:"\
  "add  esi, edx"\
  "inc  edi"\
  "dec  ecx"\
  "jg   over"\
  parm [edi][esi][ecx]\
  modify [eax ecx edx esi edi];


/*
  Draw a line of pixels
  */
void  VGAPAGEdrawLine( int x, int y, int iLen, uchar *buf )
{
  /* Each address accesses four neighboring pixels, so set
     Write Plane Enable according to which pixel we want
     to modify.  The plane is determined by the two least
     significant bits of the x-coordinate:
  */
#define BITPLANES  4
  int    i, offset, xlen = iLen;

  y *= widthBytes;   /* get buffer line start address */
  y += actStart;
  for ( i=0; i < BITPLANES; i++ )
  {
    /* cycle thru' all 4 bitplanes
     */
    _setup_plane((uchar )((x+i) & 3 ));
    offset = y + ((x+i)/ BITPLANES);

    if ( Transparent )  _FastCopyTransp( &vga[ offset ], &buf[ i ],
                                       ( iLen-i+(BITPLANES-1))/BITPLANES );
    else                _FastCopy( &vga[ offset ], &buf[ i ],
                                 ( iLen-i+(BITPLANES-1))/BITPLANES );
    if ( --xlen == 0 ) break;
  }
}




/*
  Draw a block of pixels
  */
int  VGAPAGEdrawBlock( int x, int y, int xlen, int ylen, uchar *block )
{
  int     i, clippedLen;

  clippedLen = min( xlen, width  - x );   /* do clipping */
  ylen       = min( ylen, height - y );
  if ( clippedLen <= 0 || ylen <= 0 )  return EXIT_FAILURE;

  Transparent = 0;
  for ( i=0; i < ylen; i++, y++ )
  {
    VGAPAGEdrawLine( x, y, clippedLen, block );
    block += xlen;
  }
  return EXIT_SUCCESS;
}




/* Draw block of pixels transparently
*/
int  VGAPAGEdrawTranspBlock( int x, int y, int xlen, int ylen, uchar *block )
{
  int     i, clippedLen;

  clippedLen = min( xlen, width  - x );   /* do clipping */
  ylen       = min( ylen, height - y );
  if ( clippedLen <= 0 || ylen <= 0 )  return EXIT_FAILURE;

  Transparent = 1;
  for ( i=0; i < ylen; i++, y++ )
  {
    VGAPAGEdrawLine( x, y, clippedLen, block );
    block += xlen;
  }
  return EXIT_SUCCESS;
}







extern  void _select_plane( ushort );

#pragma aux _select_plane =\
  "mov  dx, 3CEh"\
  "mov  ax, 4"\
  "out  dx, ax"\
\
  "inc  dx"\
  "mov  ax, cx"\
  "out  dx, ax"\
  parm [cx] \
  modify [ax dx];




int  VGAPAGEgetPixel( int x, int y )
{
  /* Select the plane from which we must read the pixel color
   */
  uint p;
  
  _select_plane((ushort )( x & 3 ));
  p = (uint )vga[(unsigned )( widthBytes * y ) + (x / 4) + actStart];
  return (int )p;
}




/*
  Copy contents of video buffer to given buffer
  */
void  VGAPAGEgetLine( int x, int y, int iLen, uchar *buf )
{
  /* Select the plane from which we must read the pixel color: */
  int    i, ii, xlen = iLen;
  ushort xx;

  y *= widthBytes;   /* get buffer line start address */
  y += actStart;
  for ( i=0; i < BITPLANES; i++ )
  {
    /* cycle thru' all 4 bitplanes */
    xx = (x+i) & 3;

    _select_plane( xx );
    for ( ii=i; ii < iLen; ii += BITPLANES )
      buf[ ii ] = vga[ y + (x+ii)/ BITPLANES ];
    if ( --xlen == 0 )  break;
  }
}






/*
  Get a block of pixels
  */
int  VGAPAGEgetBlock( int x, int y, int xlen, int ylen, uchar *block )
{
  int     i, clippedLen;

  clippedLen = min( xlen, width  - x );   /* do clipping */
  ylen       = min( ylen, height - y );
  if ( clippedLen <= 0 || ylen <= 0 )  return EXIT_FAILURE;

  for ( i=0; i < ylen; i++, y++ )
  {
    VGAPAGEgetLine( x, y, clippedLen, block );
    block += xlen;
  }
  return EXIT_SUCCESS;
}







void VGAPAGEset320x240( void )
{
  VGAPAGEset320x200();    /* Set the unchained version of mode 13h: */

  /* Modify the vertical sync polarity bits in the Misc. Output
   Register to achieve square aspect ratio: */

  outp( 0x3C2, 0xE3 );

  /* Modify the vertical timing registers to reflect the increased
   vertical resolution, and to center the image as good as
   possible: */
  outpw( CRTC_ADDR, 0x2C11 );   // turn off write protect
  outpw( CRTC_ADDR, 0xD06 );    // vertical total
  outpw( CRTC_ADDR, 0x3E07 );   // overflow register
  outpw( CRTC_ADDR, 0xEA10 );   // vertical retrace start
  outpw( CRTC_ADDR, 0xAC11 );   // vertical retrace end AND wr.prot
  outpw( CRTC_ADDR, 0xDF12 );   // vertical display enable end
  outpw( CRTC_ADDR, 0xE715 );   // start vertical blanking
  outpw( CRTC_ADDR, 0x616 );    // end vertical blanking

  /* Update mode info, so future operations are aware of the resolution
   */
  height = 240;
}



typedef struct
{
  unsigned port;
  uchar    index;
  uchar    value;
} Register;

Register Mode256x256[] =
  {
  { 0x3c2, 0x0, 0xe3},
  { 0x3d4, 0x0, 0x5f},
  { 0x3d4, 0x1, 0x3f},
  { 0x3d4, 0x2, 0x40},
  { 0x3d4, 0x3, 0x82},
  { 0x3d4, 0x4, 0x4a},
  { 0x3d4, 0x5, 0x9a},
  { 0x3d4, 0x6, 0x23},
  { 0x3d4, 0x7, 0xb2},
  { 0x3d4, 0x8, 0x0},
  { 0x3d4, 0x9, 0x61},
  { 0x3d4, 0x10, 0xa},
  { 0x3d4, 0x11, 0xac},
  { 0x3d4, 0x12, 0xff},
  { 0x3d4, 0x13, 0x20},
  { 0x3d4, 0x14, 0x0},
  { 0x3d4, 0x15, 0x7},
  { 0x3d4, 0x16, 0x1a},
  { 0x3d4, 0x17, 0xe3},
  { 0x3c4, 0x1, 0x1},
  { 0x3c4, 0x4, 0x6},
  { 0x3ce, 0x5, 0x40},
  { 0x3ce, 0x6, 0x5},
  { 0x3c0, 0x10, 0x41},
  { 0x3c0, 0x13, 0x0}
};
Register Mode360x270[] =
  {
  { 0x3c2, 0x0, 0xe7},
  { 0x3d4, 0x0, 0x6b},
  { 0x3d4, 0x1, 0x59},
  { 0x3d4, 0x2, 0x5a},
  { 0x3d4, 0x3, 0x8e},
  { 0x3d4, 0x4, 0x5e},
  { 0x3d4, 0x5, 0x8a},
  { 0x3d4, 0x6, 0x30},
  { 0x3d4, 0x7, 0xf0},
  { 0x3d4, 0x8, 0x0},
  { 0x3d4, 0x9, 0x61},
  { 0x3d4, 0x10, 0x20},
  { 0x3d4, 0x11, 0xa9},
  { 0x3d4, 0x12, 0x1b},
  { 0x3d4, 0x13, 0x2d},
  { 0x3d4, 0x14, 0x0},
  { 0x3d4, 0x15, 0x1f},
  { 0x3d4, 0x16, 0x2f},
  { 0x3d4, 0x17, 0xe3},
  { 0x3c4, 0x1, 0x1},
  { 0x3c4, 0x4, 0x6},
  { 0x3ce, 0x5, 0x40},
  { 0x3ce, 0x6, 0x5},
  { 0x3c0, 0x10, 0x41},
  { 0x3c0, 0x13, 0x0}
};

Register Mode400x300[] =
  {
  { 0x3c2, 0x0, 0xeb},
  { 0x3d4, 0x0, 0x89},
  { 0x3d4, 0x1, 0x63},
  { 0x3d4, 0x2, 0x64},
  { 0x3d4, 0x3, 0x8c},
  { 0x3d4, 0x4, 0x6f},
  { 0x3d4, 0x5, 0x9},
  { 0x3d4, 0x6, 0x70},
  { 0x3d4, 0x7, 0xf0},
  { 0x3d4, 0x8, 0x0},
  { 0x3d4, 0x9, 0xe0},
  { 0x3d4, 0x10, 0x58},
  { 0x3d4, 0x11, 0x8c},
  { 0x3d4, 0x12, 0x57},
  { 0x3d4, 0x13, 0x32},
  { 0x3d4, 0x14, 0x0},
  { 0x3d4, 0x15, 0x58},
  { 0x3d4, 0x16, 0x70},
  { 0x3d4, 0x17, 0xe3},
  { 0x3c4, 0x1, 0x1},
  { 0x3c4, 0x2, 0xf},
  { 0x3c4, 0x3, 0x0},
  { 0x3c4, 0x4, 0x6},
  { 0x3ce, 0x0, 0x0},
  { 0x3ce, 0x1, 0x0},
  { 0x3ce, 0x2, 0x0},
  { 0x3ce, 0x3, 0x0},
  { 0x3ce, 0x5, 0x40},
  { 0x3ce, 0x6, 0x5},
  { 0x3ce, 0x7, 0xf},
  { 0x3ce, 0x8, 0xff},
  { 0x3c0, 0x10, 0x41},
  { 0x3c0, 0x11, 0x0},
  { 0x3c0, 0x12, 0xf},
  { 0x3c0, 0x13, 0x0},
  { 0x3c0, 0x14, 0x0}
};
Register Mode400x600[] =
  {
  { 0x3c2, 0x0, 0xeb},
  { 0x3d4, 0x0, 0x89},
  { 0x3d4, 0x1, 0x63},
  { 0x3d4, 0x2, 0x64},
  { 0x3d4, 0x3, 0x8c},
  { 0x3d4, 0x4, 0x6f},
  { 0x3d4, 0x5, 0x9},
  { 0x3d4, 0x6, 0x70},
  { 0x3d4, 0x7, 0xf0},
  { 0x3d4, 0x8, 0x0},
  { 0x3d4, 0x9, 0x60},
  { 0x3d4, 0x10, 0x58},
  { 0x3d4, 0x11, 0x8c},
  { 0x3d4, 0x12, 0x57},
  { 0x3d4, 0x13, 0x32},
  { 0x3d4, 0x14, 0x0},
  { 0x3d4, 0x15, 0x58},
  { 0x3d4, 0x16, 0x70},
  { 0x3d4, 0x17, 0xe3},
  { 0x3c4, 0x1, 0x1},
  { 0x3c4, 0x2, 0xf},
  { 0x3c4, 0x3, 0x0},
  { 0x3c4, 0x4, 0x6},
  { 0x3ce, 0x0, 0x0},
  { 0x3ce, 0x1, 0x0},
  { 0x3ce, 0x2, 0x0},
  { 0x3ce, 0x3, 0x0},
  { 0x3ce, 0x5, 0x40},
  { 0x3ce, 0x6, 0x5},
  { 0x3ce, 0x7, 0xf},
  { 0x3ce, 0x8, 0xff},
  { 0x3c0, 0x10, 0x41},
  { 0x3c0, 0x11, 0x0},
  { 0x3c0, 0x12, 0xf},
  { 0x3c0, 0x13, 0x0},
  { 0x3c0, 0x14, 0x0}
};


#define ATTRCON_ADDR    0x3c0
#define MISC_ADDR       0x3c2
#define VGAENABLE_ADDR  0x3c3
#define SEQ_ADDR        0x3c4
#define GRACON_ADDR     0x3ce
#define STATUS_ADDR     0x3da

static void readyVgaRegs( void )
{
  int v;
  outp(0x3d4,0x11);
  v = inp(0x3d5) & 0x7f;
  outp(0x3d4,0x11);
  outp(0x3d5,v);
}


/*
  outReg sets a single register according to the contents of the
  passed Register structure.
*/
static void outReg( Register r )
{
  switch (r.port)
  {
  /* First handle special cases: */

  case ATTRCON_ADDR:
    inp(STATUS_ADDR);     /* reset read/write flip-flop */
    outp(ATTRCON_ADDR, r.index | 0x20);
                /* ensure VGA output is enabled */
    outp(ATTRCON_ADDR, r.value);
    break;

  case MISC_ADDR:
  case VGAENABLE_ADDR:
    outp(r.port, r.value);  /*    directly to the port */
    break;

  case SEQ_ADDR:
  case GRACON_ADDR:
  case CRTC_ADDR:
  default:          /* This is the default method: */
    outp(r.port, r.index);  /*    index to port        */
    outp(r.port+1, r.value);  /*    value to port+1        */
    break;
  }
}


/*
  outRegArray sets n registers according to the array pointed to by r.
  First, indexes 0-7 of the CRT controller are enabled for writing.
*/
void outRegArray( Register *r, int n )
{
  readyVgaRegs();
  while (n--)   outReg( *r++ );
}





/* Set actual video line lenght in number of pixels.
   The byte count in video memory is (# of pixels)/(4 bitplanes)
*/
void VGAPAGEsetVideoLineLen( int Pixels )
{
  Register r;

  ScreenInit( Pixels, height, 0 );    /* Remember video buffer dims */

  Pixels /= 8;

  r.port  = CRTC_ADDR;
  r.index = 0x13;
  r.value = (uchar )Pixels;
  outRegArray( &r, 1 );
}




/*
  Set up 360x270 256 color mode
  */
void VGAPAGEset360x270( void )
{
  _setvideomode( _MRES256COLOR );  //  Setup default graphics mode
  outRegArray( Mode360x270, sizeof( Mode360x270 )/sizeof( Register ));

  ScreenInit( 360, 270, 1 );
}



/*
  Set up 256x256 256 color mode
  */
void VGAPAGEset256x256( void )
{
  _setvideomode( _MRES256COLOR );  //  Setup default graphics mode
  outRegArray( Mode256x256, sizeof( Mode256x256 )/sizeof( Register ));
  ScreenInit( 256, 256, 1 );
}



/*
  Set up 400x300 256 color mode
  */
void VGAPAGEset400x300( void )
{
  _setvideomode( _MRES256COLOR );  //  Setup default graphics mode
  outRegArray( Mode400x300, sizeof( Mode400x300 )/sizeof( Register ));
  ScreenInit( 400, 300, 1 );
}


/*
  Set up 400x600 256 color mode
  */
void VGAPAGEset400x600( void )
{
  _setvideomode( _MRES256COLOR );  //  Setup default graphics mode
  outRegArray( Mode400x600, sizeof( Mode400x600 )/sizeof( Register ));
  ScreenInit( 400, 600, 1 );
}





/*
  Return current width and height of screen in # of pixels.
  Changed name fro VGAPAGElineLen
  */
int  VGAPAGEscreenWidth( void )
{
  return physWidth;
}

int  VGAPAGEscreenHeight( void )
{
  return height;
}



/*
  Setup default screen parameters.  If the width of a scanline of
  pixels equals the screen pixel horizontal resolution, then
  the flag  LogWidthEquPhysWidth  must be set to 1, else zero.
  */
static void ScreenInit( int w, int h, int LogWidthEquPhysWidth )
{
  width = w, height = h;


  if ( LogWidthEquPhysWidth )
    physWidth = width;


  /* Each byte addresses four pixels, so the width of a scan line
   in *bytes* is one fourth of the number of pixels on a line.
  */
  widthBytes = width / 4;

  /* By default we want screen refreshing and drawing operations
   to be based at offset 0 in the video segment.
  */
  actStart = visStart = 0;  
}







/* Routines to emulate the Microsoft moveto, lineto and setcolor
   graphics routines.  These are here only for completeness.
   */
static uchar VGAPAGEcolor = 0;
static int  VGAPAGEstartX = 0;
static int  VGAPAGEstartY = 0;
void  VGAPAGEsetcolor( uchar Color ) {  VGAPAGEcolor = Color;  }
void  VGAPAGEmoveto( int x, int y )
{
  VGAPAGEstartX = x;
  VGAPAGEstartY = y;
}
/* lineto will only work for horizontal or vertical lines, and is quite
   slow
*/
void  VGAPAGElineto( int x, int y )
{
  int delta = 1;
  int tox = x, toy = y;

  if ( x == VGAPAGEstartX )
  {
    if ( y > VGAPAGEstartY )
      delta = -1;
    while ( y != VGAPAGEstartY )
    {
      VGAPAGEputPixel( x, y, VGAPAGEcolor );
      y += delta;
    }
    VGAPAGEputPixel( x, y, VGAPAGEcolor );  // and the last 1
  }
  else if ( y == VGAPAGEstartY )
  {
    if ( x > VGAPAGEstartX )
      delta = -1;
    while ( x != VGAPAGEstartX )
    {
      VGAPAGEputPixel( x, y, VGAPAGEcolor );
      x += delta;
    }
    VGAPAGEputPixel( x, y, VGAPAGEcolor );  // and the last 1
  }
  VGAPAGEstartX = tox;   /* update new x, y location (place we went to) */
  VGAPAGEstartY = toy;
}
