/*
   Routines to draw to/read display in VGA mode 13hex 256 color mode
   written by Arve Holmbo.
   Released to public domain March 1996.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "vga256.h"
#include "mem.h"
#include "pragmas.h"

#pragma inline memcpy;


typedef void  *(*MemCpyFunc)( void  *, const void *, size_t );


static int drawBlock( int x, int y, int xlen, int ylen,
                      uchar *block, MemCpyFunc Func );


void  VGA256drawLine( int x, int y, int xlen, uchar *p )
{
  int  y2 = y << 6;
  uchar *addrs = (uchar *)0xA0000;
  addrs += ( x + y2 + (y2 << 2));
  memcpy( addrs, p, xlen );  /*Watcom does a good job here, no need for asm*/
}



/* Draw a block of pixels
  */
int  VGA256drawBlock( int x, int y, int xlen, int ylen, uchar *block )
{  return drawBlock( x, y, xlen, ylen, block, memcpy );  }
//{  return drawBlock( x, y, xlen, ylen, block, MEMcopy );  }


/* Similar to VGA256drawBlock, but draws pixels transparently
*/
int  VGA256drawTranspBlock( int x, int y, int xlen, int ylen, uchar *block )
{  return drawBlock( x, y, xlen, ylen, block, MEMtcpy );  }



/* Opposite of VGA256drawBlock
*/
int  VGA256getBlock( int x, int y, int xlen, int ylen, uchar *block )
{
  uchar  *p    = block;
  uchar *addrs = (uchar *)0xA0000;   /* VGA start address */
  int   i, clippedLen;

  // Perform clipping
  clippedLen = min( xlen, 320 - x );
  ylen       = min( ylen, 200 - y );
  if ( ylen <= 0 || clippedLen <= 0 )
    return -1;

  addrs += ( x + 320 * y );    /* Goto start address in video memory */
  for ( i=0; i < ylen; i++, y++ )
  {
    memcpy( p, addrs, clippedLen );
    addrs += 320;   /* Goto next line in display memory */
    p += xlen;      /* Goto next line in buffer */
  }
  return 0;
}




/* Copy the buffer to video memory
*/
static int drawBlock( int x, int y, int xlen, int ylen,
                      uchar *block, MemCpyFunc  MemFunc )
{
  uchar  *p    = block;
  uchar *addrs = (uchar *)0xA0000;   /* VGA start address */
  int   i, clippedLen;
  int   y2 = y << 6;

  // Perform clipping
  clippedLen = min( xlen, 320 - x );
  ylen       = min( ylen, 200 - y );
  if ( ylen <= 0 || clippedLen <= 0 )
    return -1;

  addrs += ( x + y2 + (y2 << 2));    /* Goto start address in video memory */
  for ( i=0; i < ylen; i++, y++ )
  {
    (*MemFunc)( addrs, p, clippedLen );
    addrs += 320;   /* Goto next line in display memory */
    p += xlen;      /* Goto next line in buffer */
  }
  return 0;
}
