/*
   Routines to draw to/read display in EGA/VGA 16 color modes.
   written by Arve Holmbo.
   Released to public domain March 1996.
   Code in this package is based on the excellent material found in
   "Programmers Guide to PC / PS/2 Video Systems" by Richard Wilton.
*/

#include <stdio.h>
#include <stdlib.h>

#define  VGA16_PACKAGE

#include "vga16.h"
#include "pragmas.h"



/* Table to convert an X-ccordinate to bitmask with the bit set
   that corresponds to the pixel in video memory.
   */
uchar  VGA16getBitMask[8] =
{
  0x80, 0x40, 0x20, 0x10, 8, 4, 2, 1
};

static int Transparent = 0;


extern void  _set_reset_reg( void );
extern void  _setbitmask( uchar );
extern void  _repeatsetpix( uchar * );
extern void  _setcolorreg( uchar );
extern void  _reset_regs( void );

#pragma inline _set_reset_reg;
#pragma aux    _set_reset_reg =\
  "mov  dx, 3CEh"\
  "mov  ax, 0F01h" /* Bit plane mask for Enable Set/Reset register # */\
  "out  dx, ax"    /* Outs are costly in terms of # of CPU cycles */\
  "mov  ax, 3"     /* Must keep # of outs as low as possible for speed */\
  "out  dx, ax"\
  modify [ax dx];

#pragma inline _setbitmask;
#pragma aux    _setbitmask =\
  "mov  dx, 3CEh"\
  "mov  al, 8"    /* Bitmask of pixel in video memory byte in ah */\
  "out  dx, ax"\
  modify [ax dx]\
  parm  [ah];


#ifdef OLD
#pragma inline _repeatsetpix;
#pragma aux    _repeatsetpix =\
  "mov  al, [edi]" /* Latch operation: read, then write */\
  "mov  [edi], al"\
  modify [ax]\
  parm  [edi];

#pragma inline _setcolorreg;
#pragma aux    _setcolorreg =\
  "mov  dx, 3CEh"\
  "xor  al, al"    /* Color to set in ah */\
  "out  dx, ax"    /* Update color register */\
  modify [ax dx]\
  parm  [ah];
#endif


#pragma inline _reset_regs;
#pragma aux    _reset_regs =\
  "xor  ax, ax"\
  "mov  dx, 3CEh"\
  "out  dx, ax"  /* Reset set/reset register */\
  "inc  ax"\
  "out  dx, ax"  /* Restore enable set/reset reg. */\
  "mov  al, 3"\
  "out  dx, ax"  /* Data rotate */\
  "mov  ax, 0FF08h"\
  "out  dx, ax"  /* Restore bit mask reg. */\
  modify [ax dx];



/* Set Transparent flag for 16 color VGA modes to on/off.
   With the transparent mode flag on, calls to VGA16drawLine
   will only set non-zero pixel values.
*/
void VGA16setTransparent( int flag )  {  Transparent = flag;  }

/* Setup VGA register */
void  VGA16setupRegs( void )  {  _set_reset_reg();  }

/* restore VGA registers */
void  VGA16resetRegs( void )  {  _reset_regs();  }



/* The _SetFastPix is intended to replace the following code:
      for (; src < end; src += 8, dest++ )
      {
        if ( oldcol != *src )
        {
          oldcol = *src;
          _setcolorreg( *src );   / / New color.  Update color register
        }
        _repeatsetpix( dest );
      }
  The same goes for _SetFastPixTransp with the added test to see if
  *src != 0.
  Nothe that every 8th pixel has the same bit offset in its corresponding
  byte in video mem.
*/
extern void _SetFastPix( uchar *src, uchar *dest, int cnt );
#pragma inline _SetFastPix;
#pragma aux    _SetFastPix =\
  "dec  edi"\
  "xor  al, al"\
  "mov  bl, 16"     /* Set to unused color */\
  "mov  dx, 3CEh"\
"over:"\
  "inc  edi"\
  "mov  ah, [esi]"   /* Color to set in ah */\
  "cmp  ah, bl"\
  "jz   repeat"\
    "mov  bl, ah"\
    "out  dx, ax"    /* Update color register */\
"repeat:"\
  "add  esi, 8"\
  "mov  ah, [edi]"\
  "dec  ecx"\
  "mov  [edi], ah"\
  "jg   over"\
  parm [esi][edi][ecx]\
  modify [esi edi ebx eax ecx edx];

/* Same as above, but transparent mode
*/
extern void _SetFastPixTransp( uchar *src, uchar *dest, int cnt );
#pragma inline _SetFastPixTransp;
#pragma aux    _SetFastPixTransp =\
  "dec  edi"\
  "xor  al, al"\
  "mov  bl, 16"     /* Set to unused color */\
  "mov  dx, 3CEh"\
"over:"\
  "inc  edi"\
  "mov  ah, [esi]"   /* Color to set in ah */\
  "or   ah, ah"\
  "jz   nocol"\
    "cmp  ah, bl"\
    "jz   repeat"\
      "mov  bl, ah"\
      "out  dx, ax"    /* Update color register */\
"repeat:"\
    "mov  ah, [edi]"\
    "nop"\
    "mov  [edi], ah"\
"nocol:"\
  "add  esi, 8"\
  "dec  ecx"\
  "jg   over"\
  parm [esi][edi][ecx]\
  modify [esi edi ebx eax ecx edx];


/*
  Set pixel values in EGA/VGA 16-color graphics mode with values in *buf.
  A horizontal line of pixels is set, starting at (x,y).  Line is xlen
  pixels long.
  A call MUST be made to VGA16setupRegs() prior to calling this function.
  A call MUST be made to VGA16resetRegs() after this call to clean up.
  Works for any 16-color mode there is..
*/
void  VGA16drawLine( int x, uchar *adr, int xlen, uchar *BufP )
{
  uchar *dest;
  uchar *src, *end;
  uchar  bits, oldcol = 16;  /* Just set it to a non-used color */
  int    i, iOffset;

  iOffset = x & 7;       /* offset in first byte in video memory */
  bits    = VGA16getBitMask[ iOffset ];
  end     = BufP+xlen;   /* 1st byte after xlen bytes */

  for ( i=0; i < 8; i++ )  /* Each byte in video mem adrs 8 pixels */
  {
    /* past 1st byte in video memory ? If so add 1
     */
    dest = ( i + iOffset >= 8 ) ? adr+1 : adr;
    src  = BufP+i;

    _setbitmask( bits );

    if ( !Transparent )  _SetFastPix( src, dest, (end-src+7)/8 );
    else                 _SetFastPixTransp( src, dest, (end-src+7)/8 );

    bits = ( bits == 1 ) ? 0x80 : bits >>= 1;  // Keep track of video mem.

    if (--xlen == 0 )  break;
  }
}







/* Draw the block in 640x480 16 color mode
*/
#include <svga.h>

int  VGA16drawBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  int i, clippedLen;
  uchar *adr = (uchar *)( 0xA0000 + y*80 + x/8 );  /* video address */
  SVGAdata *sd = SVGAgetData();

  // Perform clipping
  clippedLen = min( xlen, sd->XPixels - x );
  ylen       = min( ylen, sd->YPixels - y );
  if ( ylen <= 0 || clippedLen <= 0 ) { putch(7);  return EXIT_FAILURE; }

  Transparent = 0;

  VGAwaitVertRetrace();
  VGA16setupRegs();
  for ( i=0; i < ylen; i++, y++ )
  {
    VGA16drawLine( x, adr, clippedLen, src );
    adr += 80;
    src += xlen;
  }
  VGA16resetRegs();
  return EXIT_SUCCESS;
}




/* Draw the block in 640x480 16 color mode, transparent where 
   null-bytes in source
*/
int  VGA16drawTranspBlock( int x, int y, int xlen, int ylen, uchar *src )
{
  int i, clippedLen;
  uchar *adr = (uchar *)( 0xA0000 + y*80 + x/8 );

  // Perform clipping
  clippedLen = min( xlen, 640 - x );
  ylen       = min( ylen, 480 - y );
  if ( ylen <= 0 || clippedLen <= 0 )    return EXIT_FAILURE;

  Transparent = 1;
  VGA16setupRegs();
  for ( i=0; i < ylen; i++, y++ )
  {
    VGA16drawLine( x, adr, clippedLen, src );
    adr += 80;
    src += xlen;
  }
  VGA16resetRegs();
  return EXIT_SUCCESS;
}







extern void _selectbitplane( char );

#pragma aux _selectbitplane =\
  "mov  dx, 3CEh"\
  "mov  al, 4"    /* Initial bit plane #, Read Map Select register # */\
  "out  dx, ax"\
  parm [ah]\
  modify [ax dx];


/* Set the given bitplane 
*/
void VGA16selectBitPlane( char bitPlane )
{
  _selectbitplane( bitPlane );
}




/* Read xlen pixels starting at (x, y) into buffer pointed to by dest.
   Each pixel occupies one byte in supplied memory.  Works for any
   16-color mode there is..
*/
void  VGA16getLine( int x, uchar *adr, int xlen, uchar *dest )
{
  uchar bits, *dest1, *adr1;
  int   i;
  char  iBitPlane;
  static uchar  bitplanes[4] = { 1, 2, 4, 8 };

  memset( dest, 0, xlen );

  /* Get all the pixel values from one plane at a time, thereby
     reducing the number of times we have to switch between planes.
   */
  for ( iBitPlane=0; iBitPlane < 4; iBitPlane++ )
  {
    bits = VGA16getBitMask[ x & 7 ];
    _selectbitplane( iBitPlane );

    for ( i=0, dest1=dest, adr1=adr;  i < xlen;  i++, dest1++ )
    {
      if ( *adr1 & bits ) *dest1 |= bitplanes[ iBitPlane ];
      bits >>= 1;
      if ( bits == 0 )
      {
        adr1++;       /* Get next byte in video memory */
        bits = 0x80;
      }
    }
  }
}








/* Get the required memory block
*/
int  VGA16getBlock( int x, int y, int xlen, int ylen, uchar *dest )
{
  int    i, clippedLen;
  uchar *adr = (uchar *)( 0xA0000 + y*80 + x/8 );  /* Video start addrs */

  // Perform clipping
  clippedLen = min( xlen, 640 - x );
  ylen       = min( ylen, 480 - y );   // Actuall 360 for EGA..
  if ( ylen <= 0 || clippedLen <= 0 )   return EXIT_FAILURE;

  for ( i=0; i < ylen; i++, y++ )
  {
    VGA16getLine( x, adr, clippedLen, dest );
    adr  += 80;
    dest += xlen;
  }
  return EXIT_SUCCESS;
}
