/* VESA package definitions.
   Library routines support VESA SVGA modes of 256 colors.
   Examples:  640x480x256, 800x600x256, 1024x768x256
   These modes require the video card to have at least 1 Meg.
   To check the amount of memory on your card:
   1. Call  VESAinit( mode ) with the appropriate mode
      (example _SVRES256COLOR, see Watcom's graph.h)
   2. Call  ptr = VESAgetData() and check the
      value of ptr->TotalMemory.

   The library package has been compiled with the foll. switches:
      CC=wcc386
      CFLAGS= -3s -zq -bt=dos4g -j -oil+ -ort -oau -s -e6

*/

#ifndef _VESA_
#define _VESA_


#ifdef __cplusplus
extern "C" {    /* Use C linkage when in C++ mode */
#endif

#include <mytypes.h>


#ifndef TRUE
#define TRUE   1
#define FALSE  0
#endif

typedef struct   /* Data returned by the VESAgetData function */
{
  ushort  ModeAttr;      /* VESA mode attributes */
  uchar   WindowAattr;   /* Attributes window A */
  uchar   WindowBattr;   /*   "          "    B */
  ushort  Granularity;   /* Page granularity */
  ushort  WinSize;       /* Window size in kBytes */
  ushort  Asegment;      /* Window A segment */
  ushort  Bsegment;      /*  "     B    "    */
  ushort  BytesPerLine;
  uchar   Planes;        /* Bit planes in this mode */
  uchar   BitsPerPixel;
  uchar   Banks;         /* Number of memory banks */
  uchar   BankSize;
  uchar   NumOfImPages;  /* Number of video pages */
  ushort  XPixels, YPixels; /* Video resolution in x and y */
  char    MemoryModel;   /* VESA memory model of curr. mode */
  ushort  TotalMemory;   /* Total video memory in kB */

  uchar   RedMaskSiz;    
  uchar   RedFieldPos;
  uchar   GreenMaskSiz;
  uchar   GreenFieldPos;
  uchar   BlueMaskSiz;
  uchar   BlueFieldPos;

  uchar   RsvdMaskSiz;
  uchar   RsvdFieldPos;
  uchar   DirColModeInfo;

  int     PhysBasePtr;       /* Linear frame buffer start address */
  int     OffScreenMemOffs;  /* Pointer to start of off-screen memory */
  short   OffScreenMemSiz;   /* # of 1K blocks of off-screen memory */


  short   SVGA16Color;   /* TRUE if a 16 color SVGA mode in use */
  short   VGA16Color;    /* TRUE if standard 16 color EGA/VGA mode in use */
  short   VGA256Color;   /* TRUE if standard 256 color VGA mode 13h in use */
  short   VidModesOffs;  /* Pointer to list of video modes */
  short   VidModesSeg;
  short   OEMinfoOffset; /* Pointer to OEM string (real mode pointer seg:off*/
  short   OEMinfoSegment;
  char    Version[2];    /* VESA version number */
  char    OEMinfo[256];  /* Filled if VESA VBE 2.0 */
} VESAdata;


/*
  Function description:

  VESAsetMode is provided to set a video mode that is not supported by
  the Watcom library (mode 0x10D and onwards).  If you plan on using
  the Watcom graphics drawing functions you should not use VESAsetMode.
  Use instead the code following:

  mode = _SVRES256COLOR;   / * example mode: 800x600 256 color mode * /
  if ( VESAinit( mode, &status ) != EXIT_SUCCESS )
    exit( status );
  _setvideomode( mode );   / * Watcom library function: Set video mode * /

  If you plan on using a video mode not supported by Watcom, use the
  following init code:

  if ( VESAsetMode( mode, linearframebuff, &status ) != EXIT_SUCCESS )
    exit( status );

  An example to try is 800x600 64K color ( mode 0x114 ) where 64k colors
  are available using a (5:6:5) bit scheme.

  VESAqueryMode( short mode, int linearMode, int *VesaStatus ) is similar
  to VESAsetMode except that the mode is not initialised, merely, then
  internal data that the vesa package uses are initialised.  This can
  be used when the application has already set a particular video mode
  and the vesa routines need to be aware of this.
                                                                 
  VESAgetData can be called after init to check various current mode
  parameters. See the VESAdata data structure above.  Note that some of
  the parameter values are not defined for standard EGA/VGA-modes.
  VESAsetPage function sets the given page.
  VESAgetLinearAddr() returns the address to the linear frame buffer.
  VESAsetLogicalScanLen() sets the logical scan length in number of pixels. 
  VESAsetDisplayStart() sets the display start given x and y start location.
  VESAgetModeList returns the list of available VESA graphics modes.  The
  list is terminated by -1 and the user should free the buffer allocated
  for this list.
*/

int   VESAsetMode( short mode, int iUseLinearFrameBuf, int *VesaStatus );
int   VESAqueryMode( short mode, int linearMode, int *VesaStatus );
int   VESAinit( short mode, int *VesaStatus );
int   VESAgetInfo( void );
VESAdata *VESAgetData( void );
void  VESAsetPage( uint vpage );
uint  VESAgetPage( void );
int   VESAfailPrint( int reason );
void  VESAaddressingMode( short mode );
int   VESAgetPal( uint index, uint howmany, uchar *pals );
uchar *VESAgetLinearAddr( void );
int   VESAsetLogicalScanLen( int iLen );
int   VESAsetDisplayStart( int x, int y );
short *VESAgetModeList( void );
void  VESAcleanup( void );

/* The following video modes are supported:

  Watcom mode #   Numeric #   Video resolution  Colors
  _ERESCOLOR         16        EGA 640x350        16
  _VRES16COLOR       18        VGA 640x480        16
  _MRES256COLOR      19        VGA 320x200       256
  _URES256COLOR     256        SVGA 640x400      256
  _VRES256COLOR     257        SVGA 640x480      256
  _SVRES16COLOR     258        SVGA 800x600       16
  _SVRES256COLOR    259        SVGA 800x600      256
  _XRES16COLOR      260        SVGA 1024x768      16
  _XRES256COLOR     261        SVGA 1024x768     256

  The first 3 are not defined VESA modes but are included for completeness.
*/


/* Status values returned by in *VesaStatus by VESAinit:
   If VESAinit returns EXIT_FAILURE, the reason is found by checking the
   value of the *VesaStatus flag
*/
#define  VesaModeValid 0       /* Init Ok and requested mode valid */
#define  VesaModeInValid (-1)  /* VESA mode is not recognised by HW or SW */
#define  VesaInitFail    (-2)  /* graphics card probably not VESA compliant */
#define  VesaModeNotSupported (-3)  /* Mode not yet implemented */
#define  VesaInitFailLFB  (-4) /* Linear frame buffer not avail. for mode */

#ifdef __cplusplus
};
#endif

#endif
