/* Table of standard VGA palette values for 256-color modes
*/
#include <math.h>

#include "mytypes.h"
#include "pragmas.h"

static uchar PALtable[] = 
{
  0x00,0x00,0x00,  0x00,0x00,0xa8,  0x00,0xa8,0x00,  0x00,0xa8,0xa8, /* 3 */
  0xa8,0x00,0x00,  0xa8,0x00,0xa8,  0xa8,0x54,0x00,  0xa8,0xa8,0xa8, /* 7 */
  0x54,0x54,0x54,  0x54,0x54,0xfc,  0x54,0xfc,0x54,  0x54,0xfc,0xfc, /* 11 */
  0xfc,0x54,0x54,  0xfc,0x54,0xfc,  0xfc,0xfc,0x54,  0xfc,0xfc,0xfc, /* 15 */
  0x00,0x00,0x00,  0x14,0x14,0x14,  0x20,0x20,0x20,  0x2c,0x2c,0x2c, /* 19 */
  0x38,0x38,0x38,  0x44,0x44,0x44,  0x50,0x50,0x50,  0x60,0x60,0x60, /* 23 */
  0x70,0x70,0x70,  0x80,0x80,0x80,  0x90,0x90,0x90,  0xa0,0xa0,0xa0, /* 27 */
  0xb4,0xb4,0xb4,  0xc8,0xc8,0xc8,  0xe0,0xe0,0xe0,  0xfc,0xfc,0xfc, /* 31 */
  0x00,0x00,0xfc,  0x40,0x00,0xfc,  0x7c,0x00,0xfc,  0xbc,0x00,0xfc, /* 35 */
  0xfc,0x00,0xfc,  0xfc,0x00,0xbc,  0xfc,0x00,0x7c,  0xfc,0x00,0x40, /* 39 */
  0xfc,0x00,0x00,  0xfc,0x40,0x00,  0xfc,0x7c,0x00,  0xfc,0xbc,0x00, /* 43 */
  0xfc,0xfc,0x00,  0xbc,0xfc,0x00,  0x7c,0xfc,0x00,  0x40,0xfc,0x00, /* 47 */
  0x00,0xfc,0x00,  0x00,0xfc,0x40,  0x00,0xfc,0x7c,  0x00,0xfc,0xbc, /* 51 */
  0x00,0xfc,0xfc,  0x00,0xbc,0xfc,  0x00,0x7c,0xfc,  0x00,0x40,0xfc, /* 55 */
  0x7c,0x7c,0xfc,  0x9c,0x7c,0xfc,  0xbc,0x7c,0xfc,  0xdc,0x7c,0xfc, /* 59 */
  0xfc,0x7c,0xfc,  0xfc,0x7c,0xdc,  0xfc,0x7c,0xbc,  0xfc,0x7c,0x9c, /* 63 */
  0xfc,0x7c,0x7c,  0xfc,0x9c,0x7c,  0xfc,0xbc,0x7c,  0xfc,0xdc,0x7c, /* 67 */
  0xfc,0xfc,0x7c,  0xdc,0xfc,0x7c,  0xbc,0xfc,0x7c,  0x9c,0xfc,0x7c, /* 71 */
  0x7c,0xfc,0x7c,  0x7c,0xfc,0x9c,  0x7c,0xfc,0xbc,  0x7c,0xfc,0xdc, /* 75 */
  0x7c,0xfc,0xfc,  0x7c,0xdc,0xfc,  0x7c,0xbc,0xfc,  0x7c,0x9c,0xfc, /* 79 */
  0xb4,0xb4,0xfc,  0xc4,0xb4,0xfc,  0xd8,0xb4,0xfc,  0xe8,0xb4,0xfc, /* 83 */
  0xfc,0xb4,0xfc,  0xfc,0xb4,0xe8,  0xfc,0xb4,0xd8,  0xfc,0xb4,0xc4, /* 87 */
  0xfc,0xb4,0xb4,  0xfc,0xc4,0xb4,  0xfc,0xd8,0xb4,  0xfc,0xe8,0xb4, /* 91 */
  0xfc,0xfc,0xb4,  0xe8,0xfc,0xb4,  0xd8,0xfc,0xb4,  0xc4,0xfc,0xb4, /* 95 */
  0xb4,0xfc,0xb4,  0xb4,0xfc,0xc4,  0xb4,0xfc,0xd8,  0xb4,0xfc,0xe8, /* 99 */
  0xb4,0xfc,0xfc,  0xb4,0xe8,0xfc,  0xb4,0xd8,0xfc,  0xb4,0xc4,0xfc, /* 103 */
  0x00,0x00,0x70,  0x1c,0x00,0x70,  0x38,0x00,0x70,  0x54,0x00,0x70, /* 107 */
  0x70,0x00,0x70,  0x70,0x00,0x54,  0x70,0x00,0x38,  0x70,0x00,0x1c, /* 111 */
  0x70,0x00,0x00,  0x70,0x1c,0x00,  0x70,0x38,0x00,  0x70,0x54,0x00, /* 115 */
  0x70,0x70,0x00,  0x54,0x70,0x00,  0x38,0x70,0x00,  0x1c,0x70,0x00, /* 119 */
  0x00,0x70,0x00,  0x00,0x70,0x1c,  0x00,0x70,0x38,  0x00,0x70,0x54, /* 123 */
  0x00,0x70,0x70,  0x00,0x54,0x70,  0x00,0x38,0x70,  0x00,0x1c,0x70, /* 127 */
  0x38,0x38,0x70,  0x44,0x38,0x70,  0x54,0x38,0x70,  0x60,0x38,0x70, /* 131 */
  0x70,0x38,0x70,  0x70,0x38,0x60,  0x70,0x38,0x54,  0x70,0x38,0x44, /* 135 */
  0x70,0x38,0x38,  0x70,0x44,0x38,  0x70,0x54,0x38,  0x70,0x60,0x38, /* 139 */
  0x70,0x70,0x38,  0x60,0x70,0x38,  0x54,0x70,0x38,  0x44,0x70,0x38, /* 143 */
  0x38,0x70,0x38,  0x38,0x70,0x44,  0x38,0x70,0x54,  0x38,0x70,0x60, /* 147 */
  0x38,0x70,0x70,  0x38,0x60,0x70,  0x38,0x54,0x70,  0x38,0x44,0x70, /* 151 */
  0x50,0x50,0x70,  0x58,0x50,0x70,  0x60,0x50,0x70,  0x68,0x50,0x70, /* 155 */
  0x70,0x50,0x70,  0x70,0x50,0x68,  0x70,0x50,0x60,  0x70,0x50,0x58, /* 159 */
  0x70,0x50,0x50,  0x70,0x58,0x50,  0x70,0x60,0x50,  0x70,0x68,0x50, /* 163 */
  0x70,0x70,0x50,  0x68,0x70,0x50,  0x60,0x70,0x50,  0x58,0x70,0x50, /* 167 */
  0x50,0x70,0x50,  0x50,0x70,0x58,  0x50,0x70,0x60,  0x50,0x70,0x68, /* 171 */
  0x50,0x70,0x70,  0x50,0x68,0x70,  0x50,0x60,0x70,  0x50,0x58,0x70, /* 175 */
  0x00,0x00,0x40,  0x10,0x00,0x40,  0x20,0x00,0x40,  0x30,0x00,0x40, /* 179 */
  0x40,0x00,0x40,  0x40,0x00,0x30,  0x40,0x00,0x20,  0x40,0x00,0x10, /* 183 */
  0x40,0x00,0x00,  0x40,0x10,0x00,  0x40,0x20,0x00,  0x40,0x30,0x00, /* 187 */
  0x40,0x40,0x00,  0x30,0x40,0x00,  0x20,0x40,0x00,  0x10,0x40,0x00, /* 191 */
  0x00,0x40,0x00,  0x00,0x40,0x10,  0x00,0x40,0x20,  0x00,0x40,0x30, /* 195 */
  0x00,0x40,0x40,  0x00,0x30,0x40,  0x00,0x20,0x40,  0x00,0x10,0x40, /* 199 */
  0x20,0x20,0x40,  0x28,0x20,0x40,  0x30,0x20,0x40,  0x38,0x20,0x40, /* 203 */
  0x40,0x20,0x40,  0x40,0x20,0x38,  0x40,0x20,0x30,  0x40,0x20,0x28, /* 207 */
  0x40,0x20,0x20,  0x40,0x28,0x20,  0x40,0x30,0x20,  0x40,0x38,0x20, /* 211 */
  0x40,0x40,0x20,  0x38,0x40,0x20,  0x30,0x40,0x20,  0x28,0x40,0x20, /* 215 */
  0x20,0x40,0x20,  0x20,0x40,0x28,  0x20,0x40,0x30,  0x20,0x40,0x38, /* 219 */
  0x20,0x40,0x40,  0x20,0x38,0x40,  0x20,0x30,0x40,  0x20,0x28,0x40, /* 223 */
  0x2c,0x2c,0x40,  0x30,0x2c,0x40,  0x34,0x2c,0x40,  0x3c,0x2c,0x40, /* 227 */
  0x40,0x2c,0x40,  0x40,0x2c,0x3c,  0x40,0x2c,0x34,  0x40,0x2c,0x30, /* 231 */
  0x40,0x2c,0x2c,  0x40,0x30,0x2c,  0x40,0x34,0x2c,  0x40,0x3c,0x2c, /* 235 */
  0x40,0x40,0x2c,  0x3c,0x40,0x2c,  0x34,0x40,0x2c,  0x30,0x40,0x2c, /* 239 */
  0x2c,0x40,0x2c,  0x2c,0x40,0x30,  0x2c,0x40,0x34,  0x2c,0x40,0x3c, /* 243 */
  0x2c,0x40,0x40,  0x2c,0x3c,0x40,  0x2c,0x34,0x40,  0x2c,0x30,0x40, /* 247 */
  0x54,0x54,0x54,  0x54,0x54,0xfc,  0x54,0xfc,0x54,  0x54,0xfc,0xfc, /* 251 */
  0xfc,0x54,0x54,  0xfc,0x54,0xfc,  0xfc,0xfc,0x54,  0xfc,0xfc,0xfc  /* 255 */
};


/* Table to be customised by the user
*/
static uchar PALtableCustom[] = 
{
  0x00,0x00,0x00,  0x00,0x00,0xa8,  0x00,0xa8,0x00,  0x00,0xa8,0xa8, /* 3 */
  0xa8,0x00,0x00,  0xa8,0x00,0xa8,  0xa8,0x54,0x00,  0xa8,0xa8,0xa8, /* 7 */
  0x54,0x54,0x54,  0x54,0x54,0xfc,  0x54,0xfc,0x54,  0x54,0xfc,0xfc, /* 11 */
  0xfc,0x54,0x54,  0xfc,0x54,0xfc,  0xfc,0xfc,0x54,  0xfc,0xfc,0xfc, /* 15 */
  0x00,0x00,0x00,  0x14,0x14,0x14,  0x20,0x20,0x20,  0x2c,0x2c,0x2c, /* 19 */
  0x38,0x38,0x38,  0x44,0x44,0x44,  0x50,0x50,0x50,  0x60,0x60,0x60, /* 23 */
  0x70,0x70,0x70,  0x80,0x80,0x80,  0x90,0x90,0x90,  0xa0,0xa0,0xa0, /* 27 */
  0xb4,0xb4,0xb4,  0xc8,0xc8,0xc8,  0xe0,0xe0,0xe0,  0xfc,0xfc,0xfc, /* 31 */
  0x00,0x00,0xfc,  0x40,0x00,0xfc,  0x7c,0x00,0xfc,  0xbc,0x00,0xfc, /* 35 */
  0xfc,0x00,0xfc,  0xfc,0x00,0xbc,  0xfc,0x00,0x7c,  0xfc,0x00,0x40, /* 39 */
  0xfc,0x00,0x00,  0xfc,0x40,0x00,  0xfc,0x7c,0x00,  0xfc,0xbc,0x00, /* 43 */
  0xfc,0xfc,0x00,  0xbc,0xfc,0x00,  0x7c,0xfc,0x00,  0x40,0xfc,0x00, /* 47 */
  0x00,0xfc,0x00,  0x00,0xfc,0x40,  0x00,0xfc,0x7c,  0x00,0xfc,0xbc, /* 51 */
  0x00,0xfc,0xfc,  0x00,0xbc,0xfc,  0x00,0x7c,0xfc,  0x00,0x40,0xfc, /* 55 */
  0x7c,0x7c,0xfc,  0x9c,0x7c,0xfc,  0xbc,0x7c,0xfc,  0xdc,0x7c,0xfc, /* 59 */
  0xfc,0x7c,0xfc,  0xfc,0x7c,0xdc,  0xfc,0x7c,0xbc,  0xfc,0x7c,0x9c, /* 63 */
  0xfc,0x7c,0x7c,  0xfc,0x9c,0x7c,  0xfc,0xbc,0x7c,  0xfc,0xdc,0x7c, /* 67 */
  0xfc,0xfc,0x7c,  0xdc,0xfc,0x7c,  0xbc,0xfc,0x7c,  0x9c,0xfc,0x7c, /* 71 */
  0x7c,0xfc,0x7c,  0x7c,0xfc,0x9c,  0x7c,0xfc,0xbc,  0x7c,0xfc,0xdc, /* 75 */
  0x7c,0xfc,0xfc,  0x7c,0xdc,0xfc,  0x7c,0xbc,0xfc,  0x7c,0x9c,0xfc, /* 79 */
  0xb4,0xb4,0xfc,  0xc4,0xb4,0xfc,  0xd8,0xb4,0xfc,  0xe8,0xb4,0xfc, /* 83 */
  0xfc,0xb4,0xfc,  0xfc,0xb4,0xe8,  0xfc,0xb4,0xd8,  0xfc,0xb4,0xc4, /* 87 */
  0xfc,0xb4,0xb4,  0xfc,0xc4,0xb4,  0xfc,0xd8,0xb4,  0xfc,0xe8,0xb4, /* 91 */
  0xfc,0xfc,0xb4,  0xe8,0xfc,0xb4,  0xd8,0xfc,0xb4,  0xc4,0xfc,0xb4, /* 95 */
  0xb4,0xfc,0xb4,  0xb4,0xfc,0xc4,  0xb4,0xfc,0xd8,  0xb4,0xfc,0xe8, /* 99 */
  0xb4,0xfc,0xfc,  0xb4,0xe8,0xfc,  0xb4,0xd8,0xfc,  0xb4,0xc4,0xfc, /* 103 */
  0x00,0x00,0x70,  0x1c,0x00,0x70,  0x38,0x00,0x70,  0x54,0x00,0x70, /* 107 */
  0x70,0x00,0x70,  0x70,0x00,0x54,  0x70,0x00,0x38,  0x70,0x00,0x1c, /* 111 */
  0x70,0x00,0x00,  0x70,0x1c,0x00,  0x70,0x38,0x00,  0x70,0x54,0x00, /* 115 */
  0x70,0x70,0x00,  0x54,0x70,0x00,  0x38,0x70,0x00,  0x1c,0x70,0x00, /* 119 */
  0x00,0x70,0x00,  0x00,0x70,0x1c,  0x00,0x70,0x38,  0x00,0x70,0x54, /* 123 */
  0x00,0x70,0x70,  0x00,0x54,0x70,  0x00,0x38,0x70,  0x00,0x1c,0x70, /* 127 */
  0x38,0x38,0x70,  0x44,0x38,0x70,  0x54,0x38,0x70,  0x60,0x38,0x70, /* 131 */
  0x70,0x38,0x70,  0x70,0x38,0x60,  0x70,0x38,0x54,  0x70,0x38,0x44, /* 135 */
  0x70,0x38,0x38,  0x70,0x44,0x38,  0x70,0x54,0x38,  0x70,0x60,0x38, /* 139 */
  0x70,0x70,0x38,  0x60,0x70,0x38,  0x54,0x70,0x38,  0x44,0x70,0x38, /* 143 */
  0x38,0x70,0x38,  0x38,0x70,0x44,  0x38,0x70,0x54,  0x38,0x70,0x60, /* 147 */
  0x38,0x70,0x70,  0x38,0x60,0x70,  0x38,0x54,0x70,  0x38,0x44,0x70, /* 151 */
  0x50,0x50,0x70,  0x58,0x50,0x70,  0x60,0x50,0x70,  0x68,0x50,0x70, /* 155 */
  0x70,0x50,0x70,  0x70,0x50,0x68,  0x70,0x50,0x60,  0x70,0x50,0x58, /* 159 */
  0x70,0x50,0x50,  0x70,0x58,0x50,  0x70,0x60,0x50,  0x70,0x68,0x50, /* 163 */
  0x70,0x70,0x50,  0x68,0x70,0x50,  0x60,0x70,0x50,  0x58,0x70,0x50, /* 167 */
  0x50,0x70,0x50,  0x50,0x70,0x58,  0x50,0x70,0x60,  0x50,0x70,0x68, /* 171 */
  0x50,0x70,0x70,  0x50,0x68,0x70,  0x50,0x60,0x70,  0x50,0x58,0x70, /* 175 */
  0x00,0x00,0x40,  0x10,0x00,0x40,  0x20,0x00,0x40,  0x30,0x00,0x40, /* 179 */
  0x40,0x00,0x40,  0x40,0x00,0x30,  0x40,0x00,0x20,  0x40,0x00,0x10, /* 183 */
  0x40,0x00,0x00,  0x40,0x10,0x00,  0x40,0x20,0x00,  0x40,0x30,0x00, /* 187 */
  0x40,0x40,0x00,  0x30,0x40,0x00,  0x20,0x40,0x00,  0x10,0x40,0x00, /* 191 */
  0x00,0x40,0x00,  0x00,0x40,0x10,  0x00,0x40,0x20,  0x00,0x40,0x30, /* 195 */
  0x00,0x40,0x40,  0x00,0x30,0x40,  0x00,0x20,0x40,  0x00,0x10,0x40, /* 199 */
  0x20,0x20,0x40,  0x28,0x20,0x40,  0x30,0x20,0x40,  0x38,0x20,0x40, /* 203 */
  0x40,0x20,0x40,  0x40,0x20,0x38,  0x40,0x20,0x30,  0x40,0x20,0x28, /* 207 */
  0x40,0x20,0x20,  0x40,0x28,0x20,  0x40,0x30,0x20,  0x40,0x38,0x20, /* 211 */
  0x40,0x40,0x20,  0x38,0x40,0x20,  0x30,0x40,0x20,  0x28,0x40,0x20, /* 215 */
  0x20,0x40,0x20,  0x20,0x40,0x28,  0x20,0x40,0x30,  0x20,0x40,0x38, /* 219 */
  0x20,0x40,0x40,  0x20,0x38,0x40,  0x20,0x30,0x40,  0x20,0x28,0x40, /* 223 */
  0x2c,0x2c,0x40,  0x30,0x2c,0x40,  0x34,0x2c,0x40,  0x3c,0x2c,0x40, /* 227 */
  0x40,0x2c,0x40,  0x40,0x2c,0x3c,  0x40,0x2c,0x34,  0x40,0x2c,0x30, /* 231 */
  0x40,0x2c,0x2c,  0x40,0x30,0x2c,  0x40,0x34,0x2c,  0x40,0x3c,0x2c, /* 235 */
  0x40,0x40,0x2c,  0x3c,0x40,0x2c,  0x34,0x40,0x2c,  0x30,0x40,0x2c, /* 239 */
  0x2c,0x40,0x2c,  0x2c,0x40,0x30,  0x2c,0x40,0x34,  0x2c,0x40,0x3c, /* 243 */
  0x2c,0x40,0x40,  0x2c,0x3c,0x40,  0x2c,0x34,0x40,  0x2c,0x30,0x40, /* 247 */
  0x54,0x54,0x54,  0x54,0x54,0xfc,  0x54,0xfc,0x54,  0x54,0xfc,0xfc, /* 251 */
  0xfc,0x54,0x54,  0xfc,0x54,0xfc,  0xfc,0xfc,0x54,  0xfc,0xfc,0xfc  /* 255 */
};



static uchar  *currPal = PALtable;


/* Get  current palette
*/
uchar *PALgetTable( void )
{
  return  currPal;
}


/* Define new palette
*/
uchar  *PALsetCustomTable( uchar *palValues )
{
  int   i;
  char *p, tmp;

  currPal = PALtableCustom;
  memcpy( currPal, palValues, 3 * 256 );
  for ( p=currPal, i=0; i < 3*256; i+=3 )
  {
    tmp = p[i];   /* Swap R and B values */
    p[i] = p[i+2];
    p[i+2] = tmp;
  }
  return  currPal;
}

/* Set default palette
*/
uchar  *PALsetDefaultTable( void )
{
  currPal = PALtable;
  return  currPal;
}




typedef struct
{
  uchar r,g,b;
} RGBtype;


/* Find arithmetic distance between two given colors
*/
static int  Dist( RGBtype *p, RGBtype *v )
{
  int r1 = p->r;  /* Make casts from unsigned to signed */
  int r2 = v->r;
  int g1 = p->g;
  int g2 = v->g;
  int b1 = p->b;
  int b2 = v->b;
  int dist;

  r1 -= r2;  g1 -= g2;  b1 -= b2;
  dist = (int )sqrt( r1*r1 + b1*b1 + g1*g1 );
  return  dist;
}


/* Take the input palette and create an array of
   indexes into the standard VGA palette for each
   of the entries in the input palette.
   The index array is updated.
   The caller can then remap the pixel values as:
   Closest vga entry = Index[ current entry ];
   */
int PALgetRemap( uchar *pals, /* I */
                 int   *index /* I/O */)
{
  int i, iCnt, j, dist, best_dist;
  uchar  *vga = PALsetDefaultTable();
  RGBtype  *p, *v;

//  FILE *fp;
//  fp = fopen( "pals", "w" );
//  fprintf( fp, "Vga ind,  Pals ind\n" );

  p = (RGBtype *)pals;
  v = (RGBtype *)vga;
  for ( iCnt=i=0; i < 256; i++ )
  {
    index[i] = i;   /* Initialize this entry */
    best_dist = 50;
    for ( j=0; j < 256; j++ )
    {
      if (( dist = Dist( &p[i], &v[j] )) < best_dist )
      {
        best_dist = dist;
        index[i] = j;   /* Found close match */
      }
    }

//    fprintf( fp, "% 3d % 3d % 3d [% 3d],\t % 3d % 3d % 3d [% 3d] ",
//      v[ index[i] ].r, v[ index[i] ].g, v[ index[i] ].b, index[i],
//      p[ i ].r, p[ i ].g, p[ i ].b, i );
//
//    if ( index[ i ] != i )
//      fprintf( fp, " <----- change"), iCnt++;
//    fprintf( fp, "\n" );
  }

//  fprintf( fp, "\n %d changes.\n", iCnt );
//  fclose( fp );

  return 0;
}



/* Find the closest match betw. the colors from index 16-256 to the
   16 primary colors and store their indices into the 
   PAL256to16Table.  Then for a color in 256 color mode find its
   nearest color in 16-color mode from the table.
   */
static uchar  PAL256to16Table[ 256 ] = { 0 };

static void  PALreduce( void )
{
  int     i, j, iBest, best_dist, dist;
  RGBtype *p;

  p = (RGBtype *)PALgetTable();
  for ( j=0;  j < 16; j++ )  PAL256to16Table[ j ] = j;  /* Same of course */

  for ( j=16; j < 256; j++ )
  {
    iBest = 0;
    best_dist = 250;   /* Some odd number far out */
    for ( i=0; i < 16; i++ )
    {
      if (( dist = Dist( &p[i], &p[j] )) < best_dist )
      {
        best_dist = dist;
        iBest = i;   /* Found close match */
      }
    }
    PAL256to16Table[ j ] = (uchar )iBest;
  }
}


/* For a given index into the palette in 256 color mode, return
   the index of the 'closest' color in 16-color mode
   */
uchar PALget16colorIndex( uchar index )
{
  static int first = 1;
  uchar  c;

  if ( first )
  {
    first = 0;
    PALreduce();
  }
  if ( index )
    c = PAL256to16Table[ index ];
  else
    c = 0;
  return c;
}
