/*********************************************
  VMOUSE:  package containing routines to
  emulate mouse cursor in software for most
  PC video graphics modes.
  Copyright 1996 by Arve Holmb

**********************************************/

/*
 * Function prototypes/definitions for Mouse cursor control founctions
 *
 */
#ifndef _VMOUSE_H_
#define _VMOUSE_H_

#ifdef __cplusplus
extern "C" {    /* Use C linkage when in C++ mode */
#endif


#include "mytypes.h"

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif


/*
  Data that will be global and visible to the application
  */
#ifdef   VMOUSE_PACKAGE
#define  VMOUSEint  0x33     /* Mouse interrupt */
#define  VMOUSEkeyInt  0x9   /* Keyboard hardware interrupt */
#endif


/* Type used to pass parameters to the VMOUSEinit function.
   The  UseWatcomLib flag is set if the application wants VMOUSE to use the
   Watcom run-time library to draw the cursor primitives.  This is seldom
   used (by me) and not to be used in linear frame buffer mode.
   UseLFB is set to use linear frame buffer support.
   SetVideoMode is set if the application wants VMOUSEinit to set the given
   graphics mode.
   UseWatcomLib and UseLFB are mutually exclusive.
   The UseLFB may be set if the SetVideoMode flag is set.
   
   Note that the SetVideoMode flag allows the application to initialise the
   mouse either before or after a video mode has been set.  This adds some
   flexiblity.  One could then call SVGAinit prior to calling VMOUSEinit
   with the SetVideoMode flag set to false.

   AutoRedraw enables the mouse cursor to be redrawn even though the program
   is busy doing something else, like reading a file.  However, for 16-color
   and banked video modes it is best setting this to zero.
*/
typedef struct
{
  short  UseWatcomLib;
  short  UseLFB;
  short  SetVideoMode;
} VMOUSEinitType;



/* Type used to define custom cursor shapes.
   Use in conjunction with the VMOUSEsetCustomCursor()
   function.
*/
typedef struct
{
  int    xsiz, ysiz; /* Size of cursor rectangle */
  int    xhot, yhot; /* offset values for hotspot, from Upper left corner*/
  uchar *buf;        /* Buffer holding image of cursor rectangle */
} VMOUSEcurShape;


/*
 * Mouse status (really change) codes
 */
#define  VMOUSEnonEvent   0      /* Dummy, no event */
#define  VMOUSEmoved      0x1    /* Mouse was moved */
#define  VMOUSEleftPres   0x2    /* Left button pressed */
#define  VMOUSEleftRele   0x4    /* Left button released */
#define  VMOUSErightPres  0x8    /* Right button pressed */
#define  VMOUSErightRele  0x10   /* Left button released */
#define  VMOUSEmidPres    0x20   /* Middle button pressed */
#define  VMOUSEmidRele    0x40   /* Middle button released */


/* Defines for cursor shapes to be used with the
   VMOUSEsetCursor() function
   */
#define  VMOUSEcurPointer   0    /* Standard mouse pointer */
#define  VMOUSEcurClock     1    /* Clock cursor */
#define  VMOUSEcurQuestion  2    /* Question mark */
#define  VMOUSEcurCrossHair 3    /* Cross hair cursor */
#define  VMOUSEcurHand      4
#define  VMOUSEcursors      5    /* Number of built in cursors */


/* Function prototypes follow here
*/

/* VMOUSEinit initializes the mouse.  If success, EXIT_SUCCESS is
   returned, else EXIT_FAILURE is returned.
   The VGAmode is the desired video mode and one of those listed
   in SVGA.h.
   If the UseLFB flag is set, then graphics drawing will be done
   to a linear frame buffer if the graphics card or emulation software
   supports it.
   */
int  VMOUSEinit( short VGAmode,         /* Video mode */
                 VMOUSEinitType *init,  /* Various input parameters */
                 int minX, int maxX, int minY, int maxY /* mouse region */);

void     VMOUSErestore( void );
void     VMOUSEshow( void );
void     VMOUSEhide( void );
int      VMOUSEhidden( void );
void     VMOUSEsetPos( int, int );
void     VMOUSEsetHorisontalRange( int minX, int maxX );
void     VMOUSEsetVerticalRange( int minY, int maxY );
void     VMOUSEdrawCursor( void );
int      VMOUSEsetInterruptRate( int rate );
void     VMOUSEautoOn( void );
void     VMOUSEautoOff( void );


/* VMOUSEgetEvent returns buffered status codes.  If no event,
   then zero is returned.  Also returned is mouse pointer coordinates
*/
uint     VMOUSEgetEvent( int *x, int *y );
void     VMOUSEgetCoord( int *x, int *y );  /* Coords of mouse pointer */
int      VMOUSEsetCursor( int Cursor );
int      VMOUSEsetCustomCursor( VMOUSEcurShape *Shape );
int      VMOUSEsetCustomCurFullColor( VMOUSEcurShape *shape );

/* VMOUSEinvertCursor() inverts colors of the currently used standard
   cursor shape.  A second call will convert back to original.
   256 color modes only. Color indexes 254, 255 are used.
   Status is returned (EXIT_SUCCESS or EXIT_FAILURE)
*/
int     VMOUSEinvertCursor( void );


/* Functions to support cursor animation.
   Dont set iRefresh=TRUE if there is a chance the display is being
   repainted when the animation ISR is called.
*/
int  VMOUSEanimateCursor( VMOUSEcurShape *shapes,  /* Cursor images */
                          int iCnt,    /* Number of shapes to animate */
                          int iFreq,   /* Number of frames/second */
                          int iRefresh /* Drawing cursor while in ISR */);
void  VMOUSEanimateCurStop( void );




/* Flag to be ORed in with the mode parameter in the call to 
   VMOUSEinit if one wishes to use Watcom library calls to draw the
   cursor on screen instead of the built in routines.
   Those routines are slow, however, and should only be used for debugging.
   E.g.  iStat = VMOUSEinit( mode & VMOUSEuseWatcom, ... );
   This is no longer in use.
*/
#define  VMOUSEuseWatcom  0x8000




#ifdef __cplusplus
};
#endif

#endif
