/* SVGA package definitions.
   Library routines support VGA/SVGA modes of 16/256 colors.
   Examples:  640x480x 16 or 256, 800x600x 16 or 256, 1024x768x 16 or 256
   These modes require the video card to have at least 1 Meg.

   The library package has been compiled with the foll. switches:
      CC=wcc386
      CFLAGS= -3s -zq -bt=dos4g -j -oil+ -ort -oau -s -e6
*/

#ifndef _SVGA_H_
#define _SVGA_H_

#ifdef __cplusplus
extern "C" {    /* Use C linkage when in C++ mode */
#endif

#include <mytypes.h>


#ifndef TRUE
#define TRUE   1
#define FALSE  0
#endif

typedef struct   /* Data returned by the SVGAgetData function */
{
  ushort  Granularity;   /* Page granularity */
  ushort  BytesPerLine;
  int     WinSize;       /* Window size in Bytes */
  ushort  XPixels, YPixels; /* Video resolution in x and y */
  int     VidAdd;        /* Video start address */
  int     VidOffs;       /* Start address of next page */
  uchar   BitsPerPixel;  /* used by VESA modes */
  uchar   BytesPerPixel; /*  "     "       "   */
  uchar   Banks;         /* Number of banks in VESA modes */
  uchar   BankSize;      /* Bank size in VESA modes */
  ushort  Mode;
  uchar   RedMaskSiz;    
  uchar   RedFieldPos;
  uchar   GreenMaskSiz;
  uchar   GreenFieldPos;
  uchar   BlueMaskSiz;
  uchar   BlueFieldPos;

} SVGAdata;





typedef struct
{
  uchar b, g, r;   /* order of RGB bytes in memory for true color modes */
} SVGAtrueT;



typedef union
{
  uchar     vpix;  /* Pixel in VGA modes */
  SVGAtrueT tpix;  /* Pixel in True color modes */
} SVGApixT;        /* Pixel type */



/*
  Function description:

  SVGAinit initializes internal data used by the other routines in
  this package.  If the linear flag is set, then a linear frame buffer
  will be used for the graphics drawing if one can be used.
  If wflag is set, then Watcom library functions will be used for the
  graphics drawing whenever possible.  If the SetMode flag is set
  then the video mode will be entered, else if this flag is clear,
  only SVGA internal data will be initialised.

  mode = _SVRES256COLOR;   / * example mode: 800x600 256 color mode * /
  if ( SVGAinit( mode, wflag, linear, setmode, &status ) != EXIT_SUCCESS )
  {
    SVGAfailPrint( status );
    exit( status );
  }
  _setvideomode( mode );   / * Watcom library function: Set video mode * /

  The SVGAdrawBlock function copies the content of the srcBlock buffer
  to the display starting at (startx, starty) and the size of the
  displayed rectangle is xlen in x and ylen in y.
  The SVGAdrawTranspBlock is similar, but draws only the non-null pixel
  values in the source buffer.
  The SVGAgetBlock is the opposite of SVGAdrawBlock.
  SVGAgetData can be called after init to check various current mode
  parameters. See the SVGAdata data structure above.
  The SVGAfailPrint function can be used to print status message output
  by the SVGAinit function via the status parameter.
  SVGAgetNonSVGAresolution returns the x-and y-resolution in pixels for any
  non-SVGA mode.
  SVGAend() should be called once when the application exits.
*/

int  SVGAinit( short mode,
               int   iUseWatcomLib,
               int   iUseLinearFB,
               int   iDoSetMode,
               int  *SvgaStatus );
void SVGAend( void );
int  SVGAdrawBlock( int startx, int starty,
                    int xlen,   int ylen,   uchar *srcBlock );
int  SVGAdrawTranspBlock( int startx, int starty,
                          int xlen,   int ylen, uchar *srcBlock );
int  SVGAgetBlock( int startx, int starty,
                   int xlen,   int ylen,   uchar *destBlock );
SVGAdata *SVGAgetData( void );

/* Miscellaneous routines
*/
void  SVGAputPixel( int x, int y, uchar color );
int   SVGAgetPixel( int x, int y );
void  SVGAmoveto( int x, int y );
void  SVGAlineto( int x, int y, uchar Color );
int   SVGAfailPrint( int reason );
int   SVGAisTrueColorMode( void ); /* Returns TRUE if hi/true color mode set */
uchar *SVGAbufferAlloc( uint Xpix, uint Ypix );
uchar *SVGAbufferConvert( uchar *buf, uint Xpix, uint Ypix );
int   SVGAgetVideoRes( int *Xpix, int *Ypix );
int   SVGAsetPal( uchar *palValues );  /* Hi/true-color modes only */
int   SVGAsetLogicalScanLen( int iLen );
int   SVGAsetDisplayStart( int x, int y );
int   SVGAgetNonSVGAresolution( short mode, ushort *XPixels, ushort *YPixels );
int   SVGAclearScreen( void );
int   SVGAis16colorMode( void );

/* The following video modes are supported:

  Watcom mode #   Numeric #   Video resolution  Colors
  _ERESCOLOR         16        EGA   640x350      16
  _VRES16COLOR       18        VGA   640x480      16
  _MRES256COLOR      19        VGA   320x200     256
  _URES256COLOR     256        SVGA  640x400     256
  _VRES256COLOR     257        SVGA  640x480     256
  _SVRES16COLOR     258        SVGA  800x600      16
  _SVRES256COLOR    259        SVGA  800x600     256
  _XRES16COLOR      260        SVGA 1024x768      16
  _XRES256COLOR     261        SVGA 1024x768     256

  _SVGAmode1280x16  262        SVGA 1280x1024     16
  _SVGAmode1280x256 263        SVGA 1280x1024    256
  _SVGAmode320true  271        SVGA  320x200     16.8 mill. true color
  _SVGAmode640hi15  272        SVGA  640x480     15-bit hi-res mode
  _SVGAmode640hi16  273        SVGA  640x480     16-bit hi-res mode
  _SVGAmode640true  274        SVGA  640x480     16.8 mill. true color
  _SVGAmode800hi15  275        SVGA  800x600     15-bit hi-res mode
  _SVGAmode800hi16  276        SVGA  800x600     16-bit hi-res mode
  _SVGAmode800true  277        SVGA  800x600     16.8 mill. true color
  _SVGAmode1024hi15 278        SVGA 1024x768     15-bit hi-res mode
  _SVGAmode1024hi16 279        SVGA 1024x768     16-bit hi-res mode

  Non-Watcom non-SVGA modes
  _SVGAXMODE256x256  20        VGA   256x256     256
  _SVGAXMODE320x200  21        VGA   320x200     256
  _SVGAXMODE320x240  22        VGA   320x240     256
  _SVGAXMODE360x270  23        VGA   360x270     256
  _SVGAXMODE400x300  24        VGA   400x300     256
  _SVGAXMODE400x600  25        VGA   400x600     256


  The first 3 are not defined SVGA modes but are included for completeness,
  because all the 16-color modes can use the same code..
  The X-modes are included as a bonus!
*/


/* Status values returned in status by SVGAinit:
   If SVGAinit returns EXIT_FAILURE, the reason is found
   by checking the value of the *SvgaStatus flag
*/
#define  SvgaVesaModeValid 1   /* Video card supports VESA modes */
#define  SvgaModeValid     0   /* Init Ok and requested mode valid */
#define  SvgaModeInValid (-1)  /* SVGA mode is not recognised by HW or SW */
#define  SvgaInitFail    (-2)  /* graphics card probably not SVGA compliant */
#define  SvgaModeNotSupported (-3)  /* Mode not yet implemented */
#define  SvgaLFBnotAvail (-4)  /* Linear frame buffer not available */
#define  SvgaLinAdrNotAvail (-5)   /* Linear addr. for LFB not generated */


/* Video modes not supported by Watcom 10.0 drawing functions
*/
#define  _SVGAXMODE256x256   20 //VGAPAGEmode256x256, see VGAPAGE.h, if avail.
#define  _SVGAXMODE320x200   21 //VGAPAGEmode320x200
#define  _SVGAXMODE320x240   22 //VGAPAGEmode320x240  
#define  _SVGAXMODE360x270   23 //VGAPAGEmode360x270  
#define  _SVGAXMODE400x300   24 //VGAPAGEmode400x300  
#define  _SVGAXMODE400x600   25 //VGAPAGEmode400x600  

#define  _SVGAmode1280x16   262 //1280x1024 16-color
#define  _SVGAmode1280x256  263 //1280x1024 256-color

/* True color modes */
#define  _SVGAmode320true   271 //320x200 16.8 mill. true color
#define  _SVGAmode640true   274 //640x480 16.8 mill. true color
#define  _SVGAmode800true   277 //800x600 16.8 mill. true color

/* High color modes */
#define  _SVGAmode640hi15   272 //SVGA  640x480   15-bit hi-res mode
#define  _SVGAmode640hi16   273 //SVGA  640x480   16-bit hi-res mode
#define  _SVGAmode800hi15   275 //SVGA  800x600   15-bit hi-res mode
#define  _SVGAmode800hi16   276 //SVGA  800x600   16-bit hi-res mode
#define  _SVGAmode1024hi15  278 //SVGA 1024x768   15-bit hi-res mode
#define  _SVGAmode1024hi16  279 //SVGA 1024x768   16-bit hi-res mode


/* Some useful macros */
#define  SVGA256colorMode( mode ) \
  (mode == _MRES256COLOR || mode == _URES256COLOR || \
   mode == _VRES256COLOR || mode == _SVRES256COLOR || \
   mode == _XRES256COLOR || (mode >= _SVGAXMODE256x256 && \
   mode <= _SVGAXMODE400x600) || mode == _SVGAmode1280x256 )


#define SVGA16colorMode( mode ) \
  (mode <= _ERESCOLOR    || mode == _VRES16COLOR || \
   mode == _SVRES16COLOR || mode == _XRES16COLOR )


#define SVGAhiOrTrueColMode( mode ) \
  (mode >= _SVGAmode320true && mode <= _SVGAmode1024hi16)



#ifdef __cplusplus
};
#endif

#endif
