/* PM.C: Code to handle protected mode to real mode issues and
   mapping of physical addressing to linear addressing suitable
   for use with a video memory linear frame buffer.
   By  Arve Holmbo.
*/
#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <string.h>

#include "pm.h"

#define DPMIint  0x31


/* Generic 386 interrupt with segment registers
*/
static int interrpt( int vector, union REGS *r1, union REGS *r2 )
{
  struct SREGS sregs;

  segread( &sregs );
  return int386x( vector, r1, r2, &sregs );
}

#define  CallDPMI( r )   interrpt( DPMIint, r, r )




/* Functions to handle real-memory manipulations */


/* Allocate real mode memory and return pointer to it.
   Also, return the Protected mode selector and segment
*/
void far *PMallocDosMem( unsigned bytes, PM_REALPTR *ptr )
{
  ushort paragraphs;
  unsigned  temp = ( bytes + 15 ) >> 4;
  union  REGS  r;
  short        selector, segment;
  void far    *str;

  memset( &r,  0, sizeof( r ));
  paragraphs = (ushort )temp;

  r.w.ax = 0x100;        /* Allocate DOS memory */
  r.w.bx = paragraphs;   /* # of paragraphs to allocate */
  CallDPMI( &r );

  if ( r.x.cflag == 0 )
  {
    ptr->segment  = r.w.ax;
    ptr->selector = r.w.dx;
    str = MK_FP( ptr->selector, 0 );   /* Make a far pointer to allocated buf*/
  }
  else
    str = NULL;
  return str;
}




/* Free DOS memory allocated earlier via
   the AllocDosMem function
   */
void  PMfreeDosMem( PM_REALPTR * ptr )
{
  union  REGS  r;

  memset( &r,  0, sizeof( r ));
  r.w.ax = 0x101;     /* Free memory */
  r.w.dx = ptr->selector; 
  CallDPMI( &r );
}




/* Return the DPMI-generated linear start address, given a physical
   address and an address range.
*/
static uint  MapPhysical2Linear( uint PhysAddr, uint Limit )
{
  union REGS  r;
  uint  PhysOffset;

  /* Align the physical address to a 4Kb boundary and the range to a
     4Kb-1 boundary before passing the values to DPMI.  When we align the
     physical address, then we also add an extra offset into the address
     that we return.  This works well with DOS4GW and PMODEW.
   */
  PhysOffset = PhysAddr & 4095;  /* Keep lower 12 bits */
  PhysAddr  &= ~4095;            /* Mask off lower 12 bits */
  Limit = (( Limit + PhysOffset + 4096 ) & ~4095)-1;

  r.w.ax = 0x800;             /* DPMI map physical to linear */
  r.x.ebx = PhysAddr >> 16;
  r.x.ecx = PhysAddr & 0xFFFF;
  r.x.esi = Limit >> 16;
  r.x.edi = Limit & 0xFFFF;
  CallDPMI( &r );
  if ( r.w.cflag )
    return 0;
  else
    return (uint )(( r.x.ebx << 16 ) + r.x.ecx + PhysOffset );
}



/* Get linear address, given a start address.  This start address
   must be mapped to a linear address using the DPMI interface.
   Then, the datasegment selector must be changed to allow
   addressing of memory from this linear address thru' the given
   range.
*/
void *PMgetLinearAddr( uint Start, uint Range )
{
  struct SREGS  sregs;
  union  REGS   r;
  uint   LinAdr, RangeEnd, DSstartAdr;

  /* Map the memory to linear address */
  if (( LinAdr = MapPhysical2Linear( Start, Range )) == 0 ) 
  {
    if ( Start >= 1024*1024 )
      return  NULL;  /* Linear address mapping failed */
    LinAdr = Start;  /* Assume a DOS-compatibility box (Win / OS/2) */
  }

  /* Get start address for the DS selector */
  segread(&sregs);
  r.w.ax = 6;   /* DPMI get DS selector start address */
  r.w.bx = sregs.ds;
  r.x.ecx = r.x.edx = 0;
  CallDPMI( &r );
  DSstartAdr = (r.x.ecx << 16) + r.x.edx;

  LinAdr -= DSstartAdr;  /* Take into account DS start address */

  /* Expand the DS selector as required to address all of
     video memory.  If this doesn't work, then try RangeEnd = 0xFFFFFFFF;
     to get full access to all of CPU's address range (scary).
     So, if you try to access memory at the first byte above the RangeEnd
     address, then you should get a protection fault...
     Note that the 12 lower bits must be set.
   */
  RangeEnd = (LinAdr + Range) | 4095; /* Assume high above avail.sysmem */
  r.w.ax = 8;   /* DPMI set selector limit */
  r.w.bx = sregs.ds;
  r.w.cx = RangeEnd >> 16;
  r.w.dx = RangeEnd & 0xFFFF;
  CallDPMI( &r );
  if (r.w.cflag)
    return NULL;

  /* Return the start address of the memory area */
  return (void*)LinAdr;
}




/* Cleanup internals.  Could reset the selector limit here.
   Maybe some other time...
*/
void  PMcleanup( void )
{

}
