/*      dllsmp.c
 *
 * An example on how to use MIDAS modules, streams and sound effects through
 * the MIDAS DLL API.
 *
 * Copyright 1996 Petteri Kangaslampi and Jarno Paananen
 *
 * This file is part of the MIDAS Sound System, and may only be
 * used, modified and distributed under the terms of the MIDAS
 * Sound System license, LICENSE.TXT. By continuing to use,
 * modify or distribute this file you indicate that you have
 * read the license and understand and accept it fully.
*/

#if defined(__NT__) || defined(__WINDOWS__)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#define __WIN32
#endif

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include "midasdll.h"



#define NUMSAMPLECHANNELS 4
#define NUMMUSICCHANNELS 16


/****************************************************************************\
*
* Function:     void MIDASerror(void)
*
* Description:  Handles a MIDAS error - displays an error message and exits
*
\****************************************************************************/

void MIDASerror(void)
{
    int         error;

    error = MIDASgetLastError();
    printf("\nMIDAS error: %s\n", MIDASgetErrorMessage(error));
    if ( !MIDASclose() )
    {
        printf("\nBIG PANIC! MIDASclose Failed: %s\n", MIDASgetErrorMessage(
            MIDASgetLastError()));
    }
    exit(EXIT_FAILURE);
}



static MIDASmodule module;
static MIDASsample sample1, sample2;
static MIDASsamplePlayHandle playHandle1, playHandle2;


int main(void)
{
    int         key, exit = 0;

    MIDASstartup();

    setbuf(stdout, NULL);

    /* Flag that we don't have a module or effects playing: */
    module = NULL;
    sample1 = sample2 = 0;
    playHandle1 = playHandle2 = 0;

    /* Decrease the size of buffer used: */
    MIDASsetOption(MIDAS_OPTION_MIXBUFLEN, 150);
    MIDASsetOption(MIDAS_OPTION_MIXBUFBLOCKS, 4);

    /* Initialize MIDAS and start play thread: */
    if ( !MIDASinit() )
        MIDASerror();

#ifdef __WIN32
    if ( !MIDASstartPlayThread(10) )
        MIDASerror();
#endif

    /* Open all channels: */
    if ( !MIDASopenChannels(NUMSAMPLECHANNELS + NUMMUSICCHANNELS) )
        MIDASerror();

    /* The first NUMSAMPLECHANNELS are used for samples and the rest for
       music */

    /* Set automatic sample channel range: */
    if ( !MIDASsetAutoEffectChannels(0, NUMSAMPLECHANNELS) )
        MIDASerror();

    /* Load our samples: */
    if ( (sample1 = MIDASloadRawSample("..\\data\\explosi1.pcm",
        MIDAS_SAMPLE_8BIT_MONO, MIDAS_LOOP_NO)) == 0 )
        MIDASerror();
    if ( (sample2 = MIDASloadRawSample("..\\data\\laugh1.pcm",
        MIDAS_SAMPLE_8BIT_MONO, MIDAS_LOOP_YES)) == 0 )
        MIDASerror();

    /* Loop, reading user input, until we should exit: */
    while ( !exit )
    {
        puts("Keys:     1/2     Play/Stop sample 1\n"
             "          q/w     Play/Stop sample 2\n"
             "          5/6     Play/Stop module\n"
             "          Esc     Exit\n");

        key = getch();

        switch ( key )
        {
            case 27:
                exit = 1;
                break;

            case '1':
                if ( (playHandle1 = MIDASplaySample(sample1,
                    MIDAS_CHANNEL_AUTO, 0, 22050, 64, MIDAS_PAN_MIDDLE)) == 0)
                    MIDASerror();
                break;

            case '2':
                if ( playHandle1 != 0 )
                {
                    if ( !MIDASstopSample(playHandle1) )
                        MIDASerror();
                }
                break;

            case 'q':
                if ( (playHandle2 = MIDASplaySample(sample2,
                    MIDAS_CHANNEL_AUTO, 0, 16000, 64, -20)) == 0 )
                    MIDASerror();
                break;

            case 'w':
                if ( playHandle2 != 0 )
                {
                    if ( !MIDASstopSample(playHandle2) )
                        MIDASerror();
                }
                break;

            case '5':
                if ( module != NULL )
                {
                    if ( !MIDASstopModule(module) )
                        MIDASerror();
                    if ( !MIDASfreeModule(module) )
                        MIDASerror();
                }
                if ( (module = MIDASloadModule("..\\data\\templsun.xm"))
                    == NULL )
                    MIDASerror();
                if ( !MIDASplayModule(module, 0) )
                    MIDASerror();
                break;

            case '6':
                if ( module != NULL )
                {
                    if ( !MIDASstopModule(module) )
                        MIDASerror();
                    if ( !MIDASfreeModule(module) )
                        MIDASerror();

                    module = NULL;
                }
                break;
        }
    }

    /* Stop MIDAS: */
#ifdef __WIN32
    if ( !MIDASstopPlayThread() )
        MIDASerror();
#endif
    if ( !MIDASclose() )
        MIDASerror();

    return 0;
}