/* kbmlock.c -- memory locking
 * Copyright (C) 1995, 1996 Markus F.X.J. Oberhumer
 * For conditions of distribution and use, see copyright notice in kb.h 
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <kb.h>
#include <kbmlock.h>
#include "_kb.h"


/* see also: djgpp2/src/libc/go32/gopint.c
 *           djgpp2/src/libc/dpmi/api/d0600.s
 */


#if defined(KB_LOCK_ALL_CODE_START)
KB_LOCK_ALL_CODE_START(_libkb_kbmlock)
#endif


/***********************************************************************
// DPMI support for Watcom C32 (djgpp v2 compatible)
************************************************************************/

#if defined(__WATCOMC__) && defined(__KB_MSDOS32)

typedef struct {
	unsigned long handle;					/* 0, 2 */
	unsigned long size; 	/* or count */	/* 4, 6 */
	unsigned long address;					/* 8, 10 */
} __dpmi_meminfo;


/* it looks like every base address is 0 under DOS/4GW and PMODE/W */

/* DPMI 0.9 AX=0006 */
static
int __dpmi_get_segment_base_address(int _selector, unsigned long *_addr)
{
	KB_INT86_REGS regs;
	_kb_int86_regs_init_ax(&regs,0x0006);

	regs.w.bx = (unsigned short) _selector;
	regs.w.flags = 0x01;					/* be paranoid */
	KB_INT86(0x31,&regs);
	if (regs.w.flags & 0x01)				/* error if carry flag set */
		return -1;
	/* FIXME: is this correct ? */
	*_addr = regs.w.cx | ((unsigned long)regs.w.dx << 16);
	return 0;
}


static
int watcom_c32_do_lock(__dpmi_meminfo *_info, unsigned short ax)
{
	KB_INT86_REGS regs;
	_kb_int86_regs_init_ax(&regs,ax);

	/* FIXME: is this correct ? */
	regs.w.cx = (unsigned short) (_info->address);
	regs.w.bx = (unsigned short) (_info->address >> 16);
	regs.w.di = (unsigned short) (_info->size);
	regs.w.si = (unsigned short) (_info->size >> 16);

	regs.w.flags = 0x01;					/* be paranoid */
	KB_INT86(0x31,&regs);
	return (regs.w.flags & 0x01 ? -1 : 0);	/* error if carry flag set */
}

/* DPMI 0.9 AX=0600 */
static
int __dpmi_lock_linear_region(__dpmi_meminfo *_info)
{
	return watcom_c32_do_lock(_info,0x0600);
}

/* DPMI 0.9 AX=0601 */
static
int __dpmi_unlock_linear_region(__dpmi_meminfo *_info)
{
	return watcom_c32_do_lock(_info,0x0601);
}

#endif /* __WATCOMC__ */


/***********************************************************************
// low level locking - djgpp v2, Watcom C32
************************************************************************/

#if defined(__KB_MSDOS32)
#if defined(__DJGPP__) || defined(__WATCOMC__)

static int _kb_lock_mem(int seg, unsigned long lockaddr, unsigned long locksize,
						int code, int lock)
{
	int r;
    __dpmi_meminfo memregion;

#if defined(__DJGPP__)
	seg = code ? _go32_my_cs() : _go32_my_ds();
#endif

    memset(&memregion, 0, sizeof(memregion));
    if (__dpmi_get_segment_base_address(seg,&memregion.address) != 0)
		return -1;
    memregion.address += lockaddr;
    memregion.size     = locksize;

	if (lock)
		r = __dpmi_lock_linear_region(&memregion);
	else
		r = __dpmi_unlock_linear_region(&memregion);

#if defined(KB_DEBUG) && (KB_DEBUG >= 3)
	fprintf(stderr,"libkb info: %s %04x:%08lx (0x%08lx), %6ld bytes: %d\n",
		lock ? "lock " : "unlock", seg, lockaddr,
		memregion.address, locksize, r);
#endif

	return r;
}


#if defined(__DJGPP__)
#  define KB_LOCK_MEM(a,s,c)	_kb_lock_mem(0,(unsigned long)(a),s,c,1)
#  define KB_UNLOCK_MEM(a,s,c)	_kb_lock_mem(0,(unsigned long)(a),s,c,0)
#elif defined(__WATCOMC__)
#  define KB_LOCK_MEM(a,s,c)	_kb_lock_mem(FP_SEG(a),FP_OFF(a),s,c,1)
#  define KB_UNLOCK_MEM(a,s,c)	_kb_lock_mem(FP_SEG(a),FP_OFF(a),s,c,0)
#endif


#endif
#endif /* __KB_MSDOS32 */



/***********************************************************************
// high level locking
************************************************************************/

int kb_lock_code(void (*start)(void), void (*end)(void))
{
	if (start == NULL || end == NULL || start == end)
		return -1;

#if defined(KB_LOCK_MEM)
	/* warning: ANSI C forbids ordered comparisons of pointers to functions */
	{
		unsigned long s = (unsigned long) start;
		unsigned long e = (unsigned long) end;
		if (e > s)
			return KB_LOCK_MEM(start,e-s,1);
		else if (s > e)
			return KB_LOCK_MEM(end,s-e,1);
		else
			return -1;
	}
#else
	return 0;
#endif
}


int kb_lock_var(const void *addr, unsigned size)
{
	if (addr == NULL || size <= 0)
		return -1;

#if defined(KB_LOCK_MEM)
	return KB_LOCK_MEM(addr,size,0);
#else
	return 0;
#endif
}


int kb_lock_data(const void *start, const void *end)
{
	if (start == NULL || end == NULL || start == end)
		return -1;

#if defined(KB_LOCK_MEM)
	if (start < end)
		return KB_LOCK_MEM(start,(char*)end-(char*)start,0);
	else
		return KB_LOCK_MEM(end,(char*)start-(char*)end,0);
#else
	return 0;
#endif
}


/***********************************************************************
// high level unlocking
************************************************************************/

int kb_unlock_code(void (*start)(void), void (*end)(void))
{
	if (start == NULL || end == NULL || start == end)
		return -1;

#if defined(KB_UNLOCK_MEM)
	/* warning: ANSI C forbids ordered comparisons of pointers to functions */
	{
		unsigned long s = (unsigned long) start;
		unsigned long e = (unsigned long) end;
		if (e > s)
			return KB_UNLOCK_MEM(start,e-s,1);
		else if (s > e)
			return KB_UNLOCK_MEM(end,s-e,1);
		else
			return -1;
	}
#else
	return 0;
#endif
}


int kb_unlock_var(const void *addr, unsigned size)
{
	if (addr == NULL || size <= 0)
		return -1;

#if defined(KB_UNLOCK_MEM)
	return KB_UNLOCK_MEM(addr,size,0);
#else
	return 0;
#endif
}


int kb_unlock_data(const void *start, const void *end)
{
	if (start == NULL || end == NULL || start == end)
		return -1;

#if defined(KB_UNLOCK_MEM)
	if (start < end)
		return KB_UNLOCK_MEM(start,(char*)end-(char*)start,0);
	else
		return KB_UNLOCK_MEM(end,(char*)start-(char*)end,0);
#else
	return 0;
#endif
}


/***********************************************************************
// locking (lock the locking code :-)
************************************************************************/

#if defined(KB_LOCK_ALL_CODE_END)
KB_LOCK_ALL_CODE_END(_libkb_kbmlock)
#endif

int _libkb_kbmlock_lock(void)
{
	int x = 0;

#if defined(KB_LOCK_ALL_CODE)
	KB_LOCK_ALL_CODE(_libkb_kbmlock,x);
#endif

	return x;
}


/*
vi:ts=4
*/
