/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <i86.h>
#include <string.h>
#include <stdio.h>
#include <jlib.h>

#pragma pack(1)

typedef struct {
	USHORT seg, sel;
} rptr;

rptr vesa_info_rp, vesa_mode_data_rp;

static void far *dosalloc(rptr * rp, unsigned int size);
static void dosfree(rptr * rp);

typedef struct {
	char signature[4];
	USHORT version;
	ULONG OEM_name;
	ULONG capabilities;
	ULONG video_modes;
	USHORT total_mem;
	USHORT OEM_rev;
	ULONG OEM_vendor;
	ULONG OEM_product;
	ULONG OEM_prod_rev;
	UBYTE reserved[222];
	UBYTE OEM_data[256];
} vesainfo;

typedef struct {
	USHORT attributes;
	UBYTE Aattributes, Battributes;
	USHORT granularity, size, segA, segB;
	UBYTE foobar[4];	/* void (far *setpage)(); */
	USHORT scan_line_bytes;
	USHORT pixel_width;
	USHORT pixel_height;
	UBYTE cpixel_width, cpixel_height, mem_planes, bpp, num_banks, mem_model;
	UBYTE bank_size, image_pages, res1;
	UBYTE rmask;
	UBYTE rfp, gmask, gfp, bmask, bfp, res_mask, res_fp, direct_col;
	ULONG phys_ptr;
	ULONG offscr_ptr;
	short offscr_size;
	UBYTE res2[206];
} vesamodedata;

typedef struct {
	ULONG edi, esi, ebp, reserved, ebx, edx, ecx, eax;
	USHORT flags, es, ds, fs, gs, ip, cs, sp, ss;
} rm_regs;

#pragma pack()

#define PACKED_MEM	4
#define LINEAR_MODE	0x0080
#define AVAIL_MODE	0x0001
#define USE_LINEAR	0x4000
#define VBE2_SIG	(((int)'2'<<24)|((int)'E'<<16)|((int)'B'<<8)|'V')
#define VBE2_CONF	(((int)'A'<<24)|((int)'S'<<16)|((int)'E'<<8)|'V')
#define VESA_MAX_MODES	50

#define CLEAR_REGS(r)	memset(&r,0,sizeof(r))
vesamodedata far *vesa_mode_data = NULL;
vesainfo far *vesa_info = NULL;

int	vesa_remains;
int	vesa_granularity;
int	vesa_gran_mul;
int	vesa_gran1024;
int	vesa_pages;
int	vesa_width;
int	vesa_lfb = 0;
short	*vesa_modes;
char	version_string[] = "WATCOM VESA";
int	__jlib_pg = 0;
int	__jlib_mouse_visible = 0;
UBYTE	__jlib_previous_mode = 0x3;
UBYTE	__jlib_screen_initted = 0;
buffer_rec	__jlib_screen_buff;
int	__jlib_is_13h = 0;


/*+------------------------------------------------------------------------+ */
/*| Set the video mode, using a VESA 2.0 Linear buffer if available.       | */
/*| If a 320x200 mode is required and not available, emulate it w/ mode 13h| */
/*+------------------------------------------------------------------------+ */
int screen_set_video_mode(void)
{
	union REGS regs;
	struct SREGS sregs;
	rm_regs rmregs;
	int i;

	if (__jlib_screen_initted)
		return 1;

	/* Save the current mode */
	regs.h.ah = 0x0f;
	int386(0x10, &regs, &regs);
	__jlib_previous_mode = regs.h.al;

	/* allocate real mode transfer buffers */
	vesa_info = (vesainfo far *) dosalloc(&vesa_info_rp, sizeof(vesainfo));
	vesa_mode_data = (vesamodedata far *) dosalloc(&vesa_mode_data_rp, sizeof(vesamodedata));

	/* Do the common parts of initialising the direct video access buffer */
	__jlib_screen_buff.width = SCREEN_WIDTH;
	__jlib_screen_buff.height = SCREEN_HEIGHT;
	__jlib_screen_buff.offset = (UBYTE **) malloc(SCREEN_HEIGHT * sizeof(UBYTE *));
	if (__jlib_screen_buff.offset == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* try to get any VBE2 information present */
	*(int far *) vesa_info->signature = VBE2_SIG;

	/* Get VESA info */
	CLEAR_REGS(rmregs);
	CLEAR_REGS(regs);
	segread(&sregs);
	rmregs.eax = 0x4f00;
	rmregs.es = rmregs.ds = vesa_info_rp.seg;
	regs.x.eax = 0x300;
	regs.x.ebx = 0x10;
	regs.x.ecx = 0;
	sregs.es = FP_SEG(&rmregs);
	regs.x.edi = FP_OFF(&rmregs);
	int386x(0x31, &regs, &regs, &sregs);

#if SCREEN_WIDTH != 320
	if ((rmregs.eax & 0xff) != 0x4f) {
		JLIB_PRINT_DEBUG_INFO("No VESA support present.");
		return 0;
	}
#endif

	/* Use VBE 2.0+ interface if a suitable mode is available */
	if (((rmregs.eax & 0xff) == 0x4f) &&
		(*(int far *) vesa_info->signature == VBE2_CONF) &&
		(vesa_info->version >= 0x200)) {

		short *avail, *dest;

		JLIB_PRINT_DEBUG_INFO("Found VESA 2.0 interface...");

		vesa_modes = (short *) malloc(VESA_MAX_MODES * sizeof(short));

		if (vesa_modes == NULL)
			jlib_exit(jlib_msg(JLIB_ENULL));

		dest = vesa_modes;

		avail = (short *)(((vesa_info->video_modes) >> 12) + ((vesa_info->video_modes) & 0xFFFF));

		/* copy available modes */
		while (*avail != -1)
			*dest++ = *avail++;
		*dest = -1;

		/* Traverse the list until we find a suitable mode */
		avail = vesa_modes;

		while (*avail != -1) {
			/* Get mode info */
			CLEAR_REGS(rmregs);
			rmregs.es = rmregs.ds = vesa_mode_data_rp.seg;
			rmregs.edi = 0;
			rmregs.eax = 0x4f01;
			rmregs.ecx = *avail;
			CLEAR_REGS(regs);
			regs.x.edi = FP_OFF(&rmregs);
			regs.x.eax = 0x300;
			regs.x.ebx = 0x10;
			segread(&sregs);
			sregs.es = FP_SEG(&rmregs);
			int386x(0x31, &regs, &regs, &sregs);

			/* Check the mode is suitable */
			if (((rmregs.eax & 0xff) == 0x004f) &&
				(vesa_mode_data->pixel_width == SCREEN_WIDTH) &&
				(vesa_mode_data->pixel_height == SCREEN_HEIGHT) &&
				(vesa_mode_data->attributes & LINEAR_MODE) &&
				(vesa_mode_data->mem_model == PACKED_MEM) &&
				(vesa_mode_data->bpp == 8) &&
				(vesa_mode_data->mem_planes == 1)) {

				/* Set the mode */
				CLEAR_REGS(regs);
				regs.w.ax = 0x4f02;
				regs.w.bx = *avail | USE_LINEAR;
				int386(0x10, &regs, &regs);

				if (regs.w.ax == 0x004f) {
					/* Map the LFB pointer to our linear address space */
					regs.w.ax = 0x800;
					regs.w.bx = vesa_mode_data->phys_ptr >> 16;
					regs.w.cx = vesa_mode_data->phys_ptr & 0xffff;
					regs.w.si = 0x3f;
					regs.w.di = 0xffff;
					int386(0x31, &regs, &regs);

					if (!regs.x.cflag) {
						JLIB_PRINT_DEBUG_INFO("Set LFB mode OK.");

						/* create a fake buffer for the screen */
						__jlib_screen_buff.buffer = (UBYTE *) ((long) regs.w.bx << 16) + regs.w.cx;

						for (i = 0; i < SCREEN_HEIGHT; i++)
							__jlib_screen_buff.offset[i] = __jlib_screen_buff.buffer + (i * SCREEN_WIDTH);

						free(vesa_modes);
						dosfree(&vesa_info_rp);
						dosfree(&vesa_mode_data_rp);

						buff_reset_clip_region(&__jlib_screen_buff);
						vesa_lfb = __jlib_screen_initted = 1;
						return 1;	/* mode set OK */
					}
				}
				JLIB_PRINT_DEBUG_INFO("Set LFB mode failed.");
			}
			avail++;	/* try the next mode */
		}

		JLIB_PRINT_DEBUG_INFO("failed to find or set v2.0 mode: reverting to v1.2");

		free(vesa_modes);
	}

#if SCREEN_WIDTH == 320
	JLIB_PRINT_DEBUG_INFO("Using Mode 13h...");

	CLEAR_REGS(regs);
	regs.x.eax = 0x13;
	int386(0x10, &regs, &regs);

	/* create a fake buffer for the screen */
	__jlib_screen_buff.buffer = (UBYTE *)VIDEO_START;
	for (i = 0; i < SCREEN_HEIGHT; i++)
		__jlib_screen_buff.offset[i] = __jlib_screen_buff.buffer + (i * SCREEN_WIDTH);

	__jlib_is_13h = __jlib_screen_initted = 1;
	return 1;		/* mode set OK */
#else
	JLIB_PRINT_DEBUG_INFO("Trying VESA v1.2 interface...");

	free(__jlib_screen_buff.offset);

	/* Get v1.2 mode info */
	CLEAR_REGS(rmregs);
	rmregs.es = rmregs.ds = vesa_mode_data_rp.seg;
	rmregs.edi = 0;
	rmregs.eax = 0x4f01;
	rmregs.ecx = VESA_MODE;
	CLEAR_REGS(regs);
	regs.x.eax = 0x300;
	regs.x.ebx = 0x10;
	regs.x.edi = (unsigned int) &rmregs;
	segread(&sregs);
	int386x(0x31, &regs, &regs, &sregs);

	if (regs.h.al == 0) {
		vesa_granularity = vesa_mode_data->granularity;
		vesa_width = vesa_mode_data->pixel_width;
		if (vesa_width == 0)
			vesa_width = vesa_mode_data->scan_line_bytes;
		vesa_gran_mul = 64 / vesa_granularity;
		vesa_gran1024 = vesa_granularity * 1024;
		vesa_pages = SCR_SIZE / (vesa_gran1024);
		vesa_remains = SCR_SIZE - (vesa_pages * vesa_gran1024);

		/* Set the mode */
		CLEAR_REGS(regs);
		regs.w.ax = 0x4f02;
		regs.w.bx = VESA_MODE;
		int386(0x10, &regs, &regs);

		if (regs.w.ax == 0x004f) {
			dosfree(&vesa_mode_data_rp);
			dosfree(&vesa_info_rp);
			__setpg(0);
			__jlib_screen_initted = 1;
			return 1;	/* mode set OK */
		}
	}

	dosfree(&vesa_mode_data_rp);
	dosfree(&vesa_info_rp);
	return 0;		/* mode set failed */
#endif
}


/*+------------------------------------------------------------------------+ */
/*|Restore the screen to its previous state.                               | */
/*+------------------------------------------------------------------------+ */
void screen_restore_video_mode(void)
{
	if (__jlib_screen_initted) {
		union REGS regs;

		CLEAR_REGS(regs);
		regs.x.eax = __jlib_previous_mode;
		int386(0x10, &regs, &regs);
		__jlib_screen_initted = 0;
	}
}


/*+------------------------------------------------------------------------+ */
/*|Return the current page number.                                         | */
/*+------------------------------------------------------------------------+ */
int screen_get_page(void)
{
	return 1;
}


/*+------------------------------------------------------------------------+ */
/*|set the given  page number.                                             | */
/*+------------------------------------------------------------------------+ */
void screen_set_page(int page)
{
	/* do nothing */
}


/*+------------------------------------------------------------------------+ */
/*|Show the current drawing page.                                          | */
/*+------------------------------------------------------------------------+ */
void screen_show_page(int page)
{
	/* do nothing */
}


/*+------------------------------------------------------------------------+ */ 
/*|Wait for the retrace beam to move offscreen.                            | */ 
/*+------------------------------------------------------------------------+ */ 
void screen_wait_vsync(void)
{
	while((inp(0x3da)&8) != 0);
	while((inp(0x3da)&8) == 0);
}


/*+------------------------------------------------------------------------+ */
/*|Fill the screen with a given color.                                     | */
/*+------------------------------------------------------------------------+ */
void screen_fill(UBYTE color)
{
	if (vesa_lfb || __jlib_is_13h) {
		UBYTE *screen = B_BUFF_PTR(&__jlib_screen_buff);
		MEM_STORE_LONG(screen, color, SCR_SIZE);
	}
	else {
		int i;

		for (i = 0; i < vesa_pages; i++) {
			__setpg(i);
			MEM_STORE_LONG(VIDEO_START, color, vesa_gran1024);
		}

		if (vesa_remains) {
			__setpg(vesa_pages);
			__jlib_pg = vesa_pages;
			MEM_STORE_LONG(VIDEO_START, color, vesa_remains);
		}
		else
			__jlib_pg = vesa_pages - 1;
	}
}


/*+------------------------------------------------------------------------+ */
/*|Clear the screen.                                                       | */
/*+------------------------------------------------------------------------+ */
void screen_clear(void)
{
	screen_fill(0);
}


int get_best_copy_method(void *addr1, void *addr2, unsigned int len)
{
	return ALIGNMENT_STATE_1;
}


/*+------------------------------------------------------------------------+ */
/*|Allocate some dos memory.                                               | */
/*+------------------------------------------------------------------------+ */
static void far *dosalloc(rptr * rp, unsigned int size)
{
	union REGS regs;

	size = ((size + 15) & 0xfffffff0);	/* Round up to nearest paragraph */
	CLEAR_REGS(regs);
	regs.w.ax = 0x100;
	regs.w.bx = (USHORT) (size >> 4);	/* Convert bytes to pages */
	int386(0x31, &regs, &regs);

	if (!regs.x.cflag) {
		rp->seg = regs.w.ax;
		rp->sel = regs.w.dx;
		return (MK_FP(regs.w.dx, 0));
	}

	jlib_exit(jlib_msg(JLIB_ENULL));
	return NULL;
}


/*+------------------------------------------------------------------------+ */
/*|Free some allocated dos memory.                                         | */
/*+------------------------------------------------------------------------+ */
static void dosfree(rptr * rp)
{
	union REGS regs;
	regs.w.ax = 0x101;
	regs.w.dx = rp->sel;
	int386(0x31, &regs, &regs);
}


/*+------------------------------------------------------------------------+ */
/*|Return a buffer representing the screen if available.                   | */
/*+------------------------------------------------------------------------+ */
buffer_rec *screen_get_buffer(void)
{
	buffer_rec *screen_buff;
	int i;

	if (__jlib_screen_initted)
		if (vesa_lfb || __jlib_is_13h) {
			screen_buff = (buffer_rec *) malloc(sizeof(buffer_rec));

			if (screen_buff == NULL)
				jlib_exit(jlib_msg(JLIB_EMALLOC));
		
			screen_buff->width = SCREEN_WIDTH;
			screen_buff->height = SCREEN_HEIGHT;

			screen_buff->offset = (UBYTE **) malloc(SCREEN_HEIGHT * sizeof(UBYTE *));
			if (screen_buff->offset == NULL)
				jlib_exit(jlib_msg(JLIB_EMALLOC));

			for (i=0; i < SCREEN_HEIGHT; i++)
				screen_buff->offset[i] = __jlib_screen_buff.offset[i];

			screen_buff->buffer = __jlib_screen_buff.buffer;
			buff_reset_clip_region(screen_buff);

			return screen_buff;
		}
	return NULL;
}