/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
/*
 * By Lennart Steinke and Jon Griffiths
 */
#include <dos.h>
#include <pc.h>
#include <jlib.h>
#include <dpmi.h>
#include <sys/segments.h>
#include <sys/farptr.h>

#define M_HEIGHT	16
#define M_MAX_Y	15
#define M_WIDTH	10
#define M_MAX_X	9

static UBYTE __jlib_mouse_pic[] =
{
	255, 255, 0, 0, 0, 0, 0, 0, 0, 0,
	255, 15, 255, 0, 0, 0, 0, 0, 0, 0,
	255, 15, 15, 255, 0, 0, 0, 0, 0, 0,
	255, 15, 15, 15, 255, 0, 0, 0, 0, 0,
	255, 15, 15, 15, 15, 255, 0, 0, 0, 0,
	255, 15, 15, 15, 15, 15, 255, 0, 0, 0,
	255, 15, 15, 15, 15, 15, 15, 255, 0, 0,
	255, 15, 15, 15, 15, 15, 15, 15, 255, 0,
	255, 15, 15, 15, 15, 15, 15, 15, 15, 255,
	255, 15, 15, 15, 15, 15, 255, 255, 255, 0,
	255, 15, 15, 255, 15, 15, 255, 0, 0, 0,
	255, 15, 255, 0, 255, 15, 15, 255, 0, 0,
	0, 255, 0, 0, 255, 15, 15, 255, 0, 0,
	0, 0, 0, 0, 0, 255, 15, 15, 255, 0,
	0, 0, 0, 0, 0, 255, 15, 15, 255, 0,
	0, 0, 0, 0, 0, 0, 255, 255, 0, 0,
};


buffer_rec	*__mouse_pic, *__mouse_back, *__mouse_draw;
int	__mouse_old_x, __mouse_old_y;
int	__jlib_mouse_visible;
int	__jlib_mouse_initted=0;
_go32_dpmi_seginfo __jlib_mouse_isr_info;
__dpmi_regs __jlib_mouse_regs;

int already_executing = 0;

void save_mouse(void);
void draw_mouse(void);
void rest_mouse(void);
void getbox(int x, int y, int h_len, int v_len, UBYTE * src);


/*+------------------------------------------------------------------------+ */
/*|Service routine called by mouse driver to display the mouse on screen   | */
/*+------------------------------------------------------------------------+ */
void __jlib_display_mouse_handler(__dpmi_regs *regs)
{
	int x, y, width, height;

	if (already_executing)
		return;

	if (!__jlib_mouse_visible)
		return;

	already_executing = 1;

	x = regs->x.cx >> 1;
	y = regs->x.dx;

	screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_back, 0, 0, M_MAX_X, M_MAX_Y);

	if (x + M_WIDTH > SCREEN_WIDTH)
		width = SCREEN_WIDTH - x;
	else
		width = M_WIDTH;

	if (y + M_HEIGHT > SCREEN_HEIGHT)
		height = SCREEN_HEIGHT - y;
	else
		height = M_HEIGHT;

	/* save the background */
	{
		ULONG src = y * SCREEN_WIDTH + x;
		ULONG dest = (ULONG) B_BUFF_PTR(__mouse_back);

		while(height--) {
			movedata(_jlib_sel, src, _my_ds(), dest, width);
			src += SCREEN_WIDTH;
			dest += M_WIDTH;
		}
	}

	memcpy(B_BUFF_PTR(__mouse_draw), B_BUFF_PTR(__mouse_back), M_WIDTH * M_HEIGHT);

	buff_stencil_buff_toNC(__mouse_draw, 0, 0, __mouse_pic, 0, 0, M_MAX_X, M_MAX_Y);
	screen_blit_buff_to(x, y, __mouse_draw, 0, 0, M_MAX_X, M_MAX_Y);

	__mouse_old_x = x;
	__mouse_old_y = y;

	already_executing = 0;
}


/*+------------------------------------------------------------------------+ */
/*|check for the presence of a mouse.                                      | */
/*+------------------------------------------------------------------------+ */
int mouse_present(void)
{
	__dpmi_regs regs;
      
#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if(__jlib_mouse_initted)
		return MOUSE_PRESENT;

	regs.x.ax = 0;
	__dpmi_int(0x33, &regs);

	if(regs.x.ax == 0xffff) {
		__jlib_mouse_visible =0;
		__jlib_mouse_initted = 1;

		if (__jlib_is_13h) {
			mouse_show_pointer();
			return MOUSE_PRESENT;
		}

		/* Init the buffers */
		__mouse_pic=buff_init(M_WIDTH, M_HEIGHT);
		__mouse_back=buff_init(M_WIDTH, M_HEIGHT);
		__mouse_draw=buff_init(M_WIDTH, M_HEIGHT);

		memcpy(B_BUFF_PTR(__mouse_pic),__jlib_mouse_pic, B_SIZE(__mouse_pic));

		/* Set the mouse at the middle of the screen */
		mouse_set_status(SCREEN_WIDTH/2, SCREEN_HEIGHT/2);

		/* Set the area the mouse can be moved in */
		regs.x.ax = 7;
		regs.x.cx = 0;
		regs.x.dx = SCREEN_WIDTH * 2 - 1;
		__dpmi_int(0x33, &regs);

		regs.x.ax = 8;
		regs.x.cx = 0;
		regs.x.dx = SCREEN_HEIGHT - 1;
		__dpmi_int(0x33, &regs);

		/* Tell the mouse driver to call our SR whenever the mouse
		 * is moved. */
		if(__jlib_is_linear) {
			__jlib_mouse_isr_info.pm_offset=(ULONG)__jlib_display_mouse_handler;
			__jlib_mouse_isr_info.pm_selector=_my_cs();

			_go32_dpmi_allocate_real_mode_callback_retf(&__jlib_mouse_isr_info, &__jlib_mouse_regs);

			regs.x.ax = 12;
			regs.x.cx = 1;
			regs.x.es = __jlib_mouse_isr_info.rm_segment;
			regs.x.dx = __jlib_mouse_isr_info.rm_offset;
			__dpmi_int(0x33, &regs);
		}

		mouse_show_pointer();
		return MOUSE_PRESENT;
	}
	else
		return MOUSE_ABSENT;
}


/*+------------------------------------------------------------------------+ */
/*|show the pointer.                                                       | */
/*+------------------------------------------------------------------------+ */
void mouse_show_pointer (void)
{
	__dpmi_regs regs;
	int b,width,height;

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if(!__jlib_mouse_initted || __jlib_mouse_visible)
	      return;

	mouse_get_status(&__mouse_old_x, &__mouse_old_y, &b);

	if (__jlib_is_linear) {
		while(already_executing);

		already_executing = 1;

		if (__mouse_old_x + M_WIDTH > SCREEN_WIDTH)
			width = SCREEN_WIDTH - __mouse_old_x;
		else
			width = M_WIDTH;

		if (__mouse_old_y + M_HEIGHT > SCREEN_HEIGHT)
			height = SCREEN_HEIGHT - __mouse_old_y;
		else
			height = M_HEIGHT;

		/* save the background */
		{
			ULONG src = __mouse_old_y * SCREEN_WIDTH + __mouse_old_x;
			ULONG dest = (ULONG) B_BUFF_PTR(__mouse_back);

			while(height--) {
				movedata(_jlib_sel, src, _my_ds(), dest, width);
				src += SCREEN_WIDTH;
				dest += M_WIDTH;
			}
		}

		memcpy(B_BUFF_PTR(__mouse_draw), B_BUFF_PTR(__mouse_back), M_WIDTH * M_HEIGHT);

		buff_stencil_buff_toNC(__mouse_draw, 0, 0, __mouse_pic, 0, 0, M_MAX_X, M_MAX_Y);
		screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_draw, 0, 0, M_MAX_X, M_MAX_Y);
		already_executing = 0;
	}
	else
 		if (__jlib_is_13h) {
			regs.x.ax= 0x0001;
			__dpmi_int(0x33, &regs);
		}
		else {
			save_mouse();
			draw_mouse();
		}


	__jlib_mouse_visible = 1;
}


/*+------------------------------------------------------------------------+ */
/*|hide the pointer.                                                       | */
/*+------------------------------------------------------------------------+ */
void mouse_hide_pointer (void)
{
	__dpmi_regs regs;

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if((!__jlib_mouse_initted) || (!__jlib_mouse_visible))
		return;

	if (__jlib_is_linear) {
		while(already_executing);
		already_executing = 1;
		screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_back, 0, 0, M_MAX_X, M_MAX_Y);
		already_executing = 0;
	}
	else
		if (__jlib_is_13h) {
			regs.x.ax= 0x0002;
			__dpmi_int(0x33, &regs);
		}
		else
			rest_mouse();
  
	__jlib_mouse_visible= 0;
}


/*+------------------------------------------------------------------------+ */
/*|read pointer coordinates.                                               | */
/*+------------------------------------------------------------------------+ */
void mouse_get_status (int *x_pos, int *y_pos, int *b_status)
{
	__dpmi_regs regs;

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if(!__jlib_mouse_initted)
		return;

	regs.x.ax= 0x0003;
	__dpmi_int(0x33, &regs);

	*x_pos = (int)(((USHORT)regs.x.cx >> 1)&0xffff); /* halve returned coord */
	*y_pos = (int)((USHORT)regs.x.dx & 0xffff);
	*b_status = (int)(USHORT)regs.x.bx;

	if(!__jlib_is_13h && __jlib_mouse_visible && !__jlib_is_linear)
		if ((*x_pos != __mouse_old_x) || (*y_pos != __mouse_old_y)) {
			rest_mouse();
			__mouse_old_x = *x_pos;
			__mouse_old_y = *y_pos;
			save_mouse();
			draw_mouse();
		}
}


/*+------------------------------------------------------------------------+ */
/*|change pointer coordinates.                                             | */
/*+------------------------------------------------------------------------+ */
void mouse_set_status (int x, int y)
{
	__dpmi_regs regs;

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if(!__jlib_mouse_initted)
		return;

	if(!__jlib_is_linear && !__jlib_is_13h && __jlib_mouse_visible)
		screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_back,0, 0, M_MAX_X, M_MAX_Y);

	regs.x.ax= 0x0004;
	regs.x.cx = x;
	regs.x.dx = y;
	__dpmi_int(0x33, &regs);

	if (!__jlib_is_linear && !__jlib_is_13h) {
		if (__jlib_mouse_visible)
			rest_mouse();

		__mouse_old_x = x;
		__mouse_old_y = y;

		if (__jlib_mouse_visible) {
			save_mouse();
			draw_mouse();
		}
	}
}


/*+------------------------------------------------------------------------+ */
/*|shut down the mouse.                                                    | */
/*+------------------------------------------------------------------------+ */
void mouse_closedown(void)
{
	__dpmi_regs regs;

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if(!__jlib_mouse_initted)
		return;

	mouse_hide_pointer();

	regs.x.ax = 0;
	__dpmi_int(0x33, &regs);

	if (!__jlib_is_13h) {
		buff_free(__mouse_pic);
		buff_free(__mouse_back);
		buff_free(__mouse_draw);
		if (__jlib_is_linear)
			_go32_dpmi_free_real_mode_callback(&__jlib_mouse_isr_info);
	}
	__jlib_mouse_initted= 0;
}


void save_mouse(void)
{
	int xlen, ylen;

	if (__mouse_old_x + M_MAX_X > SCREEN_MAX_X)
		xlen = SCREEN_MAX_X - __mouse_old_x + 1;
	else
		xlen = M_WIDTH;

	if (__mouse_old_y + M_MAX_Y > SCREEN_MAX_Y)
		ylen = SCREEN_MAX_Y - __mouse_old_y + 1;
	else
		ylen = M_HEIGHT;

	getbox(__mouse_old_x, __mouse_old_y, xlen, ylen, B_BUFF_PTR(__mouse_back));
}


void draw_mouse(void)
{
	memcpy(B_BUFF_PTR(__mouse_draw), B_BUFF_PTR(__mouse_back), (M_WIDTH * M_HEIGHT) );
	buff_stencil_buff_to(__mouse_draw, 0, 0, __mouse_pic, 0, 0, M_MAX_X, M_MAX_Y);
	screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_draw,0, 0, M_MAX_X, M_MAX_Y);
}


void rest_mouse(void)
{
	screen_blit_buff_to(__mouse_old_x, __mouse_old_y, __mouse_back, 0, 0, M_MAX_X, M_MAX_Y);
}


void getbox(int x, int y, int h_len, int v_len, UBYTE * src)
{
	int offset, page;

	offset = y * vesa_width + x;

	if (offset == 0)
		page = 0;
	else {
		page = offset / (vesa_gran1024);
		offset = offset % (vesa_gran1024);
	}

	__setpg(page);

	for (; v_len != 0; v_len--) {
		if (offset + h_len > vesa_gran1024)
			if (offset >= vesa_gran1024) {
				page++;
				__setpg(page);
				offset = offset % vesa_gran1024;
			}
			else {
				dosmemget(0xA0000 + offset, vesa_gran1024 - offset, src);
				page++;
				__setpg(page);
				dosmemget(0xA0000, (offset + h_len) % vesa_gran1024, src + vesa_gran1024 - offset);
				offset = (offset + vesa_width) % vesa_gran1024;
				src += M_MAX_X;
				continue;
			}
		dosmemget(0xA0000 + offset, h_len, src);
		src += M_WIDTH;
		offset += vesa_width;
	}
}