/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdlib.h>
#include <stdio.h>
#include <jlib.h>

/* format of a sprite file -
   **
   **   USHORT = number of sprite frames in the file.
   **
   **   for each sprite frame:
   **      UBYTE          = width
   **      UBYTE          = height
   **      (width*height) UBYTES = data
   **      UBYTE          = number of bounding rectangles
   **      for each bounding rect:
   **          UBYTE          = x1
   **          UBYTE          = y1
   **          UBYTE          = x2
   **          UBYTE          = y2
   **      }
   **   }
   **  }

   Sprite files are stored using the device independant routines in "jio.h"
 */

#define _SHEIGHT  (sys->sprite_data[count1]->height)
#define _SWIDTH   (sys->sprite_data[count1]->width)
#define _SSIZE    (_SHEIGHT*_SWIDTH)


/*+----------------------------------------------------------------------+ */
/*|load a sprite file into a sprite system.                              | */
/*+----------------------------------------------------------------------+ */
UBYTE sprite_load(char *filename, sprite_system * sys)
{
	FILE *spritefile;
	UBYTE result;

	JLIB_ENTER("sprite_load");

#ifndef JLIB_PRODUCTION
	if (filename == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_sprite_system(sys);
#endif

	if ((spritefile = fopen(filename, "rb")) == NULL) {
		JLIB_SPRINTF("%s: File IO Error.", filename);
		JLIB_LEAVE;
		return COULDNT_OPEN;
	}
	result = sprite_load_fp(spritefile, sys);
	fclose(spritefile);

	JLIB_LEAVE;
	return result;
}


/*+----------------------------------------------------------------------+ */
/*|load a sprite file into a sprite system from an open file.            | */
/*+----------------------------------------------------------------------+ */
UBYTE sprite_load_fp(FILE * spritefile, sprite_system * sys)
{
	int num_in_file;
	int count1, i, j;
	USHORT t_ushort;

	JLIB_ENTER("sprite_load_fp");

#ifndef JLIB_PRODUCTION
	if (spritefile == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_sprite_system(sys);
#endif

	jio_read_elementary_type(spritefile, &t_ushort, sizeof(USHORT));
	num_in_file = (int) t_ushort;

	JLIB_SPRINTF("%d Sprites In File.", num_in_file);

	if ((num_in_file + SPR_NUM_LOADED(sys)) > SPR_MAX_FRAMES(sys)) {
		JLIB_SPRINTF("Too Many Frames (%d).", num_in_file);
		JLIB_LEAVE;
		return TOO_MANY_IN_FILE;
	}
	JLIB_SPRINTF("%d Frames Available.", SPR_MAX_FRAMES(sys));

	for (count1 = SPR_NUM_LOADED(sys); count1 < (SPR_NUM_LOADED(sys) + num_in_file); count1++) {
		if ((sys->sprite_data[count1] =
		     (sprite_data_rec *) malloc(sizeof(sprite_data_rec))) == NULL) {
			jlib_exit(jlib_msg(JLIB_EMALLOC));
		}
		jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->width, sizeof(UBYTE));
		jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->height, sizeof(UBYTE));

		if ((sys->sprite_data[count1]->data =
		     (UBYTE *) malloc((_SHEIGHT * _SWIDTH) * sizeof(UBYTE))) == NULL) {
			jlib_exit(jlib_msg(JLIB_EMALLOC));
		}
		fread(sys->sprite_data[count1]->data, _SSIZE, 1, spritefile);

		jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->no_rects, sizeof(UBYTE));

		/* if there are bounding rectangles, read them in */
		if (sys->sprite_data[count1]->no_rects != 0) {

			if ((sys->sprite_data[count1]->rect_coords =
			     (UBYTE *) malloc(((sys->sprite_data[count1]->no_rects) * 4) * sizeof(UBYTE))) == NULL) {
				jlib_exit(jlib_msg(JLIB_EMALLOC));
			}
			j = 0;

			for (i = 0; i < sys->sprite_data[count1]->no_rects; i++) {
				/* read in each bounding rectangles coords */
				jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
				j++;
				jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
				j++;
				jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
				j++;
				jio_read_elementary_type(spritefile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
				j++;
			}
		}
		else
			sys->sprite_data[count1]->rect_coords = NULL;


		generate_rle(sys,sys->sprite_data[count1]);
	}

	sys->number_loaded += num_in_file;

#ifndef JLIB_PRODUCTION
	for (count1 = 0; count1 < SPR_NUM_LOADED(sys); count1++)
		jlib_check_frame(sys,count1);
#endif

	sprite_optimise_mem(sys);

	JLIB_LEAVE;
	return SUCCESS;
}


void generate_rle(sprite_system *sys, sprite_data_rec * srec)
{
	int hcount, rlecount, rlesize;
	UBYTE linereparr[SPR_MAX_Y], *data;

	JLIB_ENTER("generate_rle");

	/* work out the size to malloc space for */
	rlecount = rlesize = 0;
	data = srec->data;
	for (hcount = 0; hcount < srec->height; hcount++) {
		int w = 0, len, rle = 0;

		do {
			len = 0;
			while ((w < srec->width) && (data[w] != 0) && (len < 15)) {
				rlesize++;
				len++;
				w++;
			}
			if (len)
				rle++;
			len = 0;
			while ((w < srec->width) && (data[w] == 0) && (len < 15)) {
				len++;
				w++;
			}
			if (len)
				rle++;
		} while (w < srec->width);
		linereparr[hcount] = rle;
		rlecount += rle;
		data += srec->width;
	}

	srec->pattern = (UBYTE *) calloc(rlecount + hcount, 1);

	if (srec->pattern == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	if (rlesize > sys->biggest_rle)
		sys->biggest_rle = rlesize;	/* store if the biggest */

	/* now we actually encode the data */
	rlecount = 0;
	data = srec->data;
	for (hcount = 0; hcount < srec->height; hcount++) {
		int w = 0, len;

		srec->pattern[rlecount] = linereparr[hcount];
		rlecount++;

		do {
			len = 0;
			while ((w < srec->width) && (data[w] != 0) && (len < 15)) {
				len++;
				w++;
			}
			if (len) {
				srec->pattern[rlecount] = len | 16;
				rlecount++;
			}
			len = 0;
			while ((w < srec->width) && (data[w] == 0) && (len < 15)) {
				len++;
				w++;
			}
			if (len) {
				srec->pattern[rlecount] = len;
				rlecount++;
			}
		} while (w < srec->width);
		data += srec->width;
	}

	JLIB_LEAVE;
}


/*+----------------------------------------------------------------------+ */
/*|Save a sprite system's frames into a file.                            | */
/*+----------------------------------------------------------------------+ */
int sprite_save(char *filename, sprite_system * sys)
{
	FILE *sfile;
	int result;

	JLIB_ENTER("sprite_save");

#ifndef JLIB_PRODUCTION
	if (filename == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_sprite_system(sys);
#endif

	if ((sfile = fopen(filename, "wb")) == NULL) {
		JLIB_SPRINTF("%s: File IO Error.", filename);
		JLIB_LEAVE;
		return 0;
	}
	result = sprite_save_fp(sfile, sys);
	fclose(sfile);

	JLIB_LEAVE;
	return result;
}


/*+----------------------------------------------------------------------+ */
/*|save a sprite system's frames into an open file.                      | */
/*+----------------------------------------------------------------------+ */
int sprite_save_fp(FILE * sfile, sprite_system * sys)
{
	int count1, i, j;
	USHORT t_ushort;

	JLIB_ENTER("sprite_load_fp");

#ifndef JLIB_PRODUCTION
	if (sfile == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
	jlib_check_sprite_system(sys);
#endif

	/* write number of sprite frames */
	t_ushort = (USHORT) SPR_NUM_LOADED(sys);
	jio_write_elementary_type(sfile, &t_ushort, sizeof(USHORT));

	JLIB_SPRINTF("%d Sprites to write.", t_ushort);

	for (count1 = 0; count1 < SPR_NUM_LOADED(sys); count1++)
		if (sys->sprite_data[count1] != NULL) {
			jio_write_elementary_type(sfile, &sys->sprite_data[count1]->width, sizeof(UBYTE));
			jio_write_elementary_type(sfile, &sys->sprite_data[count1]->height, sizeof(UBYTE));

			fwrite(sys->sprite_data[count1]->data, _SSIZE, 1, sfile);

			/* write out number of bounding rects */
			jio_write_elementary_type(sfile, &sys->sprite_data[count1]->no_rects, sizeof(UBYTE));

			if (sys->sprite_data[count1]->no_rects)
				for (i = 0, j = 0; i < sys->sprite_data[count1]->no_rects; i++) {
					/* write out each bounding rectangles coords */
					jio_write_elementary_type(sfile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
					j++;
					jio_write_elementary_type(sfile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
					j++;
					jio_write_elementary_type(sfile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
					j++;
					jio_write_elementary_type(sfile, &sys->sprite_data[count1]->rect_coords[j], sizeof(UBYTE));
					j++;
				}
		}
		else
			return 0;

	JLIB_LEAVE;
	return 1;
}