/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>

/* Color -> Grey conversion constants */
#define R_COMP 19595		/* .299 << 16 */
#define G_COMP 3850		/* .587 << 16 */
#define B_COMP 9437		/* .114 << 16 */

/* default palette */
UBYTE __jlib_pal[SIZEOFPAL] =
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x00, 0xa8, 0x00, 0x00, 0xa8, 0xa8,
  0xa8, 0x00, 0x00, 0xa8, 0x00, 0xa8, 0xa8, 0x54, 0x00, 0xa8, 0xa8, 0xa8,
  0x54, 0x54, 0x54, 0x54, 0x54, 0xfc, 0x54, 0xfc, 0x54, 0x54, 0xfc, 0xfc,
  0xfc, 0x54, 0x54, 0xfc, 0x54, 0xfc, 0xfc, 0xfc, 0x54, 0xfc, 0xfc, 0xfc,
  0x00, 0x00, 0x00, 0x14, 0x14, 0x14, 0x20, 0x20, 0x20, 0x2c, 0x2c, 0x2c,
  0x38, 0x38, 0x38, 0x44, 0x44, 0x44, 0x50, 0x50, 0x50, 0x60, 0x60, 0x60,
  0x70, 0x70, 0x70, 0x80, 0x80, 0x80, 0x90, 0x90, 0x90, 0xa0, 0xa0, 0xa0,
  0xb4, 0xb4, 0xb4, 0xc8, 0xc8, 0xc8, 0xe0, 0xe0, 0xe0, 0xfc, 0xfc, 0xfc,
  0x00, 0x00, 0xfc, 0x40, 0x00, 0xfc, 0x7c, 0x00, 0xfc, 0xbc, 0x00, 0xfc,
  0xfc, 0x00, 0xfc, 0xfc, 0x00, 0xbc, 0xfc, 0x00, 0x7c, 0xfc, 0x00, 0x40,
  0xfc, 0x00, 0x00, 0xfc, 0x40, 0x00, 0xfc, 0x7c, 0x00, 0xfc, 0xbc, 0x00,
  0xfc, 0xfc, 0x00, 0xbc, 0xfc, 0x00, 0x7c, 0xfc, 0x00, 0x40, 0xfc, 0x00,
  0x00, 0xfc, 0x00, 0x00, 0xfc, 0x40, 0x00, 0xfc, 0x7c, 0x00, 0xfc, 0xbc,
  0x00, 0xfc, 0xfc, 0x00, 0xbc, 0xfc, 0x00, 0x7c, 0xfc, 0x00, 0x40, 0xfc,
  0x7c, 0x7c, 0xfc, 0x9c, 0x7c, 0xfc, 0xbc, 0x7c, 0xfc, 0xdc, 0x7c, 0xfc,
  0xfc, 0x7c, 0xfc, 0xfc, 0x7c, 0xdc, 0xfc, 0x7c, 0xbc, 0xfc, 0x7c, 0x9c,
  0xfc, 0x7c, 0x7c, 0xfc, 0x9c, 0x7c, 0xfc, 0xbc, 0x7c, 0xfc, 0xdc, 0x7c,
  0xfc, 0xfc, 0x7c, 0xdc, 0xfc, 0x7c, 0xbc, 0xfc, 0x7c, 0x9c, 0xfc, 0x7c,
  0x7c, 0xfc, 0x7c, 0x7c, 0xfc, 0x9c, 0x7c, 0xfc, 0xbc, 0x7c, 0xfc, 0xdc,
  0x7c, 0xfc, 0xfc, 0x7c, 0xdc, 0xfc, 0x7c, 0xbc, 0xfc, 0x7c, 0x9c, 0xfc,
  0xb4, 0xb4, 0xfc, 0xc4, 0xb4, 0xfc, 0xd8, 0xb4, 0xfc, 0xe8, 0xb4, 0xfc,
  0xfc, 0xb4, 0xfc, 0xfc, 0xb4, 0xe8, 0xfc, 0xb4, 0xd8, 0xfc, 0xb4, 0xc4,
  0xfc, 0xb4, 0xb4, 0xfc, 0xc4, 0xb4, 0xfc, 0xd8, 0xb4, 0xfc, 0xe8, 0xb4,
  0xfc, 0xfc, 0xb4, 0xe8, 0xfc, 0xb4, 0xd8, 0xfc, 0xb4, 0xc4, 0xfc, 0xb4,
  0xb4, 0xfc, 0xb4, 0xb4, 0xfc, 0xc4, 0xb4, 0xfc, 0xd8, 0xb4, 0xfc, 0xe8,
  0xb4, 0xfc, 0xfc, 0xb4, 0xe8, 0xfc, 0xb4, 0xd8, 0xfc, 0xb4, 0xc4, 0xfc,
  0x00, 0x00, 0x70, 0x1c, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54, 0x00, 0x70,
  0x70, 0x00, 0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1c,
  0x70, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54, 0x00,
  0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1c, 0x70, 0x00,
  0x00, 0x70, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x70, 0x38, 0x00, 0x70, 0x54,
  0x00, 0x70, 0x70, 0x00, 0x54, 0x70, 0x00, 0x38, 0x70, 0x00, 0x1c, 0x70,
  0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60, 0x38, 0x70,
  0x70, 0x38, 0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44,
  0x70, 0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60, 0x38,
  0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44, 0x70, 0x38,
  0x38, 0x70, 0x38, 0x38, 0x70, 0x44, 0x38, 0x70, 0x54, 0x38, 0x70, 0x60,
  0x38, 0x70, 0x70, 0x38, 0x60, 0x70, 0x38, 0x54, 0x70, 0x38, 0x44, 0x70,
  0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68, 0x50, 0x70,
  0x70, 0x50, 0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58,
  0x70, 0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68, 0x50,
  0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58, 0x70, 0x50,
  0x50, 0x70, 0x50, 0x50, 0x70, 0x58, 0x50, 0x70, 0x60, 0x50, 0x70, 0x68,
  0x50, 0x70, 0x70, 0x50, 0x68, 0x70, 0x50, 0x60, 0x70, 0x50, 0x58, 0x70,
  0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30, 0x00, 0x40,
  0x40, 0x00, 0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10,
  0x40, 0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30, 0x00,
  0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10, 0x40, 0x00,
  0x00, 0x40, 0x00, 0x00, 0x40, 0x10, 0x00, 0x40, 0x20, 0x00, 0x40, 0x30,
  0x00, 0x40, 0x40, 0x00, 0x30, 0x40, 0x00, 0x20, 0x40, 0x00, 0x10, 0x40,
  0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38, 0x20, 0x40,
  0x40, 0x20, 0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28,
  0x40, 0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38, 0x20,
  0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28, 0x40, 0x20,
  0x20, 0x40, 0x20, 0x20, 0x40, 0x28, 0x20, 0x40, 0x30, 0x20, 0x40, 0x38,
  0x20, 0x40, 0x40, 0x20, 0x38, 0x40, 0x20, 0x30, 0x40, 0x20, 0x28, 0x40,
  0x2c, 0x2c, 0x40, 0x30, 0x2c, 0x40, 0x34, 0x2c, 0x40, 0x3c, 0x2c, 0x40,
  0x40, 0x2c, 0x40, 0x40, 0x2c, 0x3c, 0x40, 0x2c, 0x34, 0x40, 0x2c, 0x30,
  0x40, 0x2c, 0x2c, 0x40, 0x30, 0x2c, 0x40, 0x34, 0x2c, 0x40, 0x3c, 0x2c,
  0x40, 0x40, 0x2c, 0x3c, 0x40, 0x2c, 0x34, 0x40, 0x2c, 0x30, 0x40, 0x2c,
  0x2c, 0x40, 0x2c, 0x2c, 0x40, 0x30, 0x2c, 0x40, 0x34, 0x2c, 0x40, 0x3c,
  0x2c, 0x40, 0x40, 0x2c, 0x3c, 0x40, 0x2c, 0x34, 0x40, 0x2c, 0x30, 0x40,
  0x54, 0x54, 0x54, 0x54, 0x54, 0xfc, 0x54, 0xfc, 0x54, 0x54, 0xfc, 0xfc,
   0xfc, 0x54, 0x54, 0xfc, 0x54, 0xfc, 0xfc, 0xfc, 0x54, 0xfc, 0xfc, 0xfc
};



#ifdef JDEFAULT_PALETTE
/*+------------------------------------------------------------------------+ */
/*|Initialise a blank palette.                                             | */
/*+------------------------------------------------------------------------+ */
UBYTE *pal_init(void)
{
	UBYTE *tmp;

	JLIB_ENTER("pal_init");

	if ((tmp = (UBYTE *) malloc(SIZEOFPAL * sizeof(UBYTE))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	JLIB_LEAVE;

	return tmp;
}


/*+------------------------------------------------------------------------+ */
/*|Copy one palette into another.                                          | */
/*+------------------------------------------------------------------------+ */
void pal_copy(UBYTE * pal1, UBYTE * pal2)
{
	JLIB_ENTER("pal_copy");

#ifndef JLIB_PRODUCTION
	if ((pal1 == NULL) || (pal2 == NULL))
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif

	FAST_SHORT_COPY(pal1, pal2, SIZEOFPAL);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|set all colors to zero.(DEFAULT)                                        | */
/*+------------------------------------------------------------------------+ */
void screen_blank_pal(void)
{
	UBYTE temp_pal[SIZEOFPAL];

	JLIB_ENTER("screen_blank_pal");

	MEM_STORE_LONG(temp_pal, 0, SIZEOFPAL);

	screen_block_set_pal(temp_pal);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|fade a palette in.(DEFAULT)                                             | */
/*+------------------------------------------------------------------------+ */
void screen_fade_in_pal(UBYTE * pal, unsigned int delaytime)
{
	int i, j;
	UBYTE temp_pal[SIZEOFPAL];
	unsigned int st, et, tt;

	JLIB_ENTER("screen_fade_in_pal");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	MEM_STORE_LONG(temp_pal, 0, SIZEOFPAL);

	tt = delaytime >> 6;
	uclock_init();

	for (i = 1; i < 64; i++) {
		st = uclock_read();
		for (j = 0; j < SIZEOFPAL; j++) {
			temp_pal[j] = (UBYTE) ((pal[j] * i) >> 6);
		}
		screen_block_set_pal(temp_pal);
		do {
			et = uclock_read();
		} while (et < (st + tt));
	}
	screen_block_set_pal(pal);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|fade from one palette to another.(DEFAULT)                              | */
/*+------------------------------------------------------------------------+ */
void screen_fade_to_pal(UBYTE * pal1, UBYTE * pal2, unsigned int delaytime)
{
	int i, j;
	UBYTE temp_pal[SIZEOFPAL];
	unsigned int st, et, tt;

	JLIB_ENTER("screen_fade_to_pal");

#ifndef JLIB_PRODUCTION
	if ((pal1 == NULL) || (pal2 == NULL)) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	screen_block_set_pal(pal1);
	tt = delaytime >> 6;
	uclock_init();

	for (i = 1; i < 64; i++) {
		st = uclock_read();
		for (j = 0; j < SIZEOFPAL; j++) {
			if (pal1[j] > pal2[j]) {
				temp_pal[j] = (UBYTE) (pal1[j] - ((pal1[j] - pal2[j]) * i >> 6));
			} else {
				temp_pal[j] = (UBYTE) (pal1[j] + ((pal2[j] - pal1[j]) * i >> 6));
			}
		}
		screen_block_set_pal(temp_pal);
		do {
			et = uclock_read();
		} while (et < (st + tt));
	}
	screen_block_set_pal(pal2);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|fade a palette out.(DEFAULT)                                            | */
/*+------------------------------------------------------------------------+ */
void screen_fade_out_pal(UBYTE * pal, unsigned int delaytime)
{
	int i, j;
	UBYTE temp_pal[SIZEOFPAL];
	unsigned int st, et, tt;

	JLIB_ENTER("Entering screen_fade_out_pal");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	/* copy the palette into our local copy */
	FAST_SHORT_COPY(pal, temp_pal, SIZEOFPAL);

	tt = delaytime >> 6;
	uclock_init();

	for (i = 64; i > 0; i--) {
		st = uclock_read();
		for (j = 0; j < SIZEOFPAL; j++) {
			temp_pal[j] = (UBYTE) ((pal[j] * i) >> 6);
		}
		screen_block_set_pal(temp_pal);
		do {
			et = uclock_read();
		} while (et < (st + tt));
	}

	screen_blank_pal();

	JLIB_LEAVE;
}
#endif				/* JDEFAULT_PALETTE */


#ifdef JDEFAULT_BLOCK_PAL
/*+------------------------------------------------------------------------+ */
/*|set all the colors from the given palette.(DEFAULT)                     | */
/*+------------------------------------------------------------------------+ */
void screen_block_set_pal(UBYTE * pal)
{
	int i;

	JLIB_ENTER("screen_block_set_pal");

#ifndef JLIB_PRODUCTION
	if (pal == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif

	for (i = 0; i < SCREEN_NUM_COLORS; i++) {
		screen_put_pal((UBYTE) i, pal[0], pal[1], pal[2]);
		pal += 3;
	}

	JLIB_LEAVE;
}
#endif


#ifdef JDEFAULT_LOAD_PAL
/*+------------------------------------------------------------------------+ */
/*|load a palette from a file name.(DEFAULT)                               | */
/*+------------------------------------------------------------------------+ */
UBYTE *pal_load(char *fname)
{
	UBYTE *tmp;
	FILE *fp;

	JLIB_ENTER("pal_load");

#ifndef JLIB_PRODUCTION
	if (fname == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif


	/* open file */
	if ((fp = fopen(fname, "rb")) == NULL) {
		JLIB_SPRINTF("%s File Open Error.", fname);
		JLIB_LEAVE;
		return (UBYTE *) NULL;
	}
	if ((tmp = pal_load_fp(fp)) == NULL) {
		free(tmp);

		JLIB_LEAVE;
		return (UBYTE *) NULL;
	}
	fclose(fp);

	JLIB_LEAVE;
	return tmp;
}


/*+------------------------------------------------------------------------+ */
/*|load a palette from a file handle.(DEFAULT)                             | */
/*+------------------------------------------------------------------------+ */
UBYTE *pal_load_fp(FILE * fp)
{
	UBYTE *tmp;
	int ret;

	JLIB_ENTER("pal_load_fp");

#ifndef JLIB_PRODUCTION
	if (fp == NULL)
		jlib_exit(jlib_msg(JLIB_ENULL));
#endif

	/* allocate memory for the palette */
	if ((tmp = (UBYTE *) malloc(SIZEOFPAL * sizeof(UBYTE))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* read in the palette */
	if ((ret = fread(tmp, 1, (sizeof(UBYTE) * SIZEOFPAL), fp)) != SIZEOFPAL) {
		JLIB_SPRINTF("File IO Read %d bytes.", ret);
		free(tmp);

		JLIB_LEAVE;
		return (UBYTE *) NULL;
	}
	JLIB_LEAVE;
	return tmp;
}
#endif				/* JDEFAULT_LOAD_PAL */


/*+------------------------------------------------------------------------+ */
/*|convert a palette to greyscale.                                         | */
/*+------------------------------------------------------------------------+ */
void pal_to_grey(UBYTE * pal)
{
	int i;
	unsigned long conv;

	JLIB_ENTER("pal_to_grey");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	for (i = 0; i < SCREEN_NUM_COLORS; i++) {

		conv = (((R_COMP * pal[i * 3]) + (G_COMP * pal[i * 3 + 1]) + (B_COMP * pal[i * 3 + 2])) >> 16);

		pal[i * 3] = pal[i * 3 + 1] = pal[i * 3 + 2] = (UBYTE) conv;
	}

	JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|convert a palette to redscale.                                          | */
/*+------------------------------------------------------------------------+ */
void pal_to_red(UBYTE * pal)
{
	int i;
	unsigned long conv;

	JLIB_ENTER("pal_to_red");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	for (i = 0; i < SCREEN_NUM_COLORS; i++) {

		conv = (((R_COMP * pal[i * 3]) + (G_COMP * pal[i * 3 + 1]) + (B_COMP * pal[i * 3 + 2])) >> 16);

		pal[i * 3] = (UBYTE) conv;
		pal[i * 3 + 1] = pal[i * 3 + 2] = 0;
	}

	JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|convert a palette to greenscale.                                        | */
/*+------------------------------------------------------------------------+ */
void pal_to_green(UBYTE * pal)
{
	int i;
	unsigned long conv;

	JLIB_ENTER("pal_to_green");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	for (i = 0; i < SCREEN_NUM_COLORS; i++) {

		conv = (((R_COMP * pal[i * 3]) + (G_COMP * pal[i * 3 + 1]) + (B_COMP * pal[i * 3 + 2])) >> 16);

		pal[i * 3 + 1] = (UBYTE) conv;
		pal[i * 3] = pal[i * 3 + 2] = 0;
	}

	JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|convert a palette to bluescale.                                         | */
/*+------------------------------------------------------------------------+ */
void pal_to_blue(UBYTE * pal)
{
	int i;
	unsigned long conv;

	JLIB_ENTER("pal_to_blue");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	for (i = 0; i < SCREEN_NUM_COLORS; i++) {

		conv = (((R_COMP * pal[i * 3]) + (G_COMP * pal[i * 3 + 1]) + (B_COMP * pal[i * 3 + 2])) >> 16);

		pal[i * 3 + 2] = (UBYTE) conv;
		pal[i * 3] = pal[i * 3 + 1] = 0;
	}

	JLIB_LEAVE
}


/*+------------------------------------------------------------------------+ */
/*|find the closest matching colour in pal to the given rgb.               | */
/*+------------------------------------------------------------------------+ */
UBYTE pal_closest_color_rgb(UBYTE * pal, UBYTE r, UBYTE g, UBYTE b)
{
	int i, temp, best = 256;
	UBYTE rd, gd, bd, closest = 0;

	JLIB_ENTER("pal_closest_color_rgb");

#ifndef JLIB_PRODUCTION
	if (pal == NULL) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	for (i = SCREEN_NUM_COLORS - 1; i != 0; i--) {	/* skips color 0 */
		rd = ABS(r - pal[i * 3]);
		gd = ABS(g - pal[i * 3 + 1]);
		bd = ABS(b - pal[i * 3 + 2]);

		temp = MAX(MAX(rd, gd), bd);

		if (temp < best) {
			best = temp;
			closest = i;
		}
	}

	JLIB_LEAVE;
	return closest;
}


/*+------------------------------------------------------------------------+ */
/*|Get a copy of the default palette.                                      | */
/*+------------------------------------------------------------------------+ */
UBYTE *pal_get_default(void)
{
	UBYTE *pal;

	pal = pal_init();
	pal_copy(pal, __jlib_pal);
	return pal;
}

/*+------------------------------------------------------------------------+ */
/*|Create a remapping index from pal2 to pal1                              | */
/*+------------------------------------------------------------------------+ */
UBYTE *pal_create_remap_index(UBYTE * pal1, UBYTE * pal2)
{
	UBYTE *remap;
	int i;

#ifndef JLIB_PRODUCTION
	if ((pal1 == NULL) || (pal2 == NULL)) {
		jlib_exit(jlib_msg(JLIB_ENULL));
	}
#endif

	if ((remap = malloc(SCREEN_NUM_COLORS)) == NULL) {
		jlib_exit(jlib_msg(JLIB_EMALLOC));
	}
	remap[0] = 0;		/* color 0 always maps to color 0 */

	for (i = SCREEN_NUM_COLORS - 1; i != 0; i--) {	/* skips color 0 */
		remap[i] = pal_closest_color_rgb(pal1, pal2[i * 3], pal2[i * 3 + 1], pal2[i * 3 + 2]);
	}

	return remap;
}
