/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdlib.h>
#include <jlib.h>

static void masked_copy(UBYTE * s, UBYTE * d, int len);

#define	BR buffer_rec

/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer without clipping           | */
/*+------------------------------------------------------------------------+ */
void buff_blit_buff_toNC(BR *d, int x, int y, BR *s, int x1, int y1, int x2, int y2)
{
	UBYTE *src, *dest;

	JLIB_ENTER("buff_blit_buff_toNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	if (x2 < x1)	JLIB_SWAP(x1, x2);
	if (y2 < y1)	JLIB_SWAP(y1, y2);

	src = B_OFFSET(s,y1) + x1;
	dest = B_OFFSET(d,y) + x;

	y2 -= y1;       	/* vertical range */
	x2 = x2 - x1 + 1;	/* horizontal range */

	do {
		MEM_COPY_ALIGN_1(src, dest, x2);
		src += B_X_SIZE(s);
		dest += B_X_SIZE(d);
	} while(y2--);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer with clipping              | */
/*+------------------------------------------------------------------------+ */
void buff_blit_buff_to(BR *d, int x, int y, BR *s, int x1, int y1, int x2, int y2)
{
	int width, height;
	UBYTE *src, *dest;

	JLIB_ENTER("buff_blit_buff_to");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	if (x2 < x1)
		JLIB_SWAP(x1, x2);
	if (y2 < y1)
		JLIB_SWAP(y1, y2);

	/* clip src */
	if (x1 < B_CX1(s)) {
		x += (B_CX1(s) - x1);
		x1 = B_CX1(s);
	}
	else
		if (x1 > B_CX2(s)) {
			JLIB_LEAVE;
			return;
		}

	if (x2 > B_CX2(s))
		x2 = B_CX2(s);
	else
		if (x2 < B_CX1(s)) {
			JLIB_LEAVE;
			return;
		}

	if (y1 < B_CY1(s)) {
		y += (B_CY1(s) - y1);
		y1 = B_CY1(s);
	}
	else
		if (y1 > B_CY2(s)) {
			JLIB_LEAVE;
			return;
		}

	if (y2 > B_CY2(s))
		y2 = B_CY2(s);
	else
		if (y2 < B_CY1(s)) {
			JLIB_LEAVE;
			return;
		}

	height = y2 - y1 + 1;
	width = x2 - x1 + 1;

	/* clip dest */
	if (x < B_CX1(d)) {
		int cut = B_CX1(d) - x;
		x1 += cut;
		if (x1 > B_CX2(s)) {
			JLIB_LEAVE;
			return;
		}
		width -= cut;
		x = B_CX1(d);
	}
	else
		if (x > B_CX2(d)) {
			JLIB_LEAVE;
			return;
		}

	if (x + width > B_CX2(d))
		width = B_CX2(d) - x + 1;
	else
		if (x + width < B_CX1(d)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(d)) {
		int cut = B_CY1(d) - y;
		y1 += cut;
		if (y1 > B_CY2(s)) {
			JLIB_LEAVE;
			return;
		}
		height -= cut;
		y = B_CY1(d);
	}
	else
		if (y > B_CY2(d)) {
			JLIB_LEAVE;
			return;
		}

	if (y + height > B_CY2(d))
		height = B_CY2(d) - y + 1;
	else
		if (y + height < B_CY1(d)) {
			JLIB_LEAVE;
			return;
		}

	src = B_OFFSET(s, y1) + x1;
	dest = B_OFFSET(d, y) + x;

	while (height--) {
		MEM_COPY_ALIGN_1(src, dest, width);
		src += B_X_SIZE(s);
		dest += B_X_SIZE(d);
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of an image to a buffer without clipping                 | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_img_toNC(BR *d, int x, int y, image *img, int x1, int y1, int x2, int y2)
{
	JLIB_ENTER("buff_blit_img_toNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(d);
	jlib_check_image(img);
#endif

	buff_blit_buff_toNC(d, x, y, IMG_BUFFER(img), x1, y1, x2, y2);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of an image to a buffer with clipping                    | */
/*+------------------------------------------------------------------------+ */
JINLINE void buff_blit_img_to(BR *d, int x, int y, image *img, int x1, int y1, int x2, int y2)
{
	JLIB_ENTER("buff_blit_img_to");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(d);
	jlib_check_image(img);
#endif

	buff_blit_buff_to(d, x, y, IMG_BUFFER(img), x1, y1, x2, y2);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Copy a line of pixels excluding blanks (unoptimised).                   | */
/*+------------------------------------------------------------------------+ */
JINLINE static void masked_copy(UBYTE *s, UBYTE *d, int len)
{
	while (len--) {
		if (*s)
			*d = *s;
		++s;
		++d;
	}
}


/*+------------------------------------------------------------------------+ */
/*|blit a masked portion of a buffer to another buffer without clipping    | */
/*+------------------------------------------------------------------------+ */
void buff_stencil_buff_toNC(BR *d, int x, int y, BR *s, int x1, int y1, int x2, int y2)
{
	UBYTE *src, *dest;

	JLIB_ENTER("buff_blit_buff_toNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	if (x2 < x1)	JLIB_SWAP(x1, x2);
	if (y2 < y1)	JLIB_SWAP(y1, y2);

	src = B_OFFSET(s,y1) + x1;
	dest = B_OFFSET(d,y) + x;

	y2 -= y1;       	/* vertical range */
	x2 = x2 - x1 + 1;	/* horizontal range */

	do {
		masked_copy(src, dest, x2);
		src += B_X_SIZE(s);
		dest += B_X_SIZE(d);
	} while(y2--);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a masked portion of a buffer to another buffer with clipping       | */
/*+------------------------------------------------------------------------+ */
void buff_stencil_buff_to(BR *d, int x, int y, BR *s, int x1, int y1, int x2, int y2)
{
	int width, height;
	UBYTE *src, *dest;

	JLIB_ENTER("buff_blit_buff_to");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(s);
	jlib_check_buffer(d);
#endif

	if (x2 < x1)
		JLIB_SWAP(x1, x2);
	if (y2 < y1)
		JLIB_SWAP(y1, y2);

	/* clip src */
	if (x1 < B_CX1(s)) {
		x += (B_CX1(s) - x1);
		x1 = B_CX1(s);
	}
	else
		if (x1 > B_CX2(s)) {
			JLIB_LEAVE;
			return;
		}

	if (x2 > B_CX2(s))
		x2 = B_CX2(s);
	else
		if (x2 < B_CX1(s)) {
			JLIB_LEAVE;
			return;
		}

	if (y1 < B_CY1(s)) {
		y += (B_CY1(s) - y1);
		y1 = B_CY1(s);
	}
	else
		if (y1 > B_CY2(s)) {
			JLIB_LEAVE;
			return;
		}

	if (y2 > B_CY2(s))
		y2 = B_CY2(s);
	else
		if (y2 < B_CY1(s)) {
			JLIB_LEAVE;
			return;
		}

	height = y2 - y1 + 1;
	width = x2 - x1 + 1;

	/* clip dest */
	if (x < B_CX1(d)) {
		int cut = B_CX1(d) - x;
		x1 += cut;
		if (x1 > B_CX2(s)) {
			JLIB_LEAVE;
			return;
		}
		width -= cut;
		x = B_CX1(d);
	}
	else
		if (x > B_CX2(d)) {
			JLIB_LEAVE;
			return;
		}

	if (x + width > B_CX2(d))
		width = B_CX2(d) - x + 1;
	else
		if (x + width < B_CX1(d)) {
			JLIB_LEAVE;
			return;
		}

	if (y < B_CY1(d)) {
		int cut = B_CY1(d) - y;
		y1 += cut;
		if (y1 > B_CY2(s)) {
			JLIB_LEAVE;
			return;
		}
		height -= cut;
		y = B_CY1(d);
	}
	else
		if (y > B_CY2(d)) {
			JLIB_LEAVE;
			return;
		}

	if (y + height > B_CY2(d))
		height = B_CY2(d) - y + 1;
	else
		if (y + height < B_CY1(d)) {
			JLIB_LEAVE;
			return;
		}

	src = B_OFFSET(s, y1) + x1;
	dest = B_OFFSET(d, y) + x;

	while (height--) {
		masked_copy(src, dest, width);
		src += B_X_SIZE(s);
		dest += B_X_SIZE(d);
	}

	JLIB_LEAVE;
}
