/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>

/* Demonstrates most of the primitives in a clipping region */

/* Note that it is deliberately possible to crash this program since it
** uses non checking (NC) function calls when clipping is turned off.
*/

#define COLOR	100
#define COLOR2	15
#define RANDOM(u) ( rand() % ( ((u) + 1)  ) )

/* Primitives */
#define POINT		1
#define LINE		2
#define HLINE		3
#define VLINE		4
#define BOX		5
#define FBOX		6
#define CHAR		7
#define STRING		8
#define STAMP_SPR	9
#define STENCIL_SPR	10
#define STAMP_SPRC	11
#define STENCIL_SPRC	12
#define STAMP_SPRT	13
#define STENCIL_SPRT	14
#define DRSPRITE	15
#define POLY		16
#define TRIANGLE	17
#define FTRIANGLE	18
#define BLIT		19
#define CBLIT		20
#define BSTENCIL	21
#define BSCALE		22
#define CBSCALE		23
#define CIRCLE		24
#define FCIRCLE		25

char *documentation = "Lower Right L-Click=Exit; Upper Left L-Click=Clipping&Box; R-Click=Primitive";

/* coordinates of click areas */
#define	CLEFT		(SCREEN_WIDTH/4)
#define	CRIGHT		(SCREEN_WIDTH-(SCREEN_WIDTH/4))
#define CTOP		(SCREEN_HEIGHT/4)
#define CBOTTOM		(SCREEN_HEIGHT-(SCREEN_HEIGHT/4))

char *primitives[] = {
	"none",
	"point",
	"line",
	"hline",
	"vline",
	"box",
	"filled box",
	"char",
	"string",
	"stamp sprite",
	"stencil sprite",
	"stamp sprite colour",
	"stencil sprite colour",
	"stamp sprite texture",
	"stencil sprite texture",
	"draw sprite",
	"convex poly",
	"triangle",
	"filled triangle",
	"blit",
	"blit clipped at source",
	"stencil buff",
	"scaled buff",
	"scaled buff clipped at source",
	"circle",
	"filled circle"
};

int X1 = (SCREEN_WIDTH/10);
int X2 = (SCREEN_WIDTH-(SCREEN_WIDTH/10));
int Y1 = (SCREEN_HEIGHT/10);
int Y2 = (SCREEN_HEIGHT-(SCREEN_HEIGHT/10));


int main(int argc, char **argv)
{
	int xpos,ypos,buttons,primitive=POINT,nc=1;
	buffer_rec *offscreen,*texture;
	sprite_system *sprites;
	int x[4],y[4],i,frame=0;
	UBYTE *pal;

	if (argc > 2)
		JLIB_DEBUG_TRACE;
	else
		if (argc > 1)
			JLIB_DEBUG_ON;
		else
			JLIB_DEBUG_OFF;

	/* set up the screen */
	screen_set_app_title("Clipping Demo");

	if (!screen_set_video_mode())
		jlib_exit("Screen Setup failed.");
 
	sprites = sprite_init(1,64);

	if(sprites == NULL)
		jlib_exit("Initialisation Failed.");

	if(sprite_load("demo.spr",sprites) != SUCCESS)
		jlib_exit("Sprite Load Failed.");

	sprite_set_xy(sprites,0,xpos-10,ypos-10);
	sprite_set_an_frame(sprites,0,frame);
	sprite_turn_on(sprites,0);

	offscreen = buff_init(SCREEN_WIDTH,SCREEN_HEIGHT);
	buff_set_clip_region(offscreen,X1,Y1,X2,Y2);
 
	srand(time(NULL));
	texture = buff_init(SPR_MAX_X,SPR_MAX_Y);
	for(i = 0; i < B_SIZE(texture); i++) {
		UBYTE col = RANDOM(16);
		if (col > 12) col = 0;	/* Ensure lots of 0's in the buffer */
		*(B_BUFF_PTR(texture) + i) = col;
	}
	
	if (!MOUSE_AVAILABLE)
		jlib_exit("No Mouse Support.");

	if (!mouse_present())
		jlib_exit("No Mouse Installed.");

	mouse_hide_pointer();

	pal = pal_load("demo.pal");

	if (pal == NULL)
		jlib_exit("Failed to load palette.");

	screen_block_set_pal(pal);
	screen_put_pal(0,0,0,0);
	screen_put_pal(1,0,0,255);
	screen_put_pal(2,0,255,0);
	screen_put_pal(COLOR,255,0,0);
	screen_put_pal(COLOR2,255,255,255);
	
	do{
		mouse_get_status(&xpos,&ypos,&buttons);

		buff_draw_rectNC(offscreen, 0, 0, B_MAX_X(offscreen), B_MAX_Y(offscreen), 1);

		if (BUTTON_CHECK(buttons,MOUSE_B_LEFT)){
			if ((xpos < CLEFT) && (ypos < CTOP)) {
				nc = ! nc;
				X2-=20;
				Y2-=10;
				buff_set_clip_region(offscreen,X1,Y1,X2,Y2);
				do{
					mouse_get_status(&xpos,&ypos,&buttons);
				} while(BUTTON_CHECK(buttons,MOUSE_B_LEFT));
			}
			else
				/* Dont quit if button not in quit area */
				if ((xpos < CRIGHT) || (ypos < CBOTTOM))
					buttons = 0;
		}

		if(nc)
			buff_draw_stringNC(offscreen,"Clipping",0,0,COLOR2);
		else
			buff_draw_stringNC(offscreen,"Not Clipping",0,0,COLOR2);

		buff_draw_stringNC(offscreen,primitives[primitive],SCREEN_WIDTH/4,0,COLOR2);

		buff_draw_stringNC(offscreen,documentation,0,SCREEN_HEIGHT-CHAR_HEIGHT('A')-1,COLOR2);

		buff_draw_rect(offscreen,X1,Y1,X2,Y2,COLOR);
		buff_draw_rect(offscreen,X1+1,Y1+1,X2-1,Y2-1,0);

		switch(primitive){
		case POINT:
			if (nc)
				buff_draw_point(offscreen,xpos,ypos,COLOR2);
			else
				buff_draw_pointNC(offscreen,xpos,ypos,COLOR2);
			break;
		case LINE:
			if (nc) {
				buff_draw_line(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
				buff_draw_line(offscreen,xpos,ypos+25,xpos+25,ypos,COLOR2);
			}
			else {
				buff_draw_lineNC(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
				buff_draw_lineNC(offscreen,xpos,ypos+25,xpos+25,ypos,COLOR2);
			}
			break;
		case HLINE:
			if (nc)
				buff_draw_h_line(offscreen,xpos-15,ypos,xpos+15,COLOR2);
			else
				buff_draw_h_lineNC(offscreen,xpos-15,ypos,xpos+15,COLOR2);
			break;
		case VLINE:
			if (nc)
				buff_draw_v_line(offscreen,xpos,ypos-15,ypos+15,COLOR2);
			else
				buff_draw_v_lineNC(offscreen,xpos,ypos-15,ypos+15,COLOR2);
			break;
		case BOX:
			if (nc)
				buff_draw_box(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
			else
				buff_draw_boxNC(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
			break;
		case FBOX:
			if (nc)
				buff_draw_rect(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
			else
				buff_draw_rectNC(offscreen,xpos,ypos,xpos+25,ypos+25,COLOR2);
			break;
		case CHAR:
			if (nc)
				buff_draw_char(offscreen,'W',xpos,ypos,COLOR2);
			else
				buff_draw_charNC(offscreen,'W',xpos,ypos,COLOR2);
			break;
		case STRING:
			if (nc)
				buff_draw_string(offscreen,"Hello",xpos,ypos,COLOR2);
			else
				buff_draw_stringNC(offscreen,"Hello",xpos,ypos,COLOR2);
			break;
		case STAMP_SPR:
			if (nc)
				buff_stamp_sprite(sprites,1,offscreen,xpos,ypos);
			else
				buff_stamp_spriteNC(sprites,1,offscreen,xpos,ypos);
			break;
		case STENCIL_SPR: 
			if (nc)
				buff_stencil_sprite(sprites,1,offscreen,xpos,ypos);
			else
				buff_stencil_spriteNC(sprites,1,offscreen,xpos,ypos);
			break;
		case STAMP_SPRC:
			if (nc)
				buff_stamp_sprite_color(sprites,1,offscreen,xpos,ypos,COLOR2);
			else
				buff_stamp_sprite_colorNC(sprites,1,offscreen,xpos,ypos,COLOR2);
			break;
		case STENCIL_SPRC: 
			if (nc)
				buff_stencil_sprite_color(sprites,1,offscreen,xpos,ypos,COLOR2);
			else
				buff_stencil_sprite_colorNC(sprites,1,offscreen,xpos,ypos,COLOR2);
			break;
		case STENCIL_SPRT: 
			if (nc)
				buff_stencil_sprite_buff(sprites,1,offscreen,xpos,ypos,texture);
			else
				buff_stencil_sprite_buffNC(sprites,1,offscreen,xpos,ypos,texture);
			break;
		case STAMP_SPRT: 
			if (nc)
				buff_stamp_sprite_buff(sprites,1,offscreen,xpos,ypos,texture);
			else
				buff_stamp_sprite_buffNC(sprites,1,offscreen,xpos,ypos,texture);
			break;
		case DRSPRITE: 
			sprite_set_xy(sprites,0,xpos-20,ypos-20);
			sprite_set_an_frame(sprites,0,frame);
			if (nc) {
				buff_save_sprite(sprites,0,offscreen);
				buff_rest_sprite(sprites,0,offscreen);
				buff_draw_sprite(sprites,0,offscreen);
			}
			else{
				buff_save_spriteNC(sprites,0,offscreen);
				buff_rest_spriteNC(sprites,0,offscreen);
				buff_draw_spriteNC(sprites,0,offscreen);
			}
			break;
		case POLY:
			x[0] = xpos - 25;
			x[1] = x[3] = xpos;
			x[2] = xpos + 25;			
			y[1] = ypos - 25;
			y[0] = y[2] = ypos;
			y[3] = ypos + 25;			
			if (nc)
				buff_convex_poly(offscreen,4,x,y,COLOR2);
			else
				buff_convex_polyNC(offscreen,4,x,y,COLOR2);
			break;
		case TRIANGLE:
			x[0] = xpos - 25;
			x[1] = xpos;
			x[2] = xpos + 25;			
			y[1] = ypos - 25;
			y[0] = y[2] = ypos;
			if (nc)
				buff_draw_triangle(offscreen,x[0],y[0],x[1],y[1],x[2],y[2],COLOR2);
			else
				buff_draw_triangleNC(offscreen,x[0],y[0],x[1],y[1],x[2],y[2],COLOR2);
			break;
		case FTRIANGLE:
			x[0] = xpos - 25;
			x[1] = xpos;
			x[2] = xpos + 25;			
			y[1] = ypos - 25;
			y[0] = y[2] = ypos;
			if (nc)
				buff_filled_triangle(offscreen,x[0],y[0],x[1],y[1],x[2],y[2],COLOR2);
			else
				buff_filled_triangleNC(offscreen,x[0],y[0],x[1],y[1],x[2],y[2],COLOR2);
			break;
		case BLIT:
			if (nc)
				buff_blit_buff_to(offscreen,xpos-20,ypos-20,texture,0,0,B_MAX_X(texture),B_MAX_Y(texture));
			else
				buff_blit_buff_toNC(offscreen,xpos-20,ypos-20,texture,0,0,B_MAX_X(texture),B_MAX_Y(texture));
			break;
		case CBLIT:
			if (nc)
				buff_blit_buff_to(offscreen,xpos-20,ypos-20,texture,-20,-20,B_MAX_X(texture)+20,B_MAX_Y(texture)+20);
			else
				buff_blit_buff_toNC(offscreen,xpos-20,ypos-20,texture,-20,-20,B_MAX_X(texture)+20,B_MAX_Y(texture)+20);
			break;
		case BSTENCIL:
			if (nc)
				buff_stencil_buff_to(offscreen,xpos-20,ypos-20,texture,0,0,B_MAX_X(texture),B_MAX_Y(texture));
			else
				buff_stencil_buff_toNC(offscreen,xpos-20,ypos-20,texture,0,0,B_MAX_X(texture),B_MAX_Y(texture));
			break;
		case BSCALE:
			if (nc)
				buff_scale_full_buff_to(offscreen,xpos-20,ypos-20,xpos+B_MAX_X(texture),ypos+B_MAX_X(texture),texture);
			else
				buff_scale_full_buff_toNC(offscreen,xpos-20,ypos-20,xpos+B_MAX_X(texture),ypos+B_MAX_X(texture),texture);
			break;
		case CBSCALE:
			if (nc)
				buff_scale_buff_to(offscreen,xpos-20,ypos-20,xpos+B_MAX_X(texture),ypos+B_MAX_X(texture),texture, -20, -20, B_MAX_X(texture)+20, B_MAX_Y(texture)+20);
			else
				buff_scale_buff_toNC(offscreen,xpos-20,ypos-20,xpos+B_MAX_X(texture),ypos+B_MAX_X(texture),texture, -20, -20, B_MAX_X(texture)+20, B_MAX_Y(texture)+20);
			break;
		case CIRCLE:
			if (nc) {
				buff_draw_circle(offscreen,xpos,ypos,25,COLOR2);
			}
			else
				buff_draw_circleNC(offscreen,xpos,ypos,25,COLOR2);
			break;
		case FCIRCLE:
			if (nc) {
				buff_filled_circle(offscreen,xpos,ypos,25,COLOR2);
			}
			else
				buff_filled_circleNC(offscreen,xpos,ypos,25,COLOR2);
			break;
		}
    
		screen_blit_fs_buffer(offscreen);

		if (BUTTON_CHECK(buttons,MOUSE_B_RIGHT) || (BUTTON_CHECK(buttons,MOUSE_B_MIDDLE))) {
			primitive++;
			if (primitive > FCIRCLE)
				primitive = POINT;
			do{
				mouse_get_status(&xpos,&ypos,&buttons);
			}while(BUTTON_CHECK(buttons,MOUSE_B_RIGHT) || BUTTON_CHECK(buttons,MOUSE_B_MIDDLE));
		}

		frame++;
		if (frame > 5)
			frame = 1;

	} while(!BUTTON_CHECK(buttons,MOUSE_B_LEFT));

	mouse_closedown();
	sprite_free(sprites);
	buff_free(offscreen);

	screen_restore_video_mode();

	return 1;
}