/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <jlib.h>
#include <vga.h>
#include <stdlib.h>

char	version_string[] = "Linux svgalib";
int	__jlib_pages;
int	__jlib_is_linear=0;
int	__jlib_screen_initted = 0;
int	__jlib_current_page = 0;
vga_modeinfo	*__jlib_modeinfo;
buffer_rec	__jlib_screen_buff;


/*+------------------------------------------------------------------------+ */
/*| Set the video mode.                                                    | */
/*+------------------------------------------------------------------------+ */
int screen_set_video_mode(void)
{
	if (__jlib_screen_initted)
		return 1;

	JLIB_SIGNAL_SETUP;

	if (!vga_hasmode(JLIB_MODE)) {
		JLIB_PRINT_DEBUG_INFO("Mode is not available");
		return 0;
	}
	__jlib_modeinfo = vga_getmodeinfo(JLIB_MODE);
	__jlib_pages = (__jlib_modeinfo->height * __jlib_modeinfo->linewidth + 65535) >> 16;

	__jlib_screen_buff.width = SCREEN_WIDTH;
	__jlib_screen_buff.height = SCREEN_HEIGHT;

	if (!vga_init()) {
		if (!vga_setmode(JLIB_MODE)) {
			vga_setpage(0);
			/* Set linear addressing mode if possible */
#if SCREEN_WIDTH != 320
			if (__jlib_modeinfo->flags & CAPABLE_LINEAR)
				if (vga_setlinearaddressing() != -1)
					__jlib_is_linear = 1;
				else {
					JLIB_PRINT_DEBUG_INFO("vga_setlinearaddressing() failed.");
					vga_setmode(JLIB_MODE); /* reset mode */
				}
#else
				__jlib_is_linear = 1;
#endif
			__jlib_screen_initted = 1;
			__jlib_screen_buff.buffer = vga_getgraphmem();

			/* reset clipping region */
			B_CX1(&__jlib_screen_buff) = B_CY1(&__jlib_screen_buff) = 0;
			B_CX2(&__jlib_screen_buff) = B_MAX_X(&__jlib_screen_buff);
			B_CY2(&__jlib_screen_buff) = B_MAX_Y(&__jlib_screen_buff);

			if (__jlib_is_linear) {
				int i;
				/* Set up direct video write buffer */
				__jlib_screen_buff.offset = (UBYTE **) malloc(SCREEN_HEIGHT * sizeof(UBYTE *));
				if (__jlib_screen_buff.offset == NULL)
					jlib_exit(jlib_msg(JLIB_EMALLOC));
				for (i = 0; i < SCREEN_HEIGHT; i++)
					__jlib_screen_buff.offset[i] = __jlib_screen_buff.buffer + (i * SCREEN_WIDTH);
			}
			return 1;	/* mode set OK */
		}
		JLIB_PRINT_DEBUG_INFO("vga_setmode() failed.");
	}
	JLIB_PRINT_DEBUG_INFO("vga_init() failed.");

	return 0;	/* mode set failed */
}


/*+------------------------------------------------------------------------+ */
/*|restore the video mode.                                                 | */
/*+------------------------------------------------------------------------+ */
void screen_restore_video_mode(void)
{
	if (__jlib_screen_initted) {
		__jlib_screen_initted = 0;
		vga_setmode(TEXT);
	}
}


/*+------------------------------------------------------------------------+ */
/*|Return the current page number.                                         | */
/*+------------------------------------------------------------------------+ */
int screen_get_page(void)
{
#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif
	return 1;
}


/*+------------------------------------------------------------------------+ */
/*|set the given  page number.                                             | */
/*+------------------------------------------------------------------------+ */
void screen_set_page(int page)
{
#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif
	/* do nothing */
}


/*+------------------------------------------------------------------------+ */
/*|Show the current drawing page.                                          | */
/*+------------------------------------------------------------------------+ */
void screen_show_page(int page)
{
#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif
	/* do nothing */
}


/*+------------------------------------------------------------------------+ */
/*|Wait for the retrace beam to move offscreen.                            | */
/*+------------------------------------------------------------------------+ */
void screen_wait_vsync(void)
{
	JLIB_ENTER("screen_wait_vsync");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif
	vga_waitretrace();
	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|Fill the screen with a given color.                                     | */
/*+------------------------------------------------------------------------+ */
void screen_fill(UBYTE color)
{
#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if (__jlib_is_linear)
		memset(__jlib_screen_buff.buffer, color, SCR_SIZE);
	else {
		int i;
		UBYTE *dest = __jlib_screen_buff.buffer;

		for (i = 0; i < __jlib_pages; i++) {
			vga_setpage(i);
			MEM_STORE_LONG(dest, color, 65536);
		}
		__jlib_current_page = __jlib_pages - 1;
	}
}


/*+------------------------------------------------------------------------+ */
/*|Clear the screen.                                                       | */
/*+------------------------------------------------------------------------+ */
void screen_clear(void)
{
	screen_fill(0);
}


/*+------------------------------------------------------------------------+ */
/*|Return a buffer representing the screen if available.                   | */
/*+------------------------------------------------------------------------+ */
buffer_rec *screen_buff_init(void)
{
	JLIB_ENTER("screen_buff_init");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	/* Only linear modes allow direct screen access */
	if (__jlib_is_linear) {
		JLIB_LEAVE;
		return &__jlib_screen_buff;
	}
	
	JLIB_LEAVE;
	return NULL;
}