/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>
#include "lz.h"
#include "rle.h"

/* magic number that identifies jlib files */
#define JLB_MAGIC_NUMBER 0xd00d


/*+------------------------------------------------------------------------+ */
/*|Load a .JLB image file from a file name.                                | */
/*+------------------------------------------------------------------------+ */
image *image_load_jlb(char *filename)
{
	FILE *infile;
	image *img;

	JLIB_ENTER("image_load_jlb");

	/* Open the file */
	if ((infile = fopen(filename, "rb")) == NULL) {
		JLIB_SPRINTF("%s File Open Error.", filename);
		JLIB_LEAVE;
		return (image *) NULL;
	}

	img = image_load_jlb_fp(infile);

	fclose(infile);

	JLIB_LEAVE;
	return img;
}


/*+------------------------------------------------------------------------+ */
/*|Load a .JLB image file from a file pointer.                             | */
/*+------------------------------------------------------------------------+ */
image *image_load_jlb_fp(FILE *infile)
{
	image *img;
	USHORT magic_number;
	long comp_length, filler, rlen;
	UBYTE *sp, *rp;
	int i;
	ULONG thirtytwobit;
	USHORT t_ushort;

	JLIB_ENTER("image_load_jlb_fp");

	if (infile == NULL) {
		JLIB_LEAVE;
		return (image *) NULL;
	}

	/* read in header information */
	fseek(infile, 0L, 0);

	/* read magic number */
	jio_read_elementary_type(infile, &magic_number, sizeof(USHORT));

	/* check magic number */
	if (magic_number != JLB_MAGIC_NUMBER) {
		JLIB_PRINT_DEBUG_INFO("File is not a .JLB File.");
		JLIB_LEAVE;
		return (image *) NULL;
	}

	/* Allocate header record */
	img = (image *) malloc(sizeof(image));

	if (img == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* Allocate palette */
	IMG_PALETTE(img) = pal_init();

	/* Get width + height */
	jio_read_elementary_type(infile, &t_ushort, sizeof(USHORT));
	IMG_WIDTH(img) = (int) t_ushort;
	jio_read_elementary_type(infile, &t_ushort, sizeof(USHORT));
	IMG_HEIGHT(img) = (int) t_ushort;

	JLIB_SPRINTF("JLB Width: %d.", IMG_WIDTH(img));
	JLIB_SPRINTF("JLB Height: %d.", IMG_HEIGHT(img));

	image_setup(img);

	/* get length of rle data */
	jio_read_elementary_type(infile, &thirtytwobit, sizeof(ULONG));
	rlen = (long) thirtytwobit;

	/* get length of compressed data */
	jio_read_elementary_type(infile, &thirtytwobit, sizeof(ULONG));
	comp_length = (long) thirtytwobit;

	/* grab memory to hold compressed data */
	if ((sp = (UBYTE *) malloc((comp_length) * sizeof(UBYTE))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* grab memory to hold rle data */
	if ((rp = (UBYTE *) malloc(rlen)) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* read compressed data into compressed buffer */
	for (i = 0; i < comp_length; i++)
		sp[i] = (UBYTE) fgetc(infile);

	/* uncompress from the compressed buffer to the rle buffer */
	lz_decompress(sp, comp_length, rp, &filler);

	/* free the memory for the compressed buffer */
	free(sp);

	/* run length decode data into image buffer */
	rle_decode_prealloc(rp, IMG_DATA_PTR(img));

	/* get length of compressed palette data */
	jio_read_elementary_type(infile, &thirtytwobit, sizeof(ULONG));

	comp_length = (long) thirtytwobit;

	/* grab memory to hold compressed palette data */
	if ((sp = (UBYTE *) malloc((comp_length) * sizeof(UBYTE))) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* read compressed pal data into compressed pal buffer */
	for (i = 0; i < comp_length; i++)
		sp[i] = (UBYTE) fgetc(infile);

	/* uncompress from the compressed pal buff to the uncompressed buffer */
	lz_decompress(sp, comp_length, IMG_PALETTE(img), &filler);

	/* free the memory for the compressed buffer */
	free(sp);

	JLIB_LEAVE;
	return img;
}


/*+------------------------------------------------------------------------+ */
/*|Save a buffer to a .JLB image file.                                     | */
/*+------------------------------------------------------------------------+ */
int image_save_jlb(char *filename, buffer_rec * buff, UBYTE * pal)
{
	FILE *outfile;
	int retval;

	JLIB_ENTER("image_save_jlb");

	/* Open the file */
	if ((outfile = fopen(filename, "wb")) == NULL) {
		JLIB_SPRINTF("%s File Open Error.", filename);
		JLIB_LEAVE;
		return 0;
	}

	retval = image_save_jlb_fp(outfile, buff, pal);

	fclose(outfile);

	JLIB_LEAVE;
	return retval;
}


/*+------------------------------------------------------------------------+ */
/*|Save a buffer as a .JLB to a file pointer.                              | */
/*+------------------------------------------------------------------------+ */
int image_save_jlb_fp(FILE *outfile, buffer_rec * buff, UBYTE * pal)
{
	USHORT magic_number = JLB_MAGIC_NUMBER;
	long *comp_length, rlen;
	UBYTE *sp, *rp;
	int i;
	ULONG thirtytwobit;
	USHORT t_ushort;

	JLIB_ENTER("image_save_jlb_fp");

	if ((buff == NULL) || (pal == NULL) || (outfile == NULL)) {
		JLIB_PRINT_DEBUG_INFO("JLB write failed with NULL parameter(s)");
		JLIB_LEAVE;
		return 0;
	}

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
	jlib_check_palette(pal);
#endif

	/* grab memory to hold compressed length */
	comp_length = (long *) malloc(sizeof(long));

	if (comp_length == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	fseek(outfile, 0L, 0);

	/* write magic number */
	jio_write_elementary_type(outfile, &magic_number, sizeof(USHORT));

	/* put width + height */
	t_ushort = (USHORT) B_X_SIZE(buff);
	jio_write_elementary_type(outfile, &t_ushort, sizeof(USHORT));
	t_ushort = (USHORT) B_Y_SIZE(buff);;
	jio_write_elementary_type(outfile, &t_ushort, sizeof(USHORT));

	JLIB_SPRINTF("Buff Width: %d.", B_X_SIZE(buff));
	JLIB_SPRINTF("Buff Height: %d.", B_Y_SIZE(buff));

	/* run length encode data into rp */
	rlen = rle_encode(B_BUFF_PTR(buff), B_SIZE(buff), &rp);

	/* put length of rle data */
	thirtytwobit = (ULONG) rlen;
	jio_write_elementary_type(outfile, &rlen, sizeof(ULONG));

	/* grab memory to hold compressed rle data */
	if ((sp = (UBYTE *) malloc(rlen + 256)) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* compress from the rle buffer to the compressed buffer */
	lz_compress(rp, rlen, sp, comp_length);

	/* free the rle buffer */
	free(rp);

	/* put length of compressed data */
	thirtytwobit = (ULONG) * comp_length;

	jio_write_elementary_type(outfile, &thirtytwobit, sizeof(ULONG));

	/* write compressed data into the file */
	for (i = 0; i < *comp_length; i++)
		fputc(sp[i], outfile);

	/* free the memory for the compressed buffer */
	free(sp);

	/* grab memory to hold compressed palette data */
	if ((sp = (UBYTE *) malloc(256 + 768)) == NULL)
		jlib_exit(jlib_msg(JLIB_EMALLOC));

	/* compress from the uncompressed pal buff to the compressed buffer */
	lz_compress(pal, 768, sp, comp_length);

	/* put length of compressed palette data */
	thirtytwobit = (ULONG) * comp_length;

	jio_write_elementary_type(outfile, &thirtytwobit, sizeof(ULONG));

	/* write compressed data into the file */
	for (i = 0; i < *comp_length; i++)
		fputc(sp[i], outfile);

	/* free the memory for the compressed buffers */
	free(sp);
	free(comp_length);

	JLIB_LEAVE;
	return 1;
}