/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <jlib.h>
#include <vga.h>

extern vga_modeinfo	*__jlib_modeinfo;
extern buffer_rec	__jlib_screen_buff;
extern int	__jlib_current_page;


/*+------------------------------------------------------------------------+ */
/*|blit a fullscreen buffer to the screen                                  | */
/*+------------------------------------------------------------------------+ */
JINLINE void screen_blit_fs_buffer(buffer_rec * buff)
{
	JLIB_ENTER("screen_blit_fs_buffer");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
	jlib_check_buffer(buff);
#endif

	if (__jlib_is_linear)
		memcpy(__jlib_screen_buff.buffer, B_BUFF_PTR(buff),SCR_SIZE);
	else {
		int i;
		UBYTE *out = B_BUFF_PTR(buff);

		for (i = 0; i < __jlib_pages; i++) {
			vga_setpage(i);
			LONG_COPY(out, __jlib_screen_buff.buffer, 65536);
			out += 65536;
		}
		__jlib_current_page = __jlib_pages - 1;
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit part of a buffer to the screen without clipping.                   | */
/*+------------------------------------------------------------------------+ */
void screen_blit_buff_toNC(int x, int y, buffer_rec * buff, int x1, int y1, int x2, int y2)
{
	JLIB_ENTER("screen_blit_buff_toNC");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
#endif

	if (__jlib_is_linear)
		buff_blit_buff_toNC(&__jlib_screen_buff, x, y, buff, x1, y1, x2, y2);
	else {
		int v_len, h_len;
		UBYTE *src,*dest = __jlib_screen_buff.buffer;
		int offset, page;

		if (x1 > x2)
			JLIB_SWAP(x1, x2);
		if (y1 > y2)
			JLIB_SWAP(y1, y2);

		v_len = y2 - y1 + 1;
		h_len = x2 - x1 + 1;
		src = B_OFFSET(buff,y1) + x1;
		offset = y * __jlib_modeinfo->linewidth + x;
		page = offset >> 16;
		offset &= 0xffff;

		if (__jlib_current_page != page)
			vga_setpage(page);

		for (; v_len != 0; v_len--) {
			if (offset + h_len > 0x10000)
				if (offset >= 0x10000) {
					page++;
					vga_setpage(page);
					offset &= 0xffff;
				}
				 else {	/* page break within line */
					memcpy(dest + offset, src, 0x10000 - offset);
					page++;
					vga_setpage(page);
					memcpy(dest, src + 0x10000 - offset, (offset + h_len) & 0xffff);
					offset = (offset + __jlib_modeinfo->linewidth) & 0xffff;
					src += B_X_SIZE(buff);
					continue;
				}
			memcpy(dest + offset, src, h_len);
			src += B_X_SIZE(buff);
			offset += __jlib_modeinfo->linewidth;
		}
		__jlib_current_page = page;
	}
	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit part of a buffer to the screen with clipping.                      | */
/*+------------------------------------------------------------------------+ */
void screen_blit_buff_to(int x, int y, buffer_rec * buff, int x1, int y1, int x2, int y2)
{
	JLIB_ENTER("screen_blit_buff_to");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
	jlib_check_buffer(buff);
#endif

	if (__jlib_is_linear)
		buff_blit_buff_to(&__jlib_screen_buff, x, y, buff, x1, y1, x2, y2);
	else {
		int v_len, h_len;
		UBYTE *src, *dest = __jlib_screen_buff.buffer;
		int offset, page;

		if (x1 > x2) JLIB_SWAP(x1, x2);
		if (y1 > y2) JLIB_SWAP(y1, y2);

		/* clip src buffer */
		if (x1 < B_CX1(buff)) {
			x += (B_CX1(buff) - x1);
			x1 = B_CX1(buff);
		}
		else
			if (x1 > B_CX2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (x2 > B_CX2(buff))
			x2 = B_CX2(buff);
		else
			if (x2 < B_CX1(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y1 < B_CY1(buff)) {
			y += (B_CY1(buff) - y1);
			y1 = B_CY1(buff);
		}
		else
			if (y1 > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}

		if (y2 > B_CY2(buff))
			y2 = B_CY2(buff);
		else
			if (y2 < B_CY1(buff)) {
				JLIB_LEAVE;
				return;
			}

		
		/* clip to screen */
		if (x < 0) {
			x1 -= x;
			x = 0;
			if (x1 > B_CX2(buff)) {
				JLIB_LEAVE;
				return;
			}
		}
		else
			if (x > SCREEN_MAX_X) {
				JLIB_LEAVE;
				return;
			}

		if (y < 0) {
			y1 -= y;
			y = 0;
			if (y1 > B_CY2(buff)) {
				JLIB_LEAVE;
				return;
			}
		}
		else
			if (y > SCREEN_MAX_Y) {
				JLIB_LEAVE;
				return;
			}

		v_len = y2 - y1 + 1;
		h_len = x2 - x1 + 1;

		if ((x + h_len) > SCREEN_MAX_X)
			h_len = SCREEN_MAX_X - x;

		if ((y + v_len) > SCREEN_MAX_Y)
			v_len = SCREEN_MAX_Y - y;

		src = B_OFFSET(buff, y1) + x1;
		offset = y * __jlib_modeinfo->linewidth + x;
		page = offset >> 16;
		offset &= 0xffff;

		if (__jlib_current_page != page)
			vga_setpage(page);

		for (; v_len != 0; v_len--) {
			if (offset + h_len > 0x10000)
				if (offset >= 0x10000) {
					page++;
					vga_setpage(page);
					offset &= 0xffff;
				}
				else {	/* page break within line */
					memcpy(dest + offset, src, 0x10000 - offset);
					page++;
					vga_setpage(page);
					memcpy(dest, src + 0x10000 - offset, (offset + h_len) & 0xffff);
					offset = (offset + __jlib_modeinfo->linewidth) & 0xffff;
					src += B_X_SIZE(buff);
					continue;
				}
			memcpy(dest + offset, src, h_len);
			src += B_X_SIZE(buff);
			offset += __jlib_modeinfo->linewidth;
		}
		__jlib_current_page = page;
	}

	JLIB_LEAVE;
}