/* Copyright 1995-97 Jon Griffiths and Lennart Steinke. See the file
 * "jlib.doc" for copyright details. Based on input code by Lennart Steinke.
 */

#if defined (FOR_WATCOM)
#define inportb  inp
#define outportb outp
#define enable   _enable
#define disable  _disable
#endif

#define JOY_PORT  0x201
#define THRESHOLD 0xffff

#define J1_ACTIVE 0x0f
#define J2_ACTIVE 0xf0

int __jlib_num_joysticks;
logical_joy __jlib_joys[3];

/*+------------------------------------------------------------------------+ */
/*| Return the number of joysticks connected.                              | */
/*+------------------------------------------------------------------------+ */
int joystick_count(void)
{
	UBYTE val;
	int count = 0, j1 = 0, j2 = 0;

	__jlib_joys[1].joy = 0;
	__jlib_joys[2].joy = 0;

	outportb(JOY_PORT, 0xff);

	disable();

	do {
		/* give the sticks plenty of time to get back to us */
		val = inportb(JOY_PORT);
		if ((val & J1_ACTIVE) != J1_ACTIVE) {
			j1 = 1;
		}
		if ((val & J2_ACTIVE) != J2_ACTIVE) {
			j2 = 1;
		}
		count++;
	} while (count < THRESHOLD);

	enable();
	__jlib_num_joysticks = j1 + j2;
	switch (__jlib_num_joysticks) {
	case 2:
		__jlib_joys[2].joy = 2;
	case 1:
		if (j1) {
			__jlib_joys[1].joy = 1;
		} else {
			__jlib_joys[1].joy = 2;
		}
	}

	return __jlib_num_joysticks;
}


/*+------------------------------------------------------------------------+ */
/*| Initialise a given joystick.                                           | */
/*+------------------------------------------------------------------------+ */
int joystick_init(int which)
{
	if ((which < 1) || (which > 2) || (which > __jlib_num_joysticks)) {
		return 0;
	}
	return 1;
}


/*+------------------------------------------------------------------------+ */
/*| Uninitialise a given joystick.                                         | */
/*+------------------------------------------------------------------------+ */
void joystick_closedown(int which)
{

}


/*+------------------------------------------------------------------------+ */
/*| Return raw values from a joystick.                                     | */
/*+------------------------------------------------------------------------+ */
void joystick_get_status(int which, int *x_axis, int *y_axis, int *b_status)
{
	int k, jx = 0, jy = 0;
	int m1, m2, m3, m4, m5;
	UBYTE val;

#ifndef JLIB_PRODUCTION
	if ((which < 1) || (which > 2) || (which > __jlib_num_joysticks)) {
		jlib_exit("Invalid joystick");
	}
#endif

	if (__jlib_joys[which].joy == 1) {
		m4 = 1;
	} else {
		m4 = 4;
	}

	m5 = m4 << 1;
	m1 = m4 << 4;
	m2 = m5 << 4;
	m3 = m4 + m5;

	disable();

	outportb(JOY_PORT, 0xff);
	val = inportb(JOY_PORT);

	/* we could check debouncing on these returned buttons */
	if (!(val & m1)) {
		*b_status = JOY_FIRE1;
	} else {
		*b_status = 0;
	}

	if (!(val & m2)) {
		*b_status = *b_status | JOY_FIRE2;
	}
	for (k = 0; (val & m3) != 0; k++) {
		if ((val & m4) != 0)
			jx = k;
		if ((val & m5) != 0)
			jy = k;
		val = inportb(JOY_PORT);
	}

	enable();

	*x_axis = jx;
	*y_axis = jy;
}


/*+------------------------------------------------------------------------+ */
/*| Set up joystick calibration for joystick_get_direction()               | */
/*+------------------------------------------------------------------------+ */
void joystick_calibrate(int which, int l, int r, int t, int b, int x_cen, int y_cen)
{
#ifndef JLIB_PRODUCTION
	if ((which < 1) || (which > 2) || (which > __jlib_num_joysticks)) {
		jlib_exit("Invalid joystick");
	}
#endif

	/* set up mapping for coordinates to directions */
	__jlib_joys[which].x1 = l;
	__jlib_joys[which].y1 = t;
	__jlib_joys[which].x2 = r;
	__jlib_joys[which].y2 = b;
	__jlib_joys[which].xcen = x_cen;
	__jlib_joys[which].ycen = y_cen;

	/* set dead zone size as +/- 13% of stick centre */
	__jlib_joys[which].deadx1 = x_cen - ((x_cen - __jlib_joys[which].x1) >> 3);
	__jlib_joys[which].deadx2 = x_cen + ((__jlib_joys[which].x2 - x_cen) >> 3);
	__jlib_joys[which].deady1 = y_cen - ((y_cen - __jlib_joys[which].y1) >> 3);
	__jlib_joys[which].deady2 = y_cen + ((__jlib_joys[which].y2 - y_cen) >> 3);
}


/*+------------------------------------------------------------------------+ */
/*| Convert raw joystick values into a direction.                          | */
/*+------------------------------------------------------------------------+ */
int joystick_get_direction(int which, int x_axis, int y_axis)
{
	int dir = JOY_CENTER;

#ifndef JLIB_PRODUCTION
	if ((which < 1) || (which > 2) || (which > __jlib_num_joysticks)) {
		jlib_exit("Invalid joystick");
	}
#endif

	/* should we check for being > x1, < x2 etc also? */
	if (x_axis <= __jlib_joys[which].deadx1)
		dir |= JOY_WEST;
	else if (x_axis >= __jlib_joys[which].deadx2)
		dir |= JOY_EAST;
	if (y_axis <= __jlib_joys[which].deady1)
		dir |= JOY_NORTH;
	else if (y_axis >= __jlib_joys[which].deady2)
		dir |= JOY_SOUTH;

	return dir;
}
