/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
#include <jlib.h>


/*+-------------------------------------------------------------------------+ */
/*|Draw a line from x1, y1 to x2, y2 in color c without clipping            | */
/*+-------------------------------------------------------------------------+ */
void buff_draw_lineNC(buffer_rec * buff, int x1, int y1, int x2, int y2, UBYTE c)
{
	int d, x, y, ax, ay, sx, sy, dx, dy;

	JLIB_ENTER("buff_draw_lineNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if (x1 > x2) {
		JLIB_SWAP(x1, x2);
		JLIB_SWAP(y1, y2);
	}
	dx = x2 - x1;
	ax = ABS(dx) << 1;
	sx = SGN(dx);
	dy = y2 - y1;
	ay = ABS(dy) << 1;
	sy = SGN(dy);

	x = x1;
	y = y1;

	if (ax > ay) {
		d = ay - (ax >> 1);
		while (x != x2) {
			*(B_OFFSET(buff, y) + x) = c;
			if (d >= 0) {
				y += sy;
				d -= ax;
			}
			x += sx;
			d += ay;
		}
	} else {
		d = ax - (ay >> 1);
		while (y != y2) {
			*(B_OFFSET(buff, y) + x) = c;
			if (d >= 0) {
				x += sx;
				d -= ay;
			}
			y += sy;
			d += ax;
		}
	}
	*(B_OFFSET(buff, y) + x) = c;

	JLIB_LEAVE;
}


/*+-------------------------------------------------------------------------+ */
/*|draw a line from x1, y1 to x2, y2 in color c                             | */
/*+-------------------------------------------------------------------------+ */
void buff_draw_line(buffer_rec * buff, int x1, int y1, int x2, int y2, UBYTE c)
{
	int max_x, min_x, max_y, min_y, delta;

	JLIB_ENTER("buff_draw_line");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	min_x = B_CX1(buff);
	max_x = B_CX2(buff);
	min_y = B_CY1(buff);
	max_y = B_CY2(buff);

	if (x1 > x2) {
		JLIB_SWAP(x1, x2);
		JLIB_SWAP(y1, y2);
	}
	if ((x2 < min_x) || (x1 > max_x)) {
		JLIB_LEAVE;
		return;
	}
	if (y1 < y2) {
		if ((y2 < min_y) || (y1 > max_y)) {
			JLIB_LEAVE;
			return;
		}
		if (y1 < min_y) {
			delta = (x2 - x1) * (min_y - y1) / (y2 - y1);
			if ((x1 += delta) > max_x) {
				JLIB_LEAVE;
				return;
			}
			y1 = min_y;
		}
		if (y2 > max_y) {
			delta = (x2 - x1) * (y2 - max_y) / (y2 - y1);

			if ((x2 -= delta) < min_x) {
				JLIB_LEAVE;
				return;
			}
			y2 = max_y;
		}
		if (x1 < min_x) {
			delta = (y2 - y1) * (min_x - x1) / (x2 - x1);
			y1 += delta;
			x1 = min_x;
		}
		if (x2 > max_x) {
			delta = (y2 - y1) * (x2 - max_x) / (x2 - x1);
			y2 -= delta;
			x2 = max_x;
		}
	}
	else {
		if ((y1 < min_y) || (y2 > max_y)) {
			JLIB_LEAVE;
			return;
		}
		if (y1 > max_y) {
			delta = (x2 - x1) * (y1 - max_y) / (y1 - y2);
			if ((x1 += delta) > max_x) {
				JLIB_LEAVE;
				return;
			}
			y1 = max_y;
		}
		if (y2 < min_y) {
			delta = (x2 - x1) * (min_y - y2) / (y1 - y2);
			if ((x2 -= delta) < min_x) {
				JLIB_LEAVE;
				return;
			}
			y2 = min_y;
		}
		if (x1 < min_x) {
			delta = (y1 - y2) * (min_x - x1) / (x2 - x1);
			y1 -= delta;
			x1 = min_x;
		}
		if (x2 > max_x) {
			delta = (y1 - y2) * (x2 - max_x) / (x2 - x1);
			y2 += delta;
			x2 = max_x;
		}
	}

	buff_draw_lineNC(buff, x1, y1, x2, y2, c);

	JLIB_LEAVE;
}



/*+------------------------------------------------------------------------+ */
/*|draw a horizontal line from x1, y1 to x2 in color c without clipping    | */
/*+------------------------------------------------------------------------+ */
void buff_draw_h_lineNC(buffer_rec * buff, int x1, int y1, int x2, UBYTE c)
{
	UBYTE *buffer;
	int width;

	JLIB_ENTER("buff_draw_h_lineNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if (x1 > x2)
		JLIB_SWAP(x2, x1);

	buffer = B_OFFSET(buff, y1) + x1;
	width = x2 - x1 + 1;

	if (width < 17) {
		MEM_STORE_SHORT(buffer, c, width);
	} else {
		MEM_STORE_LONG(buffer, c, width);
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|draw a horizontal line from x1, y1 to x2 in color c with clipping       | */
/*+------------------------------------------------------------------------+ */
void buff_draw_h_line(buffer_rec * buff, int x1, int y1, int x2, UBYTE c)
{
	JLIB_ENTER("buff_draw_h_line");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if ((y1 < B_CY1(buff)) || (y1 > B_CY2(buff))) {
		JLIB_LEAVE;
		return;
	}

	if (x1 > x2)
		JLIB_SWAP(x2, x1);

	if (x2 > B_CX2(buff))
		x2 = B_CX2(buff);
	else
		if (x2 < B_CX1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (x1 < B_CX1(buff))
		x1 = B_CX1(buff);
	else
		if (x1 > B_CX2(buff)) {
			JLIB_LEAVE;
			return;
		}

	buff_draw_h_lineNC(buff, x1, y1, x2, c);

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|draw a vertical line from x1, y1 to x2 in color c without clipping      | */
/*+------------------------------------------------------------------------+ */
void buff_draw_v_lineNC(buffer_rec * buff, int x1, int y1, int y2, UBYTE c)
{
	UBYTE *buffer;

	JLIB_ENTER("buff_draw_v_lineNC");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if (y1 > y2)
		JLIB_SWAP(y1, y2);

	buffer = B_OFFSET(buff, y1) + x1;

	y2 = y2 - y1 + 1;

	while (y2--) {
		*buffer = c;
		buffer += B_X_SIZE(buff);
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|draw a vertical line from x1, y1 to x2 in color c with clipping         | */
/*+------------------------------------------------------------------------+ */
void buff_draw_v_line(buffer_rec * buff, int x1, int y1, int y2, UBYTE c)
{
	JLIB_ENTER("buff_draw_v_line");

#ifndef JLIB_PRODUCTION
	jlib_check_buffer(buff);
#endif

	if ((x1 < B_CX1(buff)) || (x1 > B_CX2(buff))) {
		JLIB_LEAVE;
		return;
	}

	if (y1 > y2)
		JLIB_SWAP(y1, y2);

	if (y2 > B_CY2(buff))
		y2 = B_CY2(buff);
	else
		if (y2 < B_CY1(buff)) {
			JLIB_LEAVE;
			return;
		}

	if (y1 < B_CY1(buff))
		y1 = B_CY1(buff);
	else
		if (y1 > B_CY2(buff)) {
			JLIB_LEAVE;
			return;
		}

	buff_draw_v_lineNC(buff, x1, y1, y2, c);

	JLIB_LEAVE;
}