/* Copyright 1995-97 Jon Griffiths.  See the file "jlib.doc" for details. */
/*
 * By Lennart Steinke and Jon Griffiths
 */
#include <go32.h>
#include <sys/movedata.h>
#include <sys/segments.h>
#include <jlib.h>


/*+------------------------------------------------------------------------+ */
/*|blit a fullscreen buffer to the screen                                  | */
/*+------------------------------------------------------------------------+ */
void screen_blit_fs_buffer (buffer_rec * buff)
{
	JLIB_ENTER("screen_blit_fs_buffer");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
	jlib_check_buffer(buff);
#endif

	if (__jlib_is_13h) {
		dosmemput(B_BUFF_PTR(buff), 64000, 0xA0000);
		JLIB_LEAVE;
		return;
	}

	if (__jlib_is_linear)
		movedata(_my_ds(), (int)B_BUFF_PTR(buff), _jlib_sel, 0, SCR_SIZE);
	else {
		int i;
		UBYTE *out = B_BUFF_PTR(buff);

		for (i = 0; i < vesa_pages; i++) {
			__setpg(i);
			dosmemput(out, vesa_gran1024, 0xA0000);
			out += vesa_gran1024;
		}

		if (vesa_remains) {
			__setpg(vesa_pages);
			dosmemput(out, vesa_remains, 0xA0000);
			__jlib_pg = vesa_pages;
		}
		else
			__jlib_pg = vesa_pages - 1;
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer without clipping           | */
/*+------------------------------------------------------------------------+ */
void screen_blit_buff_toNC(int x, int y, buffer_rec * sbuff, int x1, int y1, int x2, int y2)
{
	UBYTE *src;

	JLIB_ENTER("screen_blit_buff_toNC");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
	jlib_check_buffer(sbuff);
#endif

	if (x2 < x1) JLIB_SWAP(x1, x2);
	if (y2 < y1) JLIB_SWAP(y1, y2);

	src = B_OFFSET(sbuff, y1) + x1;

	if (__jlib_is_13h) {
		int dest = 0xA0000 + (y * SCREEN_WIDTH) + x;
		y2 = y2 - y1 + 1;
		x2 = x2 - x1 + 1;
		while (y2--) {
			dosmemput(src, x2, dest);
			src += B_X_SIZE(sbuff);
			dest += SCREEN_WIDTH;
		}
		JLIB_LEAVE;
		return;
	}

	if (__jlib_is_linear) {
		int dest = y * SCREEN_WIDTH + x;
		y2 = y2 - y1 + 1;
		x2 = x2 - x1 + 1;
		while (y2--) {
			movedata(_my_ds(), (int)src, _jlib_sel, dest, x2);
			src += B_X_SIZE(sbuff);
			dest += SCREEN_WIDTH;
		}
	}
	else {
		int v_len = y2 - y1 + 1, h_len = x2 - x1 + 1;
		UBYTE *src = B_OFFSET(sbuff,y1) + x1;
		int offset, page;

		offset = y * vesa_width + x;
		if (offset == 0)
			page = 0;
		else {
			page = offset / vesa_gran1024;
			offset = offset % vesa_gran1024;
		}
		__setpg(page);

		for (; v_len != 0; v_len--) {
			if (offset + h_len > vesa_gran1024)
				if (offset >= vesa_gran1024) {
					page++;
					__setpg(page);
					offset = offset % vesa_gran1024;
				}
				else {
					dosmemput(src, vesa_gran1024 - offset, 0xA0000 + offset);
					page++;
					__setpg(page);
					dosmemput(src + vesa_gran1024 - offset, (offset + h_len) % vesa_gran1024, 0xA0000);
					offset = (offset + vesa_width) % vesa_gran1024;
					src += B_X_SIZE(sbuff);
					continue;
				}
			dosmemput(src, h_len, 0xA0000 + offset);
			src += B_X_SIZE(sbuff);
			offset = offset + vesa_width;
		}
		__jlib_pg = page;
	}

	JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+ */
/*|blit a portion of a buffer to another buffer with clipping              | */
/*+------------------------------------------------------------------------+ */
void screen_blit_buff_to(int x, int y, buffer_rec * sbuff, int x1, int y1, int x2, int y2)
{
	int width, height;
	UBYTE *src;

	JLIB_ENTER("screen_blit_buff_to");

#ifndef JLIB_PRODUCTION
	if (!__jlib_screen_initted)
		jlib_exit(jlib_msg(JLIB_EINIT));
	jlib_check_buffer(sbuff);
#endif

	if (x2 < x1)
		JLIB_SWAP(x1, x2);
	if (y2 < y1)
		JLIB_SWAP(y1, y2);

	/* clip src */
	if (x1 < B_CX1(sbuff)) {
		x += (B_CX1(sbuff) - x1);
		x1 = B_CX1(sbuff);
	}
	else
		if (x1 > B_CX2(sbuff)) {
			JLIB_LEAVE;
			return;
		}

	if (x2 > B_CX2(sbuff))
		x2 = B_CX2(sbuff);
	else
		if (x2 < B_CX1(sbuff)) {
			JLIB_LEAVE;
			return;
		}

	if (y1 < B_CY1(sbuff)) {
		y += (B_CY1(sbuff) - y1);
		y1 = B_CY1(sbuff);
	}
	else
		if (y1 > B_CY2(sbuff)) {
			JLIB_LEAVE;
			return;
		}

	if (y2 > B_CY2(sbuff))
		y2 = B_CY2(sbuff);
	else
		if (y2 < B_CY1(sbuff)) {
			JLIB_LEAVE;
			return;
		}

	height = y2 - y1 + 1;
	width = x2 - x1 + 1;

	/* clip dest */
	if (x < 0) {
		x1 -= x;
		if (x1 > B_CX2(sbuff)) {
			JLIB_LEAVE;
			return;
		}
		width += x;
		x = 0;
	}
	else
		if (x > SCREEN_WIDTH) {
			JLIB_LEAVE;
			return;
		}

	if (x + width > SCREEN_MAX_X)
		width = SCREEN_MAX_X - x + 1;
	else
		if (x + width < 0) {
			JLIB_LEAVE;
			return;
		}

	if (y < 0) {
		y1 -= y;
		if (y1 > B_CY2(sbuff)) {
			JLIB_LEAVE;
			return;
		}
		height += y;
		y = 0;
	}
	else
		if (y > SCREEN_MAX_Y) {
			JLIB_LEAVE;
			return;
		}

	if (y + height > SCREEN_MAX_Y)
		height = SCREEN_MAX_Y - y + 1;
	else
		if (y + height < 0) {
			JLIB_LEAVE;
			return;
		}

	src = B_OFFSET(sbuff, y1) + x1;

	if (__jlib_is_13h) {
		int dest = 0xA0000 + (y * SCREEN_WIDTH) + x;
		while (height--) {
			dosmemput(src, width, dest);
			src += B_X_SIZE(sbuff);
			dest += SCREEN_WIDTH;
		}
		JLIB_LEAVE;
		return;
	}

	if (__jlib_is_linear) {
		int dest = y * SCREEN_WIDTH + x;
		while (height--) {
			movedata(_my_ds(), (int)src, _jlib_sel, dest, width);
			src += B_X_SIZE(sbuff);
			dest += SCREEN_WIDTH;
		}
	}
	else {
		UBYTE *src = B_OFFSET(sbuff,y1) + x1;
		int offset, page;

		offset = y * vesa_width + x;
		if (offset == 0)
			page = 0;
		else {
			page = offset / vesa_gran1024;
			offset = offset % vesa_gran1024;
		}
		__setpg(page);

		for (; height != 0; height--) {
			if (offset + width > vesa_gran1024)
				if (offset >= vesa_gran1024) {
					page++;
					__setpg(page);
					offset = offset % vesa_gran1024;
				}
				else {
					dosmemput(src, vesa_gran1024 - offset, 0xA0000 + offset);
					page++;
					__setpg(page);
					dosmemput(src + vesa_gran1024 - offset, (offset + width) % vesa_gran1024, 0xA0000);
					offset = (offset + vesa_width) % vesa_gran1024;
					src += B_X_SIZE(sbuff);
					continue;
				}
			dosmemput(src, width, 0xA0000 + offset);
			src += B_X_SIZE(sbuff);
			offset = offset + vesa_width;
		}
		__jlib_pg = page;
	}

	JLIB_LEAVE;
}