/************************************************************************/
/* sbdev.c, the sound blaster device code.                              */
/************************************************************************/
#include "sbdev.h"

extern unsigned char error[]; /* error string */
extern samrec samples[];
extern channelinforec channel[];
extern unsigned int master;
extern void mastervolume(signed int vol);

#define DEFAULT_AMPLIFICATION 256

slong *mtmp;     // 32-bit tmp buffer for mixing
slong *voltable; // volume table for mixing

struct
{
  _go32_dpmi_seginfo info,vanha; // for interrupt handling

  slong dma_buf_size,            // dma buffer size
        dma_buf_segment,         // dma buffer's real mode segment
        dma_buf_offset,          // dma buffer's offset
        dma_buf_selector;        // selector for dma buffer

  ulong base,                    // sb's base port
        irq,                     // sb's irq
        dma,                     // sb's 8-bit dma channel
        hdma,                    // sb's 16-bit dma channel
        playspeed;               // sb's output frequency

  ulong interrupts;              // the number of mixing interrupts occured

  ubyte airq,                    // the irq number to pass to ?etintvecs
        pic,                     // pic's port address
        pageport,                // pic's pageport address
        istam,                   // irq start mask
        istom,                   // irq stop mask
        dstam,                   // dma start mask
        dstom,                   // dma stop mask
        dmode;                   // dma mode

  slong dsp_reset,               // sb's port addresses :
        dsp_read_data,
        dsp_write_data,
        dsp_write_status,
        dsp_data_avail;

  ubyte installed;               // is the sb driver installed
  ulong dspversion;              // 8.8 bits. bcd stylish

  slong autoinit;                // 0 no; 1 yes
  slong _16bit;                  // 0 no; 1 yes
  slong stereo;                  // 0 no; 1 yes

  ubyte dmamaskport,             // dma controller's port addresses :
        dmaclrptrport,
        dmamodeport,
        dmabaseaddrport,
        dmacountport;

  slong ackport;                 // sb's irq ack port
  slong dmalen;                  // the length of a dma transfer
} sb;

struct
{
  slong koee; /* sb's status, actually no point for it to be here. */
  slong vuoro; /* double buffer's offset */
  ulong amplification,active_channels;
} mixerinfo;

typedef struct
{
  sbyte name[70];
  slong allow_auto_init;
  slong allow_16bit;
  slong allow_stereo;
  slong mindsp, maxdsp;
} mixingmode_t;

mixingmode_t mixingmodes[7] =
{
  {"SB 1.0,  8 bits, mono,   max 22 kHz, irq controlled dma", 0, 0, 0, 0x000, 0xFFFF},
  {"SB 1.5+, 8 bits, mono,   max 22 kHz on 1.5, else 44 kHz", 1, 0, 0, 0x200, 0xFFFF},
  {"SB pro,  8 bits, stereo, max 22 kHz, for SB pro ONLY   ", 1, 0, 1, 0x300, 0x0400},
  {"SB 16,   8 bits, stereo, max 45 kHz                    ", 1, 0, 1, 0x401, 0xFFFF},
  {"SB 16,  16 bits, mono,   max 45 kHz (faster than 8 bit)", 1, 1, 0, 0x401, 0xFFFF},
  {"SB 16,  16 bits, stereo, max 45 kHz (faster than 8 bit)", 1, 1, 1, 0x401, 0xFFFF},
  {"The best mode available                                ", 1, 1, 1, 0x000, 0xFFFF}
};

int choose_mixing_mode(unsigned short dsp) // -1 aborted, else the chosen one
{
  vali_t mmodes[7];
  int i;
  for (i = 0; i < 7; i++)
  {
    strcpy(mmodes[i].menutext, mixingmodes[i].name);
    if ( (dsp >= mixingmodes[i].mindsp) && (dsp <= mixingmodes[i].maxdsp) )
      mmodes[i].selectable = 1;
    else mmodes[i].selectable = 0;
  }
  if ((sb.hdma < 4) || (sb.hdma > 7))
  {
    mmodes[4].selectable = 0;
    mmodes[5].selectable = 0;
  }
  return valintabar(mmodes, 7, " Select ", 7);
}

typedef struct
{
  sbyte name[70];
  ulong speedhz;
} mixingspeed_t;

mixingspeed_t mixingspeeds[7] =
{
  {" 5 kHz, very fast :)     ", 5000},
  {"11 kHz, no quality       ", 11111},
  {"16 kHz, extra low quality", 16130},
  {"22 kHz, low quality      ", 21739},
  {"29 kHz, medium quality   ", 29412},
  {"37 kHz, normal quality   ", 37038},
  {"The highest kHz available", 45454}
};

int select_speed(unsigned int max)
{
  vali_t speedos[7];
  ulong i;
  slong r;
  for (i = 0; i < 7; i++)
  {
    strcpy(speedos[i].menutext, mixingspeeds[i].name);
    if (mixingspeeds[i].speedhz <= max) speedos[i].selectable = 1;
    else speedos[i].selectable = 0;
  }
  speedos[6].selectable = 1; // the last option is always available
  r = valintabar(speedos, 7, " Select ", 7);
  switch (r)
  {
   case -1:
     i = max;
     break;
   default:
     i = mixingspeeds[r].speedhz;
     break;
  }
  if (i > max) i = max;
  return i;
}

void writedsp(unsigned char value)
{
  int nerve = 100000;
  asm volatile (
  "pushfl
   cli");

  while ( ((inportb(sb.dsp_write_status) & 0x80) != 0) && (nerve > 0) )
    nerve--;
  outportb(sb.dsp_write_data, value);

  asm volatile ("popfl");
}

unsigned char readdsp()
{
 int nerve = 100000;
 unsigned char resultti;
 asm volatile (
 "pushfl;
  cli");

 while ( ((inportb(sb.dsp_data_avail) & 0x80) == 0) && (nerve > 0) )
   nerve--;
 resultti = inportb(sb.dsp_read_data);

 asm volatile ("popfl");
 return resultti;
}

void dmacontinue()
{
  if (sb._16bit == 1) writedsp(0xD6);
  else writedsp(0xD4);
}

void dmastop()
{
  if (sb._16bit == 1) writedsp(0xD5);
  else writedsp(0xD0);
}

void odota()
{
 int nerve = 100000;
 asm volatile (
 "pushfl
  cli");
 while ( ((inportb(sb.dsp_data_avail) & 0x80) == 0) && (nerve > 0) )
   nerve--;
 asm volatile("popfl");
}

signed int resetdsp(unsigned int epatoivo) // 0 success, -1 failed.
{
 ulong eptoiv = 0;
 slong reset  = -1;
 sb.dsp_reset        = sb.base + 0x06;
 sb.dsp_read_data    = sb.base + 0x0A;
 sb.dsp_write_data   = sb.base + 0x0C;
 sb.dsp_write_status = sb.base + 0x0C;
 sb.dsp_data_avail   = sb.base + 0x0E;
 disable();
 while ((reset == -1) && (eptoiv <= epatoivo))
 {
   eptoiv++;
   outportb(sb.dsp_reset, 1);
   delay(1000 / 35);
   outportb(sb.dsp_reset, 0);
   delay(1000 / 70);
   odota();
   if (readdsp() == 0xAA) reset = 0; else reset = -1;
 }
 enable();
 return reset;
}

unsigned int verify_freq(unsigned int freq)
{
  // all sb versions :
  if (freq < 5000) freq = 5000;

  // sb 1.0 and sb 1.5 :
  if ( ((sb.autoinit == 0) || (sb.dspversion <= 0x200)) && (freq > 21739) )
    freq = 21739;

  // sb pro STEREO :
  if ( (sb.stereo==1) && (sb.dspversion<0x401) && (freq>21739) )
    freq=21739;

  // sb pro and sb 2.0 :
  if ( (sb.dspversion < 0x401) && (freq > 43478) ) freq = 43478;

  // sb 16 :
  if (freq > 45454) freq = 45454;

  return freq;
}

int setfreq(unsigned int freq) // returns the frequency set.
{
  slong t;
  ulong sb_pro_stereo = 1;

  freq = verify_freq(freq);

  if ((sb.dspversion > 0x400) && (sb.autoinit == 1))
  { // with sb 16 we're able to set the playspeed directly :
    writedsp(0x41);
    writedsp(hi(freq)); // hi
    writedsp(lo(freq)); // lo
  }
  else
  { // normal set sb time constant :
    // with sb pro stereo, the freq must be doubled
    if ( (sb.stereo == 1) && (sb.dspversion < 0x401) ) sb_pro_stereo = 2;
    else sb_pro_stereo = 1;

    // calculate the time constant :
    t = (unsigned char)((256 - (1000000 / (freq * sb_pro_stereo))) & 255);

    writedsp(0xD1);
    writedsp(0x40);
    writedsp((unsigned char)(t));

    // calculate the "exact" value of the frequency set :
    freq  = ((signed int)(-1000000) / (signed int)(t - 256));
    freq /= sb_pro_stereo;
  }
  return freq;
}

void setmixer(unsigned char index, unsigned char value)
{
  outportb(sb.base + 0x04, index);
  outportb(sb.base + 0x05, value);
}

volatile void playback()
{
  ulong page, offset, len;
  ubyte commandb, modeb;

  // calculate dma controller's setup :
  // the offset must be even. the dma buffer must not cross 64kB boundaries.

  // calculate offset :
  offset = ((unsigned int)sb.dma_buf_segment << 4) +
            (unsigned int)sb.dma_buf_offset;

  // [midas rulez] in 16 bit modes the starting offset is the starting WORD :
  if (sb._16bit == 1) offset /= 2;

  page = ((ulong)((ulong)sb.dma_buf_segment +
                 ((ulong)sb.dma_buf_offset >> 4))) >> 12;

  disable(); // don't disturb us while we're messing with the ports

  outportb(sb.dmamaskport, sb.dstom);       // mask dma channel off == stop it
  outportb(sb.dmaclrptrport, 0);            // clear flipflop
  outportb(sb.dmamodeport, sb.dmode);       // write the mode.
  outportb(sb.dmabaseaddrport, lo(offset)); // write address LOw
  outportb(sb.dmabaseaddrport, hi(offset)); // write address HIgh
  len = sb.dmalen;
  if (sb.stereo == 1) len *= 2;
  outportb(sb.dmacountport, lo(len - 1));   // write transfer length LOw
  outportb(sb.dmacountport, hi(len - 1));   // write transfer length HIgh
  outportb(sb.pageport, page);              // write page number
  outportb(sb.dmamaskport, sb.dstam);       // activate dma channel

  // now program the sb :
  if ((sb.dspversion > 0x400) && (sb.autoinit == 1)) // do we have a SB16?
  { // use SB16 "generic dac" -command :
    sb.playspeed = setfreq(sb.playspeed);   // set output frequency
    commandb = 2 | 4;                       // enable fifo, use autoinit dma
    if (sb._16bit == 1)
      commandb |= 16 | 32 | 128;            // 16-bit
    else
      commandb |= 128 | 64;                 // 8-bit

    // 16-bit modes use signed, 8-bit modes unsigned data :
    if (sb._16bit == 1) modeb = 16;         // signed
    else modeb = 0;                         // unsigned

    if (sb.stereo == 1) modeb |= 32;        // set stereo mode
    writedsp(commandb);                     // write command byte
    writedsp(modeb);                        // write mode byte
    len = sb.dma_buf_size;
    if (sb.stereo == 1) len *= 2;
    writedsp(lo(len - 1));                  // write transfer length LOw
    writedsp(hi(len - 1));                  // write transfer length HIgh
  }
  else
  { // use standard sb dac commands :
    writedsp(0xD1);                         // turn on speaker
    sb.playspeed = setfreq(sb.playspeed);   // set'n'get output frequency
    if (sb.autoinit == 1)
    { // playing 8bit autoinited :
      if (sb.stereo == 1)                   // sb pro's stereo
      {                                     // so let's reprogram the mixer
        outportb(sb.base + 0x04, 0x0E);
        modeb = inportb(sb.base + 0x05);    // 0x22 is only a mask
        setmixer(0x0E, modeb | 0x22);       // sb pro ONLY: stereo, no filter
      }
                                            // sb pro's mono :
      if ((sb.stereo == 0) && (sb.dspversion >= 0x300))
      {                                     // let's reprogram the mixer :
        outportb(sb.base+0x04,0x0E);        // get the old settings
        modeb = inportb(sb.base + 0x05);    // set sb pro mixer: mono, no filter :
        setmixer(0x0E, (modeb & 0xFD) | 0x20);
      }

      writedsp(0x48);                       // set dma block size
      len = sb.dma_buf_size;
      if (sb.stereo == 1) len *= 2;
      writedsp(lo(len - 1));                // LOw
      writedsp(hi(len - 1));                // HIgh
      if (sb.dspversion == 0x200)           // normal speed (<23KHz) dma transfer.
        writedsp(0x1C);                     // sb 1.5 (dsp version 2.00)
      else writedsp(0x90);                  // high speed dma transfer
                                            // SB 2.0; dsp version 2.01+
    }
    else
    { // playing 8bit single cycle ("irq controlled dma")
      writedsp(0x14);                       // 8bit single cycle, dma dac
      writedsp(lo(sb.dma_buf_size-1));      // LOw
      writedsp(hi(sb.dma_buf_size-1));
    }
   }
  enable();
}

unsigned int get_dspversion()
{
  unsigned int tulos;
  disable();
  writedsp(0xE1);
  tulos  = ((unsigned int)readdsp()) << 8;
  tulos += readdsp();
  enable();
  return tulos;
}

inline volatile void singlecykleplayback(int from)
{
  unsigned int page, offset;
  offset = (sb.dma_buf_segment << 4) + sb.dma_buf_offset + from;
  page   = (sb.dma_buf_segment + ((sb.dma_buf_offset + from) >> 4)) >> 12;
  outportb(sb.dmamaskport, sb.dstom);
  outportb(sb.dmaclrptrport, 0);
  outportb(sb.dmamodeport, sb.dmode);
  outportb(sb.dmabaseaddrport, lo(offset));
  outportb(sb.dmabaseaddrport, hi(offset));
  outportb(sb.dmacountport, lo(sb.dmalen - 1));
  outportb(sb.dmacountport, hi(sb.dmalen - 1));
  outportb(sb.pageport, page);
  outportb(sb.dmamaskport, sb.dstam);
  writedsp(0x14); // 8bit single cycle
  writedsp(lo(sb.dma_buf_size - 1));
  writedsp(hi(sb.dma_buf_size - 1));
}

// sofware mixing code :

static volatile void bom(){} // Begin Of Mixer-stuff

 // some globals :
 slong mi,sam,monovol,rightvol,leftvol,*rtaul,*ltaul,ma,*taul,*vali;
 uword selektoori;
 uhuge bytes,bytez,fadd;
 ubyte *deita;
 ulong pit,faddl,faddh,posil,posih;

 // assembly procedures :
 extern inline volatile void mixmonoloop();
 extern inline volatile void mixstereoloop();
 extern inline volatile void clip16bitloop();
 extern inline volatile void clip8bitloop();

inline void mixmono() // mono mixer
{
  // vali is a pointer to the volumetable :
  vali = mtmp;

  // mi is the channel counter :
  for (mi = 0; mi < 33; mi++) if (channel[mi].playing == 1)
  {
    // ma is the number of dwords put to the output buffer :
    ma = 0;

    // calculate the step :
    fadd = ((uhuge)channel[mi].freq << 32)/ (uhuge)sb.playspeed;
    // this 32.32 fixed point number is added to the sample pointer
    // each 'iteration' of the loop causing the sample pointer to
    // increase either faster or slower (or at the exact same speed :)
    // thus making it play at the correct speed compared to the output
    // frequency.

    sam = channel[mi].sample;

    // make sure we're playing a valid sample :
    if ( (samples[sam].used != 1) ||
         (samples[sam].data == NULL) ||
         (channel[mi].freq == 0)
       )
      channel[mi].playing = 0;

    // calculate the index to the volumetable for our volumelevel :
    monovol = (channel[mi].volume * master * 4) & 0xFFFFFF00;
    // volume_index = volume * mastervolume / 64 * 256

    // force 64 to be the maximum volume :
    if (monovol > 0x4000) monovol = 0x4000;

    // point taul to the volumetable :
    taul  = voltable;

    // add the correct index :
    taul += monovol;

    // point deita to the sample data :
    deita = samples[sam].data;

    // while the output buffer isn't full and the channel is playing :
    while ((ma < sb.dma_buf_size) && (channel[mi].playing == 1))
    {
      // how many bytes (in 32.32 fixed) are there left until the
      // end of the sample will be reached :
      bytes = ((uhuge)samples[sam].length << 32);

      // and when shall we reach the end of the loop :
      bytez = ((uhuge)samples[sam].loop_endi << 32);

      // if the sample is looped and loop end comes before the sample
      // end then only mix to the loop end :
      if ((samples[sam].loop_endi > 0) && (bytez < bytes)) bytes = bytez;

      // how many bytes may we read from the sample data :
      bytez = bytes - channel[mi].position;

      // and how many loop iterations would that make :
      pit = bytez / fadd;

      // make sure we don't mix too much or else ..
      while ( ( (uhuge)pit * (uhuge)fadd ) >= bytez ) pit--;
      while ( ( (uhuge)pit * (uhuge)fadd ) < bytez ) pit++;

      // if the sample wants to play longer than the output buffer can
      // take for this interrupt, then only mix to the output buffer end
      if (pit > (sb.dma_buf_size - ma)) pit = (sb.dma_buf_size - ma);

      // set some variables for the assembly mixer :
      faddl = (fadd & 0xFFFFFFFF);                         // step low
      faddh = ((fadd >> 32) & 0xFFFFFFFF);                 // step high
      posil = (channel[mi].position & 0xFFFFFFFF);         // position low
      posih = ((channel[mi].position >> 32) & 0xFFFFFFFF); // position high

      // and mix :
      mixmonoloop();

      // update channel's position :
      channel[mi].position = (uhuge)posil+((uhuge)posih << 32);

      // and the number of dwords mixed to the output buffer :
      ma+=pit;

      // if we're at the end of the loop then goto the loop start :
      if ( (samples[sam].loop_endi > 0) &&
           (channel[mi].position >=
           (((uhuge)samples[sam].loop_endi << 32) - 2147483647))
         )
        channel[mi].position = ((uhuge)samples[sam].loop_start << 32);

      // if the sample is at the end then stop playing this channel :
      if (channel[mi].position >=
          (((uhuge)samples[sam].length << 32) - 2147483647) )
        channel[mi].playing = 0;
    }
  }
}

inline void mixstereo() // stereo mixer
{
  vali = mtmp;
  for (mi = 0; mi < 33; mi++) if (channel[mi].playing == 1)
  {
    ma   = 0;
    fadd = ((uhuge)channel[mi].freq << 32) / (uhuge)sb.playspeed;
    sam  = channel[mi].sample;
    if ( (samples[sam].used != 1) ||
         (samples[sam].data == NULL) ||
         (channel[mi].freq == 0)
       )
      channel[mi].playing = 0;

    leftvol = ((master * channel[mi].volume * (256 - channel[mi].pan)) >> 6 )
      & 0xFFFFFF00;
    if (leftvol > 0x4000) leftvol = 0x4000;

    rightvol = ((channel[mi].volume * channel[mi].pan * master) >> 6)
      & 0xFFFFFF00;
    if (rightvol > 0x4000) rightvol = 0x4000;

    rtaul  = voltable;
    ltaul  = voltable;
    rtaul += rightvol;
    ltaul += leftvol;
    deita  = samples[sam].data;

    while ((ma < sb.dma_buf_size * 2) && (channel[mi].playing == 1))
    {
      bytes = ((uhuge)samples[sam].length << 32);
      bytez = ((uhuge)samples[sam].loop_endi<< 32);
      if ((samples[sam].loop_endi > 0) && (bytez < bytes)) bytes = bytez;

      bytez = bytes - channel[mi].position;
      pit = bytez / fadd;

      while ( ( (uhuge)pit * (uhuge)fadd ) >= bytez ) pit--;
      while ( ( (uhuge)pit * (uhuge)fadd ) < bytez ) pit++;

      if (pit > (sb.dma_buf_size - (ma / 2)))
        pit = (sb.dma_buf_size - (ma / 2));

      faddl = (fadd & 0xFFFFFFFF);
      faddh = ((fadd >> 32) & 0xFFFFFFFF);
      posil = (channel[mi].position & 0xFFFFFFFF);
      posih = ((channel[mi].position >> 32) & 0xFFFFFFFF);

      mixstereoloop();

      channel[mi].position = (uhuge)posil + ((uhuge)posih << 32);
      ma += pit * 2;

      if ( (samples[sam].loop_endi > 0) &&
           (channel[mi].position >=
           (((uhuge)samples[sam].loop_endi << 32) - 2147483647))
         )
        channel[mi].position = ((uhuge)samples[sam].loop_start << 32);

      if (channel[mi].position >=
          (((uhuge)samples[sam].length << 32) - 2147483647) )
        channel[mi].playing = 0;
    }
  }
}

inline void mix_n_clip_n_move_to_dma_buffer()
{
 signed int outputsamples, i;

 // clear the mixing buffer :
 for (i = 0; i < 6000; i++) mtmp[i] = 0;

 // do mixing :
 if (sb.stereo == 0) mixmono();
 else mixstereo();

 outputsamples = sb.dma_buf_size;
 if (sb.stereo == 1) outputsamples *= 2;

 if (sb._16bit == 0) // output: 8 bits
 {
   selektoori = sb.dma_buf_selector;
   pit        = outputsamples;
   vali       = mtmp;
   posil      = sb.dma_buf_offset + mixerinfo.vuoro;
   clip8bitloop(); // sar'n'clip'n'move to dma_buffer
 }
 else if (sb._16bit == 1) // output: 16 bits
 {
   selektoori = sb.dma_buf_selector;
   pit        = outputsamples;
   vali       = mtmp;
   posil      = sb.dma_buf_offset + (mixerinfo.vuoro * 2);
   clip16bitloop(); // clip'n'move to dma_buffer
 }

 if (sb.autoinit == 1) // doublebuffer
 {
   // toggle it :
   if (mixerinfo.vuoro == 0) mixerinfo.vuoro = sb.dma_buf_size;
   else mixerinfo.vuoro = 0;

   if (sb.stereo == 1) mixerinfo.vuoro *= 2;
 }
}

static void keskeytys() // interrupt handler
{
 if (sb.autoinit == 0) // there's a bug somewhere in the non-autoinit code.
 {
   singlecykleplayback(mixerinfo.vuoro); // restart dma transfer

   if (mixerinfo.vuoro == 0)
     mixerinfo.vuoro = sb.dma_buf_size;
   else
     mixerinfo.vuoro = 0;
 }

 mixerinfo.koee = inportb(sb.ackport); // sb ack

 mix_n_clip_n_move_to_dma_buffer();

 sb.interrupts++;

 // if the irq is a high irq, then send ack to both pics, first to the pic2 :
 if (sb.irq > 7) outportb(0xA0, 0x20);
 outportb(0x20, 0x20);

 // set interrupt flag on before exiting :
 asm volatile("sti");
}

static volatile void eom(){} // end of mixer-stuff

int sb_get_settings()
{
 int len, pos;
 char *blaster = getenv("BLASTER");
 if (blaster == NULL)
 {
   strcat(error, "env. read: no blaster env. variable\n");
   return -1;
 }

 sb_settings.base = 0;
 sb_settings.irq  = 0;
 sb_settings.dma  = 0;
 sb_settings.hdma = 0;

 len = strlen(blaster);

 // get baseport address :
 pos = strcspn(blaster, "A");
 if (pos >= len) pos = strcspn(blaster, "a");

 if (pos >= len)
 {
   strcat(error, "env. read: no base. (A220 etc.)\n");
   return -1;
 }

 sb_settings.base = (blaster[pos + 1] - 48) * 0x100 +
                    (blaster[pos + 2] - 48) * 0x010 +
                    (blaster[pos + 3] - 48);

 if ( (sb_settings.base < 0x210) || (sb_settings.base > 0x280) )
 {
   strcat(error, "env read: no base. (A220 etc.)\n");
   return -1;
 }

 // get irq number :
 pos = strcspn(blaster, "I");
 if (pos >= len) pos = strcspn(blaster, "i");

 if (pos >= len)
 {
   strcat(error, "env. read: no irq. (I5 etc.)\n");
   return -1;
 }

 sb_settings.irq = (blaster[pos + 1] - 48);
 if ( (sb_settings.irq == 1) && (blaster[pos + 2] == '0') )
   sb_settings.irq = 10;
 if ( (sb_settings.irq != 2) &&
      (sb_settings.irq != 3) &&
      (sb_settings.irq != 5) &&
      (sb_settings.irq != 7) &&
      (sb_settings.irq != 10)
    )
 {
   strcat(error, "env. read: no irq. (I5 etc.)\n");
   return -1;
 }

 // get 8 bit dma channel :
 pos = strcspn(blaster, "D");
 if (pos >= len) pos = strcspn(blaster, "d");

 if (pos >= len)
 {
   strcat(error, "env. read: no dma. (D1 etc.)\n");
   return -1;
 }

 sb_settings.dma = (blaster[pos + 1] - 48);
 if ( (sb_settings.dma != 0) &&
      (sb_settings.dma != 1) &&
      (sb_settings.dma != 3)
    )
 {
   strcat(error, "env. read: no dma. (D1 etc.)\n");
   return -1;
 }

 pos = strcspn(blaster, "H");
 if (pos >= len) pos = strcspn(blaster, "h");

 if (pos < len)
 {
  sb_settings.hdma = (blaster[pos + 1] - 48);
  if ( (sb_settings.hdma != 5) &&
       (sb_settings.hdma != 6) &&
       (sb_settings.hdma != 7)
     )
    sb_settings.hdma = 0x00;
 }

 return 0;
}

int soundblaster_init(int ask_user) // 1 ask; 0 don't ask
{
 ulong blocksizee;
 slong i;
 slong allow_auto_init = 1;
 slong allow_16bit     = 1;
 slong allow_stereo    = 1;

 if (sb.installed == 1)
 {
   strcat(error, "sb_init: already inited.\n");
   return -1;
 }

 // reset channels :
 for (i = 0; i < 33; i++)
 {
   channel[i].playing  = 0;
   channel[i].sample   = 0;
   channel[i].position = 0;
   channel[i].freq     = 0;
   channel[i].volume   = 0;
 }

 // get sb's settings from the BLASTER environment variable :
 if (sb_get_settings() == -1)
 {
   strcat(error, "sb_init: blaster env. variable not (properly) set.\n");
   return -1;
 }

 sb.base      = sb_settings.base;
 sb.irq       = sb_settings.irq;
 sb.dma       = sb_settings.dma;
 sb.hdma      = sb_settings.hdma;
 sb.playspeed = 45454;

 resetdsp(10);           // stop the playback :)
 if (resetdsp(10) == -1) // reset dsp.
 {
   strcat(error, "sb_init: no sb.\n");
   return -1;
 }

 // get dsp version :
 do {
   sb.dspversion = get_dspversion();
 } while (sb.dspversion != get_dspversion());

 // if the caller want's to prompt the settings from the user :
 if (ask_user == 1)
 {
   i = choose_mixing_mode(sb.dspversion);  // select mixing mode
   if ((i == -1) || (i < 0) || (i > 6))
   {
     strcat(error,"sb_init: aborted by user.\n");
     return -1;
   }
   allow_16bit     = mixingmodes[i].allow_16bit;
   allow_auto_init = mixingmodes[i].allow_auto_init;
   allow_stereo    = mixingmodes[i].allow_stereo;
 }

 // sb 1.5 and better (dsp 2.0) can do autoinit :
 if (sb.dspversion >= 0x200)
   sb.autoinit = 1; // dsp version 2.00 supports auto-init dma transfer
 else
   sb.autoinit = 0; // sb 1.0 does not.

 if ((sb.dspversion > 0x400) && (sb.hdma > 3) && (sb.hdma < 8))
   sb._16bit = 1;   // SB16 can play 16 bit
 else
   sb._16bit = 0;   // the others can't

 if (allow_16bit == 0) sb._16bit = 0; // force 8-bit mode

 if ((sb.dspversion > 0x400) && (allow_stereo == 1))
   sb.stereo = 1;   // sb 16, stereo
 else
   sb.stereo = 0;

 if ( (sb.dspversion >= 0x300) && (allow_stereo == 1))
   sb.stereo = 1;   // sb pro, stereo

 if (allow_auto_init == 0)
 {                  // force
   sb._16bit   = 0; // 8 bit
   sb.autoinit = 0; // irq controlled (single cycle) dma
   sb.stereo   = 0; // mono
 }

 // set and verify the default frequency :
 sb.playspeed = verify_freq(sb.playspeed);

 // if the caller wants us to ask the user
 if (ask_user == 1)
   sb.playspeed = verify_freq( select_speed(sb.playspeed) );

 // calculate blocksize so that it is apprx. 1/50 seconds in length.
 blocksizee = ((sb.playspeed / 50) + 3) & 0xFFFFFFFC;

 if (blocksizee > 2048) blocksizee = 2048;
 sb.dma_buf_size = blocksizee;

 // if 16 bit output is used, the output buffer has to be doubled
 if (sb._16bit == 1) sb.dma_buf_size = sb.dma_buf_size * 2;

 // if stereo is used, the output buffer has to be doubled
 if (sb.stereo == 1) sb.dma_buf_size *= 2;

 // allocate dos memory for the dma buffer :
 sb.dma_buf_segment =
   __dpmi_allocate_dos_memory(((sb.dma_buf_size * 4 + 15) >> 4),
     &sb.dma_buf_selector);

 // ok, actually we really didn't want to allocate it :
 if ((sb.dma_buf_segment > 0xFFFF) || (sb.dma_buf_segment < 0))
 {
   strcat(error, "sb_init: out of base memory.\n");
   return -1;
 }

 // clear the dma buffer :
 for (i = 0; i < (sb.dma_buf_size * 4); i++)
   _farpokeb(sb.dma_buf_selector, i, 0);

 sb.dma_buf_offset = 0;

 // check if the dma_buffer crosses a page boundary :
 if ((ulong)(((((ulong)sb.dma_buf_segment << 4) +
   (ulong)sb.dma_buf_offset) & 0xFFFF) +
   (ulong)sb.dma_buf_size * 2) > 0x0000FFFF)
   (sb.dma_buf_offset = sb.dma_buf_size * 2); // if it does, move it.

 // make the dma_buf_size indicate the number of words :
 if (sb._16bit == 1) sb.dma_buf_size = sb.dma_buf_size / 2;
 // and the number of rl-elements :
 if (sb.stereo == 1) sb.dma_buf_size = sb.dma_buf_size / 2;

 switch (sb._16bit)
 {
  case 0: // 8 bit transfer :
   sb.dmamaskport     = 0x0A;
   sb.dmaclrptrport   = 0x0C;
   sb.dmamodeport     = 0x0B;
   sb.dmabaseaddrport = (2 * sb.dma);
   sb.dmacountport    = (1 + 2 * sb.dma);
   switch (sb.dma)
   {
     case 0:
       sb.pageport = 0x87;
       break;
     case 1:
       sb.pageport = 0x83;
       break;
     case 2:
       sb.pageport = 0x81;
       break;
     case 3:
       sb.pageport = 0x82;
       break;
   }
   sb.dstam   = sb.dma;
   sb.dstom   = sb.dma | 0x04;
   sb.ackport = (sb.base + 0x0E);

   if (sb.autoinit == 1)
     sb.dmode = (sb.dma | 0x58);
   else
     sb.dmode = (sb.dma | 0x48);
   break;

  case 1: // 16 bit playing :
   sb.dmamaskport     = 0xd4;
   sb.dmaclrptrport   = 0xd8;
   sb.dmamodeport     = 0xd6;
   sb.dmabaseaddrport = 0xc0 + 4 * (sb.hdma - 4);
   sb.dmacountport    = 0xc2 + 4 * (sb.hdma - 4);
   switch (sb.hdma)
   {
     case 4: // never used
       sb.pageport = 0x8F;
       break;
     case 5:
       sb.pageport = 0x8B;
       break;
     case 6:
       sb.pageport = 0x89;
       break;
     case 7:
       sb.pageport = 0x8A;
       break;
   }
   sb.dstam  = (sb.hdma - 4);
   sb.dstom  = (sb.hdma - 4) | 0x04;
   sb.dmode  = (sb.hdma - 4) | 0x58;
   sb.ackport= sb.base + 0x0F;
   break;
 }

 if (sb.autoinit == 1)
   sb.dmalen = (sb.dma_buf_size * 2);
 else
   sb.dmalen = sb.dma_buf_size;

 if (sb.irq <= 7)
 {
   sb.airq = sb.irq + 0x08;
   sb.pic  = 0x21;
 }
 else
 {
   sb.airq = 0x70 + sb.irq - 0x08;
   sb.pic  = 0xA1;
 }

 sb.istom        = 1 << (sb.irq % 8);
 sb.istam        = (sb.istom ^ 0xFF);
 mixerinfo.vuoro = 0;
 sb.interrupts   = 0;

 disable();

 // disable sb's interrupt :
 outportb(sb.pic, (inportb(sb.pic)|sb.istom) );

 // get the old sb interrupt handler :
 _go32_dpmi_get_protected_mode_interrupt_vector(sb.airq,&sb.vanha);

 // set up the new handler :
 sb.info.pm_offset=(unsigned long int)keskeytys;
 sb.info.pm_selector=_my_cs();
 disable();
 _go32_dpmi_allocate_iret_wrapper(&sb.info);
 _go32_dpmi_set_protected_mode_interrupt_vector(sb.airq,&sb.info);

 // ack everything :
 if (sb.irq>7) outportb(0xA0,0x20);
 outportb(0x20,0x20);
 inportb(sb.ackport);

 sb.installed=1;

 // enable sb's irq :
 if (sb.irq > 7) outportb(0x21, inportb(0x21) & 0xFB); // irq 2
 outportb(sb.pic, (inportb(sb.pic) & sb.istam) );      // sb's irq

 if (sb._16bit == 1) writedsp(0xD5); // halt 16 bit dma operation
 else
 {
   writedsp(0xD0);  // halt 8 bit dma operation
   writedsp(0xD3);  // disable speaker
 }

 enable();   // enable interrupts

 playback(); // start playback
 return 0;   // everything ok
}

int sb_deinit()
{
 int tulos;
 if (sb.installed != 1)
 {
   strcat(error, "sb_deinit: sb driver not installed.\n");
   return -1;
 }

 disable();

 // stop playback (for sure):
 resetdsp(5);
 resetdsp(5);

 outportb(sb.dmamaskport, sb.dstom);
 if (sb._16bit == 1) writedsp(0xd5); // stop 16 bit dma
 else
 { // stop 8 bit dma and close speaker
   writedsp(0xD0);
   writedsp(0xD3);
 }

 disable();

 // disallow the sb interrupt :
 outportb(sb.pic, (inportb(sb.pic) | sb.istom) );

 // restore the old interrupt vector:
 tulos = _go32_dpmi_set_protected_mode_interrupt_vector(sb.airq,&sb.vanha);

 // free the reserver base memory :
 tulos += __dpmi_free_dos_memory(sb.dma_buf_selector);

 // free volumetable and tmp mixing buffer :
 if (voltable != NULL) free(voltable);
 if (mtmp != NULL) free(mtmp);
 inportb(sb.ackport);

 enable();

 sb.installed = 0;
 return tulos;
}

int sb_playsample(int channaa, int sample, int volume, int frequ, int ppan, int __offset)
{
  if (sample >= 130) return -1;
  if (samples[sample].used != 1) return -1;
  if (channaa > 32) return -1;
  asm volatile (
  "pushfl
   cli");
  channel[channaa].playing = 0;
  channel[channaa].sample  = sample;

  if (volume == -1) channel[channaa].volume = samples[sample].volume;
  else if (volume != -2) channel[channaa].volume = volume;

  if (frequ == -1) channel[channaa].freq = samples[sample].c2spd;
  else if (frequ != -2) channel[channaa].freq = frequ;

  if (ppan == -1) channel[channaa].pan = samples[sample].pan;
  else if (ppan != -2) channel[channaa].pan = ppan;

  channel[channaa].position = __offset;
  channel[channaa].position <<= 32;
  if (__offset < samples[sample].length) channel[channaa].playing = 1;
  asm volatile ("popfl");
  return 0;
}

void sb_stopchannel(int channaa)
{
  channel[channaa].playing = 0;
}

void sb_set_volume(int channaa, int _volume)
{
  if (_volume < 0) _volume = 0;
  else if (_volume > 64) _volume = 64;
  channel[channaa].volume = _volume;
}

void sb_set_pan(int channaa, int _pan)
{
  if (_pan > 256) _pan = 256;
  else if (_pan < 0) _pan = 0;
  channel[channaa].pan = _pan;
}

void sb_set_freq(unsigned int channaa, unsigned int _freq)
{
  channel[channaa].freq = _freq;
}

int sb_loadsample_8_bits_raw_from_memory(sbyte *data, int samplenro,
      int frequ, int pan, int loopstart, int loopend,
      int length, char ssigned)
{
  int i;
  if (data == NULL)
  {
    strcat(error, "sb_lsmp_mem: no data.\n");
    return -1;
  }
  samples[samplenro].length = length;
  samples[samplenro].data = (char *)malloc(samples[samplenro].length + 4);
  if (samples[samplenro].data == NULL)
  {
    strcat(error, "sb_lsmp: out of memory.\n");
    return -1;
  }
  memset(samples[samplenro].data, 0, samples[samplenro].length + 4);
  _go32_dpmi_lock_data(samples[samplenro].data, samples[samplenro].length + 4);

  memcpy(samples[samplenro].data, data, length);

  for (i = 0; i < samples[samplenro].length; i++)
    samples[samplenro].data[i] ^= ssigned;

  samples[samplenro].c2spd       = frequ;
  samples[samplenro].loop_start  = loopstart;
  samples[samplenro].loop_endi   = loopend;
  samples[samplenro].loop_length = loopend - loopstart;
  samples[samplenro].pan         = pan;
  samples[samplenro].used        = 1;
  return 0;
}


int sb_loadsample_8_bits_raw_from_opened_file_curpos(FILE *f,
  int samplenro, int frequ, int pan, int loopstart, int loopend,
  int read_length, char ssigned)
{
  int i;
  if (f == NULL)
  {
    strcat(error, "sb_lsmp: file not open.\n");
    return -1;
  }
  samples[samplenro].length = read_length;
  samples[samplenro].data = (char *)malloc(samples[samplenro].length + 4);
  if (samples[samplenro].data == NULL)
  {
    strcat(error, "sb_lsmp: out of memory.\n");
    return -1;
  }
  memset(samples[samplenro].data, 0, samples[samplenro].length + 4);
  _go32_dpmi_lock_data(samples[samplenro].data, samples[samplenro].length + 4);

  if ((-1) == fread(samples[samplenro].data, 1, samples[samplenro].length, f))
  {
    strcat(error, "sb_lsmp: load error.\n");
    return -1;
  }

  for (i = 0; i < samples[samplenro].length; i++)
    samples[samplenro].data[i] ^= ssigned;

  samples[samplenro].c2spd       = frequ;
  samples[samplenro].loop_start  = loopstart;
  samples[samplenro].loop_endi   = loopend;
  samples[samplenro].loop_length = loopend - loopstart;
  samples[samplenro].pan         = pan;
  samples[samplenro].used        = 1;
  return 0;
}

int sb_loadsample_signed_8_bits_raw(char *filename, int samplenro, int frequ,
  int pan, int loopstart, int loopend)
{
  FILE *f;
  signed int rr;

  if (samples[samplenro].used == 1)
  {
    strcat(error, "sb_lsmp: sample slot already in use.\n");
    return -1;
  }

  f = fopen(filename, "rb");
  if (f == NULL)
  {
    unsigned char tempp[50] = "sb_lsmp: file ("; // hard?
    strcat(tempp, filename);
    strcat(tempp, ") not found.\n");
    strcat(error, tempp);
    return -1;
  }
  samples[samplenro].length = filelength(fileno(f));
  rr = sb_loadsample_8_bits_raw_from_opened_file_curpos(f, samplenro, frequ,
    pan, loopstart, loopend, samples[samplenro].length, 0x80);

  fclose(f);
  return rr;
}


int sb_freesample(int samplenro)
{
 if ((samplenro < 0) || (samplenro > 129))
   return -1;
 if ((samples[samplenro].used == 0) || (samples[samplenro].data == NULL))
   return -1;
 samples[samplenro].used = 0;
 free(samples[samplenro].data);
 return 0;
}

int amp(int active_channels)
{
 return 1028 - (active_channels * 7);
}

void set_amplification(signed int level)
{
 signed int b, c;

 if (level < 33) level = 33;
 else if (level > 16384) level = 16384;

 mixerinfo.amplification = level;

 level = (level * amp(mixerinfo.active_channels)) / 1000;

 // calculate the volume table :
 for (b = 0; b < 65; b++)
   for (c = 0; c < 256; c++)
   {
    voltable[(b << 8) | c]=((shuge)level * (shuge)b * (shuge)(c - 128)) / 256;
   }
}

void sb_set_channels(int __channels)
{
  if (__channels > 32) __channels = 32;
  else if (__channels < 4) __channels = 4;
  mixerinfo.active_channels = __channels;
  set_amplification(mixerinfo.amplification);
}

int sb_get_channels()
{
  return mixerinfo.active_channels;
}

int sb_memory()
{
 return 8192*1024;
}

char sbmodel[6][3] =
{
  "1.0",
  "1.5",
  "2.0",
  "pro",
  "16\0",
  "???"
};

int get_sbmodel()
{
  if (sb.dspversion < 0x200) return 0;
  if (sb.dspversion == 0x200) return 1;
  if (sb.dspversion < 0x300) return 2;
  if (sb.dspversion < 0x401) return 3;
  if (sb.dspversion > 0x400) return 4;
  return 5;
}


// IN: 1 yes, do ask; 0 no, don't ask
// OUT: 0 successful, -1 failed.
int sb_init(int askuser)
{
 printf("\rsbdev v1.04, Sound Blaster 1.0/1.5/2.0/pro/16/compatibles \n");

 voltable = (signed int*)malloc(16640 * 4);
 if (voltable == NULL)
 {
   strcat(error, "sb_init: out of memory.\n");
   return -1;
 }

 mtmp = (signed int*)malloc(6000 * 4);
 if (mtmp == NULL)
 {
   strcat(error, "sb_init: out of memory.\n");
   free(voltable);
   return -1;
 }

 mixerinfo.active_channels = 32;
 set_amplification(DEFAULT_AMPLIFICATION);
 sb_set_channels(32);

 _go32_dpmi_lock_code(bom, ((char *)eom - (char *)bom));
 _go32_dpmi_lock_data(&mixerinfo, sizeof(mixerinfo));
 _go32_dpmi_lock_data(&samples[0], sizeof(samrec) * 130);
 _go32_dpmi_lock_data(&channel[0], sizeof(channelinforec) * 33);
 _go32_dpmi_lock_data(voltable, 16640 * 4);
 _go32_dpmi_lock_data(mtmp, 6000 * 4);
 _go32_dpmi_lock_data(&mi, 4);
 _go32_dpmi_lock_data(&sam, 4);
 _go32_dpmi_lock_data(&monovol, 4);
 _go32_dpmi_lock_data(&rightvol, 4);
 _go32_dpmi_lock_data(&leftvol, 4);
 _go32_dpmi_lock_data(&rtaul, 4);
 _go32_dpmi_lock_data(&ltaul, 4);
 _go32_dpmi_lock_data(&ma, 4);
 _go32_dpmi_lock_data(&taul, 4);
 _go32_dpmi_lock_data(&vali, 4);
 _go32_dpmi_lock_data(&selektoori, 2);
 _go32_dpmi_lock_data(&bytes, 8);
 _go32_dpmi_lock_data(&bytez, 8);
 _go32_dpmi_lock_data(&fadd, 8);
 _go32_dpmi_lock_data(&deita, 4);
 _go32_dpmi_lock_data(&pit, 4);
 _go32_dpmi_lock_data(&faddl, 4);
 _go32_dpmi_lock_data(&faddh, 4);
 _go32_dpmi_lock_data(&posil, 4);
 _go32_dpmi_lock_data(&posih, 4);
 _go32_dpmi_lock_data(&sb, sizeof(sb));

 if (soundblaster_init(askuser) != 0)
 {
   if (voltable != NULL) free(voltable);
   if (mtmp != NULL) free(mtmp);
   strcat(error, "sb_init: sb initialization failed.\n");
   return -1;
 }

 printf("sb %s: A%Xh I%d D%d", sbmodel[get_sbmodel()], sb.base, sb.irq,
   sb.dma);
 if (sb._16bit == 1) printf(" H%d", sb.hdma);

 printf(", output: ");
 if (sb._16bit == 1) printf("16 bits, "); else printf("8 bits, ");
 if (sb.stereo == 1) printf("stereo, "); else printf("mono, ");
 printf("%d Hz, ", sb.playspeed);

 if (sb.autoinit == 1) printf("autoinit dma.\n"); else
   printf("irq controlled dma.\n");

 sb_set_channels(24);
 mastervolume(64);

 return 0;
}

// end of file
