/**
 ** VESA.C ---- the GRX 2.0 VESA BIOS interface
 **
 ** Copyright (c) 1995 Csaba Biegl, 820 Stirrup Dr, Nashville, TN 37221
 ** [e-mail: csaba@vuse.vanderbilt.edu] See "doc/copying.cb" for details.
 **/

#include <string.h>

#include "grdriver.h"
#include "libgrx.h"
#include "arith.h"
#include "int86.h"
#include "ioport.h"
#include "memfill.h"
#include "memcopy.h"
#include "vesa.h"

#define  NUM_MODES    30		/* max # of supported modes */
#define  NUM_EXTS     8			/* max # of mode extensions */

static GrVideoMode    modes[NUM_MODES];
static GrVideoModeExt exts[NUM_EXTS];

static char hicolor32K  = FALSE;
static char protbanking = FALSE;
static char fast256	= 0;
static int  VESAbanksft = (-1);
static int  VESArdbank;
static int  VESAwrbank;
static int  VESAversion;
static void far (*VESAbankfn)(void);

static void setbank(int bk)
{
	Int86Regs r;
	if(protbanking) {
#	    ifdef __TURBOC__
		_DX = bk << VESAbanksft;
		_BX = VESAwrbank;
		_AX = VESA_FUNC + VESA_PAGE_CTL;
		(*VESAbankfn)();
		if(VESArdbank >= 0) {
		    _DX = bk << VESAbanksft;
		    _BX = VESArdbank;
		    _AX = VESA_FUNC + VESA_PAGE_CTL;
		    (*VESAbankfn)();
		}
		return;
#	    endif
#	    ifdef __GO32__
		/*
		 * To be done by a DP(IG)MI hacker. I have no intention of
		 * of supporting Billy by becoming one.
		 */
#	    endif
	}
	sttzero(&r);
	IREG_AX(r) = VESA_FUNC + VESA_PAGE_CTL;
	IREG_BX(r) = VESAwrbank;
	IREG_DX(r) = bk << VESAbanksft;
	int10(&r);
	if(VESArdbank >= 0) {
	    IREG_AX(r) = VESA_FUNC + VESA_PAGE_CTL;
	    IREG_BX(r) = VESArdbank;
	    IREG_DX(r) = bk << VESAbanksft;
	    int10(&r);
	}
	setup_far_selector(SCRN->gc_selector);
}

static void setrwbanks(int rb,int wb)
{
	Int86Regs r;
	if(protbanking) {
#	    ifdef __TURBOC__
		_DX = wb << VESAbanksft;
		_BX = VESAwrbank;
		_AX = VESA_FUNC + VESA_PAGE_CTL;
		(*VESAbankfn)();
		_DX = rb << VESAbanksft;
		_BX = VESArdbank;
		_AX = VESA_FUNC + VESA_PAGE_CTL;
		(*VESAbankfn)();
		return;
#	    endif
#	    ifdef __GO32__
		/*
		 * To be done by a DP(IG)MI hacker. I have no intention of
		 * of supporting Billy by becoming one.
		 */
#	    endif
	}
	sttzero(&r);
	IREG_AX(r) = VESA_FUNC + VESA_PAGE_CTL;
	IREG_BX(r) = VESAwrbank;
	IREG_DX(r) = wb << VESAbanksft;
	int10(&r);
	IREG_AX(r) = VESA_FUNC + VESA_PAGE_CTL;
	IREG_BX(r) = VESArdbank;
	IREG_DX(r) = rb << VESAbanksft;
	int10(&r);
	setup_far_selector(SCRN->gc_selector);
}

static int detect(void)
{
	if(_GrViDrvDetectVGA()) {
	    VESAvgaInfoBlock blk;
	    if(_GrViDrvVESAgetVGAinfo(&blk)) return(TRUE);
	}
	return(FALSE);
}

static int build_video_mode(VESAmodeInfoBlock *ip,GrVideoMode *mp,GrVideoModeExt *ep)
{
	int banksft = 0;
	int rdbank  = (-1);
	int wrbank  = (-1);
	mp->present    = TRUE;
	mp->width      = ip->XResolution;
	mp->height     = ip->YResolution;
	mp->bpp	       = ip->BitsPerPixel;
	mp->lineoffset = ip->BytesPerScanLine;
	mp->extinfo    = NULL;
	mp->privdata   = 0;
	if(!(ip->ModeAttributes & MODE_ISGRAPHICS)) {
	    mp->extinfo = &_GrViDrvEGAVGAtextModeExt;
	    return(TRUE);
	}
	if(ip->WinSize != 64) return(FALSE);
	while((ip->WinGranularity << banksft) < 64) banksft++;
	if((ip->WinGranularity << banksft) != 64) return(FALSE);
	if(ip->WinAAttributes & WIN_SUPPORTED) {
	    if(ip->WinAAttributes & WIN_WRITABLE) wrbank = 0;
	    if(ip->WinAAttributes & WIN_READABLE) rdbank = 0;
	}
	if(ip->WinBAttributes & WIN_SUPPORTED) {
	    if(ip->WinBAttributes & WIN_WRITABLE) wrbank = 1;
	    if(ip->WinBAttributes & WIN_READABLE) rdbank = 1;
	}
	if(wrbank < 0) return(FALSE);
	if(rdbank >= 0) {
	    if(rdbank == wrbank) rdbank = (-1);
	    if(ip->WinASegment != ip->WinBSegment) rdbank = (-1);
	}
	if(VESAbanksft >= 0) {
	    if(banksft != VESAbanksft) return(FALSE);
	    if(wrbank  != VESAwrbank)  return(FALSE);
	    if(rdbank  != VESArdbank)  return(FALSE);
	}
	VESAbanksft = banksft;
	VESAwrbank  = wrbank;
	VESArdbank  = rdbank;
	if(protbanking) {
	    if(VESAbankfn && (VESAbankfn != ip->WinFuncPtr)) protbanking = FALSE;
	    VESAbankfn = ip->WinFuncPtr;
	}
	ep->mode       = GR_frameUndef;
	ep->drv	       = NULL;
	ep->frame      = MK_FP((wrbank ? ip->WinBSegment : ip->WinASegment),0);
	ep->cprec[0]   = ep->cprec[1] = ep->cprec[2] = 6;
	ep->cpos[0]    = ep->cpos[1]  = ep->cpos[2]  = 0;
	ep->flags      = 0;
	ep->setup      = _GrViDrvSetEGAVGAmode;
	ep->setvsize   = _GrViDrvVESAsetVirtualSize;
	ep->scroll     = _GrViDrvVESAvirtualScroll;
	ep->setbank    = setbank;
	ep->setrwbanks = (rdbank >= 0) ? setrwbanks : NULL;
	ep->loadcolor  = NULL;
	switch(ip->BitsPerPixel) {
	  case 4:
	    if(ip->MemoryModel != MODEL_4PLANE) return(FALSE);
	    if(ip->NumberOfPlanes != 4) return(FALSE);
	    ep->mode	  = GR_frameSVGA4;
	    ep->loadcolor = _GrViDrvLoadColorVGA4;
	    break;
	  case 8:
	    if(ip->MemoryModel != MODEL_PACKED) return(FALSE);
	    if(ip->NumberOfPlanes != 1) return(FALSE);
	    ep->mode	  = GR_frameSVGA8;
	    ep->loadcolor = _GrViDrvLoadColorVGA8;
	    ep->flags	  = fast256;
	    break;
	  case 15:
	  case 16:
	  case 24:
	    if((ip->MemoryModel != MODEL_PACKED) &&
	       (ip->MemoryModel != MODEL_DIRECT)) return(FALSE);
	    if(ip->NumberOfPlanes != 1) return(FALSE);
	    ep->mode = (ip->BitsPerPixel == 24) ? GR_frameSVGA24 : GR_frameSVGA16;
	    if(VESAversion < VESA_VERSION(1,2)) {
		if(ip->BitsPerPixel == 24) {
		    ep->cprec[0] = 8; ep->cpos[0] = 16;
		    ep->cprec[1] = 8; ep->cpos[1] = 8;
		    ep->cprec[2] = 8; ep->cpos[2] = 0;
		    break;
		}
		if((ip->BitsPerPixel == 15) || hicolor32K) {
		    mp->bpp	 = 15;
		    ep->cprec[0] = 5; ep->cpos[0] = 10;
		    ep->cprec[1] = 5; ep->cpos[1] = 5;
		    ep->cprec[2] = 5; ep->cpos[2] = 0;
		    break;
		}
		mp->bpp	     = 16;
		ep->cprec[0] = 5; ep->cpos[0] = 11;
		ep->cprec[1] = 6; ep->cpos[1] = 5;
		ep->cprec[2] = 5; ep->cpos[2] = 0;
		break;
	    }
	    ep->cprec[0] = ip->RedMaskSize;   ep->cpos[0] = ip->RedMaskPos;
	    ep->cprec[1] = ip->GreenMaskSize; ep->cpos[1] = ip->GreenMaskPos;
	    ep->cprec[2] = ip->BlueMaskSize;  ep->cpos[2] = ip->BlueMaskPos;
	    break;
	  default:
	    return(FALSE);
	}
	return(TRUE);
}

static void add_video_mode(
    GrVideoMode *mp,  GrVideoModeExt *ep,
    GrVideoMode **mpp,GrVideoModeExt **epp
){
	if(*mpp < &modes[NUM_MODES]) {
	    if(!mp->extinfo) {
		GrVideoModeExt *etp = &exts[0];
		while(etp < *epp) {
		    if(memcmp(etp,ep,sizeof(GrVideoModeExt)) == 0) {
			mp->extinfo = etp;
			break;
		    }
		    etp++;
		}
		if(!mp->extinfo) {
		    if(etp >= &exts[NUM_EXTS]) return;
		    sttcopy(etp,ep);
		    mp->extinfo = etp;
		    *epp = ++etp;
		}
	    }
	    sttcopy(*mpp,mp);
	    (*mpp)++;
	}
}

static int get_tweaked_text_mode(GrVideoMode *mp,GrVideoMode *etable)
{
	if(etable < &modes[NUM_MODES]) {
	    GrVideoMode *m1,*m2;
	    for(m1 = modes; m1 < etable; m1++) {
		if((m1->present) &&
		   (m1->extinfo) &&
		   (m1->extinfo->mode == GR_frameText) &&
		   (m1->height == 25) &&
		   (m1->width > 80)) {
		    VESAmodeInfoBlock mdinfo;
		    if((_GrViDrvVESAgetModeInfo(m1->mode,&mdinfo)) &&
		       (mdinfo.ModeAttributes & MODE_EXTINFO) &&
		       (mdinfo.YCharSize == 16)) {
			int h,exists = FALSE;
			for(h = 28; h <= 50; h += (50 - 28)) {
			    for(m2 = modes; m2 < etable; m2++) {
				if((m2->present) &&
				   (m2->extinfo) &&
				   (m2->extinfo->mode == GR_frameText) &&
				   (m2->height == h) &&
				   (m2->width == m1->width) &&
				   (m2->bpp == m1->bpp)) {
				    exists = TRUE;
				    break;
				}
			    }
			    if(!exists) {
				sttcopy(mp,m1);
				mp->height  = h;
				mp->extinfo = &_GrViDrvEGAVGAcustomTextModeExt;
				return(TRUE);
			    }
			}
		    }
		}
	    }
	}
	return(FALSE);
}

static int init(char *options)
{
	if(_GrViDrvInitEGAVGA(options)) {
	    VESAvgaInfoBlock blk;
	    memzero(modes,sizeof(modes));
	    if(_GrViDrvVESAgetVGAinfo(&blk)) {
		VESAmodeInfoBlock mdinfo;
		GrVideoMode	  mode,*modep = &modes[0];
		GrVideoModeExt	  ext, *extp  = &exts[0];
		short far	 *mp;
		hicolor32K  = FALSE;
		protbanking = FALSE;
		fast256	    = 0;
		VESAbanksft = (-1);
		VESAversion = blk.VESAversion;
		if(options) while(*options != '\0') {
		    switch(*options) {
		      case '5':
			hicolor32K  = TRUE;
			break;
		      case 'p':
		      case 'P':
			protbanking = TRUE;
			break;
		      case 'f':
		      case 'F':
			fast256 = GR_VMODEF_FAST_SVGA8;
			break;
		    }
		    options++;
		}
		for(mp = blk.VideoModePtr; *mp != (-1); mp++) {
		    mode.mode = *mp;
		    if(!_GrViDrvVESAgetModeInfo(*mp,&mdinfo))	continue;
		    if(!(mdinfo.ModeAttributes & MODE_EXTINFO)) continue;
		    if(!build_video_mode(&mdinfo,&mode,&ext))	continue;
		    add_video_mode(&mode,&ext,&modep,&extp);
		}
		while(get_tweaked_text_mode(&mode,modep)) {
		    add_video_mode(&mode,&ext,&modep,&extp);
		}
	    }
	    return(TRUE);
	}
	return(FALSE);
}

GrVideoDriver _GrVideoDriverVESA = {
    "VESA",				/* name */
    GR_VGA,				/* adapter type */
    &_GrVideoDriverSTDVGA,		/* inherit modes from this driver */
    modes,				/* mode table */
    itemsof(modes),			/* # of modes */
    detect,				/* detection routine */
    init,				/* initialization routine */
    _GrViDrvResetEGAVGA			/* reset routine */
};


