// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

// Scan converter for convex and clipped polygons.
// Called by FillPoly.
// Could also be used by other polygon filling functions.

#include <math.h>
#include "gfx/bitmap.h"
#include "gfx/poly.h"
#include "misc/error.h"

#ifndef MAX_POLYGON_HEIGHT
#define MAX_POLYGON_HEIGHT 480
#endif


DWORD PolyEdgeX[MAX_POLYGON_HEIGHT*2+1];     // buffer for x - values

// Determine the egde of the CONVEX! polygon.
// Store the left and right x-coordinate of each horizontal pixelline
// in the PolyEdgeX array.
// The function returns the number of the top line.
// PolyEdgeX[0]     = x value of left edge pixel value << 16 in top line.
//          [1]     =    "       right                "              
//          [k*2]   =            left                        in line top+k.
//          [k*2+1] =            right
// Read the hi - word to get the proper value.
// if (PolyEdgeX[k*2]<0) then this line does not belong to the polygon.
int ScanConvertPoly(Polygon &Poly)
  {
  #ifdef DEBUG
  if (Poly.Vertices<3) ErrorExit("ScanPoly()  Need at least 3 vertices.");
  #endif
  int Clockwise=1; // flag for the arrangement of vertices: 1=clockwise, -1=counterclockwise
  const int LastIndex=Poly.Vertices-1;
  Vector2D * const LastVertex=&Poly.Vertex[LastIndex];
  // search the vertex with the minimum y-coordinate (the upmost vertex)
  Vector2D *TopVertex;
  int TopIndex,TopY,BottomY;
  TopIndex=0;
  TopY=BottomY=Poly.Vertex[0].y;
  Vector2D *Vertex=TopVertex=Poly.Vertex;
  for (int Index=1;Index<Poly.Vertices;Index++)
    {
    Vertex++;
    if (Vertex->y<TopY)
      {
      TopVertex=Vertex;
      TopIndex=Index; // this vertex is higher
      TopY=Vertex->y;
      }
    else
      if (Vertex->y>BottomY)
        BottomY=Vertex->y;
    }
  #ifdef DEBUG
  if (TopY<0) ErrorExit("ScanPoly()  Polygon is not clipped (y<0).");
  if (BottomY-TopY+1>=MAX_POLYGON_HEIGHT) ErrorExit("ScanPoly()  Polygon is too large (change MAX_POLYHEIGHT)");
  #endif
  if (BottomY!=TopY) // the algorithm only works correct if the polygon is higher than one pixel
    {
    // now we need to know if the vertices are in clockwise or counterclockwise
    // order.
    if (!(Poly.Flags&(Polygon::CounterClockwise|Polygon::CounterClockwise)))
      {
      // only find out the order if the order has not yet been computed.
      Vector2D *CheckVertex=Poly.Vertex;
      Index=2;
      do
        {
        int ux=CheckVertex[2].x-CheckVertex[1].x;
        int uy=CheckVertex[2].y-CheckVertex[1].y;
        int vx=CheckVertex[1].x-CheckVertex[0].x;
        int vy=CheckVertex[1].y-CheckVertex[0].y;
        long prod=ux*vy-vx*uy;
        if (prod!=0) // otherwise the three vertices are on the same line -> check next pair
          {
          Clockwise=prod<0?1:-1;
          break;
          }
        CheckVertex++;
        }
      while (Index<Poly.Vertices);
      }
    else
      {
      // The order of this polygon has been precomputed (see polygon.h).
      Clockwise=Poly.Flags&Polygon::Clockwise?1:-1;
      }
    // Calculate the x-coordinates of the right edge of the polygon by
    // walking thru the vertices clockwise from the top-vertex.
    int VertexIndex2=TopIndex;
    Vector2D *Vertex1,*Vertex2=TopVertex;
    int Direction=Clockwise;      // walk along the right edge
    long x;                       // x has to be recalled below
    do
      {
      Vertex1=Vertex2;
      // walk on to the next vertex
      if (Direction>0)
        if (VertexIndex2!=LastIndex)
          {
          VertexIndex2++;
          Vertex2++;
          }
        else
          {
          VertexIndex2=0;
          Vertex2=Poly.Vertex;
          }
      else
        if (VertexIndex2!=0)
          {
          VertexIndex2--;
          Vertex2--;
          }
        else
          {
          VertexIndex2=LastIndex;
          Vertex2=LastVertex;
          }
      int dy=Vertex2->y-Vertex1->y;
      if (dy>0) // we can forget horizontal lines (dy==0); if dy<0 then the poly is not convex
        {
        int Pixels=dy;                     // e.g. calculate 5 pixels in a line with dy=5 (that line contains 6 horiz scanlines)
        long xAddPerLine=0;
        x=Vertex1->x<<16;                   // fixed point (8bit after the point)
        int Diff=Vertex2->x-Vertex1->x;
        if (Diff!=0)
          {
          if (Diff>0) Diff++; else Diff--;
          xAddPerLine=(Diff<<16)/(dy+1);
          // Adjust AddPerLine.
          if (xAddPerLine>0)
            {
            x+=xAddPerLine;
            if (xAddPerLine>=1L<<16) x-=1<<16; // steep line adjustment
            }
          else
            x+=(1L<<16)-1;    // prevent round down
          }
        // Calculate x-coords.
        int yIndex=Vertex1->y*2+1;
        /**********
        // This here is the inner loop that computes the x-coords.
        do
          {
          PolyEdgeRightX[yIndex]=x>>16;
          yIndex+=2;
          x+=xAddPerLine;
          }
        while (--Pixels);
        // we speed the process up: 
        // 1. omit the shift, we store the unshifted value, so the
        //    filling process must read the hi word to get the proper value
        // 2. unroll the loop
        *******/
        const int Log2BlockSize=2; // handle 2^2=4 lines per loop
        int BlocksToGo=Pixels>>Log2BlockSize;
        while (BlocksToGo--) // unrolled loop
          {
          PolyEdgeX[yIndex+0]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+2]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+4]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+6]=x;x+=xAddPerLine;
          yIndex+=8;
          }
        if (Pixels&2)
          {
          PolyEdgeX[yIndex+0]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+2]=x;x+=xAddPerLine;
          yIndex+=4;
          }
        if (Pixels&1)
          {
          PolyEdgeX[yIndex  ]=x;x+=xAddPerLine;
          }
        }
      }
    while (Vertex2->y<BottomY);
    PolyEdgeX[Vertex2->y*2+1]=x;  // set last x coord.; dont just take vertex2.x (is obvious when the slope of the last line is small, line flat)
    // Calculate the x-coordinates of the left edge of the polygon by
    // walking thru the vertices clockwise from the top-vertex.
    VertexIndex2=TopIndex;
    Vertex2=TopVertex;
    Direction=-Clockwise;          // walk along the right edge
    do
      {
      Vertex1=Vertex2;
      // walk on to the next vertex
      if (Direction>0)
        if (VertexIndex2!=LastIndex)
          {
          VertexIndex2++;
          Vertex2++;
          }
        else
          {
          VertexIndex2=0;
          Vertex2=Poly.Vertex;
          }
      else
        if (VertexIndex2!=0)
          {
          VertexIndex2--;
          Vertex2--;
          }
        else
          {
          VertexIndex2=LastIndex;
          Vertex2=LastVertex;
          }
      int dy=Vertex2->y-Vertex1->y;
      if (dy>0) // we can forget horizontal lines (dy==0); if dy<0 then the poly is not convex
        {
        int Pixels=dy;                     // e.g. calculate 5 pixels in a line with dy=5 (that line contains 6 horiz scanlines)
        long xAddPerLine=0;
        x=Vertex1->x<<16;                   // fixed point (8bit after the point)
        int Diff=Vertex2->x-Vertex1->x;
        if (Diff!=0)
          {
          if (Diff>0) Diff++; else Diff--;
          xAddPerLine=(Diff<<16)/(dy+1);
          // Adjust AddPerLine.
          if (xAddPerLine<0) x+=xAddPerLine+(1<<16); // prevent round down
          }
        // Calculate x-coords.
        int yIndex=Vertex1->y*2;
        const int Log2BlockSize=2; // blocks of four lines
        int BlocksToGo=Pixels>>Log2BlockSize;
        while (BlocksToGo--) // unrolled loop
          {
          PolyEdgeX[yIndex+0]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+2]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+4]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+6]=x;x+=xAddPerLine;
          yIndex+=8;
          }
        if (Pixels&2)
          {
          PolyEdgeX[yIndex+0]=x;x+=xAddPerLine;
          PolyEdgeX[yIndex+2]=x;x+=xAddPerLine;
          yIndex+=4;
          }
        if (Pixels&1)
          {
          PolyEdgeX[yIndex  ]=x;x+=xAddPerLine;
          }
        }
      }
    while (Vertex2->y<BottomY);
    PolyEdgeX[Vertex2->y*2]=x;         // set last x coord.; dont just take vertex2.x (is obvious when the slope of the last line is small, line flat)
    PolyEdgeX[(Vertex2->y+1)*2]=-1;    // mark end
    }
  else
    {
    // This is the special case the upper algorithm got problems with:
    // the polygon is a horizontal line. Search min and max x to
    // draw the correct size.
    int xmin,xmax;
    Vertex=Poly.Vertex;
    xmin=xmax=Vertex->x;
    for (Index=1;Index<Poly.Vertices;Index++)
      {
      Vertex++;
      if (Vertex->x<xmin)
        xmin=Vertex->x;
      else
        if (Vertex->x>xmax)
          xmax=Vertex->x;
      }
    PolyEdgeX[TopY*2  ]=xmin<<16;
    PolyEdgeX[TopY*2+1]=xmax<<16;
    PolyEdgeX[TopY*2+2]=-1; // mark end
    }
  return(TopY);
  }

