// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include <fstream.h>
#include "misc/mem.h"
#include "compiler/types.h"
#include "gfx/bitmap.h"

struct PCXHeader
  {
  BYTE manufacturer;   // don't care
  BYTE version;        // 5 = version 3.0
  BYTE encoding;       // 1 = RLE
  BYTE bits_per_pixel; // per plane
  WORD  xmin,ymin;
  WORD  xmax,ymax;
  WORD  hres;
  WORD  vres;
  BYTE palette16[48];  // only used for 16colors
  BYTE reserved;
  BYTE color_planes;
  WORD  bytes_per_line;
  WORD  palette_type;
  BYTE filler[58];
  };

int SavePCX(Bitmap &Bitmap,char *Name)
  {
  if (!Bitmap.IsValid()) return(-5);
  int Error=0;
  fstream PCXFile;
  PCXFile.open(Name,ios::out|ios::binary);
  if (PCXFile.is_open())
    {
    // check if bitmap can be saved
    if (!(Bitmap.ClassID()==Bitmap::ClassPL1
     ||   Bitmap.ClassID()==Bitmap::ClassPP8
     ||   Bitmap.ClassID()==Bitmap::ClassDC888))
      return(-5);
    // save header
    PCXHeader Header;
    Header.manufacturer=10;
    Header.version=5;
    Header.encoding=1;
    Header.xmin=0;
    Header.ymin=0;
    Header.xmax=Bitmap.Width()-1;
    Header.ymax=Bitmap.Height()-1;
    Header.hres=Bitmap.Width();
    Header.vres=Bitmap.Height();
    memset(Header.palette16,0,48);
    Header.reserved=0;
    switch (Bitmap.ClassID())
      {
      case Bitmap::ClassPL1:
        Header.bits_per_pixel=1; // per plane
        Header.color_planes=1;
        Header.bytes_per_line=Bitmap.WidthBytes(); // per plane
        Header.palette_type=1;
        break;
      case Bitmap::ClassPP8:
        Header.bits_per_pixel=8; // per plane
        Header.color_planes=1;
        Header.bytes_per_line=Bitmap.WidthBytes();
        Header.palette_type=2;
        break;
      case Bitmap::ClassDC888:
        Header.bits_per_pixel=8;
        Header.color_planes=3;
        Header.bytes_per_line=Bitmap.Width(); // per plane
        Header.palette_type=1;
        break;
      }
    memset(Header.filler,0,58);
    PCXFile.write((char*)&Header,sizeof(struct PCXHeader));
    // save pic
    unsigned char *BitmapPtr=Bitmap.Bits();
    int PCXBytesPerLine=Header.bytes_per_line*Header.color_planes;
    unsigned char *PCXLineData=new unsigned char[PCXBytesPerLine]; // alloc buffer for TrueColor transformation
    for (int Line=0;Line<Bitmap.Height();Line++)
      {
      // copy the line that is to be written into the output buffer
      if (Bitmap.ClassID()!=Bitmap::ClassDC888)
        {
        memcpy(PCXLineData,BitmapPtr,Bitmap.WidthBytes());
        BitmapPtr+=Bitmap.WidthBytes();
        }
      else
        // the first bytes_per_line bytes are red, then green, then blue.
        // Only DC888 supported.
        for (int x=0;x<Bitmap.Width();x++)
          {
          PCXLineData[x]=BitmapPtr[2]; // red
          PCXLineData[x+Header.bytes_per_line]=BitmapPtr[1]; // green
          PCXLineData[x+Header.bytes_per_line+Header.bytes_per_line]=BitmapPtr[0]; // blue
          BitmapPtr+=3;
          }
      // encode scanline and write it to the file
      unsigned char *PCXPtr=PCXLineData;
      int x=0;
      while (x<PCXBytesPerLine)
        {
        unsigned char Byte=*PCXPtr++;
        unsigned char Count=1;
        x++;
        while (*PCXPtr==Byte && Count<63 && x<PCXBytesPerLine) { Count++;PCXPtr++;x++; }
        if (Count>1 || Byte>=0xc0)
          {
          Count|=0xc0;
          PCXFile.put(Count);
          }
        PCXFile.put(Byte);
        if (!PCXFile.good()) return(-4);
        }
      }
    delete PCXLineData;
    if (Bitmap.ClassID()==Bitmap::ClassPP8)
      {
      // save palette
      PCXFile.put((char)0x0c); // mark
      const int Colors=256;
      for (int j=0;j<Colors;j++)
        {
        ColorRef Color=Bitmap.SharedPalette().GetColor(j);
        PCXFile.put((char)RGB8Red(Color));
        PCXFile.put((char)RGB8Green(Color));
        PCXFile.put((char)RGB8Blue(Color));
        if (!PCXFile.good()) return(-4);
        }
      }
    PCXFile.close();
    }
  else
    Error=-1; // create/open error
  return(Error);
  }
