// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#ifndef PALETTE_H
#define PALETTE_H

#include "misc/error.h"
#include "compiler/types.h"

typedef unsigned long ColorRef; // may be paletteindex, RGB24 value or colortable index
  const unsigned long CRRGB8=1<<31;        // ColorRefType
  const unsigned long CRPaletteIndex=0;    //    "
inline unsigned long ColorRefType(ColorRef c) { return(c&0xff000000); };
const int RGBBits=8;
inline int RGB8Red  (ColorRef RGB) { return((RGB>>16)&0xff); };
inline int RGB8Green(ColorRef RGB) { return((RGB>>8 )&0xff); };
inline int RGB8Blue (ColorRef RGB) { return((RGB    )&0xff); };
inline ColorRef RGB8(int r,int g,int b) { return((((UBYTE)r)<<16)|(((UBYTE)g)<<8)|((UBYTE)b)|CRRGB8); };



// Palettes are used by bitmaps, one palette can be used by several
// bitmaps, that saves memory and increases graphics speed (cause no
// color mapping is necessary when working with two color compatible
// bitmaps)
class Palette
  {
  public:
  enum Class {                              
    ClassInvalid=0,
    ClassColorArray=1,       
    ClassDirectColor1555=2,
    ClassDirectColor565=3,
    ClassDirectColor888=4, };
  enum { FuncClasses=8 };
  enum Flag {
    AutoDelete=1 };
  ///////////////////// protected section //////////////
  protected:
  // internal data
  int flags;
  Class classID;                         // different from Bitmap::classID, see below
  long colors;
  ColorRef *color;
  int users;
  int (*notifysetcolor)(Palette&,ColorRef,ColorRef); // is called whenever a color in the 'color' array is to be changed, if 0 is returned, then no color change will occur
  invalidate();
  ColorRef getnearestcolor(ColorRef RGB);
  // these functions are handling palette access
  friend void SetColorNoDC(Palette &Pal,ColorRef PalIndex,ColorRef RGB);
  friend void SetColorInvalid(Palette &Pal,ColorRef PalIndex,ColorRef RGB);
  friend ColorRef GetColorNoDC(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetColorDC1555(Palette &Pal,ColorRef PalIndex); // 32K (std direct color)
  friend ColorRef GetColorDC565(Palette &Pal,ColorRef PalIndex);  // 64K (")
  friend ColorRef GetColorDC888(Palette &Pal,ColorRef PalIndex);  // 16M (")
  friend ColorRef GetColorInvalid(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetPaletteIndexNoDC(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetPaletteIndexDC1555(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetPaletteIndexDC565(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetPaletteIndexDC888(Palette &Pal,ColorRef PalIndex);
  friend ColorRef GetPaletteIndexInvalid(Palette &Pal,ColorRef PalIndex);
  static void (*SetColorFunc[])(Palette &Pal,ColorRef PalIndex,ColorRef RGB);
  static ColorRef (*GetColorFunc[])(Palette &Palette,ColorRef PalIndex);
  static ColorRef (*GetPaletteIndexFunc[])(Palette &Palette,ColorRef RGB);
  public:
  /////////////////////// user functions /////////////////////
  Palette() { invalidate(); };
  Palette(Class ClassID,long ColorNum=0); // the ClassID's except ClassColorArray are probably of no use for the user.
  Palette(Palette &Pal);                       
  Palette &operator =(Palette &Src);
  ~Palette();
  // get infos
  int IsValid() const { return(classID!=ClassInvalid); };
  Class ClassID() { return classID;};
  long Colors() const { return(colors); };
  // color table managment and color transformation
  int SetColor(ColorRef PalIndex,ColorRef RGB) { if ((*notifysetcolor)(*this,PalIndex,RGB)) { SetColorFunc[classID](*this,PalIndex,RGB);return(1); } else return(0); }; // 1 - OK, color has been set; 0 - notifier didn't allow the color change
  ColorRef GetColor(ColorRef PalIndex) { return(GetColorFunc[classID](*this,PalIndex)); };
  ColorRef GetPaletteIndex(ColorRef RGB) { return(GetPaletteIndexFunc[classID](*this,RGB)); };
  // users
  void AddUser() { users++; };
  void RemoveUser() { users--; if (flags&AutoDelete && users<=0) delete this; };
  int Users() const { return(users); }
  // flags
  void SetFlags(int Flags) { flags|=Flags; };
  void UnsetFlags(int Flags) { flags&=~Flags; };
  int Flags() const { return(flags); };
  // SetColor notifier (used by graph drivers)
  int (*NotifySetColor())(Palette&,ColorRef,ColorRef) { return(notifysetcolor); };
  void NotifySetColor(int (*notifier)(Palette&,ColorRef,ColorRef)) { notifysetcolor=notifier; };
  };

#endif
