// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include "gfx/bitmap.h"
#include "gfx/blit.h"
#include "misc/error.h"

// Array of functions for all combinations of bitmap classes.
void (*CopyRectMaskedFunc[Bitmap::Classes*Bitmap::Classes])(Bitmap& Dst,int DstX,int DstY,Bitmap& Src,Rect &SrcRect);

void CopyRectMaskedInvalid(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect);
void CopyRectMaskedGeneral(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect);

static class AutoInitCopyRectMFunc
  {
  public:
  AutoInitCopyRectMFunc()
    {
    for (int i=0;i<Bitmap::Classes*Bitmap::Classes;i++) CopyRectMaskedFunc[i]=CopyRectMaskedGeneral;
    for (    i=0;i<Bitmap::Classes;i++) { CopyRectMaskedFunc[i]=CopyRectMaskedFunc[i*Bitmap::Classes]=CopyRectMaskedInvalid; };
    #if defined(ACCEL_CopyRectMasked_PP8PP8) || defined(ACCEL_PP8)
    CopyRectMaskedFunc[(Bitmap::ClassPP8<<Bitmap::Log2Classes)+Bitmap::ClassPP8]=CopyRectMaskedPP8_PP8;
    #endif
    #if defined(ACCEL_CopyRectMasked_PP8PL1) 
    CopyRectMaskedFunc[(Bitmap::ClassPP8<<Bitmap::Log2Classes)+Bitmap::ClassPP8]=CopyRectMaskedPP8_PL1;
    #endif
    #if defined(ACCEL_CopyRectMasked_DC565PP8) 
    CopyRectMaskedFunc[(Bitmap::ClassPP8<<Bitmap::Log2Classes)+Bitmap::ClassPP8]=CopyRectMaskedDC565_PP8;
    #endif
    };
  } InitCopyRectMFunc;


void CopyRectMaskedInvalid(Bitmap&,int,int,Bitmap&,Rect&)
  {
  ErrorExit("CopyRectMasked  invalid bitmap.");
  }
void CopyRectMaskedGeneral(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect)
  {
  #ifdef DEBUG
  int RectWidth=SrcRect.Right-SrcRect.Left+1;
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorExit("CopyRectMasked  rect exceeds bitmap.");
  #endif
  short sy=SrcRect.Top;
  while (sy<=SrcRect.Bottom)
    {
    short sx=SrcRect.Left;
    short dx=dx0;
    while (sx<=SrcRect.Right)
      {
      ColorRef PaleletteIndex=Src.GetPixel(sx,sy);
      if (PaletteIndex!=0) Dst.SetPixel(dx,dy,PaletteIndex);
      dx++;
      sx++;
      }
    dy++;
    sy++;
    }
  }



#if defined(ACCEL_CopyRectMasked_PP8PP8) || defined(ACCEL_PP8)
void CopyRectMaskedPackedPixel8(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect)
  {
  int RectWidth=SrcRect.Right-SrcRect.Left+1;
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  int SrcDiff,DstDiff;
  SrcDiff=Src.WidthBytes()-RectWidth;
  DstDiff=Dst.WidthBytes()-RectWidth;
  #ifdef DEBUG
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorExit("CopyRectMasked  rect exceeds bitmap.");
  #endif
  unsigned char *SrcPtr,*DstPtr;
  SrcPtr=Src.Bits()+SrcRect.Top*Src.WidthBytes()+SrcRect.Left;
  DstPtr=Dst.Bits()+dy*Dst.WidthBytes()+dx0;
  if (&Dst.SharedPalette()==&Src.SharedPalette())
    {
    int yToGo=RectHeight;
    while (yToGo>0)
      {
      int xToGo=RectWidth;
      while (xToGo>0)
        {
        if (*SrcPtr!=0) *DstPtr=*SrcPtr;
        SrcPtr++;
        DstPtr++;
        xToGo--;
        }
      SrcPtr+=SrcDiff;
      DstPtr+=DstDiff;
      yToGo--;
      }
    }
  }
#endif


#if defined(ACCEL_CopyRectMasked_PP8PL1) 
void CopyRectMaskedPP8_PL1(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect)
  {
  int RectWidth=SrcRect.Right-SrcRect.Left+1;
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  int DstDiff;
  DstDiff=Dst.WidthBytes()-RectWidth;
  #ifdef DEBUG
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorExit("CopyRectMasked  rect exceeds bitmap.");
  #endif
  ColorRef Color1=Dst.SharedPalette().GetPaletteIndex(Src.SharedPalette().GetColor(1));
  unsigned char *SrcPtr0,*DstPtr;
  SrcPtr0=Src.Bits()+SrcRect.Top*Src.WidthBytes()+(SrcRect.Left>>3);
  DstPtr=Dst.Bits()+dy*Dst.WidthBytes()+dx0;
  int SrcMask0=0x80>>(SrcRect.Left&7);
  int yToGo=RectHeight;
  while (yToGo>0)
    {
    unsigned char *SrcPtr=SrcPtr0;
    unsigned char SrcByte=*SrcPtr++;
    int SrcMask=SrcMask0;
    int xToGo=RectWidth;
    while (xToGo>0)
      {
      if (SrcByte&SrcMask) *DstPtr=Color1;
      if (SrcMask!=1)
        {
        SrcMask>>=1;
        }
      else
        {
        SrcByte=*SrcPtr++;
        SrcMask=0x80;
        }
      DstPtr++;
      xToGo--;
      }
    SrcPtr0+=Src.WidthBytes();
    DstPtr+=DstDiff;
    yToGo--;
    }
  }
#endif

#if defined(ACCEL_CopyRectMasked_DC565PP8)
void CopyRectMaskedDC565_PL1(Bitmap& Dst,int dx0,int dy,Bitmap& Src,Rect &SrcRect)
  {
  int RectWidth=SrcRect.Right-SrcRect.Left+1;    // width always in pixels
  int RectHeight=SrcRect.Bottom-SrcRect.Top+1;
  int DstDiff;
  DstDiff=Dst.WidthBytes()-RectWidth;
  #ifdef DEBUG
  if (dx0<0          || dy<0          || dx0+RectWidth>Dst.Width() || dy+RectHeight>Dst.Height()
   || SrcRect.Left<0 || SrcRect.Top<0 || SrcRect.Right>=Src.Width() || SrcRect.Bottom>=Src.Height()) ErrorExit("CopyRectMasked  rect exceeds bitmap.");
  #endif
  ColorRef Color1=Dst.SharedPalette().GetPaletteIndex(Src.SharedPalette().GetColor(1));
//!!!!!!!faster
  unsigned char *SrcPtr0,*DstPtr;
  SrcPtr0=Src.Bits()+SrcRect.Top*Src.WidthBytes()+(SrcRect.Left>>3);
  DstPtr=Dst.Bits()+dy*Dst.WidthBytes()+dx0;
  int SrcMask0=0x80>>(SrcRect.Left&7);
  int yToGo=RectHeight;
  while (yToGo>0)
    {
    unsigned char *SrcPtr=SrcPtr0;
    unsigned char SrcByte=*SrcPtr++;
    int SrcMask=SrcMask0;
    int xToGo=RectWidth;
    while (xToGo>0)
      {
      if (SrcByte&SrcMask) *((WORD*)DstPtr)=Color1;
      if (SrcMask!=1)
        {
        SrcMask>>=1;
        }
      else
        {
        SrcByte=*SrcPtr++;
        SrcMask=0x80;
        }
      DstPtr++;
      xToGo--;
      }
    SrcPtr0+=Src.WidthBytes();
    DstPtr+=DstDiff;
    yToGo--;
    }
  }
#endif
