#include <mem.h>
#include "misc/mem.h"
#include "misc/error.h"
#include "compiler/types.h"
#include "gfx/bitmap.h"

// ClassXXX[Bitmap::Class] for constructor
static int Class2Planes[] = { 0,1,1,1,1,1 };
static int Class2BitsPerPixel[] = { 0,1,8,15,16,24 };
static Palette::Class Class2PaletteClass[] = { Palette::ClassInvalid,Palette::ClassColorArray,Palette::ClassColorArray,Palette::ClassDirectColor1555,Palette::ClassDirectColor565,Palette::ClassDirectColor888 };
static int Class2CanUseStandardPalette[] = { 0,0,0,1,1,1 }; // DirectColor modes can use the same palette, cause palette color modification is impossible
static Palette PaletteInvalid,
  PaletteDirectColor1555(Palette::ClassDirectColor1555,1<<15),
  PaletteDirectColor565(Palette::ClassDirectColor565 ,1<<16),
  PaletteDirectColor888(Palette::ClassDirectColor888 ,1<<24);
static Palette *Class2StandardPalette[Bitmap::Classes] =
  {
  &PaletteInvalid,&PaletteInvalid,&PaletteInvalid,
  &PaletteDirectColor1555,&PaletteDirectColor565,&PaletteDirectColor888
  };

void Bitmap::Construct(int w,int h,Class ID) 
  {
  width=w;
  height=h;
  if (width<0 || height<0) ErrorHandler.Abort("Bitmap::constructor  size negative.");
  if (ID<1 || ID>5) ErrorHandler.Abort("Bitmap::construct  unknown class");
  Delete();
  classID=ID;
  planes=Class2Planes[classID];
  bitsperpixel=Class2BitsPerPixel[classID];
  bitsperpixelandplane=bitsperpixel/planes;
  widthbytes=(bitsperpixelandplane*width+7)>>3;
  // widthbytes must be DWORD aligned (widthbytes%4==0), then we can use faster blitting
  // functions
  widthbytes+=(4-widthbytes)&3;
  planebytes=height*widthbytes;
  imagebytes=planes*planebytes;
  bits=new unsigned char[imagebytes];
  if (Class2CanUseStandardPalette[classID])
    usedpalette=Class2StandardPalette[classID];
  else
    {
    usedpalette=new ::Palette(Class2PaletteClass[classID],1<<bitsperpixel);
    // this palette must be deleted when it is no longer of use for
    // the bitmap system, that means when no bitmap uses it any more.
    usedpalette->SetFlags(Palette::AutoDelete);
    }
  usedpalette->AddUser();
  }
// w,h,bpp,memmodel and usedpalette(=0) must have been set.
//   dc modes: rgb masksizes/fieldpos must have been set.
// Deletes old bitmap and invalidates it.
void Bitmap::invalidate()
  {
  // Invalidate bitmap:
  // Forbid, Set/GetPixel, Palette.GetSetColor/PaletteIndex.
  usedpalette=Class2StandardPalette[ClassInvalid];
  bits=0;
  classID=ClassInvalid;
  }
Bitmap::~Bitmap()
  {
  if (classID!=ClassInvalid)
    {
    usedpalette->RemoveUser();
    delete bits;
    invalidate();
    }
  };
Bitmap& Bitmap::operator=(const Bitmap& Bitmap)
  {
  Delete();
  classID=Bitmap.classID;
  width=Bitmap.width;
  height=Bitmap.height;
  widthbytes=Bitmap.widthbytes;
  planes=Bitmap.planes;
  bitsperpixel=Bitmap.bitsperpixel;
  bitsperpixelandplane=Bitmap.bitsperpixelandplane;
  planebytes=Bitmap.planebytes;
  imagebytes=Bitmap.imagebytes;
  usedpalette=Bitmap.usedpalette;
  bitsperpixel=Bitmap.bitsperpixel;
  bits=new unsigned char[imagebytes];
  memcpy(bits,Bitmap.bits,imagebytes);
  usedpalette->AddUser();
  return(*this);
  }
int Bitmap::UsePalette(::Palette &Pal)
  {
  #ifdef DEBUG
  if (classID==ClassInvalid || usedpalette==0) ErrorHandler.Abort("Bitmap::UsePalette  bitmap invalid");
  if (usedpalette->Colors()!=Pal.Colors()) ErrorHandler.Abort("Bitmap::UsePalette  different bitmap types.");
  #endif
  if (usedpalette!=&Pal)
    {
    usedpalette->RemoveUser();
    usedpalette=&Pal;
    usedpalette->AddUser();
    }
  return(0);                 // OK
  }







// Set/Getpixel in linear Bitmaps.
void SetPixelPL1(Bitmap &Bitmap,int x,int y,ColorRef Color)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  UBYTE Mask=(UBYTE)(0x80>>(x&7));
  UBYTE *Adr=&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x>>3)];
  if (Color) *Adr|=Mask; else *Adr&=~Mask;
  }
void SetPixelPP8(Bitmap &Bitmap,int x,int y,ColorRef Color)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  Bitmap.Bits()[y*Bitmap.WidthBytes()+x]=(BYTE)Color;
  }
void SetPixelDC1555(Bitmap &Bitmap,int x,int y,ColorRef Color)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  *(UWORD*)(&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)])=(BYTE)Color;
  }
void SetPixelDC565(Bitmap &Bitmap,int x,int y,ColorRef Color)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  *(UWORD*)(&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)])=(BYTE)Color;
  }
void SetPixelDC888(Bitmap &Bitmap,int x,int y,ColorRef Color)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  DWORD *Adr=(DWORD*)(&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)+x]);
  *Adr&=0xff000000;
  Color&=0x00ffffff;
  *Adr|=Color;
  }
ColorRef GetPixelPL1(Bitmap &Bitmap,int x,int y)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  BYTE DataByte=Bitmap.Bits()[y*Bitmap.WidthBytes()+(x>>3)];
  return((DataByte>>(7-(x&7)))&1);
  }
ColorRef GetPixelPP8(Bitmap &Bitmap,int x,int y)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  return(Bitmap.Bits()[y*Bitmap.WidthBytes()+x]);
  }
ColorRef GetPixelDC1555(Bitmap &Bitmap,int x,int y)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  return(*(UWORD*)(&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)]));
  }
ColorRef GetPixelDC565(Bitmap &Bitmap,int x,int y)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  return(*(UWORD*)(&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)]));
  }
ColorRef GetPixelDC888(Bitmap &Bitmap,int x,int y)
  {
  #ifdef DEBUG
  if (x<0 || y<0 || x>Bitmap.Width() || y>Bitmap.Height()) ErrorHandler.Abort("Bitmap::Set/GetPixel  invalid coordinates.");
  #endif
  unsigned char *Adr=&Bitmap.Bits()[y*Bitmap.WidthBytes()+(x<<1)+x];
  return(*((UDWORD*)(Adr))&0x00ffffff);
  }
void SetPixelInvalid(Bitmap&,int,int,ColorRef)
  {
  ErrorHandler.Abort("Bitmap::SetPixel invalid");
  }
ColorRef GetPixelInvalid(Bitmap&,int,int)
  {
  ErrorHandler.Abort("Bitmap::GetPixel invalid");
  return(0);
  }


// tables: map class -> func
static void (*Bitmap::SetPixelFunc[Bitmap::Classes])(Bitmap& Src,int x,int y,ColorRef color) =
  {
  SetPixelInvalid,SetPixelPL1,SetPixelPP8,SetPixelDC1555,
  SetPixelDC565,SetPixelDC888,
  };
static ColorRef (*Bitmap::GetPixelFunc[Bitmap::Classes])(Bitmap& Dst,int x,int y) =
  {
  GetPixelInvalid,GetPixelPL1,GetPixelPP8,GetPixelDC1555,
  GetPixelDC565,GetPixelDC888,
  };



