// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#ifndef VESADRV_H
#define VESADRV_H

// Low level graph library.
// VESADriver is a class for lowest level graph mode handling:
// It performs:
//   get infos about graph modes
//   init/restore graph modes
//   copy a part of a bitmap onto the screen and copy a part of the screen into a bitmap.
// (that means that bitmaps can be used as off screen buffers)

#include <i86.h>
#include <dos.h>
#include "compiler/types.h"
#include "gfx/bitmap.h"
#include "gfx/palette.h"
#include "gfx/rect.h"

class VideoDriver
  {
  public:
  enum VideoMode {
    // standard VGA/VESA modes (VGA modes except mode13 are really useless,
    // so I don't support them here any more).
    VGA320x200x256=0x013,     // these modes are supported by almost every card
    VESA640x400x256=0x100,    // "
    VESA640x480x256=0x101,    // "
    VESA800x600x16 =0x102,    // "
    VESA800x600x256=0x103,    // "
    VESA1024x768x16=0x104,    // "
    VESA1024x768x256=0x105,   // "
    VESA1280x1024x16=0x106,
    VESA1280x1024x256=0x107,
    VESA320x200x32K=0x10D,    // std 32K modes always 1:5:5:5 (unused:xxx)
    VESA320x200x64K=0x10E,    // 5:6:5
    VESA320x200x16M=0x10F,    // 8:8:8
    VESA640x480x32K=0x110,    // often supported mode
    VESA640x480x64K=0x111,    // "
    VESA640x480x16M=0x112,    // "
    VESA800x600x32K=0x113,    // "
    VESA800x600x64K=0x114,    // "
    VESA800x600x16M=0x115,
    VESA1024x768x32K=0x116,
    VESA1024x768x64K=0x117,
    VESA1024x768x16M=0x118,
    VESA1280x1024x32K=0x119,
    VESA1280x1024x64K=0x11a,
    VESA1280x1024x16M=0x11b,
    };
  enum Class {
    ClassInvalid=0,
    ClassPackedPixel8Linear=1,
    ClassPackedPixel8Windowing=2,
    ClassDirectColor1555Windowing=3,
    ClassDirectColor565Windowing=4,
    ClassDirectColor888Windowing=5 };
  struct ModeInfo
    {
    int Width,Height,BitsPerPixel;
    };
  private:
  static int installed_drivers;
  int vga_support;
  int vesa_support;
  int supportedmodes,supportedvesamodes;
  int *supportedmode;
  int initialmode;
  int currentmode;
  int widthbytes;
  UBYTE *winptr;
  Class classID;
  Palette *usedpalette;
  int (*orig_notifysetcolor)(Palette&,ColorRef,ColorRef);
  // cache info that might interest the application
  int width,height;  // in pixels
  Bitmap::Class bitmapclassID; // class ID of compatible bitmap (these can be copied onto the screen using BlitToScreen)
  // windowing=paging info
  // Videomem is parted into virtual windows. Each of these is
  // VESAInfo.WinSize kB big (normally 64 or 128 kB).
  int usedwinAB;      // 0=A or 1=B (see VESA info)
  int currentvirtualwin;
  int offset2virtualwin; // virtualwinNr=videomemoffset>>offset2virtualwin
  int virtualwin2win;    // (VESA)win=virtualwin<<virtualwin2win;
  int virtualwinbytes;   // norm. 64/128 kB 
  struct VirtualWinInfo
    {
    int LastY; // the last pixel that belongs to that virtual win is on this line
    int Pixels; // number of pixels on this line that belong to this win
    long StartOffset;  // in video mem = virtualwin*virtualwinsize
    } *virtualwininfo; // each possible window has one entry
  void virtualwinswitch(int VirtualWinNr);
  // private functions
  void setcolor(int PaletteIndex,int Bits,int R,int G,int B);
  int getmodeinfo(int Mode,void *ptrVESAModeInfoBlock);
  static void (*BlitToScreenFunc[])(VideoDriver &Video,int dstX,int dstY,Bitmap &Src,const Rect &SrcRect);
  static void (*GrabFromScreenFunc[])(VideoDriver &Video,int dstX,int dstY,Bitmap &Dst,const Rect &DstRect);
  friend int VideoNotifySetColor(Palette&,ColorRef,ColorRef);
  friend void BlitToScreenInvalid(VideoDriver&,int,int,Bitmap&,const Rect&);
  friend void BlitToScreenPP8L(VideoDriver&,int,int,Bitmap&,const Rect&);
  friend void BlitToScreenPP8W(VideoDriver&,int,int,Bitmap&,const Rect&);
  friend void BlitToScreenDC1555W(VideoDriver&,int,int,Bitmap&,const Rect&);
  friend void BlitToScreenDC565W(VideoDriver&,int,int,Bitmap&,const Rect&);
  friend void BlitToScreenDC888W(VideoDriver&,int,int,Bitmap&,const Rect&);
  ///////////////////// public section /////////////////////
  public:
  VideoDriver();
  ~VideoDriver();
  // graph mode management
  int SetMode(int Mode);
  int RestoreInitialMode();
  void UsePalette(Palette &Pal);
  // get infos
  int IsVGASupported() const { return(vga_support); };
  int IsVESASupported() const { return(vesa_support); };   // 0 - no, other - yes
  int SupportedModes() const { return(supportedmodes); };
  int SupportedMode(int Index) const;                     // return mode numbers of all supported modes (0<=Index<=SupportedModes)
  int IsSupportedMode(int ModeNr) const;                  
  ModeInfo GetModeInfo(int ModeNr);
  int CurrentMode() const { return(currentmode); };
  Palette &Palette() { return(*usedpalette); };
  // info about current mode
  int Width() const { return(width); };
  int Height() const { return(height); };
  Bitmap::Class ClassID() const { return(bitmapclassID); };
  // interface to bitmap class
  void BlitToScreen(int dstX,int dstY,Bitmap &Src,const Rect &SrcRect) { BlitToScreenFunc[classID](*this,dstX,dstY,Src,SrcRect); };
  };

extern VideoDriver Video;   // only a single global instance of this class is needed, this instance is automatically created.

#endif
