// Copyright Kjell Schubert unbu@rz.uni-karlsruhe.de

#include <i86.h>
#include <dos.h>
#include <conio.h>
#include <iostream.h>
#include "device/vesadrv.h"
#include "gfx/bitmap.h"
#include "misc/error.h"

VideoDriver Video;   // the only VESADriver instance

int VideoDriver::installed_drivers;
void (*VideoDriver::BlitToScreenFunc[])(VideoDriver &Video,int dstX,int dstY,Bitmap &Src,const Rect &SrcRect) =
  {
  BlitToScreenInvalid,

  #ifdef VIDEOSUPPORT_PP8L
  BlitToScreenPP8L,
  #else 
  BlitToScreenInvalid,
  #endif

  #ifdef VIDEOSUPPORT_PP8W
  BlitToScreenPP8W,
  #else 
  BlitToScreenInvalid,
  #endif

  #ifdef VIDEOSUPPORT_DC1555
  BlitToScreenDC1555_565,
  #else 
  BlitToScreenInvalid,
  #endif

  #ifdef VIDEOSUPPORT_DC565
  BlitToScreenDC1555_565,
  #else 
  BlitToScreenInvalid,
  #endif

  #ifdef VIDEOSUPPORT_DC888
  BlitToScreenDC888,
  #else 
  BlitToScreenInvalid,
  #endif
  };

struct VESAInfoBlock       // passed by VESA card
  {
  BYTE VESASignature[4];
  WORD VESAVersion;
  DWORD OEMStringPtr;
  BYTE Capabilities[4];     //   ; capabilities of the video environment
  DWORD VideoModePtr;       //  pointer to supported Super VGA modes
  WORD TotalMemory;         // Number of 64kb memory blocks on board
  };
struct VESAModeInfoBlock
  {
  WORD ModeAttributes;
  enum {
    ModeSupport=1,
    OutputFuncSupport=4,
    ColorMode=8,
    GraphMode=16, };
  BYTE WinAAttributes;
  BYTE WinBAttributes;
  enum {
    WinSupport=1,
    WinReadable=2,
    WinWritable=3,
    WinReadWrite=WinSupport|WinReadable|WinWritable, };
  WORD WinGranularity;    // window granularity in kB
  WORD WinSize;           // window size
  UWORD WinASegment;       // window A start segment, b800h for text mode
  UWORD WinBSegment;       // window B start segment
  DWORD WinFuncPtr;       // pointer to window function 4f05 (faster paging)
  WORD BytesPerScanLine;  // bytes per scan line in one plane
  WORD XResolution;       // in pixels
  WORD YResolution;
  BYTE XCharSize;
  BYTE YCharSize;
  BYTE NumberOfPlanes;    // 1 for packed pixel modes
  BYTE BitsPerPixel;      // e.g. 4 for 16color mode; bitsperpixelperplane=bitsperpixel/planes
  BYTE NumberOfBanks;     // 1 is unbanked, scanlines are grouped in banks
  BYTE MemoryModel;
  enum {
    TextMode=0,
    CGA=1,
    Hercules=2,
    Planar=3,             // e.g. 16color modes with 4 planes
    PackedPixel=4,        // e.g. 256 color mode 0x13 with one plane (for modes with palette, that means <=256 colors?)
    NonChain256=5,        // ???
    DirectColor=6,        // e.g. 32K,64K,16M SVGA modes
    YUV=7, };             // ???
  BYTE BankSize;           // bank size in kb
  BYTE NumberOfImagePages; // additional images! zero if only one image possible
  BYTE Reserved1;
  BYTE RedMaskSize;        // only for DirectColor and YUV
  BYTE RedFieldPosition;
  BYTE GreenMaskSize;
  BYTE GreenFieldPosition;
  BYTE BlueMaskSize;
  BYTE BlueFieldPosition;
  BYTE RsvdMaskSize;
  BYTE DirectColorModeInfo;
  enum {
    ColorRampProgrammable=1,// palette needed
    ReservedFieldUsable=2, };  // e.g. rsvd field in 1:5:5:5 mode (bit 15) may be usable by the application
  };


static const int SupportedVGAModes = 1;
static const int SupportedVGAMode[] = { 0x13 };
static const VESAModeInfoBlock SupportedVGAModeInfo[SupportedVGAModes] =
  {
  VESAModeInfoBlock::ModeSupport|VESAModeInfoBlock::ColorMode|VESAModeInfoBlock::GraphMode,
  VESAModeInfoBlock::WinReadWrite,0, 64,64,0xa000,0, 0,
  320,320,200,8,8,1,8,0,VESAModeInfoBlock::PackedPixel,0,0,1,
  0,0,0,0,0,0,0,VESAModeInfoBlock::ColorRampProgrammable
  };
const char ProtectedVGAModeInfo[] = { 0,0x2b,0x2a,0x25,0x2c,0x2c,0,0x33,0x23,0x28,0x35,0x22,0x25,0x32,0x34 };

struct DCBitmapInfo
  {
  int RedMaskSize;
  int RedFieldPosition;
  int GreenMaskSize;
  int GreenFieldPosition;
  int BlueMaskSize;
  int BlueFieldPosition;
  Bitmap::Class BitmapClassID;
  Palette::Class PaletteClassID;
  VideoDriver::Class VideoClassID;
  };
// arrays for converting vesa mode to bitmap/palette class ID
static const int DCSupportedBitmaps=3;
static DCBitmapInfo DCSupportedBitmap[DCSupportedBitmaps] =
  {
  5,10,5,5,5,0, Bitmap::ClassDirectColor1555,Palette::ClassDirectColor1555, VideoDriver::ClassDirectColor1555Windowing,
  5,11,6,5,5,0, Bitmap::ClassDirectColor565, Palette::ClassDirectColor565, VideoDriver::ClassDirectColor565Windowing,
  8,16,8,8,8,0, Bitmap::ClassDirectColor888, Palette::ClassDirectColor888, VideoDriver::ClassDirectColor888Windowing, 
  };

struct rminfo {             // DPMI struct
  UDWORD EDI;
  UDWORD ESI;
  UDWORD EBP;
  UDWORD reserved_by_system;
  UDWORD EBX;
  UDWORD EDX;
  UDWORD ECX;
  UDWORD EAX;
  UWORD flags;
  UWORD ES,DS,FS,GS,IP,CS,SP,SS;
  };

static void CloseGraphOnError() { Video.RestoreInitialMode(); }

// notify procedure called whenever the user changes an entry in the
// palette that is bound to the graph driver
int VideoNotifySetColor(Palette&,ColorRef PalIndex,ColorRef Color)
  {
  Video.setcolor(PalIndex,8,RGB8Red(Color),RGB8Green(Color),RGB8Blue(Color));
  return(1); // yes, modify color table entry in palette
  }

VideoDriver::VideoDriver()
  {
  #ifdef DEBUG
  cout << "vesa init start...";
  #endif
  if (installed_drivers!=0) ErrorHandler.Abort("VideoDriver::VideoDriver  two drivers collide.");
  installed_drivers++;
  // Init VESA
  usedpalette=0,virtualwininfo=0;
  supportedvesamodes=0;
  struct rminfo RMI;
  union REGS regs;
  struct SREGS sregs;
  const int DPMI=0x31;
  // DPMI call 100h allocates DOS memory
  memset(&sregs,0,sizeof(sregs));
  regs.w.ax=0x0100;
  regs.w.bx=32; // 512 Byte
  int386x(DPMI, &regs, &regs, &sregs);
  WORD segment=regs.w.ax;
  WORD selector=regs.w.dx;
  // Set up real-mode call structure
  memset(&RMI,0,sizeof(RMI));
  RMI.EAX=0x00004f00; // call service 4fh, func 0
  RMI.ES=segment;     // put DOS seg:off into ES:DI
  RMI.EDI=0;
  // Use DMPI call 300h to issue the DOS interrupt
  regs.w.ax = 0x0300;
  regs.h.bl = 0x10;
  regs.h.bh = 0;
  regs.w.cx = 0;
  sregs.es = FP_SEG(&RMI);
  regs.x.edi = FP_OFF(&RMI);
  int386x(DPMI, &regs, &regs, &sregs );
  vesa_support=0;
  supportedmode=0;
  if (RMI.EAX==0x4f)
    {
    vesa_support=1;
    // VESA found.
    // save interesting data from the block
    VESAInfoBlock VESAInfo;
    const char far *info=(const char far*)(MK_FP(selector,0));
    _fmemcpy(&VESAInfo,info,sizeof(VESAInfo));
    WORD *AvailMode=(WORD *)(((VESAInfo.VideoModePtr>>12)&0xffff0)+(VESAInfo.VideoModePtr&0xffff));
    supportedvesamodes=0;
    while (AvailMode[supportedvesamodes]!=-1) supportedvesamodes++;
    supportedmode=new int[supportedvesamodes+SupportedVGAModes]; 
    for (int i=0;i<supportedvesamodes;i++) supportedmode[i]=AvailMode[i]; // copy without -1
    }
  else
    supportedmode=new int[SupportedVGAModes];
  supportedmodes=supportedvesamodes;
  #ifdef DEBUG
  cout << (vesa_support?"  VESA supported...":"  no VESA driver detected...");
  #endif
  // Free DOS mem
  regs.w.ax=0x0101;
  regs.w.dx=selector;
  int386x(DPMI,&regs,&regs,&sregs);
  // Check if VGA installed.
  regs.w.ax=0x1a00;
  int386x(0x10,&regs,&regs,&sregs);
  if (regs.h.al==0x1a) // vga supported ?
    vga_support=regs.h.bl; 
  else
    vga_support=0;
  if (vga_support)
    {
    // append the predefined and supported vga modes to the mode info array
    for (int Index=0;Index<SupportedVGAModes;Index++) supportedmode[Index+supportedmodes]=SupportedVGAMode[Index];
    supportedmodes+=SupportedVGAModes;
    }
  #ifdef DEBUG
  cout << (vga_support?"  VGA supported...":"  no VGA card found...");
  #endif
  // Recall initial gfx mode.
  regs.h.ah=0x4f;
  regs.h.al=3;
  int386x(0x10,&regs,&regs,&sregs);
  initialmode=regs.w.bx;
  currentmode=-1;
  // Set mode.
  ErrorHandler.AddCleanUpProc(CloseGraphOnError);
  classID=VideoDriver::ClassInvalid;
  if (SetMode(initialmode)!=0) currentmode=initialmode; // reset mode to avoid that destructor blanks the screen even if no other gfx mode has been set
  #ifdef DEBUG
  cout << "vesa init done\n";
  #endif
  }
VideoDriver::~VideoDriver()
  {
  #ifdef DEBUG
  cout << "destructing graph driver .. ";
  #endif
  ErrorHandler.RemoveCleanUpProc(CloseGraphOnError);
  if (initialmode!=currentmode) RestoreInitialMode();
  delete supportedmode;
  delete virtualwininfo;
  supportedmode=0;
  #ifdef DEBUG
  cout << "done\n";
  #endif
  }
int VideoDriver::RestoreInitialMode()
  {
  if (currentmode!=initialmode)
    {
    SetMode(initialmode);
    if (currentmode!=initialmode)
      {
      // in case the initial mode is unsupported.
      union REGS regs;
      regs.w.ax=0x4f02;
      regs.w.bx=(WORD)initialmode;
      int386(0x10,&regs,&regs);
      currentmode=initialmode;
      }
    }
  return(0);
  }
int VideoDriver::SupportedMode(int Index) const
  {
  #ifdef DEBUG
  if (Index<0 || Index>=supportedmodes) return(-1);
  #endif
  return(supportedmode[Index]);
  }
int VideoDriver::IsSupportedMode(int Mode) const
  {
  for (int i=0;i<supportedmodes;i++) if (supportedmode[i]==Mode) return(1);
  return(0);
  }
int VideoDriver::getmodeinfo(int Mode,void *ptrVESAModeInfoBlock)
  {
  int Index=0;
  while (Index<supportedmodes && supportedmode[Index]!=Mode) Index++;
  if (Index<supportedmodes)
    {
    // mode is supported by the graphic card
    if (Index<supportedvesamodes)
      {
      // it is a vesa mode
      struct rminfo RMI;
      union REGS regs;
      struct SREGS sregs;
      const int DPMI=0x31;
      // DPMI call 100h allocates DOS memory
      memset(&sregs,0,sizeof(sregs));
      regs.w.ax=0x0100;
      regs.w.bx=32; // 512 Byte
      int386x(DPMI, &regs, &regs, &sregs);
      WORD segment=regs.w.ax;
      WORD selector=regs.w.dx;
      // Set up real-mode call structure
      memset(&RMI,0,sizeof(RMI));
      RMI.EAX=0x00004f01; // call service 4fh, func 0
      RMI.ECX=Mode;
      RMI.ES=segment;     // put DOS seg:off into ES:DI
      RMI.EDI=0;
      // Use DMPI call 300h to issue the DOS interrupt
      regs.w.ax = 0x0300;
      regs.h.bl = 0x10;
      regs.h.bh = 0;
      regs.w.cx = 0;
      sregs.es = FP_SEG(&RMI);
      regs.x.edi = FP_OFF(&RMI);
      int386x(DPMI, &regs, &regs, &sregs );
      // Copy info.
      const char far *info=(const char far*)(MK_FP(selector,0));
      _fmemcpy(ptrVESAModeInfoBlock,info,sizeof(struct VESAModeInfoBlock));
      // Free DOS mem
      regs.w.ax=0x0101;
      regs.w.dx=selector;
      int386x(DPMI,&regs,&regs,&sregs);
      if (RMI.EAX==0x4f)  // VESA Mode supported.
        return(0);
      return(RMI.EAX); // VESA error
      }
    else
      {
      // it is a supported VGA mode
      memcpy(ptrVESAModeInfoBlock,&SupportedVGAModeInfo[Index-supportedvesamodes],sizeof(struct VESAModeInfoBlock));
      return(0); // OK
      }
    }
  return(-1); // unsupported mode
  }
VideoDriver::ModeInfo VideoDriver::GetModeInfo(int ModeID) 
  {
  ModeInfo Info;
  VESAModeInfoBlock FullInfo;
  if (getmodeinfo(ModeID,&FullInfo)==0) // otherwise mode is not supported by the card
    {
    Info.Width=FullInfo.XResolution;
    Info.Height=FullInfo.YResolution;
    Info.BitsPerPixel=FullInfo.BitsPerPixel;
    }
  else
    {
    Info.Width=Info.Height=Info.BitsPerPixel=0;
    }
  return Info;
  }
static int trunclog2(unsigned int num)
  {
  int log=INTBITS-1;
  unsigned int Mask=1<<(INTBITS-1);
  while ((num&Mask)==0) { Mask>>=1;log--; };
  return(log);
  }
int VideoDriver::SetMode(int Mode)
  {
  if (Mode!=currentmode)
    {
    VESAModeInfoBlock ModeInfo;
    if (getmodeinfo(Mode,&ModeInfo)==0) // otherwise mode is not supported by the card
      {
      // I need a window to write to (only for windowing=paging modes)
      if (!(ModeInfo.ModeAttributes&VESAModeInfoBlock::GraphMode)) return(-2);
      if (ModeInfo.WinAAttributes&VESAModeInfoBlock::WinReadWrite!=VESAModeInfoBlock::WinReadWrite
       && ModeInfo.WinBAttributes&VESAModeInfoBlock::WinReadWrite!=VESAModeInfoBlock::WinReadWrite) return(-4);
      // check if the mode is supported by the 'Bitmap' class interface
      // if not, then keep the current mode and return with an error code.
      if (ModeInfo.MemoryModel!=VESAModeInfoBlock::PackedPixel && ModeInfo.MemoryModel!=VESAModeInfoBlock::DirectColor) return(-3); // Planar not supported
      if (ModeInfo.MemoryModel==VESAModeInfoBlock::PackedPixel && ModeInfo.BitsPerPixel!=8) return(-5); // only PackedPixel256 sup
      DCBitmapInfo *DCMode=0;
      if (ModeInfo.MemoryModel==VESAModeInfoBlock::DirectColor)
        {
        DCBitmapInfo * Sup=&DCSupportedBitmap[0];
        int Index=0;
        while (Index<DCSupportedBitmaps)
          {
          if (Sup->RedMaskSize==ModeInfo.RedMaskSize
           && Sup->GreenMaskSize==ModeInfo.GreenMaskSize
           && Sup->BlueMaskSize==ModeInfo.BlueMaskSize
           && Sup->RedFieldPosition==ModeInfo.RedFieldPosition
           && Sup->GreenFieldPosition==ModeInfo.GreenFieldPosition
           && Sup->BlueFieldPosition==ModeInfo.BlueFieldPosition)
            {
            DCMode=Sup;
            break;
            }
          Sup++;
          Index++;
          }
        if (DCMode==0) return(-6); // this DC mode is not supported
        }
      // OK, mode supported.
      /////////////////////////////////// Init /////////////////////////////
      // Set mode.
      for (int ModeIndex=0;ModeIndex<supportedmodes;ModeIndex++) if (supportedmode[ModeIndex]==Mode) break;
      union REGS regs;
      if (ModeIndex<supportedvesamodes)
        {
        regs.w.ax=0x4f02;       // let the vesa driver set the mode
        regs.w.bx=(WORD)Mode;
        }
      else
        {
        regs.h.ah=0x00;
        regs.h.al=(BYTE)Mode;         // let std vga set the mode
        }
      int386(0x10,&regs,&regs);  // set mode
      currentmode=Mode;
      // set the proper class ID (needed for choosing the right blitting
      // procedures)
      if (DCMode)
        classID=DCMode->VideoClassID;
      else
        if (ModeInfo.BytesPerScanLine*ModeInfo.YResolution<(ModeInfo.WinSize<<10)) // is windowing=paging necessary?
          classID=ClassPackedPixel8Linear; // no windowing needed (linear mode)
        else
          classID=ClassPackedPixel8Windowing; // windowing needed
      width=ModeInfo.XResolution;
      height=ModeInfo.YResolution;
      if (DCMode)
        bitmapclassID=DCMode->BitmapClassID;
      else
        if (ModeInfo.BitsPerPixel==8)
          bitmapclassID=Bitmap::ClassPackedPixel8;
        else
          bitmapclassID=Bitmap::ClassInvalid;
      bitmapclassID=Bitmap::ClassPackedPixel8;
      // precompute some values for faster window switching (=paging)
      if (ModeInfo.WinAAttributes&VESAModeInfoBlock::WinReadWrite==VESAModeInfoBlock::WinReadWrite)
        usedwinAB=0;
      else
        usedwinAB=1;
      winptr=(UBYTE*)((usedwinAB==0?ModeInfo.WinASegment:ModeInfo.WinBSegment)<<4);
      widthbytes=ModeInfo.BytesPerScanLine;
      // switch to window 0 (beginning of video mem)
      currentvirtualwin=0;
      {union REGS regs;
      regs.w.ax=0x4f05;
      regs.w.bx=(WORD)usedwinAB;
      regs.w.dx=0;
      int386(0x10,&regs,&regs);}
      // switching precomputation.
      virtualwinbytes=ModeInfo.WinSize<<10; // kB -> bytes
      int windowspervirtualwindow=ModeInfo.WinGranularity!=0?ModeInfo.WinSize/ModeInfo.WinGranularity:0;
      offset2virtualwin=trunclog2(virtualwinbytes);
      virtualwin2win=trunclog2(windowspervirtualwindow);
      delete virtualwininfo;
      int virtualwindows=widthbytes*ModeInfo.YResolution/virtualwinbytes+1;
      virtualwininfo=new VirtualWinInfo[virtualwindows+1];
      VirtualWinInfo *WinInfo=virtualwininfo;
      for (int Win=0;Win<=virtualwindows;Win++)
        {
        WinInfo->StartOffset=Win<<offset2virtualwin; // =Win*virtualwinbytes;
        int LastOffset=WinInfo->StartOffset+virtualwinbytes-1;
        WinInfo->LastY=LastOffset/widthbytes;
        WinInfo->Pixels=LastOffset%widthbytes+1;
        WinInfo++;
        }
      // Create a palette for the screen
      Palette::Class PaletteClass;
      if (ModeInfo.MemoryModel==VESAModeInfoBlock::DirectColor)
        PaletteClass=DCMode->PaletteClassID;
      else
        PaletteClass=Palette::ClassColorArray;
      ::Palette *NewPalette=new ::Palette(PaletteClass,1<<ModeInfo.BitsPerPixel);
      NewPalette->SetFlags(Palette::AutoDelete);
      UsePalette(*NewPalette);
      /**** future: check for PM interface (VBE2.0) and linear addressing
      regs.w.ax=0x4f0a;
      regs.h.bl=0;
      int386(0x10,&regs,&regs);
      test=regs.w.ax;
      ****/
      return(0); // OK, new mode has been set
      }
    return(-1);
    }
  return(0); // OK mode was already set
  }
void VideoDriver::UsePalette(::Palette &Pal)
  {
  if (usedpalette)
    {
    #ifdef DEBUG
    if (Pal.Colors()!=usedpalette->Colors() || Pal.ClassID()!=usedpalette->ClassID()) ErrorHandler.Abort("VideoDriver::UsePalette  incompatible palette.");
    #endif
    // release old palette
    usedpalette->NotifySetColor(orig_notifysetcolor); // set the original notifier
    usedpalette->RemoveUser(); // the graphic driver no longer needs the palette
    }
  // prepare new palette: the graphic driver must get to know when a
  // color in this palette changes.
  usedpalette=&Pal;
  orig_notifysetcolor=Pal.NotifySetColor(); // remember the original notifier of the new palette
  Pal.NotifySetColor(VideoNotifySetColor);
  Pal.AddUser();
  // copy new colors to DAC
  for (int i=0;i<Pal.Colors();i++)
    {
    ColorRef Color=Pal.GetColor(i);    
    setcolor(i,8,RGB8Red(Color),RGB8Green(Color),RGB8Blue(Color));
    }
  }
void VideoDriver::setcolor(int palindex,int bits,int r,int g,int b)
  {
  if (bits>6) // 6bit DAC presumed
    {
    const int shift=bits-6;
    r>>=shift;
    g>>=shift;
    b>>=shift;
    }
  else
    if (bits<6)
      {
      const int shift=6-bits;
      r<<=shift;
      g<<=shift;
      b<<=shift;
      }
  #ifndef NO_DAC_ACCESS
  const int DAC_WRITE=0x3c8;  // FarbRegNr 0-256 zum Schreiben von R,G,B in DAC_DATA Reg. */
  const int DAC_DATA =0x3c9;  // R,G,B-Wert schreiben
  outp(DAC_WRITE,palindex);
  outp(DAC_DATA,r);
  outp(DAC_DATA,g);
  outp(DAC_DATA,b);
  #else
  union REGS regs;
  regs.h.cl=b;
  regs.h.dh=r;
  regs.h.ch=g;
  regs.w.bx=palindex;
  regs.w.ax=0x1010;
  int386(0x10,&regs,&regs); 
  #endif
  }
/*
void wretrace();
#pragma aux wretrace = \
  "  push edx"\
  "  mov edx,03daH"\
  "Loop1:"\
  "  sub eax,eax"\ 
  "  in al,dx"\
  "  test al,8"\
  "  jne Loop1"\
  "  mov edx,03daH"\
  "Loop2:"\
  "  sub eax,eax"\ 
  "  in al,dx"\
  "  test al,8"\
  "  jne Loop2"\
  "  pop edx"\
  modify [eax]
*/

