#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>
#include <malloc.h>
#include <direct.h>
#include <sys\stat.h>

#include "fmod.h"

#define version 06

UBYTE fgcolor=7, bgcolor=0;         // fore/background colours
UDWORD techindex=1, techcount=1;    // technicality index counters
UBYTE select=0, solo=255;           // select- position of channel cursor
WORD screenpos=0, screensize;       // position of screen and size of screen
									// buffer in bytes
#define setfgcolor(x) fgcolor=x
#define setbgcolor(x) bgcolor=x

char *s;                            // miscellaneous string buffer
char *screen;                       // screen buffer
char *pattern;                      // buffer for 1 pattern

// ======================
// TURBO C SPECIFIC STUFF
// ======================
#ifdef __TURBOC__

#define myint int86

void interrupt ( *oldkeyhandler)(__CPPARGS);
void interrupt ( *olderrorhandler)(__CPPARGS);

char far *vga = (char far *)MK_FP(0xb800, 0);

void waitvrt(void) {
	asm  mov dx,0x3da         // move status port address # to dx
VRT:
	asm {
		in   al,dx          // get value from port and put in al
		test al,8           // test al with bit 3 (8h)
		jnz  VRT            // wait until Verticle Retrace starts
	}
NVRT:
	asm {
		in   al,dx          // get value from port and put in al
		test al,8           // test al with bit 3 (8h)
		jz   NVRT           // wait until Verticle Retrace Ends
	}
}

UWORD kbhit() {
	asm {
		mov  ah, 1
		int  0x16
		jz   nokey:
	}
	return 1;
nokey:
	return 0;
}

UDWORD memfree() {
	return coreleft();
}


// =====================
// WATCOM SPECIFIC STUFF
// =====================
#else

char *vga = 0xb8000;
void (interrupt far *oldkeyhandler)(void);
void (interrupt far *olderrorhandler)(void);

#pragma aux waitvrt="mov dx,3dah",\
					"wait1: in al,dx",\
					"test al,8",\
					"jz wait1",\
					modify[al dx];

#define myint int386
struct meminfo {
	unsigned LargestBlockAvail;
	unsigned MaxUnlockedPage;
	unsigned LargestLockablePage;
	unsigned LinAddrSpace;
	unsigned NumFreePagesAvail;
	unsigned NumPhysicalPagesFree;
	unsigned TotalPhysicalPages;
	unsigned FreeLinAddrSpace;
	unsigned SizeOfPageFile;
	unsigned Reserved[3];
} MemInfo;

// this is a PMODE type memfree
UDWORD memfree() {
	union REGS regs;
	struct SREGS sregs;

	regs.x.eax = 0x00000500;
	memset( &sregs, 0, sizeof(sregs) );
	sregs.es = FP_SEG( &MemInfo );
	regs.x.edi = FP_OFF( &MemInfo );

	int386x( 0x31, &regs, &regs, &sregs );
	return MemInfo.LargestBlockAvail;
}

#endif


/****************************************************************************
 *    Name : setkeyrate                                                     *
 * Purpose : to increase the speed of response from the keyboard by reducing*
 *           the delay between keystrokes, and increasing the key repeat    *
 *           rate.                                                          *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : union REGS regs - register pack holding values to be passed to *
 *                             BIOS int 16h.                                *
 ****************************************************************************/
void setkeyrate() {
	union REGS regs;

	regs.h.ah = 3;
	regs.h.al = 5;
	regs.h.bl = 0;
	regs.h.bh = 0;
	myint(0x16, &regs, &regs);
}


/****************************************************************************
 *    Name : getkey                                                         *
 * Purpose : A more useful getkey, as it returns the scancode of the key    *
 *           pressed, and not the ascii value.  handy for arrow keys etc.   *
 *  Passed : -                                                              *
 * Returns : regs.h.ah - the key pressed scancode                           *
 *  Locals : union REGS regs - register pack holding values to be passed to *
 *                             BIOS int 16h.                                *
 ****************************************************************************/
UBYTE getkey() {
	union REGS regs;
	regs.h.ah = 0;
	regs.h.al = 0;
	myint(0x16, &regs, &regs);
	return regs.h.ah;
}


/****************************************************************************
 *    Name : setrgbpalette                                                  *
 * Purpose : To set an RGB hardware palette entry for a specified colour.   *
 *  Passed : UBYTE col - the specified colour to be changed                 *
 *           UBYTE r   - the red component of the RGB triplet to be set     *
 *           UBYTE g   - the green component of the RGB triplet to be set   *
 *           UBYTE b   - the blue component of the RGB triplet to be set    *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void setrgbpalette(UBYTE col, UBYTE r, UBYTE g, UBYTE b) {
	outp(0x3c8, col);
	outp(0x3c9, r);
	outp(0x3c9, g);
	outp(0x3c9, b);
}

/****************************************************************************
 *    Name : set8050mode                                                    *
 * Purpose : To set 80x50 text mode, by tweaking normal text mode 3         *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : union REGS regs - register pack holding values to be passed to *
 *                             BIOS int 10h.                                *
 ****************************************************************************/
void set8050mode() {
	union REGS regs;

	// set mode
	regs.h.ah = 0;
	regs.h.al = 3;
	myint(0x10, &regs, &regs);

	// tweak a bit to get 50 lines.
	regs.h.ah = 0x12;
	regs.h.al = 0x02;
	regs.h.bl = 0x30;
	myint(0x10, &regs, &regs);

	regs.h.ah = 0x11;
	regs.h.al = 0x12;
	regs.h.bl = 0;
	myint(0x10, &regs, &regs);

	// now turn the cursor off!
	regs.h.ah = 1;
	regs.h.al = 3;
	regs.h.ch = 16;
	regs.h.cl = 0;
	myint(0x10, &regs, &regs);
}


/****************************************************************************
 *    Name : set8025mode                                                    *
 * Purpose : set normal 80x25 text mode, clearing it as well.               *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : union REGS regs - register pack holding values to be passed to *
 *                             BIOS int 10h.                                *
 ****************************************************************************/
void set8025mode() {
	union REGS regs;

	regs.h.ah = 0;
	regs.h.al = 3;
	myint(0x10, &regs, &regs);

	system("cls");
}


/****************************************************************************
 *    Name : updatescreen                                                   *
 * Purpose : To copy the offscreen buffer to the screen.                    *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UWORD offset - the offset in the offscreen buffer to copy to   *
 *                          the screen                                      *
 ****************************************************************************/
void updatescreen() {
	UWORD offst = screenpos*160;

	memcpy(vga, screen+offst, 8000);
}


/****************************************************************************
 *    Name : clearscreen                                                    *
 * Purpose : to blank out the offscreen buffer and copy it to the screen,   *
 *           therefore clearing the screen.                                 *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void clearscreen() {
	screenpos =0;
	memset(screen, 0, screensize);
	updatescreen();
}


/****************************************************************************
 *    Name : eputch                                                         *
 * Purpose : to put a character to the offscreen buffer using fgcolor, and  *
 *           bgcolor.  For it to actually appear an updatescreen() is needed*
 *  Passed : UBYTE x - the x position on the screen buffer to put the char  *
 *           UBYTE y - the y position on the screen buffer to put the char  *
 *           UBYTE val - the actuall value to write                         *
 * Returns : -                                                              *
 *  Locals : UWORD offst - a calculated value using the x and y values, to  *
 *                         tell us where to put the character in the        *
 *                         ofscreen buffer.  Remember it is 2 bytes per char*
 ****************************************************************************/
void eputch(UBYTE x, UBYTE y, UBYTE val) {
	UWORD offst = ((y - 1) * 160) + ((x - 1) << 1);

	screen[offst] = val;
	screen[offst+1] = (bgcolor<<4) + fgcolor;
}


/****************************************************************************
 *    Name : eputstr                                                        *
 * Purpose : to write a screen to the offscreen buffer                      *
 *  Passed : UBYTE x - the x position to write the string at                *
 *           UBYTE y - the y position to write the string at                *
 *           char *msg - the string to write                                *
 * Returns : -                                                              *
 *  Locals : UBYTE count - variable to count through each character of the  *
 *                         string                                           *
 ****************************************************************************/
void eputstr(UBYTE x, UBYTE y, char *msg) {
	UBYTE count;
	for (count=0; *(msg+count)!='\0'; count++) eputch(x+count,y,*(msg+count));
}


/****************************************************************************
 *    Name : eputstrw                                                       *
 * Purpose : to write len number of bytes of a string to the virtual screen *
 *  Passed : UBYTE x - the x position to write the string at                *
 *           UBYTE y - the y position to write the string at                *
 *           char *msg - the string to write                                *
 *           UBYTE len - the number of bytes to copy                        *
 * Returns : -                                                              *
 *  Locals : UBYTE count - variable to count through each character of the  *
 *                         string                                           *
 ****************************************************************************/
void eputstrw(UBYTE x, UBYTE y, char *msg, UBYTE len) {
	UBYTE count;

	for (count=0; count<len; count++) eputch(x+count,y,*(msg+count));
}


/****************************************************************************
 *    Name : eputchp                                                        *
 * Purpose : to write a character to the virtual pattern buffer instead of  *
 *           the virtual screen                                             *
 *  Passed : UWORD x - the x position to write the string at                *
 *           UWORD y - the y position to write the string at                *
 *           UBYTE val - the character to write                             *
 * Returns : -                                                              *
 *  Locals : UWORD offst - the literal offset in the buffer to write the    *
 *                          character too.                                  *
 ****************************************************************************/
void eputchp(UWORD x, UWORD y, UBYTE val) {
	UWORD offst = (y * (6U+(FM.channels*22))) + ((x - 1) << 1);

	pattern[offst] = val;
	pattern[offst+1] = (bgcolor<<4) + fgcolor;
}


/****************************************************************************
 *    Name : eputstrp                                                       *
 * Purpose : to write a string to the virtual pattern buffer instead of the *
 *           virtual screen                                                 *
 *  Passed : UWORD x - the x position to write the string at                *
 *           UWORD y - the y position to write the string at                *
 *           char *msg - the string to write                                *
 * Returns : -                                                              *
 *  Locals : UBYTE count - a counter variable                               *
 ****************************************************************************/
void eputstrp(UWORD x, UWORD y, char *msg) {
	UBYTE count;
	for (count=0; *(msg+count)!='\0'; count++) eputchp(x+count,y,*(msg+count));
}


/****************************************************************************
 *    Name : textsquare                                                     *
 * Purpose : To draw a white square + border on virtual screen              *
 *  Passed : UBYTE left - left x position                                   *
 *           UBYTE top - upper y position                                   *
 *           UBYTE right - right x position                                 *
 *           UBYTE bottom - bottom y position                               *
 *           UBYTE type - 1 = single line, 2 = double line                  *
 *           char *heading - string containing heading - will be centred    *
 * Returns :                                                                *
 *  Locals : UWORD count - counter variable                                 *
 ****************************************************************************/
void textsquare(UBYTE left, UBYTE top, UBYTE right, UBYTE bottom, UBYTE type)
{
	WORD count=0;

	if (type==1) eputch(left, top, 218);
	else eputch(left, top, 201);
	for (count=left+1; count < right; count++) {
		if (type==1) eputch(count, top, 196);
		else eputch(count, top, 205);
	}
	if (type==1) eputch(right, top, 191);
	else eputch(right, top, 187);
	for (count=top+1; count < bottom; count++) {
		if (type==1) eputch(left, count, 179);
		else eputch(left, count, 186);
		if (type==1) eputch(right, count, 179);
		else eputch(right, count, 186);
	}
	if (type==1) eputch(left, bottom, 192);
	else eputch(left, bottom, 200);
	for (count=left+1; count < right; count++) {
		if (type==1) eputch(count, bottom, 196);
		else eputch(count, bottom, 205);
	}
	if (type==1) eputch(right, bottom, 217);
	else eputch(right, bottom, 188);
}


/****************************************************************************
 *    Name : textbox                                                        *
 * Purpose : To draw a MessageBox square with a heading and border          *
 *  Passed : UBYTE x - left x position                                      *
 *           UBYTE y - upper y position                                     *
 *           UBYTE x2 - right x position                                    *
 *           UBYTE y2 - bottom y position                                   *
 *           UBYTE type - 1 = single line, 2 = double line                  *
 *           char *heading - string containing heading - will be centred    *
 * Returns : -                                                              *
 *  Locals : WORD count,count2 - count variables                            *
 ****************************************************************************/
void textbox(UBYTE x, UBYTE y, UBYTE x2, UBYTE y2, UBYTE type, char *heading) {
	WORD count, count2;

	setbgcolor(7);
	setfgcolor(15);
	textsquare(x,y,x2,y2,type);
	for (count2=y+1; count2<y2; count2++) {
		for (count=x+1; count<x2; count++) eputch(count, count2, 32);
	}
	eputstr(((x+x2)/2)-(strlen(heading)/2),y,heading);
	setfgcolor(1);
	for (count=x; count<=x2; count++) eputch(count+1, y2+1, 219);
	for (count=y+1; count<y2+1; count++) eputch(x2+1,count,219);
}


/****************************************************************************
 *    Name : interrupt keyhandler                                           *
 * Purpose : To replace the usuall keyboard Interrupt Service Routine and   *
 *           catch when F12 is pressed.  If it is then the speed is reset.  *
 *           Seeing as it is on the keyboard ISR, then it will catch this   *
 *           key at any time.. Even in DOS Shell.                           *
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals : UBYTE key - the key pressed                                    *
 ****************************************************************************/
void interrupt keyhandler(__CPPARGS) {
	UBYTE key;
	key = inp(0x60);
	if (key == 88) SetTimerSpeed();
	oldkeyhandler();
}


/****************************************************************************
 *    Name : Error                                                          *
 * Purpose : General error handler for different error codes                *
 *  Passed : UBYTE code - the type of error                                 *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void Error(UBYTE code) {
	printf("ERROR #%d : ", code);

	switch (code) {
		case ERR_NO_ENV :
			 printf("ULTRASND Environment variable not found..");
			 break;
		case ERR_NO_FILE :
			 printf("Module not found.");
			 break;
		case ERR_UNKNOWN_FORMAT :
			 printf("Unknown format.");
			 break;
		case ERR_GUS_DRAM_FULL :
			 printf("Out of GUS DRAM!..");
			 break;
		case ERR_OUT_OF_MEM :
			 printf("Not enough memory!..");
			 break;
		case ERR_FILE_CORRUPT :
			 printf("Error loading file - file corrupted");
			 break;
		default : printf("Unexpected error.");
	};
	printf("\r\n");
	exit(1);
}


/****************************************************************************
 *    Name : help                                                           *
 * Purpose : To display help boxes                                          *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : WORD count - counter variable                                  *
 ****************************************************************************/
void help() {
	WORD count;

	// roll screen up to top!
	while (screenpos) {
		waitvrt();
		updatescreen();
		screenpos--;
	}

	setrgbpalette(1,0,0,14);
	setrgbpalette(7,42,42,42);
	setrgbpalette(57, 16, 16, 63);                  // color 9
	setrgbpalette(63,63,63,63);                     // color 15

	textbox(5,4,75,39,2,"While Playing");
	eputstr(7, 5," + -         - Change Master volume.");
	eputstr(7, 6," F1 - F9     - Set Master volume.");
	eputstr(7, 7," F10         - Show MTM song message");
	eputstr(7, 8," F12         - Reset timer speed anywhere including DOS shell");
	eputstr(7, 9," [ ]         - Change speed faster, slower.");
	eputstr(7,10," ; '         - Change BPM slower, faster.");
	eputstr(7,11," , . l r     - Change Channel panning left, right.");
	eputstr(7,12," 1 thru 0    - Mute a channel. (1-10 only)");
	eputstr(7,13," Enter       * Main Screen.  (Sample information etc)");
	eputstr(7,14," c           - Pan Cycle effect!. (stop/start) Main screen only");
	eputstr(7,15," d           - DOS shell.");
	eputstr(7,16," f           * File Selector.  (escape returns to interface)");
	eputstr(7,17," h           - Help.");
	eputstr(7,18," i           * Technical [I]nformation Screen.");
	eputstr(7,19," m           - inverses mute/unmute status of all channels.");
	eputstr(7,20," n           - play next song (In wildcard mode)");
	eputstr(7,21," p           - Pause/Unpause song.  Muting is disable during pause");
	eputstr(7,22," s           - Solo/unsolo a channel.");
	eputstr(7,23," t           * [T]racker Screen.");
	eputstr(7,24," v           * [V]U/Blob screen.");
	eputstr(7,25," ESC         - Quit.");
	eputstr(7,26," Up arrow    - Move channel cursor up.");
	eputstr(7,27," Down arrow  - Move channel cursor down.");
	eputstr(7,28," Right arrow - Move forward a pattern.");
	eputstr(7,29," Left arrow  - Move back a pattern.");
	eputstr(7,30," Page Up     - Scroll Screen up.");
	eputstr(7,31," Page Down   - Scroll Screen down.");
	eputstr(7,32," Home        - Scroll Screen all the way to the top.");
	eputstr(7,33," End         - Scroll Screen all the way to the bottom.");
	eputstr(7,34," SPACE       - mute/unmute a channel.");
	eputstr(7,35," TAB         - (Main screen) Speed up glow scroller.");
	eputstr(7,36,"               (Blob/VU screen) Toggle between channel numbers/blobs");
	eputstr(7,37,"               (Tracker screen) Move channel cursor forward.");
	eputstr(7,38,"               (Technical Screen) Move channel cursor forward.");

	textbox(5,42,75,48,2,"Contact Information");
	eputstr(7,43," email       - firelght@suburbia.apana.org.au");
	eputstr(7,44," Post        - Brett Paterson,");
	eputstr(7,45,"               48/A Parr st,");
	eputstr(7,46,"               Leongatha, 3953.");
	eputstr(7,47,"               Victoria, Australia.");

	do {
		for (count=0; count<FM.channels; count++) {
			if (!pause) {
					techcount+=2;
					if (restart[count]) techindex++;
					if (geffect[count]) techindex++;
			}
		}
		waitvrt();
		updatescreen();
	} while (!kbhit());
	getkey();
}

/****************************************************************************
 *    Name : showmessage                                                    *
 * Purpose : To display the song message - MTM only                         *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : WORD count - counter variable                                  *
 *           WORD x - x position of text as it appears on the message box   *
 *           WORD y - y position of text as it appears on the message box   *
 ****************************************************************************/
void showmessage() {
	WORD count=0, x=0,y=0;

	// roll screen up to top!
	while (screenpos) {
		waitvrt();
		updatescreen();
		screenpos--;
	}

	setrgbpalette(1,0,0,14);
	setrgbpalette(7,42,42,42);
	setrgbpalette(57, 16, 16, 63);                  		// color 9
	setrgbpalette(63,63,63,63);                             // color 15

	textbox(19,10,62,40,2,"Song Message");
	setfgcolor(1);
	while (SongMSG[count] != -1) {
		if (x > 39) { x=0; y++; }
		eputch(21+x, 12+y, SongMSG[count]);
		count++;
		x++;
	}
	updatescreen();

	do {
		for (count=0; count<FM.channels; count++) {
			if (!pause) {
					techcount+=2;
					if (restart[count]) techindex++;
					if (geffect[count]) techindex++;
			}
		}
		waitvrt();
		updatescreen();
	} while (!kbhit());
	getkey();
}


/****************************************************************************
 *    Name : processkey                                                     *
 * Purpose : Process generic keystrokes which could occur in any screen     *
 *  Passed : UBYTE key - the key to be processed                            *
 *           UBYTE sampleview - position on screen samples at drawn, used in*
 *                             checking with scrolling bounds.. i think :)  *
 * Returns : If there should be a redraw.  0 = no, 1 = yes                  *
 *  Locals : UBYTE count - a counter variable                               *
 ****************************************************************************/
UBYTE processkey(UBYTE key, UBYTE sampleview) {
	UBYTE count;

	switch (key) {                                              // KEY
		case 12 : globalvol--;                                  // -
				  if (globalvol < 0) globalvol = 0;
				  break;
		case 13 : globalvol++;                                  // +
				  if (globalvol > 64) globalvol = 64;
				  break;
		case 25 : pause=~pause;
				  for(count=0; count<FM.channels; count++) {    // p
					  if (pause) GUSSetVolume(count, 0);
					  else GUSSetVolume(count, volume[count]*globalvol/64);
				  }
				  break;
		case 26 : speed--;                                      // [
				  if (speed < 1) speed = 1;
				  break;
		case 27 : speed++;                                      // ]
				  break;
		case 39 : bpm--;                                        // ;
				  if (bpm < 24) bpm=24;
				  SetTimerSpeed();
				  break;
		case 40 : bpm++;                                        // '
				  if (bpm > 254) bpm = 254;
				  SetTimerSpeed();
				  break;
		case 75 : ord--;                                        // <-ARROW
				  row=0;
				  if (ord < 0) ord = 0;
				  break;
		case 77 : ord++;                                        // ->ARROW
				  row=0;
				  if (ord >= FM.songlength) ord = FM.restart;
				  break;
		case 50 : if (!pause) {                                 // m
						for(count=0; count<FM.channels; count++) {
							mute[count]=~mute[count];
							GUSSetVolume(count, volume[count]*globalvol/64);
						}
				  }
				  break;
		case 57 : if (!pause) {                                 // SPACE
						mute[select]=~mute[select];
					  GUSSetVolume(count, volume[select]*globalvol/64);
				  }
				  break;
		case 31 : if (!pause) {                                 // s
					for(count=0; count<FM.channels; count++) {
						if (count==select || solo==select) mute[count]=0;
						else mute[count]=255;
						GUSSetVolume(count, volume[count]*globalvol/64);
					}
				  }
				  if (solo == select) solo=255;
				  else solo = select;
				  break;
		case 38 : panval[select] = 0;                           // l
				  GUSSetBalance(select, panval[select]);
				  break;
		case 19 : panval[select] = 0xFF;                        // r
				  GUSSetBalance(select, panval[select]);
				  break;
		case 51 : panval[select]--;                             // ,
				  if (panval[select]<0) panval[select]=0;
				  GUSSetBalance(select, panval[select]);
				  break;
		case 52 : panval[select]++;                             // .
				  if (panval[select]>0xFF) panval[select]=0xFF;
				  GUSSetBalance(select, panval[select]);
				  break;
		case 73 : if (sampleview+FM.numinsts <50 || !sampleview) break;// PAGEUP
				  screenpos--;
				  if (screenpos < 0) screenpos=0;
				  break;
		case 81 : if (sampleview+FM.numinsts <50 || !sampleview) break;// PAGEDN
				  screenpos++;
				  if (screenpos > FM.numinsts-(50-sampleview)) screenpos=FM.numinsts-(50-sampleview);
				  break;
		case 71 : if (!sampleview) break;
				  while (screenpos) {                           // HOME
					waitvrt();
					updatescreen();
					screenpos--;
				  }
				  break;
		case 79 : if (!sampleview) break;
				  while (screenpos < (FM.numinsts-(50-sampleview))) {
					waitvrt();                                  // END
					updatescreen();
					screenpos++;
				  }
				  break;
		case 35 : help();                                       // h
				  return 1;
		case 68 : if (SongMSG[0] == -1) return 0;               // F11
				  showmessage();
				  return 1;
		case 32 : set8025mode();                                // d
				  printf("Type 'EXIT' to return to FireMOD.\r\n");
				  printf("(Hit F12 at any time to reset song speed)\r\n");
				  system("COMMAND");
				  SetTimerSpeed();
				  set8050mode();
				  return 1;
	};

	// keys 1 through to 0, to mute a channel.
	if (!pause) {
		if (key > 1 && key < 12) {
			mute[key-2] = ~mute[key-2];
			GUSSetVolume(key-2, volume[key-2]*globalvol/64);
		}
	}

	// keys F1 through to F9, set master volume.
	if (key >= 59 && key <= 67) globalvol = ((key-59) * 8);
	return 0;
}


/****************************************************************************
 *    Name : infoscreen                                                     *
 * Purpose : Technical information Interface screen                         *
 *  Passed : UBYTE *key - address of keystroke, so it will be returned to   *
 *                        caller function                                   *
 * Returns : ERR error code, 0 if no errors                                 *
 *  Locals : BYTE cursor - position of glowing channel cursor on screen     *
 *           WORD temp - temporary variable to hold results of some lengthly*
 *                       mathematical equations                             *
 *           UBYTE count - counter variable                                 *
 *           UBYTE r,g,b - red blue and green values for GLOW(tm) technique *
 *           UBYTE glow - variable that tells glower what to do             *
 *           UBYTE min, sec - clock variables                               *
 *           UDWORD gusused - value holding the amoun of GUS dram used       *
 *           char *efflook - lookup table holding effects numbers according *
 *                           to what format is being played, ie MOD or S3M  *
 *           char *notetab - a table holding note strings, ie 'C- C# G#'    *
 ****************************************************************************/
UBYTE infoscreen(UBYTE *key) {
	BYTE cursor;
	WORD temp;
	UBYTE count, r=64,g=64,b=64,glow=0, min, sec;
	UDWORD gusused;
	char *efflook, *notetab;

	if ((efflook=(char *)calloc(100,1))==NULL) return ERR_OUT_OF_MEM;
	if ((notetab=(char *)calloc(24,1))==NULL) {
		free(efflook);
		return ERR_OUT_OF_MEM;
	}

	if (memcmp(FM.type, "S3M", 3) == 0)
		strcpy(efflook, "  ? ? G H ? ? E 8 O ? B ? C   T A D E F I K H Q U V SAS0? ? S1S3S2SBS4S8? ? ? SCSDSESFJ ");
	else strcpy(efflook,"  1 2 3 4 5 6 7 8 9 A B C D   F ? ? ? ? ? ? ? ? ? ? ? E0E1E2E3E4E5E6E7E8E9EAEBECEDEESF0 ");
	strcpy(notetab, "C-C#D-D#E-F-F#G-G#A-A#B-");

redraw:
	clearscreen();
	setrgbpalette(0,8,16,24);               // color 0
	setrgbpalette(1,0,0,14);                // color 1
	setrgbpalette(2,10,10,63);              // color 2
	setrgbpalette(3,24,0,0);                // color 3
	setrgbpalette(4,32,5,5);                // color 4
	setrgbpalette(5,5,10,20);               // color 5
	setrgbpalette(7,42,42,42);              // color 7
	setrgbpalette(56, 16, 8, 63);           // color 8
	setrgbpalette(57, 16, 16, 63);          // color 9
	setrgbpalette(60, 48, 12, 12);          // color 12
	setrgbpalette(61,32,0,0);               // color 13
	setrgbpalette(63,63,63,63);             // color 15

	// set up heading window
	textbox(1,1,79,3,2,"FireMOD 1.06. CopyRight (c) FireLight 1995");
	setfgcolor(9);
	eputstr(3, 2, "Name : ");
	eputstr(47,2, "Size : ");
	eputstr(61,2, "Mem Free : ");
	setfgcolor(1);
	eputstr(10,2,FM.name);
	sprintf(s,"%dkb", filelen/1024); eputstr(54,2, s);
	sprintf(s,"%dkb", memfree()/1024); eputstr(72,2,s);
	setbgcolor(0);

	// set up info window
	textbox(1,38,18,49,1,"Info");
	setbgcolor(7);
	setfgcolor(9);
	eputstr(2,40,"   Type :");
	eputstr(2,41,"  Clock :");
	eputstr(2,42,"  Speed :");
	eputstr(2,43,"  Order :");
	eputstr(2,44,"Pattern :");
	eputstr(2,45,"    Row :");
	eputstr(2,46," Volume :");
	eputstr(2,47,"TechIdx :");
	eputstr(2,48,"MixRate :");
	setfgcolor(1);
	sprintf(s, "%u", divisor); eputstr(11,48,s);
	setfgcolor(13);
	eputstr(11,40,FM.type);

	// set up interface
	textbox(1,5,79,35,2,"Technical Channel Information");
	setfgcolor(15);
	eputstr(3,6,"Channel");
	setfgcolor(9);
	eputstr(3,7,"mute");
	eputstr(3,8,"volume");
	eputstr(3,9,"amigaval");
	eputstr(3,10,"panning");
	eputstr(3,11,"inst #");
	eputstr(3,12,"note");
	eputstr(3,13,"last freq");
	eputstr(3,14,"restart?");
	eputstr(3,15,"effect");
	eputstr(3,16,"s/offset");
	eputstr(3,17,"porta val");
	eputstr(3,18,"vol slide");
	eputstr(3,19,"retrig x");
	eputstr(3,20,"retrig y");
	eputstr(3,21,"porta to");
	eputstr(3,22,"porta spd");
	eputstr(3,23,"vibr pos");
	eputstr(3,24,"vibr speed");
	eputstr(3,25,"vibr depth");
	eputstr(3,26,"trem pos");
	eputstr(3,27,"trem speed");
	eputstr(3,28,"trem depth");
	eputstr(3,29,"patloop to");
	eputstr(3,30,"patloop #");
	eputstr(3,31,"wave cntrl");
	eputstr(3,32,"trem on/of");
	eputstr(3,33,"trem count");
	eputstr(3,34,"patn delay");

	textbox(21,38,79,49,1,"Miscellaneous");
	setbgcolor(7);
	setfgcolor(1);
	eputstr(23,40," Pattern data mem :            GUS DRAM used  : ");
	eputstr(23,41,"                              +GUS DRAM free  : ");
	eputstr(23,42,"                              =Total GUS DRAM : ");
	setfgcolor(3);
	eputstr(23,47,"                                                   moo?");
	setfgcolor(1);
	sprintf(s,"%04dkb", (320L*FM.channels*(FM.numpats+1))>>10); eputstr(43,40,s);
	// work out the gus dram used/free/total
	sprintf(s,"%04dkb", (gusdram >> 10)+1); eputstr(71,42,s);
	gusused =0;
	for (count=0; count<FM.numinsts; count++) {
		if ((inst[count]->offset + inst[count]->length) > gusused) {
			gusused = inst[count]->offset + inst[count]->length;
		}
	}
	sprintf(s,"%04dkb", gusused/1024+1); eputstr(71, 40, s);
	sprintf(s,"%04dkb", (gusdram-gusused)/1024); eputstr(71, 41, s);

	setfgcolor(1);
	do {
		if (kbhit()) *key = getkey();
		else *key=0;
		if (processkey(*key, 0)) goto redraw;

		// The glowing text scroller
		if (glow == 0) { g--; b++; if (b>124) glow = 1; }
		if (glow == 1) { b--; r++; if (r>124) glow = 2; }
		if (glow == 2) { r--; g++; if (g>124) glow = 0; }
		setrgbpalette(3, r>>1,g>>1,b>>1);

		switch (*key) {
			case 15 : select++;
					  if (select >= FM.channels) select =0;
					  break;
		};

		// clock
		sec = (fclock / 1000000L) % 60;
		min = (fclock / 1000000L) / 60;
		if (min > 59) {
			fclock=0;
			sec =0;
			min =0;
		}

		// print out data for info box.
		setfgcolor(1);
		setbgcolor(7);
		if (pause) { setfgcolor(12); eputstr(11,41,"PAUSE"); setfgcolor(1);}
		else { sprintf(s,"%02d:%02d", min,sec); eputstr(11,41,s); }
		sprintf(s,"%03d/%03d", speed,bpm); eputstr(11,42,s);
		sprintf(s,"%03d/%03d", ord, FM.songlength-1); eputstr(11,43,s);
		sprintf(s,"%02d/%02d", FM.order[ord], FM.numpats); eputstr(11,44,s);
		sprintf(s,"%02d/%02d", row, FM.patlen[FM.order[ord]]);eputstr(11,45,s);
		sprintf(s,"%02d/64", globalvol);   eputstr(11,46,s);
		sprintf(s,"%06.2f%%", ((float)techindex/techcount)*100); eputstr(11,47,s);

		// which channel to position the pattern data on according to cursor
		cursor = select-12;
		if (cursor < 0) cursor = 0;
		for (count=cursor; count < FM.channels && count<(cursor+13); count++) {
			if (!pause) {
				techcount+=2;
				if (restart[count]) techindex++;
				if (geffect[count]) techindex++;
			}
			setfgcolor(4);
			if (count == select) setbgcolor(3);
			else setbgcolor(7);

			temp = 14+((count-cursor) * 5);
			sprintf(s,"%-2d", count+1);  eputstr(temp, 6, s);
			setfgcolor(1);
			setbgcolor(7);
			if (mute[count]) eputstr(temp,7,"yes");
			else eputstr(temp,7,"no ");
			sprintf(s,"%02d",volume[count]);  eputstr(temp,8,s);
			sprintf(s,"%-5u",freq[count]);    eputstr(temp,9,s);
			sprintf(s,"%02X",panval[count]);  eputstr(temp,10,s);
			sprintf(s,"%u",lastins[count]); eputstr(temp,11,s);
			eputstrw(temp,12, notetab+((lastnot[count]%12)*2) ,2);
			sprintf(s,"%u",lastnot[count]/12); eputstr(temp+2, 12, s);
			sprintf(s,"%-5u",lastper[count]); eputstr(temp,13,s);
			if (restart[count]) eputstr(temp,14,"yes");
			else eputstr(temp,14,"no ");
			eputstrw(temp,15,efflook+(geffect[count]*2),2);
			sprintf(s,"%04X",soffset[count]); eputstr(temp,16,s);
			sprintf(s,"%02X",lastpor[count]); eputstr(temp,17,s);
			sprintf(s,"%02X",lastvsl[count]); eputstr(temp,18,s);
			sprintf(s,"%d",retrigx[count]); eputstr(temp,19,s);
			sprintf(s,"%d",retrigy[count]); eputstr(temp,20,s);
			sprintf(s,"%d",porto[count]);   eputstr(temp,21,s);
			sprintf(s,"%02X",portsp[count]);  eputstr(temp,22,s);
			sprintf(s,"%-3d",vibpos[count]);  eputstr(temp,23,s);
			sprintf(s,"%X",vibspe[count]);  eputstr(temp,24,s);
			sprintf(s,"%X",vibdep[count]);  eputstr(temp,25,s);
			sprintf(s,"%-3d",trempos[count]); eputstr(temp,26,s);
			sprintf(s,"%X",tremspe[count]); eputstr(temp,27,s);
			sprintf(s,"%X",tremdep[count]); eputstr(temp,28,s);
			sprintf(s,"%-2d",patlooprow[count]); eputstr(temp,29,s);
			sprintf(s,"%-2d",patloopno[count]); eputstr(temp,30,s);
			sprintf(s,"%02X",wavecon[count]); eputstr(temp,31,s);
			sprintf(s,"%-2X",tremparm[count]);  eputstr(temp,32,s);
			sprintf(s,"%-2d",tremor[count]);  eputstr(temp,33,s);
		}
		sprintf(s,"%02d", patdelay); eputstr(14,34,s);

		waitvrt();
		updatescreen();
	} while(*key!=1 && *key!=28 && *key!=47 && *key!=20 && *key!=49 && *key!=33);

	free(notetab);
	free(efflook);

	return 0;	// no errors
}


/****************************************************************************
 *    Name : trackerscreen                                                  *
 * Purpose : Tracker Interface screen - shows pattern data scrolling past   *
 *           in realtime.
 *  Passed : UBYTE *key - address of keystroke, so it will be returned to   *
 *                        caller function                                   *
 * Returns : ERR error code, returns 0 if no errors                         *
 *  Locals : BYTE cursor - position of glowing channel cursor on screen     *
 *           UBYTE rowcount - counter variable for the number of rows in a  *
 *                            pattern, whilst the offscreen pattern buffer  *
 *                            is being updated.                             *
 *           UBYTE min, sec - clock variables                               *
 *           UBYTE lastord - a 'previous' order value so we know only to    *
 *                           update to a new pattern if it has changed      *
 *           UBYTE count2 - a counter variable for 0-255 values             *
 *           UBYTE r,g,b - red blue and green values for GLOW(tm) technique *
 *           UBYTE glow - variable that tells glower what to do             *
 *           WORD rstart - position on screen to start displaying pattern   *
 *                         data                                             *
 *           WORD rend - position on screen to end displaying pattern data  *
 *           WORD rpatof - row in pattern data to start displaying from     *
 *           WORD count - signed count value for big values.                *
 *           UWORD trackoff - offset value in pattern data according to row,*
 *                            pattern number etc.                           *
 *           UWORD temp - temporary value                                   *
 *           char efflook[66] - lookup table holding effects numbers        *
 *                              according to what format is being played    *
 *           Note *trackcurr - a pointer that points at the pattern data for*
 *                             this interface screen                        *
 *           char *notetab[12] - a table holding 12 note strings, ie 'C# G#'*
 *   Notes : How this works.  It draws a pattern into an offscreen buffer   *
 *           every time a new pattern comes around, and it just             *
 *           scrolls/blit this offscreen buffer up the screen, according to *
 *           the current row being played.                                  *
 ****************************************************************************/
UBYTE trackerscreen(UBYTE *key) {
	BYTE cursor;
	UBYTE rowcount, min, sec, lastord=255, count2,r=64,g=64,b=64,glow=0;
	WORD rstart,rend,rpatof, count;
	UWORD trackoff, temp;
	char *notetab[12] = {
	"C-", "C#", "D-", "D#", "E-", "F-", "F#", "G-", "G#", "A-", "A#", "B-"};
	Note *trackcurr;
	char efflook[66];

	// set up appropriate effect
	if (memcmp(FM.type, "S3M", 3) == 0)
		memcpy(efflook, "0??GH??E8ODB?CSTADEFIKLQUVS",27);
	else memcpy(efflook,"0123456789ABCDEF", 16);

redraw:
	clearscreen();
	setrgbpalette(0,8,16,24);               // color 0
	setrgbpalette(1,0,0,14);                // color 1
	setrgbpalette(2,10,10,63);              // color 2
	setrgbpalette(3,24,0,0);                // color 3
	setrgbpalette(4,32,5,5);                // color 4
	setrgbpalette(5,5,10,20);               // color 5
	setrgbpalette(7,42,42,42);              // color 7
	setrgbpalette(56, 16, 8, 63);           // color 8
	setrgbpalette(57, 16, 16, 63);          // color 9
	setrgbpalette(60, 48, 12, 12);          // color 12
	setrgbpalette(61,32,0,0);               // color 13
	setrgbpalette(63,63,63,63);             // color 15

	// set up heading window
	textbox(1,1,79,3,2,"FireMOD 1.06. CopyRight (c) FireLight 1995");
	setfgcolor(9);
	eputstr(3, 2, "Name : ");
	eputstr(47,2, "Size : ");
	eputstr(61,2, "Mem Free : ");
	setfgcolor(1);
	eputstr(10,2,FM.name);
	sprintf(s,"%dkb", filelen/1024); eputstr(54,2, s);
	sprintf(s,"%dkb", memfree()/1024); eputstr(72,2,s);

	// set up info box at bottom of screen
	textbox(1,46,79,49,2,"Info");
	setbgcolor(7);
	setfgcolor(9);

	eputstr(3,47,"   Type :");
	eputstr(23,47,"Channel :");
	eputstr(43,47,"  Clock :");
	eputstr(61,47," Volume :");
	eputstr( 3,48,"  Speed :");
	eputstr(23,48,"  Order :");
	eputstr(43,48,"Pattern :");
	eputstr(61,48,"TechIdx :");

	setfgcolor(1);
	setbgcolor(7);
	eputstr(13,47,FM.type);

	do {
		if (kbhit()) *key = getkey();
		else *key=0;

		if (processkey(*key, 0)) goto redraw;

		switch (*key) {
			case 15 : select++;
					  if (select >= FM.channels) select =0;
					  break;
		}

		// The glowing text scroller
		if (glow == 0) { g--; b++; if (b>124) glow = 1; }
		if (glow == 1) { b--; r++; if (r>124) glow = 2; }
		if (glow == 2) { r--; g++; if (g>124) glow = 0; }
		setrgbpalette(3, r>>1,g>>1,b>>1);

		// which channel to position the pattern data on according to cursor
		cursor = select-6;
		if (cursor < 0) cursor = 0;

		// Work out clock value in terms of minutes and seconds
		sec = (fclock / 1000000L) % 60;
		min = (fclock / 1000000L) / 60;
		if (min > 59) {
			fclock=0;
			sec =0;
			min =0;
		}

		// calculate the technicality index meter.
		for (count=0; count<FM.channels; count++) {
			if (!pause) {
				techcount+=2;
				if (restart[count]) techindex++;
				if (geffect[count]) techindex++;
			}
		}

		// update info box
		setfgcolor(1);
		setbgcolor(7);
		if (pause) { setfgcolor(12); eputstr(53,47,"PAUSE"); setfgcolor(1);}
		else { sprintf(s,"%02d:%02d", min,sec); eputstr(53,47,s); }
		sprintf(s,"%02d/%02d", select+1,FM.channels); eputstr(33,47,s);
		sprintf(s,"%03d/%03d", speed,bpm); eputstr(13,48,s);
		sprintf(s,"%03d/%03d", ord, FM.songlength-1); eputstr(33,48,s);
		sprintf(s,"%02d/%02d", FM.order[ord], FM.numpats); eputstr(53,48,s);
		sprintf(s,"%02d/64", globalvol);   eputstr(71,47,s);
		sprintf(s,"%06.2f%%", ((float)techindex/techcount)*100); eputstr(71,48,s);
		setbgcolor(0);

		// work out where to position pattern on the screen (top to bottom)
		rstart=25-row;             		// starts at row 25 then scrolls up
		if (rstart < 4) {
			rstart =4;                  // stop at row 4
			rpatof = row-21;            // now print from this pattern row #
		}
		else rpatof=0;					// we still print from pattern row 0
		if (row < 45) rend = 44;
		else rend = 45+(45-row);

		// If pattern changed then update this pattern buffer to the new one.
		// This is the slow part so it may skip a few rows with lag, at the
		// start of every pattern.  It only does it once an order tho.
		if (ord != lastord) {

			// first draw stupid little glowing channel numbers at the top
			// of the pattern
			setfgcolor(3);
			for (count=0; count<FM.channels; count++) {
				sprintf(s,"%-2d", count+1);
				eputstrp(8+(count*11), 0, s);
				eputstrp(8+(count*11), 65, s);
			}

			// now do the pattern data
			for (rowcount=0; rowcount<FM.patlen[FM.order[ord]]; rowcount++) {
				setfgcolor(15);
				temp = rowcount+1;
				sprintf(s,"%02d", rowcount); eputstrp(1, temp, s);
				trackoff = 5*FM.channels*rowcount;

				for (count=0; count<FM.channels; count++) {
					// point our little note structure to patbuff
					trackcurr = (Note *)(patbuff[FM.order[ord]] + trackoff);

					// draw note/keyoff or nonote
					if (trackcurr->note == NONOTE) sprintf(s,"---");
					else if (trackcurr->note == KEYOFF) sprintf(s,"^^^");
					else sprintf(s,"%s%u", notetab[trackcurr->note%12], trackcurr->note/12);
					setfgcolor(7);
					eputstrp(4+(count*11), temp, s);

					// draw number
					sprintf(s,"%02d",trackcurr->number);
					setfgcolor(15);
					eputstrp(7+(count*11), temp, s);

					// draw volume column
					if (trackcurr->volume == 0xFF) sprintf(s,"--");
					else sprintf(s,"%02X",trackcurr->volume);
					setfgcolor(12);
					eputstrp(9+(count*11), temp,s);

					// draw effect (use a look up table to tell between mod
					// and s3m effects)
					setfgcolor(7);
					eputchp(11+(count*11), temp, efflook[trackcurr->effect]);

					// draw effect parameter
					sprintf(s,"%02X", trackcurr->eparm);
					eputstrp(12+(count*11), temp, s);

					// go to next note
					trackoff += 5;
				}
			}
			lastord = ord;
		}

		// now that's done all we have to do is scroll it up the screen like
		// a bitmap.

		// pattern data will take up from row 4 to row 44 on the screen.
		for (count = 4; count <= 44; count++) {
			// if row is not meant to hold any data print blank line.
			if (count > rend || count < rstart) memset(screen+(count*160), 0, 160);
			// else draw a row
			else {
				// start at beginning of row and print row number
				temp = (6U+(FM.channels*22U))*(count-rstart+rpatof);
				memcpy(screen+(160U*count), pattern+temp,6);

				// now go to first column of that row that is visible on the
				// screen
				temp += (6U+(cursor*22));

				// draw the 7 channels
				for (count2=cursor; count2<(cursor+7) && count2<FM.channels; count2++) {
					if (!mute[count2]) {
						 memcpy(screen+(160U*count)+6U+(22U*(count2-cursor)), pattern+temp, 22);
					}
					else memset(screen+(160U*count)+6U+(22U*(count2-cursor)), 0, 22);
					temp+=22;
				}
			}
		}

		// draw black highlight bar
		for (count =4001; count<4161; count+=2) screen[count] |= 16;
		// draw red cursor
		temp = select;
		if (temp > 6) temp=6;
		for (count =4007+(temp*22); count < 4027+(temp*22); count+=2) screen[count] |= 32;

		// blit the whole screen now to vga
		updatescreen();

	} while(*key!=1 && *key!=28 && *key!=47 && *key!=23 && *key!=49 && *key!=33);

	return 0;	// no error
}


/****************************************************************************
 *    Name : guidisplay                                                     *
 * Purpose : To run the VU/Blob screen                                      *
 *  Passed : UBYTE *key - address of the previous keystroke                 *
 * Returns : an errorcode                                                   *
 *  Locals : BYTE flag - this value says whether to display blobs or numbers*
 *			 UBYTE col[8] - remapped colour values                          *
 *			 UBYTE bar[32] - 32 channels' worth of bop bar levels           *
 *			 UBYTE min, sec - clock variables                               *
 *			 UBYTE sampleview - y position on screen to display samples     *
 *			 WORD x[32], y[32] - x and y positions for each Blob on screen  *
 *			 WORD a,b,c - attributes for each blob.. volume, freq & sample  *
 *			 WORD count, count2 - you know the deal                         *
 ****************************************************************************/
UBYTE guidisplay(UBYTE *key) {
	BYTE flag =0;
	UBYTE col[8], bar[32], min,sec, sampleview=0;
	WORD x[32],y[32],a,b,c,count,count2;

	// start blobs out in unseen places...
	for (count=0; count<32; count++) {x[count]=2; y[count]=18; bar[count]=0;}

	if (FM.channels > 11) sampleview=FM.channels+9;
	else sampleview=19;

redraw:
	clearscreen();

	// blob colours
	col[0] =59; setrgbpalette(col[0], 7,16, 23);        // color 11
	col[1] =2;  setrgbpalette(col[1], 15,16, 24);       // color 2
	col[2] =3;  setrgbpalette(col[2], 23,16, 24);       // color 3
	col[3] =4;  setrgbpalette(col[3], 31,16, 24);       // color 4
	col[4] =5;  setrgbpalette(col[4], 39,16, 24);       // color 5
	col[5] =60; setrgbpalette(col[5], 46,16, 24);       // color 12
	col[6] =56; setrgbpalette(col[6], 54,16, 24);       // color 8
	col[7] =58; setrgbpalette(col[7], 62,16, 24);       // color 10

	// other screen colors
	setrgbpalette(0,8,16,24);                           // color 0
	setrgbpalette(1,0,0,14);                            // color 1
	setrgbpalette(7,42,42,42);                          // color 7
	setrgbpalette(57, 16, 16, 63);                      // color 9
	setrgbpalette(61,32,0,0);                           // color 13
	setrgbpalette(63,63,63,63);                         // color 15

	setfgcolor(1);
	setbgcolor(0);
	textsquare(1,1,80,sampleview+FM.numinsts,1);

	textbox(1,1,79,3,2,"FireMOD 1.06. CopyRight (c) FireLight 1995");
	setfgcolor(9);
	eputstr(3, 2, "Name : ");
	eputstr(47,2, "Size : ");
	eputstr(61,2, "Mem Free : ");
	setfgcolor(1);
	eputstr(10,2,FM.name);
	sprintf(s,"%dkb", filelen/1024); eputstr(54,2, s);
	sprintf(s,"%dkb", memfree()/1024); eputstr(72,2,s);

	textbox(1,5,18,16,1,"Info");
	setbgcolor(7);
	setfgcolor(9);
	eputstr(2,7,"   Type :");
	eputstr(2,8,"  Clock :");
	eputstr(2,9,"  Speed :");
	eputstr(2,10,"  Order :");
	eputstr(2,11,"Pattern :");
	eputstr(2,12,"    Row :");
	eputstr(2,13," Volume :");
	eputstr(2,14,"TechIdx :");
	eputstr(2,15,"MixRate :");
	setfgcolor(1);
	sprintf(s, "%u", divisor); eputstr(11,15,s);
	setfgcolor(13);
	eputstr(11,7,FM.type);

	// draw sample list
	setbgcolor(0);
	for (count=0; count<FM.numinsts; count++) {
		setfgcolor(7);
		eputstrw(2, sampleview+count, inst[count]->name, 23);
		setfgcolor(1);
		eputch(24, sampleview+count, '');
	}

	// set up bopbar window
	textbox(20,5,79,6+FM.channels,1,"Bop Bars");
	setfgcolor(15);
	for (count=0; count<FM.channels; count++) { sprintf(s,"%2d",count+1); eputstr(22,count+6,s);  }
	setfgcolor(1);
	for (count=0; count < FM.channels; count++)
		for (count2=25; count2<79; count2++)
			eputch(count2, 6+count, 254);

	setbgcolor(7);
	setfgcolor(13);
	eputch(21,6+select,'');                // draw channel marker dot

	do {
		for (count=0; count < FM.channels; count++) {
			if (!pause) {
				techcount+=2;
				if (restart[count]) techindex++;
				if (geffect[count]) techindex++;
			}

			// 3624 = B#2 to B#8 = 56... sorry C-0 to B-2 you miss out
			a = 22+((3624-freq[count])/62);

			if (a<25) a=25;
			if (a>78) a=78;
			b = lastins[count];
			c = volume[count];

			setbgcolor(0);
			setfgcolor(0);
			eputstr(x[count], y[count], "  ");
			x[count]=a;
			y[count]=b+sampleview;

			if (c > 7) {
				if (!flag) {
					setfgcolor(col[((c>>3)-1)]&15);
					eputch(x[count], y[count], 219);
				}
				else {
					setfgcolor(col[((c>>3)-1)]&15);
					sprintf(s,"%02d",count+1); eputstr(x[count], y[count],s);
				}
			}
			else {
				setbgcolor(0);
				setfgcolor(0);
				eputstr(x[count], y[count], "  ");
			}

			// set volume bar positions
			if (restart[count]) bar[count]=volume[count];
			if (bar[count] > 0 && !pause) bar[count]--;
			switch (geffect[count]) {
				case  0xA :
				case  0xC :
				case  0x5 :
				case  0x6 :
				case  0x9 :
				case 0x11 :
				case 0x15 :
				case 0x16 : bar[count]=volume[count];
							break;
			}

			// Blaster volume bar technique (tm) ;)
			setfgcolor(7);
			setbgcolor(7);
			if (mute[count] == 0) {
				setbgcolor(7);
				setfgcolor(9);
				for (count2=1; count2<55; count2++) {
					if (count2 > bar[count] *.88) {
						setfgcolor(1);
						eputch(count2+24, count+6, 254);
					}
					else {
						if (count2<45) setfgcolor(9);
						else if (count2<53) setfgcolor(14);
						else setfgcolor(10);
						eputch(count2+24, count+6, 254);
					}
				}
			} else for (count2=1; count2<55; count2++)
				eputch(count2+24, count+6, 254);
		}

		// get a key
		if (kbhit()) *key = getkey();
		else *key=0;

		setbgcolor(7);
		if (processkey(*key, sampleview)) goto redraw;
		switch (*key) {                                           // KEY
			case 15 : flag=~flag;                                // TAB
					  break;
			case 72 : eputch(21,6+select,' ');                   // UP ARROW
					  select--;
					  if (select==255) select=FM.channels-1;
					  setfgcolor(13);
					  eputch(21,6+select,'');
					  break;
			case 80 : eputch(21,6+select,' ');                                              // DOWN ARROW
					  select++;
					  if (select==FM.channels) select=0;
					  setfgcolor(13);
					  eputch(21,6+select,'');
					  break;
		};

		// clock
		sec = (fclock / 1000000L) % 60;
		min = (fclock / 1000000L) / 60;
		if (min > 59) {
			fclock=0;
			sec =0;
			min =0;
		}

		// print out data for info box.
		setfgcolor(1);
		if (pause) { setfgcolor(12); eputstr(11,8,"PAUSE"); setfgcolor(1);}
		else { sprintf(s,"%02d:%02d", min,sec);   eputstr(11,8,s); }
		sprintf(s,"%03d/%03d", speed,bpm); eputstr(11,9,s);
		sprintf(s,"%03d/%03d", ord, FM.songlength-1); eputstr(11,10,s);
		sprintf(s,"%02d/%02d", FM.order[ord], FM.numpats); eputstr(11,11,s);
		sprintf(s,"%02d/%02d", row, FM.patlen[FM.order[ord]]);eputstr(11,12,s);
		sprintf(s,"%02d/64", globalvol);   eputstr(11,13,s);
		sprintf(s,"%06.2f%%", ((float)techindex/techcount)*100); eputstr(11,14,s);

		waitvrt();
		updatescreen();
	} while(*key!=1 && *key!=28 && *key!=20 && *key!=23 && *key!=49 && *key!=33);

	return 0;		// no errors
}


/****************************************************************************
 *    Name : errorhandler                                                   *
 * Purpose : To catch an error where there is not a disk in a floppy drive  *
 *           before DOS does with that stupid abort,retry,fail message      *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 *    Note : THIS DOESNT WORK IN WATCOM/PMODE FOR SOME REASON AND I COULDNT *
 *           BE FUCKED WORKING OUT WHY - IM SICK OF FMOD :)                 *
 ****************************************************************************/
void interrupt errorhandler(__CPPARGS) {
	textbox(26,24,52,27,1,"Disk Error!");
	eputstr(31,25,"Please insert disk");
	eputstr(31,26," then press a key");
	updatescreen();

	getkey();

	outp(0x20, 0x20);
}


/****************************************************************************
 *    Name : sort_by_type                                                   *
 * Purpose : comparison function for qsort(), to sort by directories/files  *
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals :                                                                *
 ****************************************************************************/
int sort_by_type( const void *a, const void *b) {
	typedef struct {
		char name[13];
		UBYTE type;
		DWORD size;
	} fileblock;

	fileblock *fblocka = (fileblock *)a;
	fileblock *fblockb = (fileblock *)b;

	return (fblocka->type<fblockb->type)?1:(fblocka->type>fblockb->type)?-1:0;
}


/****************************************************************************
 *    Name : sort_by_size1                                                  *
 * Purpose : comparison function for qsort(), to sort by file sizes         *
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals :                                                                *
 ****************************************************************************/
int sort_by_size1( const void *a, const void *b) {
	typedef struct {
		char name[13];
		UBYTE type;
		DWORD size;
	} fileblock;

	fileblock *fblocka = (fileblock *)a;
	fileblock *fblockb = (fileblock *)b;

	return (fblocka->size>fblockb->size)?1:(fblocka->size<fblockb->size)?-1:0;
}


/****************************************************************************
 *    Name : sort_by_size2                                                  *
 * Purpose : comparison function for qsort(), to sort by size backwards     *
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals :                                                                *
 ****************************************************************************/
int sort_by_size2( const void *a, const void *b) {
	typedef struct {
		char name[13];
		UBYTE type;
		DWORD size;
	} fileblock;

	fileblock *fblocka = (fileblock *)a;
	fileblock *fblockb = (fileblock *)b;

	return (fblocka->size<fblockb->size)?1:(fblocka->size>fblockb->size)?-1:0;
}

/****************************************************************************
 *    Name : sort_by_name                                                   *
 * Purpose : comparison function for qsort(), to sort file names alphabet'ly*
 *  Passed :                                                                *
 * Returns :                                                                *
 *  Locals :                                                                *
 ****************************************************************************/
int sort_by_name( const void *a, const void *b) {
	typedef struct {
		char name[13];
		UBYTE type;
		DWORD size;
	} fileblock;

	fileblock *fblocka = (fileblock *)a;
	fileblock *fblockb = (fileblock *)b;

	return memcmp(fblocka->name,fblockb->name,13);
}


/****************************************************************************
 *    Name : FileSelector                                                   *
 * Purpose : To display a file listing/drive box etc and allow user to load *
 *           songs via this method                                          *
 *  Passed : -                                                              *
 * Returns : errorcode                                                      *
 *  Locals : eek- im not commenting all those variables                     *
 ****************************************************************************/
UBYTE FileSelector() {
	UBYTE ERR=0, drive[26], key=0;
	UWORD done, dirlen,count, numdirs, numdrives;
	WORD poscursor=0, posview=0, posselect=0;
	unsigned maxdrives;
	char *fbuff, currdir[100];
	struct find_t ffblk;
	struct stat statbuf;

	typedef struct {
		char name[13];
		UBYTE type;
		DWORD size;
	} fileblock;
	fileblock *fblock;

	UBYTE keytodrive[26] = {
	   17,23, 5,18,20,25,21, 9,15,16,		// QWERTYUIOP
		1,19, 4, 6, 7, 8,10,11,12,			// ASDFGHJKL
	   26,24, 3,22, 2,14,13					// ZXCVBNM
	};


	clearscreen();

	// set some default palette values
	setrgbpalette(0,8,16,24);               // color 0
	setrgbpalette(1,0,0,14);                // color 1
	setrgbpalette(2,10,10,63);              // color 2
	setrgbpalette(3,24,0,0);                // color 3
	setrgbpalette(4,5,30,30);               // color 4
	setrgbpalette(5,5,10,20);               // color 5
	setrgbpalette(7,42,42,42);              // color 7
	setrgbpalette(56, 16, 8, 63);           // color 8
	setrgbpalette(57, 16, 16, 63);          // color 9
	setrgbpalette(60, 48, 12, 12);          // color 12
	setrgbpalette(61,32,0,0);               // color 13
	setrgbpalette(63,63,63,63);             // color 15

	// set up heading window
	textbox(1,1,79,3,2,"FireMOD 1.06. CopyRight (c) FireLight 1995");
	setfgcolor(9);
	eputstr(3, 2, "Name : ");
	eputstr(45,2, "Size : ");
	eputstr(61,2, "Mem Free : ");
	setbgcolor(0);

	textbox(1,6,79,8,1,"Path");

	// Set up drive box - working out which drives are valid and which aren't
	// First check CMOS for floppy drives by using port read/writes
	numdrives=0;
	outp(0x70, 0x10);
	if (inp(0x71) >> 4) { drive[0] = 1; numdrives++; }	// drive A
	outp(0x70, 0x10);
	if (inp(0x71) & 0xF) { drive[1] = 1; numdrives++; }	// drive B

	// Now check which hard disk partitions exist by touching them 1 at a time
	for (count=2; count<26; count++) {
		sprintf(s,"%c:\\", count+'A');
		if (stat(s, &statbuf) != 0) drive[count] = 0;
		else {
			drive[count] = 1;
			numdrives++;
		}
	}

	textbox(8,16,16,17+numdrives, 1,"Drives");

	textbox(59,16,79,31,2,"Help");
	setfgcolor(1);
	eputstr(61,18,"   Sort by File");
	eputstr(61,19,"   Sort by Name");
	eputstr(61,20,"   Sort by Size 1");
	eputstr(61,21,"   Sort by Size 2");
	eputstr(61,23,"To choose a drive");
	eputstr(61,24,"just press the");
	eputstr(61,25,"appropriate letter");
	eputstr(61,26,"ie.  `C` = C:\\");
	eputstr(61,28,"Arrows/page up/dn");
	eputstr(61,29,"to move cursor");
	setfgcolor(4);
	eputstr(61,18,"F1");
	eputstr(61,19,"F2");
	eputstr(61,20,"F3");
	eputstr(61,21,"F4");

	textbox(25,11,54,49,2,"FileList");

restart:
	setfgcolor(1);

	sprintf(s,"%dkb", memfree()/1024); eputstr(72,2,s);

	// first get the size of directory, and then allocate memory for it
	dirlen =0;
	done = _dos_findfirst("*.*", _A_NORMAL | _A_SUBDIR, &ffblk);
	if (done) goto nofiles;
	while (!done) {
		dirlen++;
		done = _dos_findnext(&ffblk);
	}

	// Allocate the memory (sorry about the `+100` part, I spent literally
	// a WHOLE solid day looking for a memory corrupting crash bug.. I knew
	// it was something to do with going outside fbuff's limits, but I never
	// pinpointed it.. so here is a hack fix.. just allocate it 1.8kb more
	// memory - it seems to fix it, and 1800 bytes isn't a lot to waste)
	if ((fbuff = (char *)calloc(sizeof(fileblock)*(dirlen+100), 1)) == NULL) {
		return ERR_OUT_OF_MEM;
	}

	// start again, and find the first file and give the wildcard
	dirlen=0;
	numdirs=0;
	done = _dos_findfirst("*.*", _A_NORMAL | _A_SUBDIR, &ffblk);
	while (!done) {
		dirlen++;
		fblock = (fileblock *)(fbuff+ ( sizeof(fileblock)*(dirlen-1) ) );
		strcpy(fblock->name, ffblk.name);
		fblock->size = ffblk.size;
		if (ffblk.attrib & _A_SUBDIR) {
			 fblock->type = 1;
			 numdirs++;
		}
		else fblock->type =0;

		done = _dos_findnext(&ffblk);
	}

	// sort the directory list by the type (subdirs at top, files at bottom)
	qsort((void *)fbuff, dirlen, sizeof(fileblock), sort_by_type);
	// now sort only the subdirectories alphabetically
	qsort((void *)fbuff, numdirs, sizeof(fileblock), sort_by_name);


nofiles:

	setfgcolor(1);
	getcwd(currdir, 100);
	sprintf(s,"%-75s",currdir); eputstr(3,7,s);

	// Draw up 'drives' box on the screen
	for (count=0; count<numdrives; count++) {
		if (drive[count]) {
			setfgcolor(1);
			sprintf(s, "%c:", 'A'+count);
			eputstr(12, 17+count, s);

			if (s[0] == currdir[0]) {
				setfgcolor(4);
				eputch(10, 17+count, '');
				setfgcolor(1);
			}
			else eputch(10, 17+count, ' ');

		}
	}

	do {
		if (kbhit()) key = getkey();
		else key=0;

		if (dirlen) {
			switch (key) {
				case 59 : free(fbuff);	    // sort by file
						  goto restart;
				case 60 : qsort((void *)(fbuff+(numdirs*sizeof(fileblock))),
							 dirlen-numdirs, sizeof(fileblock), sort_by_name);
						  break;
				case 61 : qsort((void *)(fbuff+(numdirs*sizeof(fileblock))),
							 dirlen-numdirs, sizeof(fileblock), sort_by_size1);
						  break;
				case 62 : qsort((void *)(fbuff+(numdirs*sizeof(fileblock))),
							 dirlen-numdirs, sizeof(fileblock), sort_by_size2);
						  break;
				case 73 : posselect-=15;
						  poscursor-=15;

						  if (posselect < 0) posselect =0;

						  if (poscursor < 0) {
							posview += poscursor;
							poscursor = 0;
							if (posview < 0) posview=0;
						  }
						  break;
				case 81 : posselect+=15;
						  poscursor+=15;

						  if (posselect >= (dirlen-1)) posselect = dirlen-1;

						  if (poscursor > 36) {
							posview += (poscursor-37);
							if (posview > (dirlen-37)) posview=dirlen-37;
							poscursor = 36;
						  }
						  if (poscursor >= (dirlen-1)) {
							poscursor = dirlen-1;
							posview = 0;
						  }

						  break;
				case 72 : if (posselect == 0) break;
						  posselect--;
						  poscursor--;
						  if (poscursor < 0) {
							posview --;
							poscursor = 0;
						  }
						  break;
				case 80 : if (posselect == dirlen-1) break;
						  posselect++;
						  poscursor++;
						  if (poscursor == 37) {
							posview++;
							poscursor = 36;
						  }
						  break;
				case 28 : fblock = (fileblock *)					// ENTER
								   (fbuff + (sizeof(fileblock)*(posselect)));
						  if (fblock->type == 1) {	        // if it is a dir
							  chdir(fblock->name);
							  free(fbuff);
							  poscursor=0;
							  posselect=0;
							  posview=0;
							  setbgcolor(7);
							  for (count=0; count<37; count++)
								eputstr(27,count+12,"                           ");
							  goto restart;
						  }
						  else {             // else try to load the filename
							  textbox(26,24,52,26,1,"Message");
							  setfgcolor(1);
							  sprintf(s,"Loading %s...", fblock->name);
							  eputstr(28,25,s);
							  updatescreen();

							  // If a song is currently playing STOP it
							  if (FM.playing) StopMOD();
							  sprintf(s,"%dkb", memfree()/1024);
							  eputstr(72,2,s);
							  updatescreen();

							  // try and load a new song
							  ERR = LoadSong("", fblock->name, 1);

							  // check for errors
							  if (ERR) {
								  textbox(26,24,52,26,1,"Error!");
								  switch (ERR) {
									  case 1 : eputstr(28,25,"No Gravis UltraSound?");
											   break;
									  case 2 : eputstr(28,25,"Module not found.");
											   break;
									  case 3 : eputstr(28,25,"Unknown format.");
											   break;
									  case 4 : eputstr(28,25,"Out of GUS DRAM!");
											   break;
									  case 5 : eputstr(28,25,"Not enough memory!");
											   break;
									  case 6 : eputstr(28,25,"File corrupt.");
											   break;
									  default : eputstr(28,25,"Unexpected error.");
											   break;
								  }
								  sprintf(s,"%dkb", memfree()/1024);
								  eputstr(72,2,s);
								  updatescreen();
								  delay(1000);
								  ERR=0;
								  break;
							  }
							  else {
								  GUSReset(FM.channels);
								  PlayMOD();
								  techindex=1;
								  techcount=1;
								  free(fbuff);
								  return 0;				// call up interface
							  }
						  }
			};
		}

		// keys QWERTYUIOP keys ASDFGHJKL and keys ZXCVBNM
		// for CHANGING TO A NEW DRIVE
		if ((key >= 16 && key <=25) || (key >= 30 && key <=38) ||
			(key >= 44 && key <=50)) {

			if (key >= 16 && key <=25) done = keytodrive[key-16];
			if (key >= 30 && key <=38) done = keytodrive[key-20];
			if (key >= 44 && key <=50) done = keytodrive[key-25];

			// prepare program to trap floppy disk errors
			// (DOESNT WORK IN PMODE - STILL GET ABORT,RETRY,FAIL)
			olderrorhandler = _dos_getvect(0x24);
			_dos_setvect(0x24, errorhandler);

			// try and change to specified disk
			if (drive[done-1]) {
again:
				sprintf(s,"%c:\\", count+'A');
				if (stat(s, &statbuf) != 0) goto again;
				else _dos_setdrive(done, &maxdrives);
			}
			// restore old floppy disk error handler
			_dos_setvect(0x24, olderrorhandler);

			if (dirlen) free(fbuff);
			poscursor=0;
			posselect=0;
			posview=0;
			setbgcolor(7);
			for (count=0; count<37; count++)
				eputstr(27,count+12,"                           ");
			goto restart;
		}

		if (dirlen) {
			// Update the file list box
			for (count=0; count<37; count++) {
				if (count < dirlen) {
					fblock = (fileblock *)
							 (fbuff + (sizeof(fileblock) * (posview+count)));

					if (poscursor == count) {
						setfgcolor(1);
						sprintf(s,"%6dkb", fblock->size/1024); eputstr(52,2, s);
						sprintf(s,"%-15s", fblock->name); eputstr(10,2,s);
						setbgcolor(4);
					}
					else setbgcolor(7);
					if (fblock->type == 1) setfgcolor(2);
					else setfgcolor(1);

					sprintf(s," %-13s    ", fblock->name); eputstr(27,count+12, s);
					sprintf(s,"%8ld", fblock->size); eputstr(45,count+12, s);
					setbgcolor(7);
					eputch(26,count+12,' ');
					eputch(53,count+12,' ');
				}
				else eputstr(26,count+12,"                            ");
			}
		}

		// update technicality index if mod is playing
		if (FM.playing) {
			for (count=0; count<FM.channels; count++) {
				if (!pause) {
						techcount+=2;
						if (restart[count]) techindex++;
						if (geffect[count]) techindex++;
				}
			}
		}

		// refresh the screen
		waitvrt();
		updatescreen();

	} while (key!=1);

	free(fbuff);
	return 0;		// no errors
}


/****************************************************************************
 *    Name : Interface                                                      *
 * Purpose : The main interface screen.  Everything starts from here usually*
 *  Passed : UBYTE *quit - address of quit value so it can change on return *
 * Returns : error code                                                     *
 *  Locals : not commenting all that either                                 *
 ****************************************************************************/
UBYTE Interface(UBYTE *quit) {
	UBYTE count, min,sec,key, panrow=0, fullchan[32], pandir[32];
	UBYTE r=64,g=64,b=64,glow=0,sampleview=0, pcycle=0;
	UDWORD messcount=0;
	char notetab[24], *effstr, *message;

	// allocate memory for pointers (ALWAYS allocate memory for pointers, no
	// matter how tiny they are.. unallocated pointers always cause trouble)
startagain:
	if ((s = (char *)calloc(100,1))==NULL) return ERR_OUT_OF_MEM;
	if ((pattern = (char *)calloc((6U+(FM.channels*22))*66,1))==NULL) {
		free(s);
		return ERR_OUT_OF_MEM;
	}
	if ((effstr = (char *)calloc(800,1))==NULL) {
		free(s);
		free(pattern);
		return ERR_OUT_OF_MEM;
	}
	if ((message = (char *)calloc(2000,1))==NULL) {
		free(s);
		free(pattern);
		free(effstr);
		return ERR_OUT_OF_MEM;
	}

	// hook up keyboard handler for F12 - reset speed function
	oldkeyhandler = _dos_getvect(0x9);  // back up the old keyboard's handler
	_dos_setvect(0x9, keyhandler);      // now set our new one on the keyboard

	// initialize default values
	memset(fullchan, 0, 32);
	for (count=0; count<32; count++) {
		if (panval[count]<0x80) pandir[count]=0;
		else pandir[count]=1;
	}
	strcpy(notetab, "C-C#D-D#E-F-F#G-G#A-A#B-");
	strcpy(effstr, "                  Porta Up          Porta Down        Porta to Note     Vibrato           Porta+Vol Slide   Vibrato+Vol Slide Tremolo           Set Pan           Sample Offset     Volume Slide      Pattern Jump      Set Volume        Pattern Break                       Set Speed         Set Speed         Volume Slide      Porta Down        Porta Up          Tremor            Vibrato+Vol Slide Porta+Vol Slide   Retrig+Vol Slide  Fine Vibrato      Global Volume     Stereo Control    (Set Amiga Filter)Fine Porta Up     Fine Porta Down   (Glissando Ctrl)  Set Vib Waveform  Set Fine Tune     Pattern Loop      Set Trem Waveform Set 16 Pos PanningRetrig Note       Fine VolSlide Up  Fine VolSlide DownCut Note          Delay Note        Pattern Delay     (Invert Loop)     Arpeggio          ");

	strcpy(message,"                                                             FireMOD 1.06.,");
	strcat(message," CopyRight (c) FireLight, 1995...........  (Press TAB to shuttle this");
	strcat(message," text forward/pause will pause interface)  Moo this player is kool huh");
	strcat(message," hehehe...  Hey I like this tune....  Dig those GUS clicks.  Well here");
	strcat(message," it is, the new version of FMOD you have all been waiting for!   (yeah");
	strcat(message," right :)... Anyway it has a HEAP of new stuff and I am now reasonably");
	strcat(message," happy with it as a player.  It has quite a few new features including");
	strcat(message," s3m/mtm/far/669 etc support, and the new tracker screen, info screen,");
	strcat(message," file selector, etc.  It uses about 8500 lines of C, and is quite a lot");
	strcat(message," faster than the old version, especially in the interface, going on that");
	strcat(message," I had to slow this scroller down about 4 times just to keep it at the");
	strcat(message," same speed :)..   The purpose of this player is to serve as example");
	strcat(message," code for the FMODDOC 2 tutorial package.  It explains HOW");
	strcat(message," to actually code a mod player from scratch, not leaving out any detail.");
	strcat(message,"  The main document, FMODDOC.TXT, is now over 110 kilobytes of text!  No");
	strcat(message," other document *even comes NEAR*, and there's free player code!  There");
	strcat(message," are also big texts on S3M, Sound Blaster routines and more!  It goes");
	strcat(message," into everything from the loader to interrupt handlers, and the effects");
	strcat(message," section actually explains HOW to code the effects, not just some vague");
	strcat(message," reference on what they do etc.   For example, the description for the");
	strcat(message," vibrato command, or effect 4xy, Is about 9 pages long!  Where else have");
	strcat(message," you seen such detailed coding information?  Get it from wherever you got");
	strcat(message," this player as FMODDOC2.ZIP. If you want to contact me, about my music or");
	strcat(message," this player, then do so by the means displayed in the help, or from the");
	strcat(message," info when you press escape to quit");
	strcat(message,".. bye!                                                                                                ");

	// find out where to position the sample information on the screen
	// according to how many channels there are.
	if (FM.channels > 11) sampleview=FM.channels+10;
	else sampleview=19;

	// allocate memory for offscreen buffer (for the interface)
	screensize = (sampleview+FM.numinsts) * 160;
	if (screensize < 8000) screensize = 8000;         // we want all 80x50 chars
	if ((screen = (char *)calloc(screensize,1))==NULL) {
		free(s);
		free(pattern);
		free(effstr);
		free(message);
		return ERR_OUT_OF_MEM;
	}

	// set text mode
	set8050mode();

	// speed up keystroke rate
	setkeyrate();

	if (*quit == 33) {
		key = 33;
		goto redokey;
	}
redraw:
	clearscreen();
	setfgcolor(3);
	setbgcolor(0);
	if (sampleview+FM.numinsts < 50) eputstr(76,50,"gL0w");

	setrgbpalette(0,8,16,24);               // color 0
	setrgbpalette(1,0,0,14);                // color 1
	setrgbpalette(2,10,10,63);              // color 2
	setrgbpalette(3,21,42,63);              // color 3
	setrgbpalette(4,32,5,5);                // color 4
	setrgbpalette(5,5,10,20);               // color 5
	setrgbpalette(7,42,42,42);              // color 7
	setrgbpalette(56, 16, 8, 63);           // color 8
	setrgbpalette(57, 16, 16, 63);          // color 9
	setrgbpalette(60, 42,16, 24);           // color 12
	setrgbpalette(61,32,0,0);               // color 13
	setrgbpalette(63,63,63,63);             // color 15

	// set up sample info window
	setbgcolor(0);
	setfgcolor(7);
	for (count=0; count<FM.numinsts; count++) {
		sprintf(s,"%02u    %-28s %2u %5u %6ld %5ld %5ld%7ld %7ld",
				count+1, inst[count]->name, inst[count]->volume,
				(UWORD)inst[count]->middlec, inst[count]->length,
				inst[count]->loopstart,inst[count]->loopend,
				inst[count]->offset,
				inst[count]->offset+inst[count]->length);
		eputstr(2,count+sampleview, s);
	}
	setfgcolor(15);
	setbgcolor(5);
	eputstr(2,sampleview-1,"S# C# Sample name                  Vol midC   Len  LoopB LoopE GUSBegn GUSEnd");

	// set up playtime window
	setbgcolor(0);
	setfgcolor(1);
	textsquare(1,1,80,sampleview+FM.numinsts,1);
	textbox(20,5,79,6+FM.channels,1,"Pan # Sample  Vol Note Effect ");
	setfgcolor(15);
	setbgcolor(7);
	for (count=0; count < FM.channels; count++) {
		sprintf(s," %d", count+1); eputstr(24,count+6, s);
	}

	// set up heading window
	textbox(1,1,79,3,2,"FireMOD 1.06. CopyRight (c) FireLight 1995");
	setfgcolor(9);
	eputstr(3, 2, "Name : ");
	eputstr(47,2, "Size : ");
	eputstr(61,2, "Mem Free : ");
	setfgcolor(1);
	eputstr(10,2,FM.name);
	sprintf(s,"%dkb", filelen/1024); eputstr(54,2, s);
	sprintf(s,"%dkb", memfree()/1024); eputstr(72,2,s);

	// set up info window
	textbox(1,5,18,16,1,"Info");
	setbgcolor(7);
	setfgcolor(9);
	eputstr(2,7,"   Type :");
	eputstr(2,8,"  Clock :");
	eputstr(2,9,"  Speed :");
	eputstr(2,10,"  Order :");
	eputstr(2,11,"Pattern :");
	eputstr(2,12,"    Row :");
	eputstr(2,13," Volume :");
	eputstr(2,14,"TechIdx :");
	eputstr(2,15,"MixRate :");
	setfgcolor(1);
	sprintf(s, "%u", divisor); eputstr(11,15,s);
	setfgcolor(13);
	eputstr(11,7,FM.type);

	setbgcolor(0);
	setfgcolor(1);
	for (count=0; count < FM.numinsts; count++)
		eputch(5,count+sampleview, 254);           // draw little light squares.

	setbgcolor(7);
	setfgcolor(4);
	eputch(24,6+select,'');                       // draw channel cursor dot

	do {
		if (kbhit()) key = getkey();
		else key=0;

		// process generic keystrokes that apply to all screens (volume etc)
		if (processkey(key, sampleview)) goto redraw;

		setbgcolor(7);
		// process keys *specific* to this screen
redokey:
		switch (key) {                                              // KEY
			case 15 : messcount+=300;                               // TAB
					  break;
			case 46 : pcycle = ~pcycle;                             // c
					  break;
			case 72 : eputch(24,6+select,' ');                                              // up
					  select--;
					  if (select==255) select=FM.channels-1;
					  setfgcolor(4);
					  eputch(24,6+select,'');
					  break;
			case 80 : eputch(24,6+select,' ');                                              // down
					  select++;
					  if (select==FM.channels) select=0;
					  setfgcolor(4);
					  eputch(24,6+select,'');
					  break;
			case 47 : guidisplay(&key);                                // v
					  if (key == 28) goto redraw;
					  else goto redokey;
			case 20 : trackerscreen(&key);                       	   // t
					  if (key == 28) goto redraw;
					  else goto redokey;
			case 23 : infoscreen(&key);                        		   // i
					  if (key == 28) goto redraw;
					  else goto redokey;
			case 33 : FileSelector();            				       // f

					  if (!FM.playing) {
						key = 1;			// if no song is playing then
						goto redokey;		// tell interface to QUIT
					  }

					  free(screen);     // clean up memory becuase the
					  free(s);          // new tune will probably have
					  free(message);    // totally different attributes
					  free(pattern);    // ie # of channels, # of samples etc
					  free(effstr);
					  _dos_setvect(0x9,oldkeyhandler);

					  quit =0;
					  goto startagain;
		};

		// Work out clock value in terms of minutes and seconds
		sec = (fclock / 1000000L) % 60;
		min = (fclock / 1000000L) / 60;
		if (min > 59) {
			fclock=0;
			sec =0;
			min =0;
		}

		// The glowing text scroller
		if (glow == 0) { g--; b++; if (b>124) glow = 1; }
		if (glow == 1) { b--; r++; if (r>124) glow = 2; }
		if (glow == 2) { r--; g++; if (g>124) glow = 0; }
		setrgbpalette(3, r>>1,g>>1,b>>1);
		setfgcolor(3);
		setbgcolor(0);
		if (sampleview==19) eputstrw(20,17, message+(messcount >> 8), 60);
		else eputstrw(2,sampleview-2, message+(messcount >> 8), 78);
		messcount+= 10;
		if ((messcount >> 8) > strlen(message)-80) messcount=0;

		// PANCYCLE (TM)
		if (pcycle && row!=panrow) {
			for (count=0; count<FM.channels; count++) {
				panrow=row;
				if (pandir[count]) panval[count]+=16;
				else panval[count]-=16;
				if (panval[count]<0) {panval[count]+=32; pandir[count]=1;}
				if (panval[count]>0xFF) {panval[count]-=32; pandir[count]=0;}
				GUSSetBalance(count, panval[count]);
			}
		}

		// print out data for info box.
		setfgcolor(1);
		setbgcolor(7);
		if (pause) { setfgcolor(12); eputstr(11,8,"PAUSE"); setfgcolor(1);}
		else { sprintf(s,"%02d:%02d", min,sec); eputstr(11,8,s); }
		sprintf(s,"%03d/%03d", speed,bpm); eputstr(11,9,s);
		sprintf(s,"%03d/%03d", ord, FM.songlength-1); eputstr(11,10,s);
		sprintf(s,"%02d/%02d", FM.order[ord], FM.numpats); eputstr(11,11,s);
		sprintf(s,"%02d/%02d", row, FM.patlen[FM.order[ord]]);eputstr(11,12,s);
		sprintf(s,"%02d/64", globalvol);   eputstr(11,13,s);
		sprintf(s,"%06.2f%%", ((float)techindex/techcount)*100); eputstr(11,14,s);

		// print channel specific information
		for (count=0; count < FM.channels; count++) {
			if (!pause) {
				techcount+=2;
				if (restart[count]) techindex++;
				if (geffect[count]) techindex++;
			}
			if (!mute[count]) {
				if (freq[count] && (lastins[count] < FM.numinsts)) {
					if (restart[count]) setfgcolor(8);
					else setfgcolor(1);
					setbgcolor(7);
					sprintf(s, "%02X", panval[count]);
					eputstr(22,6+count,s);
					sprintf(s, "%02d", lastins[count]+1);
					eputstr(27,6+count,s);
					eputstrw(30,6+count, inst[lastins[count]]->name,23);

					sprintf(s,"%02d", volume[count]); eputstr(54, 6+count, s);
					eputstrw(57, 6+count, notetab+((lastnot[count]%12)*2),2);
					sprintf(s,"%u", lastnot[count]/12); eputstr(59, 6+count, s);
					eputstrw(61, 6+count, effstr+((geffect[count])*18),18);

					setbgcolor(0);
					setfgcolor(4);
					eputch(5, sampleview+fullchan[count],254);
					eputch(6, sampleview+fullchan[count],' ');
					fullchan[count]=lastins[count];
					setfgcolor(14);
					sprintf(s, "%02d", count+1); eputstr(5,sampleview+fullchan[count],s);
				}
				else {
					setbgcolor(7);
					setfgcolor(1);
					sprintf(s, "%02X", panval[count]);
					eputstr(22, 6+count, s);
					eputstr(27,6+count,"00                         00 C-0");
					eputstrw(61, 6+count, effstr+((geffect[count])*18),18);
				}
			}
			else {
				setfgcolor(4);
				setbgcolor(7);
				eputstr(22,6+count,"--");
				eputstr(27,6+count,"-- --------[MUTE]--------  -- --- ------------------");
			}

		}
		// draw screen buffer to vga.
		updatescreen();
	} while(key!=1 && key!=49);

	// free all allocated memory for interface
	free(s);
	free(message);
	free(screen);
	free(pattern);
	free(effstr);

	// return old keyboard handler to 9h
	_dos_setvect(0x9,oldkeyhandler);

	// reset video mode
	set8025mode();

	// if 'N' was pressed, dont tell main() to exit, but load another tune.
	if (key==49) {
		quit = 0;
		return 0;
	}

	printf("You have just used FireMod 1.06 by FireLight.  I can be contacted below.\r\n");
	printf("Post : Brett Paterson,                 Email : firelght@suburbia.apana.org.au\r\n");
	printf("       48/A Parr st,\r\n");
	printf("       Leongatha, 3953.\r\n");
	printf("       Victoria, Australia\r\n");

	*quit = 1;				// 1 = quit for good
	return 0;               // no errors
}


/****************************************************************************
 *    Name : Main                                                           *
 * Purpose : To go where no man has gone before                             *
 *  Passed : int argc - number of command line arguments                    *
 *           char *argv[] - list of strings, which are the command line args*
 * Returns : -                                                              *
 *  Locals : char *ptr - pointer to contain ULTRASND environment variable   *
 *           struct find_t ffblk - wildcard fileblock structure             *
 *           UBYTE quit - flag whether to quit or not, if not look for      *
 *                        another file to load                              *
 *           UBYTE ERR - convenient little error value to store an errcode  *
 *           WORD done - value to say whether wildcards have finished or not*
 *           char currdir - string containing the current directory         *
 *           unsigned maxdrives - dummy value for getcwd to dump it's max   *
 *                                 number of drives into.. never used       *
 ****************************************************************************/
void main(int argc, char *argv[]) {
	char *ptr;
	struct find_t ffblk;
	UBYTE quit=0, ERR=0;
	WORD done;
	char currdir[100];
	unsigned maxdrives;

	// GUS detection routines
	ptr = getenv("ULTRASND");
	if (ptr == NULL) Error(ERR_NO_ENV);
	else if (sscanf(ptr,"%hx,", &Base)!=1) Error(ERR_NO_ENV);
	printf("Found GUS at Base Port %x, ", Base);
	gusdram = GUSFindMem();
	printf("with %d kb DRAM\r\n", gusdram/1024+1);

	// get current drive+directory
	getcwd(currdir, 100);

	// Command Line routines
	if (argc > 1) {

		// single mod (with autoextension detection - ie. no '*' or '?')
		if ((strchr(argv[1],'\*') == NULL) && (strchr(argv[1],'\?')) == NULL) {
			printf("Loading %s...\r\n", argv[1]);

			ERR = LoadSong("",argv[1], 0);
			if (ERR) Error(ERR);
			GUSReset(FM.channels);
			PlayMOD();
			ERR = Interface(&quit);
			StopMOD();
			if (ERR) Error(ERR);
		}

		// wildcards!
		else {
startagain:
			done = _dos_findfirst(argv[1], _A_NORMAL, &ffblk);
			if (done) Error(ERR_NO_FILE);
			while (!done && !quit) {
				printf("Loading %s...\r\n", ffblk.name);
				ERR = LoadSong(argv[1], ffblk.name, 1);
				if (ERR) {
					if (ERR == ERR_UNKNOWN_FORMAT || ERR == ERR_NO_FILE)
						printf("Error trying to load, skipping...\r\n");
					else Error(ERR);
				}
				else {
					GUSReset(FM.channels);
					techindex=1;
					techcount=1;
					PlayMOD();
					ERR = Interface(&quit);
					StopMOD();
					if (ERR) Error(ERR);
				}
				done = _dos_findnext(&ffblk);
			}
			if (!quit) goto startagain;
		}
	}
	else {
		quit = 33;
		ERR = Interface(&quit);  // giving interface 33 calls up fileselector
		if (ERR) Error(ERR);
		if (FM.playing) StopMOD();
	}

	_dos_setdrive(currdir[0] - 'A' + 1, &maxdrives);
	chdir(currdir);
}
