// This is the header file for the database analysis package
// Robin Abbott, 6/9/91, 16/8/95
// Class for a database

#ifndef DBH		// Prevent multiple includes

#define DBH 1

#define VERSIONMAJOR 2
#define VERSIONMINOR 3

#ifdef _Windows				// Windows Specials
 #ifndef WINDOWS_H
  #include <windows.h>
 #endif
 #ifdef __DLL__				// DLL
  #define EXPORT _export
  #define FD _export
 #else					// Bog Standard Windows
  #define FD far
  #define EXPORT huge
 #endif
#else					// Nope, it's DOS
 #define FD far
 #define EXPORT
#endif

#ifndef FILE		// stdio
 #include <stdio.h>
#endif

// #ifndef streambuf	// Stream
// #include <stream.hpp>
// #endif

#ifndef time_t		// Time
 #include <time.h>
#endif

#ifndef strcmp		// String
 #include <string.h>
 #ifndef strncmpl				// Different Compilers !
  #define strcmpl(a,b) stricmp(a,b)
  #define strncmpl(a,b,n) strnicmp(a,b,n)
 #endif
#endif

// The following define is the number of index cluster nodes stored
// in memory for each index. Each cluster uses 550 bytes in memory
// so the default value (20) uses up to 11k for each index. Reduce
// this if using a  large number of indexes & running out of memory
// at the expense of speed in adding or modifying records

#define MAXCLUSM	20


// Here are the selection types

enum seltype {FIRST=-100, LAST, NEXT, PREVIOUS};	// Selection types
enum errors  {NOFILE=1, NOINDEX, NOMEM, CANTSEL,
	      NOWRUNR, WRFAIL, RECNOTSET, MEMOERR,
	      EXPRERR,NOKEY,NOKEYHIGHER,INVIND,DUPFIELD,
	      INVFIELD,INCOMP,OORD,RECNOTFND,ERRINTREE}; 	// Errors
enum rectype {NOTDEL=0,DEL,ALL,READIND};			// Record Types
enum wrtype  {OVER,NEW};
enum errors2 {DELREC,FLDOOR,DBINV,NODELKEY,NORECSP,
	      NODBSP,INVDB,ERINDW,INVFILE,ERONW,INVSEEK,
	      NOMEMSP,ERONR,INVMEMO};			// User errors
enum dates   {USDATE,UKDATE};

// Operator type substitutions

#define OPINT		1		// Operand type integer
#define OPSTR		2		// Operand type string
#define OPDATE		4
#define OPLOG		8

#define MAXFLD 	255	// Maximum number of fields
#define NOIND	""	// No index for record

// Here are the classes

class EXPORT database
{
 int valid;		 // Status of database
 FILE *dbfp;             // file pointer to database
 FILE *dbtp;		 // file pointer to memo file
 long nrec;              // No. of records
 unsigned int reclen;    // Length of a record
 unsigned int recstart;  	 // Offset of first record
 class EXPORT field ** fielda;	 // Points to 1st field in array of fields
 int nfield;             	 // Number of fields in database
 int maxfieldl;		 // Maximum field length in database
 int change;		 // Database has changed flag
 char *exwork;		 // Holds tokenised expression
 char *memow;		 // Points to memo work space, 0 if no memo fields
 int memowl;		 // Memo work space length in blocks of 512
 struct index *findex;	 	 // Points to 1st index structure
 class  EXPORT record *firstrec; // Points to 1st record using this database

 struct index *getindex(char *name);	   // Get pointer to index by name
 void indsort(FILE *,int,int,int,long,int,void (*progress)(int,long),int order);
 void database::buildtree(FILE *fp,int,int,int,long,long &);

 public:

 char ers[81];		 // Reports the error in an expression
 dates dateformat;	 // Date format in use by database

 class expval *ex;

 database();				// Construct new database
 database(char *name,char *index=0);	// Construct with file+index
 ~database(void);

 int  addindex(char *fname);
 int  addfield(char *name,int type,
	       int ln=1,int rdp=0);	// Add field to new database
 int  addfield(class field *fp);	// Duplicate field to new database
 int  buildindex(char *expr,char *fn,void (*progress)(int,long)=0,int order=250);
					// Build an index
 field *getfield(int n)
       {return(fielda[n-1]);}           // Return field n, 1<=n<=nfield
 field *getfield(char *name);		// Get a pointer to a field by name
 char *getindkey(char *name);		// Get index key
 int getindtype(char *name);		// Get index type, OPINT or OPSTR
 long getnrec(void) {return(nrec);}     // Number of records
 int  getreclen(void) {return(reclen);} // Total record length
 int  getnfield(void) {return(nfield);} // Number of fields
 void getversion(int &j,int &n)
		      {j=VERSIONMAJOR; n=VERSIONMINOR;}		// Version
 int  isvalid(void) {return(valid);}	// Return valid flag
 void setdateformat(dates f) {dateformat=f;}	// Set date format
 int  subindex(char *iname);				// Subtract an index
 int  verifyindex(char *iname,int depth=1,
		  void (*progress)(int test,long recnum)=0,
		  int order=100);			// Verify an index
 int  write(char *name);				// Write new database

 friend class record;			// Let record get at our privates
};

// Class for a field

class EXPORT field
{
 int number;		// Number of this field in the record
 char name[11];         // Name of this field
 char namecop[11];	// Copy for accessor function
 char type;             // Type of field
 unsigned char len;     // Length of field on screen
 int rdp;               // digits to the right of the dp
 int recpos;		// Position of field in record

public:
 field(int number,char *name,char type,int length,int rdp,
       int recpos);
 ~field(void);
                                       // Functions to get privates
 int getnumber(void) {return(number);} // Get number of this field in record
 char *getname(void) {strcpy(namecop,name);
                      return(namecop);}   // Get name of field
 char gettype(void) {return(type);}    // Get type of field
 int getlen(void) {return(len);}       // Get length of field
 int getrdp(void) {return(rdp);}       // Get right of dp of field

 friend class database;                // Let database get at our members
 friend class record;
 friend class expval;
};

// Class for a record

class EXPORT record
{
 char *recbuf;			// Buffer which holds record
 char *recbufo;			// Buffer holding unmodified record
 char *rbp;			// Points to buffer to be used in eval
 char *fieldwork;		// Buffer holds text of last accessed field
 class database *db;		// Points to data base owning this record
 int delstate;			// Holds delete state of record
 long rn;			// Holds record number in dbf file
 index *oi;			// Owning index if one exists
 struct indpt *curind;		// Current index pointer if in use
 int indflg;			// Flag to show checking an index expression
 char lkey[128];		// Last key check expression
 int ltype;			// Last type of key check expression
 int lseltype;			// Last select type, OPINT or OPSTR
 record *nextind;		// Next record using same index as this
 record *next;			// Next record using this database
 record *prev;			// Previous record using this database

 int doset(char *v,field *fld);	// Actually set a field to a string
 long getind(long n);	 	// Find record in index file
 void killind();		// Kill all indclus records in current record
 void delkey(char *o,index *i);	// Delete key value from index
 void inskey(char *n,index *i); // Insert new key value
 void wmemo(int type);		// Write a memo when writing a record

public:

 record(class database &db,char *indname=0);
 ~record(void);
 void operator=(record &s);		// Copy record entirely

 int eval(char *expr,void *result,int &rtype);	// Evaluate expression
 int eval(void *result,int &rtype);		// Evaluate prev. expression
 int indchk(char *exp,int &rtype); 		// check an index expression
 char *indname();				// Return index name
 int seldbf(long n);				// Select record in dbf
 int select(long n,int type=NOTDEL,int df=0);   // Select rec number
 int select(int field,int val,long n,
			     int type=NOTDEL); // Select when fn==val
 int select(int field,char *s,long n,
			     int type=NOTDEL); // Select when field==s
 int select(int field,void *s,
	    int (*comp)(void *,void *),
	    long n,int type=NOTDEL);	// User def sel-> comp()==0
 int select(char *expr,long n,
			     int type=NOTDEL); // Select when expr is true
 char *getfield(int n,int tf=1);	// Get field by number
 char *getfield(char *name,int tf=1);	// Get field by name
 long getrecnum() {return(rn);}	// Current record number
 int  getdelstate() {return(delstate);} // Delete state of record
 void setdelstate(int sval);
 int  selkey(char *value,int type=NOTDEL,int num=OPSTR);   // Select by key
 int  selkey(double value,int type=NOTDEL);
 int  selkey();				// Find other records with key
 int  setfield(char *name,char *value);	// Set new value for field
 int  setfield(char *name,double value);
 int  setfield(int number,char *value);	// Set new value for field
 int  setfield(int number,double value);
 int  write(int type=OVER);		// Write rec to dbf

 friend class expval;	// Let expval get at our privates
 friend class database;	// Let database get at our privates
};


// Miscellaneous routines

void   FD dber(int ern);			  // Print error
void   FD dbfer(int ern);			  // Print fatal error
time_t FD gettime(char *string);	  	  // ANSI time from dbase date
char*  FD getdate(time_t time);	  		  // dbase date from ANSI time
void   FD gnums(char *date,int &d,int &m,int &y); // Date to day,month,year
char*  FD ltrim(char *string);	  		  // Trim lead spaces
char*  FD rtrim(char *string);	  		  // Trim trailing spaces
char*  FD soundex(char *d,char *s);    		  // Soundex of s to d
int    FD strcmpdb(char *s1,char *s2,int len=-1); // dBase compare
char*  FD swapdata(char *s,int c='~'); 		  // Swap data in a string
char*  FD trim(char *string);	     		  // Trim trailing spaces

// The following functions are error checking versions of the standard
// file handling functions

int Fwrite(void *,int,int,FILE*);
int Fread(void *,int,int,FILE*);
int Fgetc(FILE *);
int Fputc(int,FILE *);
int Fseek(FILE *,long,int);

// Global variables

extern int eroff;				  // error off flag

#endif
