/*
 * BCC2GRX  -  Interfacing Borland based graphics programs to LIBGRX
 * Copyright (C) 1993-96 by Hartmut Schirmer
 *
 * This library is copyrighted (see above). It might be used and
 * distributed freely as long as all copyright notices are left
 * intact.
 *
 * You may not distribute any changed versions of BCC2GRX without
 * written permission by Hartmut Schirmer.
 *
 * You are permitted to distribute an application linked with BCC2GRX
 * in binary only, provided that the documentation of the program:
 *
 *    a)   informs the user that BCC2GRX is used in the program, AND
 *
 *    b)   provides the user with the necessary information about
 *         how to obtain BCC2GRX. (i.e. ftp site, etc..)
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Contact :                Hartmut Schirmer
 *                          Feldstrasse 118
 *                  D-24105 Kiel
 *                          Germany
 *
 * e-mail : hsc@techfak.uni-kiel.d400.de
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef DEBUG_TEXT
  #define PTRDIFF(a,b) ( (int)( ((char *)(a)) - ((char *)(b)) ) )
  #include <stdarg.h>
  FILE *debug = NULL;

  static void Printf(const char *fmt, ...) {
    va_list ptr;
    if (debug == NULL)
      debug = fopen("__text__.deb", "wt");
    if (debug == NULL)
      return;
    va_start(ptr, fmt);
    vfprintf( debug, fmt, ptr);
    va_end(ptr);
  }
#endif

typedef unsigned char  uchar;
typedef   signed char  schar;
#ifndef __linux__
typedef unsigned short ushort;
#endif

#include <unistd.h>
#include "bccgrx00.h"

#define FirstUserFont    11
#define LastUserFont     (FirstUserFont+9)
#define FirstGrxFont     (LastUserFont+1)
#define LastGrxFont      (FirstGrxFont+9)
#define NrFonts          (LastGrxFont+1)
#define PreSkip          0x080
#define ZERO2ONE(chrsze) ((chrsze) ? : 1)

#define BITMAP(f) ((f)==DEFAULT_FONT || ((f)>=FirstGrxFont && (f)<=LastGrxFont))

typedef struct {
  uchar width;
  ushort *cmd;
} CharInfo;

static int                      _height;
static int                      _multx, _divx, _multy, _divy;
static int                      usr_multx=1, usr_divx=1,
				usr_multy=1, usr_divy=1;
static int _mult[11] = { 1, 3, 2, 3, 1, 4, 5, 2, 5, 3, 4};
static int _div[11]  = { 1, 5, 3, 4, 1, 3, 3, 1, 2, 1, 1};
static CharInfo                *chi[LastUserFont+1] = { NULL };
static int                      Heights[LastUserFont+1];
static CharInfo                *fntptr = NULL;
static void                    *Fonts[NrFonts];
static struct textsettingstype  TXT;
static GrTextOption             gr_Text;
#ifdef GRX_VERSION
static GrFont *DefaultFonts[11];
#endif

#ifdef __linux__
#include <ctype.h>
static char *strlwr(char *s)
{
  char *p = s;

  while (*p != '\0') {
    *p = tolower(*p);
    ++p;
  }
  return s;
}
#endif

static void do_init(void)
{
  static int Init = FALSE;
  int i;

  if (Init) return;
  for (i=0; i < NrFonts; ++i)
    Fonts[i] = NULL;

#ifdef GRX_VERSION_broken
  for (i=2; i < sizeof(DefaultFonts)/sizeof(GrFont *); ++i)
    DefaultFonts[i] = NULL;
  Fonts[DEFAULT_FONT] =
  DefaultFonts[0] =
  DefaultFonts[1] = &GrFont_PC8x8;
#else
  Fonts[DEFAULT_FONT] = (void *) GrLoadFont("@:pc8x8.fnt");
#endif
  gr_Text.txo_font    = (GrFont *)Fonts[DEFAULT_FONT];
  gr_Text.txo_chrtype = GR_BYTE_TEXT;

  TXT.font      = DEFAULT_FONT;
  TXT.direction = HORIZ_DIR;
  TXT.charsize  = 1;
  TXT.horiz     = LEFT_TEXT;
  TXT.vert      = TOP_TEXT;
  usr_multx = usr_divx =
  usr_multy = usr_divy =
     _multx =    _divx =
     _multy =    _divy = 1;

  Init = TRUE;
}


static int _installgrxfont(int start, int stop, char *name)
{
  do_init();
  while (start < stop && Fonts[start] != NULL) ++start;
  if (start >= stop)
    return grNoFontMem;
#ifdef GRX_VERSION
  if ( name[0] == '@' && name[1] == ':')
    name += 2;
#endif
  Fonts[start] = (void *) GrLoadFont(name);
  if (Fonts[start] == NULL)
    return grFontNotFound;
  return start;
}

/* -------------------------------------------------------------- */
#define PACKED __attribute ((packed))

typedef char FontNameTyp[4];

typedef struct FontFileHeader {
  ushort      header_size PACKED;   /* Version 2.0 Header Format   */
  FontNameTyp font_name PACKED;     /* Font Internal Name          */
  ushort      font_size PACKED;     /* filesize in byte            */
  uchar       font_major PACKED,    /* Driver Version Information  */
	      font_minor PACKED;
  uchar       min_major PACKED,     /* BGI Revision Information    */
	      min_minor PACKED;
} FontFileHeader;

typedef struct FontHeaderTyp {
  char   sig PACKED;            /* SIGNATURE byte                        */
  ushort nchrs PACKED;          /* number of characters in file          */
  char   unused1 PACKED;        /* Currently Undefined                   */
  uchar  firstch PACKED;        /* first character in file               */
  ushort cdefs PACKED;          /* offset to char definitions            */
  uchar  scan_flag PACKED;      /* <> 0 if set is scanable               */
  uchar  org_to_cap PACKED;     /* Height from origin to top of capitol  */
  uchar  org_to_base PACKED;    /* Height from origin to baseline        */
  schar  org_to_dec PACKED;     /* Height from origin to bot of decender */
  uchar  unused2[0x5] PACKED;   /* Currently undefined                   */
} FontHeaderTyp;

static int GetFontInfo(void *Font, CharInfo *fntptr, int *height) {
  int i, LstChar;
  char *cp;
  ushort *Offsets;
  uchar *Widths;
  char *Data;
  FontFileHeader *ffh;
  FontHeaderTyp *fht;

  cp = (char *)Font;
  i = 256;
  while (*cp != '\x1a' ) { /* \x1a = ^Z */
    ++cp;
    --i;
    if (i == 0)   /* Error, no ^Z at end of copyright */
      return FALSE;
  }
  ++cp;
  ffh = (FontFileHeader *)cp;
  fht = (FontHeaderTyp *)((char *)Font + ffh->header_size);
  if (fht->sig != '+')
    return FALSE; /* Magic failed */
  if (fht->scan_flag) {
    /* font may have DO_SCAN op, anything we should do ? */
  }
  Offsets = (ushort *)((char *)fht + sizeof(FontHeaderTyp));
  Widths  = (uchar *)Offsets + 2 * (int)fht->nchrs;
  Data    = (char *)Font + fht->cdefs + ffh->header_size;
  LstChar = fht->firstch + fht->nchrs - 1;

  *height = (int)fht->org_to_cap - (int)fht->org_to_dec;
  for (i=fht->firstch; i <= LstChar; ++i) {
    fntptr[i].width = Widths[i - fht->firstch];
    fntptr[i].cmd   = (ushort *)(Data + Offsets[i - fht->firstch]);
  }
  #ifdef DEBUG_TEXT
    Printf("FileHead at %04x\n", PTRDIFF(ffh, Font));
    Printf("FontHead at %04x\n", PTRDIFF(fht, Font));
    Printf("Offsets  at %04x\n", PTRDIFF(Offsets, Font));
    Printf("Widths   at %04x\n", PTRDIFF(Widths, Font));
    Printf("Data     at %04x\n", PTRDIFF(Data, Font));
    Printf("Last  char    %2x\n",LstChar);
    Printf("\n");
    Printf("header_size %3x\n", ffh->header_size);
    Printf("font_name   ");
    for (i=0; i < sizeof(FontNameTyp); ++i)
      Printf("%c", ffh->font_name[i]);
    Printf("\n");
    Printf("font_size %5x\n", ffh->font_size);
    Printf("driver version %d.%d\n", ffh->font_major,
				    ffh->font_minor);
    Printf("bgi version %d.%d\n", ffh->min_major,
				    ffh->min_minor);
    Printf("\n");
    Printf("sig        : %c\n", fht->sig);
    Printf("nchrs      : %u\n", fht->nchrs);
    Printf("firstch    : %u\n", fht->firstch);
    Printf("cdefs      : %x\n", fht->cdefs);
    Printf("scan_flag  : %d\n", fht->scan_flag);
    Printf("org_to_cap : %u\n", fht->org_to_cap);
    Printf("org_to_base: %u\n", fht->org_to_base);
    Printf("org_to_dec : %d\n", fht->org_to_dec);
  #endif
  return TRUE;
}

static int SetFont(int fnt)
{
  if (Fonts[fnt] == NULL)
    return FALSE;

  if (chi[fnt] == NULL) {
    chi[fnt] = calloc(256,sizeof(CharInfo));
    if (chi[fnt] == NULL)
       return FALSE;
    fntptr = chi[fnt];

    if (!GetFontInfo(Fonts[fnt], fntptr, &Heights[fnt]))
      return FALSE;
  }
  fntptr = chi[fnt];
  _height = Heights[fnt];
  return TRUE;
}

static char *StdFonts[BOLD_FONT+1] = {
  "", "TRIP.CHR", "LITT.CHR", "SANS.CHR", "GOTH.CHR", "SCRI.CHR",
      "SIMP.CHR", "TSCR.CHR", "LCOM.CHR", "EURO.CHR", "BOLD.CHR" };

static int _registerfont( int start, int stop, void *font)
{
  int    i;
  char  *Header;
  char  *name;

  do_init();
  Header = (char *)font + PreSkip;
  if ( memcmp( font, "PK\x8\x8",4)!=0 || *Header != '+')
    return grInvalidFont;

  name = (char *) font;
  while (*name != '\x1a') {
    if ( name-(char *)font > 128)
      return grInvalidFont;
    ++name;
  }
  name += 3;

  for (i=1; i <= BOLD_FONT; ++i)
    if (memcmp( name, StdFonts[i], 4) == 0)
      break;
  if (i > BOLD_FONT) {
    i = start;
    while ( i <= stop && Fonts[i] != NULL)
      ++i;
    if (i > stop)
      return grNoFontMem;
  }
  Fonts[i] = font;
  return i;
}

static int _installfont( int start, int stop, char *name)
{
  FILE *ff;
  long  size;
  void *font;
  int   res;
  char *temp = alloca(strlen(name)+1+4);

#ifdef __linux__
#  define CHG_CHAR '\\'
#  define NEW_CHAR '/'
#else
#  define CHG_CHAR '/'
#  define NEW_CHAR '\\'
#endif

  if (temp != NULL) {
    int have_ext = FALSE;
    strcpy(temp, name);
    name = temp;
    while (*temp != '\0') {
      if (*temp == CHG_CHAR) *temp = NEW_CHAR;
			else *temp = tolower(*temp);
      if (*temp == NEW_CHAR) have_ext = FALSE;
			else have_ext |= *temp == '.';
      ++temp;
    }
    if (!have_ext)
      strcat(name, ".chr");
  }

  ff = fopen( name, "rb");
  if (ff == NULL)
    return grFileNotFound;
  fseek( ff, 0, SEEK_END);
  size = ftell(ff);
  fseek( ff, 0, SEEK_SET);
  font = malloc( (size_t) size);
  if (font == NULL) {
    fclose( ff);
    return grNoFontMem;
  }
  fread( font, (size_t) size, 1, ff);
  fclose( ff);
  res = _registerfont(start, stop, font);
  if (res < 0)
    free( font);
  return res;
}

int registerbgifont(void *font)
{
  return _registerfont( FirstUserFont, LastUserFont, font);
}

int installuserfont(char *name)
{
  char *loc_name;

  loc_name = alloca(strlen(name)+5);
  if (loc_name != NULL) {
    strcpy(loc_name, name);
    strlwr( loc_name);
    if (strstr(loc_name, ".fnt") != NULL)
      return _installgrxfont( FirstGrxFont, LastGrxFont, name);
  }
  return _installfont( FirstUserFont, LastUserFont, name);
}

/* ----------------------------------------------------------------- */

int textheight(char *textstring)
{
  _DO_INIT_CHECK_RV(0);
  do_init();
  if (TXT.font == DEFAULT_FONT)
    return 8*ZERO2ONE(TXT.charsize);
  if (TXT.font >= FirstGrxFont && TXT.font <= LastGrxFont) {
    gr_Text.txo_font = (GrFont *)Fonts[TXT.font];
    gr_Text.txo_direct = GR_TEXT_RIGHT;
    return GrStringHeight(textstring, strlen(textstring), &gr_Text);
  }
  return _height * _multy / _divy;
}

/* ----------------------------------------------------------------- */
int textwidth(char *textstring)
{
  int sum;

  _DO_INIT_CHECK_RV(0);
  do_init();
  if (TXT.font == DEFAULT_FONT)
    return 8*ZERO2ONE(TXT.charsize) * strlen(textstring);
  if (TXT.font >= FirstGrxFont && TXT.font <= LastGrxFont) {
    gr_Text.txo_font = (GrFont *)Fonts[TXT.font];
    gr_Text.txo_direct = GR_TEXT_RIGHT;
    return GrStringWidth(textstring, strlen(textstring), &gr_Text);
  }
  sum = 0;
  while (*textstring != '\0')
    sum += fntptr[*((uchar *)textstring++)].width * _multx / _divx;
  return sum;
}

/* ----------------------------------------------------------------- */
inline static int xoff(ushort x) {
  /* Csaba did it this way, well done */
  return ((int)(signed char)((x) << 1) >> 1);
}

inline static int yoff(ushort y) {
  return - ((int)(signed char)((y) >> 7) >> 1);
}

/* ----------------------------------------------------------------- */
static void _otxt_vec(int *xx, int *yy, int XX, int YY, uchar *textstring)
{
  if (__gr_TextLineStyle)
    LNE.lno_color= COL|WR;
  if (TXT.direction == HORIZ_DIR) {
    int     _XX, x, y, nx, ny, w;
    ushort *dc;

    switch (TXT.horiz) {
      case CENTER_TEXT : XX -= textwidth(textstring) / 2; break;
      case RIGHT_TEXT  : XX -= textwidth(textstring);     break;
      default          : break;
    }
    switch (TXT.vert) {
      case CENTER_TEXT : YY += textheight(textstring) / 2; break;
      case TOP_TEXT    : YY += textheight(textstring);     break;
      default          : break;
    }
    _XX = XX;
    x = y = 0;
    while (*textstring != '\0') {
      w  = fntptr[*textstring].width;
      dc = fntptr[*textstring].cmd;
      while (dc != NULL) {
	switch ( *dc & 0x8080) {
	  case 0x0000 : dc = NULL;
			XX += w * _multx / _divx;
			break;
	  case 0x8000 : /* DO_SCAN op, any font using this ? */
			++dc;
			break;
	  case 0x0080 : x = xoff(*dc) * _multx / _divx;
			y = yoff(*dc) * _multy / _divy;
			++dc;
			break;
	  case 0x8080 : nx = xoff(*dc) * _multx / _divx;
			ny = yoff(*dc) * _multy / _divy;
			if (__gr_TextLineStyle)
			  GrCustomLine( XX+x, YY+y, XX+nx, YY+ny, &LNE);
			else
			  GrLine( XX+x, YY+y, XX+nx, YY+ny, COL);
			x = nx;
			y = ny;
			++dc;
			break;
	}
      }
      ++textstring;
    }
    *xx += XX-_XX;
  } else {
    int     _YY, x, y, nx, ny, w;
    ushort *dc;

    switch (TXT.horiz) {
      case LEFT_TEXT   : XX += textheight(textstring);     break;
      case CENTER_TEXT : XX += textheight(textstring) / 2; break;
      default          : break;
    }
    switch (TXT.vert) {
      case CENTER_TEXT : YY += textwidth(textstring) / 2; break;
      case TOP_TEXT    : YY += textwidth(textstring);     break;
      default          : break;
    }
    _YY = YY;
    x = y = 0;
    while (*textstring != '\0') {
      w  = fntptr[*textstring].width;
      dc = fntptr[*(textstring++)].cmd;
      while (dc != NULL) {
	switch ( *dc & 0x8080) {
	  case 0x0000 : dc = NULL;
			YY -= w * _multx / _divx;
			break;
	  case 0x8000 : /* DO_SCAN op, any font using this ? */
			++dc;
			break;
	  case 0x0080 : y = -xoff(*dc) * _multx / _divx;
			x =  yoff(*dc) * _multy / _divy;
			++dc;
			break;
	  case 0x8080 : ny = -xoff(*dc) * _multx / _divx;
			nx =  yoff(*dc) * _multy / _divy;
			if (__gr_TextLineStyle)
			  GrCustomLine( XX+x, YY+y, XX+nx, YY+ny, &LNE);
			else
			  GrLine( XX+x, YY+y, XX+nx, YY+ny, COL);
			x = nx;
			y = ny;
			++dc;
			break;
	}
      }
    }
    *yy -= YY-_YY;
  }
}

/* ----------------------------------------------------------------- */
static void _otxt_bit(GrFont *fnt, int *xx, int *yy, int XX, int YY, uchar *txt)
{
  int len;

  if ( (gr_Text.txo_font = fnt) == NULL) {
    ERR = grError;
    return;
  }
#ifndef GRX_VERSION
  gr_Text.txo_xmag      =
  gr_Text.txo_ymag      = ZERO2ONE( TXT.charsize);
#endif
  gr_Text.txo_fgcolor.v = COL;
  gr_Text.txo_bgcolor.v = GrNOCOLOR;
  gr_Text.txo_direct    = (TXT.direction == HORIZ_DIR) ? GR_TEXT_RIGHT:GR_TEXT_UP;
  switch (TXT.horiz) {
    case LEFT_TEXT   : gr_Text.txo_xalign = GR_ALIGN_LEFT;   break;
    case RIGHT_TEXT  : gr_Text.txo_xalign = GR_ALIGN_RIGHT;  break;
    case CENTER_TEXT :
    default          : gr_Text.txo_xalign = GR_ALIGN_CENTER; break;
  }
  switch (TXT.vert) {
    case BOTTOM_TEXT : gr_Text.txo_yalign = GR_ALIGN_BOTTOM; break;
    case TOP_TEXT    : gr_Text.txo_yalign = GR_ALIGN_TOP;    break;
    case CENTER_TEXT :
    default          : gr_Text.txo_yalign = GR_ALIGN_CENTER; break;
  }
  len = strlen(txt);
  GrDrawString( txt, len, XX, YY, &gr_Text);
  if (TXT.direction == HORIZ_DIR)
    *xx += GrStringWidth(txt, len, &gr_Text);
  else {
    gr_Text.txo_direct = GR_TEXT_RIGHT;
    *yy -= GrStringWidth(txt,len, &gr_Text);
  }
}

/* ----------------------------------------------------------------- */
static void _outtextxy(int *xx, int *yy, int XX, int YY, uchar *textstring)
{
  _DO_INIT_CHECK;
  do_init();
#ifdef GRX_VERSION
  if (TXT.font==DEFAULT_FONT) {
    if (DefaultFonts[TXT.charsize] == NULL)
      DefaultFonts[TXT.charsize] =
	GrBuildConvertedFont(
	  DefaultFonts[1],
	  GR_FONTCVT_RESIZE,
	  8*ZERO2ONE(TXT.charsize),
	  8*ZERO2ONE(TXT.charsize),
	  0, 0);
    _otxt_bit(DefaultFonts[TXT.charsize],xx,yy,XX,YY,textstring);
  } else
#endif
  if (BITMAP(TXT.font))
    _otxt_bit((GrFont *)Fonts[TXT.font],xx,yy,XX,YY,textstring);
  else
    _otxt_vec(xx,yy,XX,YY,textstring);
}

/* ----------------------------------------------------------------- */
void outtext( char *textstring)
{
  _outtextxy(&X, &Y, X+VL, Y+VT, (uchar *)textstring);
}

/* ----------------------------------------------------------------- */
void outtextxy(int x, int y, char *textstring)
{
  _outtextxy( &x, &y, x+VL, y+VT, (uchar *)textstring);
}

/* ----------------------------------------------------------------- */
void gettextsettings(struct textsettingstype *texttypeinfo)
{
  _DO_INIT_CHECK;
  do_init();
  memcpy( texttypeinfo, &TXT, sizeof(TXT));
}

/* ----------------------------------------------------------------- */
void settextjustify(int horiz, int vert)
{
  _DO_INIT_CHECK;
  do_init();
  TXT.horiz = horiz;
  TXT.vert  = vert;
}

/* ----------------------------------------------------------------- */
void settextstyle(int font, int direction, int charsize)
{
  _DO_INIT_CHECK;
  do_init();
  if (font < DEFAULT_FONT || font >= NrFonts ||
      (font > BOLD_FONT && Fonts[font] == NULL)) {
    ERR = grInvalidFontNum;
    return;
  }
  if (BITMAP(font) && charsize < 1) charsize = 1;
  if (charsize < 0)  charsize =  4; /* 100% */
  if (charsize > 10) charsize = 10;

  if (!BITMAP(font)) {
    if (Fonts[font] == NULL) {
      char fn[256], *cp;

      strcpy(fn, (*__gr_BGICHR != '\0' ? __gr_BGICHR : ".\\"));
      cp = fn;
      while ( *cp != '\0') ++cp;
      if ( *(--cp) != '\\' && *cp != '/') {
	*(++cp) = '\\';
	*(cp+1) = '\0';
      }
      strcat( fn, StdFonts[font]);
      _installfont( font, font, fn);
    }
    if (!SetFont( font)) {
      ERR = grFontNotFound;
      font = DEFAULT_FONT;
      charsize = 1;
    }
  }
  TXT.font         = font;
  TXT.direction    = direction;
  TXT.charsize     = charsize;
  if (charsize == USER_CHAR_SIZE) {
    _multx = usr_multx;
    _divx  = usr_divx;
    _multy = usr_multy;
    _divy  = usr_divy;
  } else {
    _multx = _multy  = _mult[charsize];
    _divx  = _divy   = _div[charsize];
  }
}


/* ----------------------------------------------------------------- */
void setusercharsize(int multx, int divx, int multy, int divy)
{
  _DO_INIT_CHECK;
  do_init();
  if (divx <= 0 || divy <= 0 || multx < 0 || multy < 0 || BITMAP(TXT.font)) {
    ERR = grError;
    return;
  }
  TXT.charsize = USER_CHAR_SIZE;
  _multx = usr_multx = multx;
  _divx  = usr_divx  = divx;
  _multy = usr_multy = multy;
  _divy  = usr_divy  = divy;
}
