/*
-----------------------------------------------------------------------------
int  MedianImage(IMAGE * image,RECT *r,WORD kw,WORD kh,IMAGE * new_image)

Parameters:

   *hpFB - Frame Buffer data
   *r    - Rectangular region to process
   kw - Number of coloums
   kh - Number of rows
   *hpFBdst  - Output Frame Buffer

Description:

Performs a median filtering operation on the image and places the output in
a secondary image buffer.
-----------------------------------------------------------------------------
*/
IMAGE *MedianImage(IMAGE *image)
   {
   WORD      x,y,kw,kh;
   WORD      nPixels, iMedian;
   BYTE      *pixbuf;
   RECT      r,r1,r2;
   IMAGE    *new_image=NULL;

   kw = 3;
   kh = 3;

   // Image must be at least the same size as the kernel.
   if (image->hres < kw || image->vres < kh)
      return(new_image);

	// Create a new image to hold the output data.
	new_image = DuplicateImage(image);
   CopyImageData(image,new_image);
   CopyImagePal(image,new_image);

   // Determine the largest rectangle that can be processed.
   SetRect(&r1,kw/2,kw/2,image->hres-kw/2,image->vres-kw/2);

   // Intersect that with the selected region.
   IntersectRect(&r2,&r1,&image->rc);

   nPixels = (kw*kh);
   iMedian = (kw*kh)/2;

   // Allocate memory for pixel buffer
   pixbuf = (BYTE *) calloc(nPixels,(WORD)sizeof(BYTE));
   if (!pixbuf)
      return(new_image);

   for (y=r2.top; y<r2.bottom; y++)
      for (x=r2.left; x<r2.right; x++)
         {
         // Copy a rectangular kernel of the image
         SetRect(&r,x-kw/2,y-kh/2,x+kw/2+1,y+kh/2+1);
         GetImageRect(image,&r,pixbuf);

         // Quick sort the brightness values into ascending order
			qsort((void *)pixbuf,nPixels,sizeof(BYTE),ByteCompare);

         // Pick out the median or middle value as that for the pixel.
         // and set the new pixel in the target image
			SetImagePixel(new_image,x,y,pixbuf[iMedian]);
         }

   // Free up the pixel value buffer.
   free(pixbuf);

   // Return the new filtered image
   return(new_image);
   }

/*
-----------------------------------------------------------------------------
IMAGE *SobelImage(IMAGE *image,WORD Threshold,BOOL Overlay)

Parameters:

   *hpFB - Frame Buffer data
   *r    - Rectangular region to process
   Threshold - Threshold value
   Overlay   - overlay value
   *hpFBdst  - Output Frame Buffer

Description:

Performs a sobel edge detection operation on the image and places the output
in a secondary image buffer.
-----------------------------------------------------------------------------
*/
IMAGE *SobelImage(IMAGE *image,WORD Threshold,BOOL Overlay)
   {
   BYTE       PtA, PtB, PtC, PtD, PtE, PtF, PtG, PtH, PtI;
   WORD       x,y,x0,y0,x1,y1;
   WORD       LineAEIAveAbove, LineAEIAveBelow, LineAEIMaxDif;
   WORD       LineBEHAveAbove, LineBEHAveBelow, LineBEHMaxDif;
   WORD       LineCEGAveAbove, LineCEGAveBelow, LineCEGMaxDif;
   WORD       LineDEFAveAbove, LineDEFAveBelow, LineDEFMaxDif;
   WORD       MaxDif;
   IMAGE    *new_image=NULL;

   x0 = image->rc.left;
   y0 = image->rc.top;
   x1 = image->rc.right;
   y1 = image->rc.bottom;

	// Create a new image to hold the output data.
	new_image = DuplicateImage(image);
   CopyImageData(image,new_image);
   CopyImagePal(image,new_image);

   // Compensate for edge effects of 3x3 pixel neighborhood.
   if (x0<1) x0 = 1;
   if (y0<1) y0 = 1;
   if (x1>image->hres-2) y1=image->hres-2;
   if (y1>image->vres-2) y1=image->vres-2;

   for (y=y0; y<y1; y++)
      for (x=x0; x<x1; x++)
         {
			// Get each pixel in 3x3 neighborhood.
         GetImagePixel(image,x-1,y-1,&PtA);
         GetImagePixel(image,x  ,y-1,&PtB);
         GetImagePixel(image,x+1,y-1,&PtC);
         GetImagePixel(image,x-1,y  ,&PtD);
         GetImagePixel(image,x  ,y  ,&PtE);
			GetImagePixel(image,x+1,y  ,&PtF);
         GetImagePixel(image,x-1,y+1,&PtG);
         GetImagePixel(image,x  ,y+1,&PtH);
			GetImagePixel(image,x+1,y+1,&PtI);

         // Calculate average above and below the line.
         // Take the absolute value of the difference.
         LineAEIAveBelow = (WORD)(PtD+PtG+PtH)/3;
         LineAEIAveAbove = (WORD)(PtB+PtC+PtF)/3;
         LineAEIMaxDif = abs(LineAEIAveBelow-LineAEIAveAbove);

         LineBEHAveBelow = (WORD)(PtA+PtD+PtG)/3;
         LineBEHAveAbove = (WORD)(PtC+PtF+PtI)/3;
         LineBEHMaxDif = abs(LineBEHAveBelow-LineBEHAveAbove);

         LineCEGAveBelow = (WORD)(PtF+PtH+PtI)/3;
         LineCEGAveAbove = (WORD)(PtA+PtB+PtD)/3;
         LineCEGMaxDif = abs(LineCEGAveBelow-LineCEGAveAbove);

         LineDEFAveBelow = (WORD)(PtG+PtH+PtI)/3;
         LineDEFAveAbove = (WORD)(PtA+PtB+PtC)/3;
         LineDEFMaxDif = abs(LineDEFAveBelow-LineDEFAveAbove);

         // Find the maximum value of the absolute differences
         // from the four possibilities.
         MaxDif = MAX(LineAEIMaxDif,LineBEHMaxDif);
         MaxDif = MAX(LineCEGMaxDif,MaxDif);
         MaxDif = MAX(LineDEFMaxDif,MaxDif);

         /*
         If maximum difference is above the threshold, set
         the pixel of interest (center pixel) to white. If
         below the threshold optionally copy the input image
         to the output image. This copying is controlled by
         the parameter Overlay.
         */

         if (MaxDif >= Threshold)
            SetImagePixel(new_image,x,y,255);
         else if (Overlay)
            SetImagePixel(new_image,x,y,PtE);
         }

   return(new_image);
   }


/*
-----------------------------------------------------------------------------
int  FBArRConvolve(IMAGE * s_image,RECT *r,short *mask,WORD kw,WORD kh,WORD Scale,BOOL av_flag,IMAGE * d_image)

Parameters:

   *hpFB - Frame buffer data
   *r    - rectangular region to process
   *mask - Convolution kernel data
   kw    - width of mask
   kh    - height of mask
   Scale - scaling factor
   av_flag - absolute value flag
   *hpFBdst - target Frame Buffer

Description:

Real Number Convolution Function. This convolution function is only used when
the kernel entries are floating point numbers instead of integers. Because of 
the floating point operations envolved, this function is substantially slower
than the already slow integer version above.
-----------------------------------------------------------------------------
*/
int  FBArRConvolve(IMAGE * s_image,RECT *r,double *mask, WORD kw,
                WORD kh, WORD Scale,BOOL av_flag, IMAGE * d_image)
   {
   BYTE      dn;
   WORD       x,y,w,h,x0,y0,kx, ky;
   RECT      r0;
   double Sum;
   double *mask2;
   BYTE huge *hpFBsrc;
   BYTE huge *hpFBdst;

   if (r->left<0 || r->top<0 ||
       r->right > s_image->hres-1 || r->bottom > s_image->vres-1)   /* Check x,y bounds     */
      return(-1);

   hpFBsrc = (BYTE huge *)GlobalLock(s_image->hData);      /* Lock the IMAGE memory   */
   hpFBdst = (BYTE huge *)GlobalLock(d_image->hData);      /* Lock the IMAGE memory   */

   w = r->right - r->left;
   h = r->bottom - r->top;

   // Image must be at least the same size as the kernel.
   if (w < kw || h < kh)
      return(-1);

   /*
   Clearing the output buffer to white will show the
   boarder areas not touched by the convolution. It also
   provides a nice white frame for the output image.
   */

   SetRect(&r0,0,0,s_image->hres-1,s_image->vres-1);
   SetImageRect(d_image,&r0,255);

   for (y=r->top+kh/2; y<r->bottom-kh/2; y++)
      {
      y0 = y - kh/2;
      for (x=r->left+kw/2; x<r->right-kw/2; x++)
         {
         x0 = x - kw/2;
         Sum = 0.0;
         mask2 = mask;
         for (kx=0; kx<kw; kx++)
            for (ky=0; ky<kh; ky++)
               {
               dn = *(hpFBsrc+(DWORD)(y0+ky)*s_image->scansize+x0+kx);
               Sum += dn * (*mask2++);
               }

         /* If absolute value is requested */
         if (av_flag)
            Sum = fabs(Sum);

         /* Summation performed. Scale and range Sum */
         Sum /= (double)(1<<Scale);
         Sum = (Sum <   0) ?   0 : Sum;
         Sum = (Sum > 255) ? 255 : Sum;

         *(hpFBdst+(DWORD)y*s_image->hres+x) = (BYTE)Sum;
         }
      }

   GlobalUnlock(s_image->hData);                            /* Unlock the IMAGE memory */
   GlobalUnlock(d_image->hData);                            /* Unlock the IMAGE memory */

   return(0);
   }

/*
-------------------------------------------------------------------------------
int  CopyImageData(IMAGE *s_image,IMAGE *d_image)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Copies the source frame buffer data to the destination frame buffer

Parameters:

   s_image - pointer to source image
   d_image - pointer to destination image

Return value:

   none

-------------------------------------------------------------------------------
*/
void CopyImageData(IMAGE *s_image,IMAGE *d_image)
   {
	int  x,y,w,h;
   //BYTE huge *hpDataSrc;
   //BYTE huge *hpDataDst;

	//hpDataSrc = (BYTE huge *)GlobalLock(s_image->hData);    /* Lock the IMAGE memory   */
   //hpDataDst = (BYTE huge *)GlobalLock(d_image->hData);    /* Lock the IMAGE memory   */

   w = (s_image->hres>d_image->hres) ? d_image->hres : s_image->hres;
	h = (s_image->vres>d_image->vres) ? d_image->vres : s_image->vres;
   for (y=0; y<h; ++y)
      CopyImageRow(s_image,d_image,0,y,w);

      //_fmemcpy(MK_HPIMAGE(d_image,hpDataDst,0,y),MK_HPIMAGE(s_image,hpDataSrc,0,y),w);

      /*
		for (x=0; x<w; ++x)
         {
         *(MK_HPIMAGE(d_image,hpDataDst,x,y)) =
         *(MK_HPIMAGE(s_image,hpDataSrc,x,y));
         //*(hpDataDst+(DWORD)(d_image->vres-1-y)*d_image->scansize+x) =
         //*(hpDataSrc+(DWORD)(s_image->vres-1-y)*s_image->scansize+x);
         }
      */

   //GlobalUnlock(s_image->hData);                         /* Unlock the IMAGE memory */
   //GlobalUnlock(d_image->hData);                         /* Unlock the IMAGE memory */
	}

   /*
	int  x,y,w,h;

   w = (s_image->hres>d_image->hres) ? d_image->hres : s_image->hres;
	h = (s_image->vres>d_image->vres) ? d_image->vres : s_image->vres;
   for (y=0; y<h; ++y)
      CopyImageRow(s_image,d_image,0,y,w);
   */



