/* Copyright (C) 1991, 1995 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gxclread.c */
/* Command list reading for Ghostscript. */
#include "memory_.h"
#include "gx.h"
#include "gpcheck.h"
#include "gserrors.h"
#include "gsbitops.h"
#include "gxdevice.h"
#include "gsdevice.h"			/* for gs_deviceinitialmatrix */
#include "gxdevmem.h"			/* must precede gxcldev.h */
#include "gxcldev.h"

/* Imported from gxclist.c */
int clist_flush_buffer(P1(gx_device_clist *));

/* Print a bitmap for tracing */
#ifdef DEBUG
private void
cmd_print_bits(const byte *data, int height, int raster)
{	int i, j;
	for ( i = 0; i < height; i++ )
	   {	const byte *row = data + i * raster;
		dprintf("[L]");
		for ( j = 0; j < raster; j++ )
		  dprintf1(" %02x", row[j]);
		dputc('\n');
	   }
}
#else
#  define cmd_print_bits(data, height, raster)
#endif

/* ------ Reading/rendering ------ */

private int clist_render_init(P1(gx_device_clist *));
private int clist_render(P4(gx_device_clist *, gx_device *, int, gs_memory_t *));
#define clist_read(cf, data, len) clist_fread_chars(data, len, cf)

/* Copy a scan line to the client.  This is where rendering gets done. */
int
clist_get_bits(gx_device *dev, int y, byte *str, byte **actual_data)
{	gx_device_memory *mdev = &cdev->mdev;
	/* Initialize for rendering if we haven't done so yet. */
	if ( cdev->ymin < 0 )
	{	int code = clist_render_init(cdev);
		if ( code < 0 ) return code;
	}
	/* Render a band if necessary, and copy it incrementally. */
	if ( !(y >= cdev->ymin && y < cdev->ymax) )
	   {	int band = y / mdev->height;
		int code;
		clist_rewind(cdev->bfile, false);
		(*dev_proc(mdev, open_device))((gx_device *)mdev);	/* reinitialize */
		/* We have to pick some allocator for rendering.... */
		code = clist_render(cdev, (gx_device *)mdev, band,
				    &gs_memory_default);
		if ( code < 0 )
		  return code;
		cdev->ymin = band * mdev->height;
		cdev->ymax = cdev->ymin + mdev->height;
	   }
	return (*dev_proc(mdev, get_bits))((gx_device *)mdev,
				y - cdev->ymin, str, actual_data);
}

#undef cdev

/* Initialize for reading. */
private int
clist_render_init(gx_device_clist *cdev)
{	gx_device *target = cdev->target;
	byte *base = cdev->mdev.base;	/* save */
	int depth = target->color_info.depth;
	uint raster = gx_device_raster(target, 1);
	const gx_device_memory *mdev =
	  gdev_mem_device_for_bits(depth);
	int code;

	if ( mdev == 0 )
	  return_error(gs_error_rangecheck);
	code = clist_flush_buffer(cdev);		/* flush buffer */
	if ( code < 0 ) return code;
	/* Write the terminating entry in the block file. */
	/* Note that because of copypage, there may be many such entries. */
	   {	cmd_block cb;
		cb.band = cmd_band_end;
		cb.pos = clist_ftell(cdev->cfile);
		clist_fwrite_chars(&cb, sizeof(cb), cdev->bfile);
		cdev->bfile_end_pos = clist_ftell(cdev->bfile);
	   }
	gs_make_mem_device(&cdev->mdev, mdev, 0, 0, target);
	cdev->mdev.base = base;		/* restore */
	/* The matrix in the memory device is irrelevant, */
	/* because all we do with the device is call the device-level */
	/* output procedures, but we may as well set it to */
	/* something halfway reasonable. */
	gs_deviceinitialmatrix(target, &cdev->mdev.initial_matrix);
	cdev->mdev.width = target->width;
	cdev->mdev.height = cdev->band_height;
	cdev->mdev.raster = raster;
	cdev->ymin = cdev->ymax = 0;
#ifdef DEBUG
if ( gs_debug_c('l') )
	{	extern void cmd_print_stats(P0());
		cmd_print_stats();
	}
#endif
	return 0;
}

/* Render one band to a specified target device. */
typedef byte _ss *cb_ptr;
#define cmd_getw(var, p)\
{ int _shift = 0; for ( var = 0; var += (int)(*p & 0x7f) << _shift,\
    *p++ > 0x7f; _shift += 7 ) ;\
}
private cb_ptr cmd_read_rect(P3(int, gx_cmd_rect *, cb_ptr));
private void clist_unpack_short_bits(P4(byte *, int, int, uint));
private int clist_decode_path(P3(gx_path *, stream_cursor_read *, gs_fixed_point *));
private int
clist_render(gx_device_clist *cdev, gx_device *tdev, int band,
  gs_memory_t *mem)
{	byte cbuf[cbuf_size];
		/* 'bits' is for short copy_mono bits and copy_mono_rle, */
		/* must be aligned */
	long bits[max(255, cbuf_size / sizeof(long) + 1)];
	register cb_ptr cbp;
	cb_ptr cb_limit;
	cb_ptr cb_end;
	clist_file_ptr file = cdev->cfile;
	clist_file_ptr bfile = cdev->bfile;
	int y0 = band * cdev->band_height;
	gx_clist_state state;
	gx_tile_bitmap state_tile;
	int state_tile_depth;
	uint tile_bits_size;		/* size of bits of each tile */
	gs_int_point tile_phase;
	gs_fixed_point ppos;
	gx_path path;
	gx_line_params line;
	float flatness;
	cmd_block b_this;
	long pos;
	uint left;
	int code = 0;

#define cmd_read(ptr, rsize, cbp)\
  if ( cb_end - cbp >= (rsize) )\
    memcpy(ptr, cbp, rsize), cbp += rsize;\
  else\
   { uint cleft = cb_end - cbp, rleft = (rsize) - cleft;\
     memcpy(ptr, cbp, cleft);\
     clist_read(file, ptr + cleft, rleft); left -= rleft;\
     cbp = cb_end;\
   }
#define cmd_read_short_bits(ptr, bw, ht, ras, cbp)\
  cmd_read(ptr, (bw) * (ht), cbp);\
  clist_unpack_short_bits(ptr, bw, ht, ras)

	state = cls_initial;
	state_tile.id = gx_no_bitmap_id;
	tile_phase.x = tile_phase.y = 0;
	ppos.x = fixed_0;
	ppos.y = int2fixed(-y0);
	gx_path_init(&path, mem);
	{ static const gx_line_params line_initial =
	    { gx_line_params_initial };
	  line = line_initial;
	}
	flatness = 1.0;
trd:	clist_read(bfile, &b_this, sizeof(b_this));
top:	/* Find the next run of commands for this band. */
	if ( b_this.band == cmd_band_end &&
	     clist_ftell(bfile) == cdev->bfile_end_pos
	   )
	  return 0;		/* end of bfile */
	if ( !(b_this.band == band || b_this.band == cmd_band_all) )
	  goto trd;
	pos = b_this.pos;
	clist_read(bfile, &b_this, sizeof(b_this));
	clist_fseek(file, pos, SEEK_SET);
	left = (uint)(b_this.pos - pos);
	cb_limit = cbuf + (cbuf_size - cmd_largest_size);
	cb_end = cbuf + cbuf_size;
	cbp = cb_end;
	while ( !code )
	   {	int op;
		uint bytes, tile_index;
		int data_x, raster;
		cb_ptr source;
		gx_color_index colors[2];
		gx_color_index _ss *pcolor;

		/* Make sure the buffer contains a full command. */
		/* We can get here with cb_limit reset to force reading, */
		/* so we must be sure to recompute cb_limit. */
		if ( cbp > cb_limit )
		{	uint nread;
			memmove(cbuf, cbp, cb_end - cbp);
			cbp = cbuf + (cb_end - cbp);
			nread = cb_end - cbp;
			if ( nread > left ) nread = left;
			clist_read(file, cbp, nread);
			cb_end = cbp + nread;
			cbp = cbuf;
			left -= nread;
			cb_limit = cbuf + (cbuf_size - cmd_largest_size);
			if ( cb_limit > cb_end )
			  cb_limit = cb_end;
			process_interrupts();
		}
		op = *cbp++;
#ifdef DEBUG
		if ( gs_debug_c('L') )
		  { if ( op < 0x10 )
		      dprintf1("[L]%s:\n", cmd_misc_op_names[op]);
		    else
		      dprintf2("[L]%s %d:\n",
			       cmd_op_names[op >> 4], op & 0xf);
		  }
#endif
		switch ( op >> 4 )
		   {
		case cmd_op_misc >> 4:
			switch ( op )
			   {
			case cmd_opv_end_run:
				goto top;
			case cmd_opv_set_tile_size:
				state_tile_depth = 1;
sts:				cmd_getw(state_tile.size.x, cbp);
				cmd_getw(state_tile.size.y, cbp);
				cmd_getw(state_tile.rep_width, cbp);
				cmd_getw(state_tile.rep_height, cbp);
				state_tile.raster =
				  bitmap_raster(state_tile.size.x *
						state_tile_depth);
				cdev->tile_slot_size = tile_bits_size =
				  state_tile.raster * state_tile.size.y;
				break;
			case cmd_opv_set_tile_size_colored:
				state_tile_depth = cdev->color_info.depth;
				goto sts;
			case cmd_opv_set_tile_phase:
				cmd_getw(state.tile_phase.x, cbp);
				cmd_getw(state.tile_phase.y, cbp);
				break;
			case cmd_opv_set_tile_index:
				tile_index = *cbp++;
sti:				state_tile.data =
				  (byte *)tile_slot_ptr(cdev, tile_index);
				if_debug1('L', "[L]  tile_index = %u\n",
					  tile_index);
				continue;
			case cmd_opv_set_tile_index_long:
				cmd_getw(tile_index, cbp);
				goto sti;
			case cmd_opv_set_tile_bits:
				tile_index = *cbp++;
stb:				state_tile.data =
				  (byte *)tile_slot_ptr(cdev, tile_index);
				if_debug1('L', "[L]  tile_index = %u\n",
					  tile_index);
			    {	int bwidth =
				  (state_tile.rep_width *
				   state_tile_depth + 7) >> 3;
				if ( bwidth <= 2 )
				  {	cmd_read_short_bits(state_tile.data,
					  bwidth, state_tile.rep_height,
					  state_tile.raster, cbp);
				  }
				else
				  {	bytes = tile_bits_size;
					cmd_read(state_tile.data, bytes, cbp);
				  }
			    }
				if ( state_tile.size.x > state_tile.rep_width )
				  bits_replicate_horizontally(state_tile.data,
				    state_tile.rep_width,
				    state_tile.rep_height, state_tile.raster,
				    state_tile.size.x, state_tile.raster);
				if ( state_tile.size.y > state_tile.rep_height )
				  bits_replicate_vertically(state_tile.data,
				    state_tile.rep_height, state_tile.raster,
				    state_tile.size.y);
#ifdef DEBUG
if ( gs_debug_c('L') )
				cmd_print_bits(state_tile.data,
					       state_tile.size.y,
					       state_tile.raster);
#endif
				continue;
			case cmd_opv_set_tile_bits_long:
				cmd_getw(tile_index, cbp);
				goto stb;
			case cmd_opv_set_lop:
				cmd_getw(state.lop, cbp);
				continue;
			case cmd_opv_set_tile_color0:
				pcolor = &state.tile_colors[0];
				goto set_c;
			case cmd_opv_set_tile_color1:
				pcolor = &state.tile_colors[1];
				goto set_c;
			case cmd_opv_enable_lop:
				state.lop_enabled = 1;
				continue;
			case cmd_opv_disable_lop:
				state.lop_enabled = 0;
				continue;
			case cmd_opv_set_line_width:
				{ float width;
				  memcpy(&width, cbp, sizeof(float));
				  gx_set_line_width(&line, width);
				}
				cbp += sizeof(float);
				continue;
			case cmd_opv_set_cap_join:
				line.cap = (gs_line_cap)(*cbp >> 4);
				line.join = (gs_line_join)(*cbp & 0xf);
				continue;
			case cmd_opv_set_flatness:
				memcpy(&flatness, cbp, sizeof(float));
				cbp += sizeof(float);
				continue;
			default:
				goto bad_op;
			   }
			tile_phase.x = state.tile_phase.x % state_tile.size.x;
			tile_phase.y = (state.tile_phase.y + y0) % state_tile.size.y;
			continue;
		case cmd_op_set_color0 >> 4:
			pcolor = &state.colors[0];
			goto set_color;
		case cmd_op_set_color1 >> 4:
			pcolor = &state.colors[1];
set_color:		switch ( op & 0xf )
			 {
			 case 0:
			   break;
			 case 8:	/* special handling because this may */
					/* require more bits than depth */
			   *pcolor = gx_no_color_index;
			   continue;
			 default:
			   *pcolor += (gx_color_index)(long)((op & 0xf) - 8);
			   continue;
			 }
set_c:			{ gx_color_index color = 0;
			  switch ( (cdev->color_info.depth + 7) >> 3 )
			    {
			    case 4: color |= (gx_color_index)*cbp++ << 24;
			    case 3: color |= (gx_color_index)*cbp++ << 16;
			    case 2: color |= (gx_color_index)*cbp++ << 8;
			    case 1: color |= (gx_color_index)*cbp++;
			    }
			  *pcolor = color;
			}
			continue;
		case cmd_op_copy_mono >> 4:
			if ( op & 0xf )
			   {	cmd_getw(state.rect.x, cbp);
				cmd_getw(state.rect.y, cbp);
				state.rect.width = *cbp++;
				state.rect.height = *cbp++;
				break;
			   }
			/* falls through */
		case cmd_op_fill_rect >> 4:
		case cmd_op_tile_rect >> 4:
		case cmd_op_copy_mono_rle >> 4:
		case cmd_op_copy_color >> 4:
		case cmd_op_copy_alpha >> 4:
			cbp = cmd_read_rect(op, &state.rect, cbp);
			break;
		case cmd_op_fill_rect_short >> 4:
		case cmd_op_tile_rect_short >> 4:
			state.rect.x += *cbp + cmd_min_short;
			state.rect.width += cbp[1] + cmd_min_short;
			if ( op & 0xf )
			   {	state.rect.height += (op & 0xf) + cmd_min_tiny;
				cbp += 2;
			   }
			else
			   {	state.rect.y += cbp[2] + cmd_min_short;
				state.rect.height += cbp[3] + cmd_min_short;
				cbp += 4;
			   }
			break;
		case cmd_op_fill_rect_tiny >> 4:
		case cmd_op_tile_rect_tiny >> 4:
		   {	int txy = *cbp++;
			state.rect.x += (txy >> 4) + cmd_min_tiny;
			state.rect.y += (txy & 0xf) + cmd_min_tiny;
			state.rect.width += (op & 0xf) + cmd_min_tiny;
		   }	break;
		case cmd_op_delta_tile_bits >> 4:
		   {	byte *new_data;
			cmd_getw(tile_index, cbp);
			if_debug1('L', "[L]  tile_index = %u\n", tile_index);
			new_data = (byte *)tile_slot_ptr(cdev, tile_index);
			memcpy(new_data, state_tile.data, tile_bits_size);
			state_tile.data = new_data;
			do
			   {	uint offset = *cbp;
				if ( offset < 0x80 )
				  { if_debug4('L',
					      "[L]  [%3u] %s%02x   %02x\n",
					      offset, (offset & 1 ? "  " : ""),
					      new_data[offset],
					      new_data[offset] ^ cbp[1]);
				    new_data[offset] ^= cbp[1];
				    cbp += 2;
				  }
				else
				  { offset -= 0x80;
				    if_debug5('L', "[L]  [%3u] %02x%02x %02x%02x\n",
					      offset,
					      new_data[offset],
					      new_data[offset + 1],
					      new_data[offset] ^ cbp[1],
					      new_data[offset + 1] ^ cbp[2]);
				    new_data[offset] ^= cbp[1];
				    new_data[offset + 1] ^= cbp[2];
				    cbp += 3;
				  }
			   }
			while ( op-- & 0xf );
		   }	continue;
		case cmd_op_delta_tile_index >> 4:
			state_tile.data +=
			  ((op & 0xf) - 8) * cdev->tile_slot_size;
			if_debug1('L', "[L]  tile_index = %d\n",
				  (state_tile.data - cdev->tile_data ) /
				   cdev->tile_slot_size);
			continue;
		case cmd_op_path >> 4:
		  {	uint nbytes = *cbp;
			stream_cursor_read cr;
			if ( cb_end - cbp < nbytes + 1 )
			  { /* Read the entire path into the buffer. */
			    cb_limit = cbp - 1;
			    continue;
			  }
			cr.limit = (cr.ptr = cbp) + nbytes;
			code = clist_decode_path(&path, &cr, &ppos);
			if ( code >= 0 )
			  switch ( op )
			    {
			    case cmd_opv_append:
				break;
			    case cmd_opv_fill:
			    case cmd_opv_eofill:
			    case cmd_opv_stroke:
			    case cmd_opv_clip:
			    case cmd_opv_eoclip:
				/****** NOT IMPLEMENTED YET ******/
				goto bad_op;
			    }
			if ( code < 0 )
			  goto out;
			cbp = (cb_ptr)cr.limit + 1;
		  }	break;
		default:
bad_op:			lprintf5("Bad op %02x band %d file pos %ld buf pos %d/%d\n",
				 op, band, clist_ftell(file), (int)(cbp - cbuf), (int)(cb_end - cbuf));
			   {	cb_ptr pp;
				for ( pp = cbuf; pp < cb_end; pp += 10 )
				  { lprintf1("%4d:", (int)(pp - cbuf));
				    lprintf10(" %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
					pp[0], pp[1], pp[2], pp[3], pp[4],
					pp[5], pp[6], pp[7], pp[8], pp[9]);
				  }
			   }
			code = gs_note_error(gs_error_Fatal);
			goto out;
		   }
		if_debug4('L', "[L]  x=%d y=%d w=%d h=%d\n",
			  state.rect.x, state.rect.y, state.rect.width,
			  state.rect.height);
		switch ( op >> 4 )
		   {
		case cmd_op_fill_rect >> 4:
		case cmd_op_fill_rect_short >> 4:
		case cmd_op_fill_rect_tiny >> 4:
			if ( !state.lop_enabled )
			  { code = (*dev_proc(tdev, fill_rectangle))
			      (tdev, state.rect.x, state.rect.y - y0,
			       state.rect.width, state.rect.height,
			       state.colors[1]);
			    break;
			  }
			source = NULL;
			data_x = 0;
			raster = 0;
			colors[0] = colors[1] = state.colors[1];
			pcolor = colors;
do_rop:			code = (*dev_proc(tdev, copy_rop))
			  (tdev, source, data_x, raster, gx_no_bitmap_id,
			   pcolor, &state_tile,
			   (state.tile_colors[0] == gx_no_color_index &&
			    state.tile_colors[1] == gx_no_color_index ?
			    NULL : state.tile_colors),
			   state.rect.x, state.rect.y - y0,
			   state.rect.width, state.rect.height,
			   tile_phase.x, tile_phase.y, state.lop);
			break;
		case cmd_op_tile_rect >> 4:
		case cmd_op_tile_rect_short >> 4:
		case cmd_op_tile_rect_tiny >> 4:
			/* Currently we don't use lop with tile_rectangle. */
			code = (*dev_proc(tdev, tile_rectangle))
			  (tdev, &state_tile,
			   state.rect.x, state.rect.y - y0,
			   state.rect.width, state.rect.height,
			   state.tile_colors[0], state.tile_colors[1],
			   tile_phase.x, tile_phase.y);
			break;
		case cmd_op_copy_mono >> 4:
			if ( op & 0xf )
			   {	data_x = (op & 0xf) - 1;
				raster = align_bitmap_mod;
				cmd_read_short_bits((byte *)bits,
				  (data_x + state.rect.width + 7) >> 3,
				  state.rect.height, align_bitmap_mod, cbp);
				source = (byte *)bits;
				goto copy;
			   }
			/* falls through */
		case cmd_op_copy_mono_rle >> 4:
		case cmd_op_copy_color >> 4:
		case cmd_op_copy_alpha >> 4:
			cmd_getw(data_x, cbp);
			cmd_getw(raster, cbp);
			bytes = state.rect.height * raster;
			/* copy_mono[_rle], copy_color and copy_alpha */
			/* ensure that the bits will fit in a single buffer. */
#ifdef DEBUG
			if ( bytes > cbuf_size )
			{	lprintf2("bitmap size exceeds buffer!  raster=%d height=%d\n",
					raster, state.rect.height);
				code = gs_note_error(gs_error_ioerror);
				goto out;
			}
#endif
			if ( op >> 4 == (byte)cmd_op_copy_mono_rle >> 4 )
			  {	/* Decompress the image data. */
				stream_RLD_state sstate;
				stream_cursor_read r;
				stream_cursor_write w;
				/* We don't know the data length a priori, */
				/* so to be conservative, we read */
				/* the lesser of the uncompressed size and */
				/* the amount left in this command run. */
				uint cleft = cb_end - cbp;
				if ( cleft < bytes && left != 0 )
				  {	uint nread = cbuf_size - cleft;
					memmove(cbuf, cbp, cleft);
					if ( nread > left ) nread = left;
					clist_read(file, cbuf + cleft, nread);
					left -= nread;
					cb_end = cbuf + cleft + nread;
					cbp = cbuf;
				  }
				sstate.EndOfData = true;
				s_RLD_init_inline(&sstate);
				r.ptr = cbp - 1;
				r.limit = cb_end - 1;
				w.ptr = (byte *)bits - 1;
				w.limit = w.ptr + sizeof(bits);
				/* The process procedure can't fail. */
				(*s_RLD_template.process)
				  ((stream_state *)&sstate, &r, &w, true);
				cbp = (cb_ptr)r.ptr + 1;
				source = (byte *)bits;
			  }
			else
			  {	cmd_read(cbuf, bytes, cbp);
				source = cbuf;
			  }
copy:
#ifdef DEBUG
if ( gs_debug_c('L') )
   {			dprintf2("[L]  data_x=%d raster=%d\n",
				 data_x, raster);
			cmd_print_bits(source, state.rect.height, raster);
   }
#endif
			if ( state.lop_enabled )
			  { /****** WRONG IF copy_alpha ******/
			    pcolor =
			      (op >> 4 == (byte)cmd_op_copy_color >> 4 ?
			       NULL : state.colors);
			    goto do_rop;
			  }
			switch ( op >> 4 )
			  {
			  case (byte)cmd_op_copy_mono >> 4:
			  case (byte)cmd_op_copy_mono_rle >> 4:
			    code = (*dev_proc(tdev, copy_mono))
			      (tdev, source, data_x, raster, gx_no_bitmap_id,
			       state.rect.x, state.rect.y - y0,
			       state.rect.width, state.rect.height,
			       state.colors[0], state.colors[1]);
			    break;
			  case (byte)cmd_op_copy_color >> 4:
			    code = (*dev_proc(tdev, copy_color))
			      (tdev, source, data_x, raster, gx_no_bitmap_id,
			       state.rect.x, state.rect.y - y0,
			       state.rect.width, state.rect.height);
			    break;
			  case (byte)cmd_op_copy_alpha >> 4:
			    code = (*dev_proc(tdev, copy_alpha))
			      (tdev, source, data_x, raster, gx_no_bitmap_id,
			       state.rect.x, state.rect.y - y0,
			       state.rect.width, state.rect.height,
			       state.colors[1], op & 0xf);
			    break;
			  default:	/* can't happen */
			    goto bad_op;
			  }
			break;
		default:		/* can't happen */
			goto bad_op;
		   }
	   }
	/* Clean up before we exit. */
out:	gx_path_release(&path);
	if ( code < 0 )
	  return_error(code);
	else
	  return code;
}

/* Unpack a short bitmap */
private void
clist_unpack_short_bits(byte *data, register int bwidth, int height,
  uint raster)
{	uint bytes = bwidth * height;
	byte *pdata = data + bytes;
	byte *udata = data + height * raster;
	while ( --height > 0 )		/* first row is in place already */
	   {	udata -= raster, pdata -= bwidth;
		switch ( bwidth )
		   {
		case 3: udata[2] = pdata[2];
		case 2: udata[1] = pdata[1];
		case 1: udata[0] = pdata[0];
		   }
	   }
}

/* Read a rectangle. */
private cb_ptr
cmd_read_rect(int op, register gx_cmd_rect *prect, register cb_ptr cbp)
{	cmd_getw(prect->x, cbp);
	if ( op & 0xf )
	  prect->y += ((op >> 2) & 3) - 2;
	else
	  { cmd_getw(prect->y, cbp);
	  }
	cmd_getw(prect->width, cbp);
	if ( op & 0xf )
	  prect->height += (op & 3) - 2;
	else
	  { cmd_getw(prect->height, cbp);
	  }
	return cbp;
}

/* Decode a command list path. */
private int
clist_decode_path(gx_path *ppath, stream_cursor_read *pr, gs_fixed_point *ppos)
{	const byte *p = pr->ptr;
	const byte *end = pr->limit;
	fixed px = ppos->x, py = ppos->y;
	int code = 0;

	while ( p < end )
	   {	fixed vs[6];
#define A vs[0]
#define B vs[1]
#define C vs[2]
#define D vs[3]
#define E vs[4]
#define F vs[5]
		int opcode = *++p;
		int i;

		for ( i = 0; i < clist_path_op_num_operands[opcode]; ++i )
		   {	fixed v;
			int b = p[1];
			if ( b < 0x80 )
			  v = (b ^ 0x40) - 0x40, p += 3;
			else if ( b < 0xc0 )
			  v = (((b ^ 0xa0) - 0x20) << 8) + p[2], p += 4;
			else
			  { v = (p[2] ^ 0x80) - 0x80;
			    p += 2;
			    for ( b = 0; b < sizeof(fixed) - 3; ++b )
			      v = (v << 8) + *++p;
			    p += 2;
			  }
			vs[i] = (v << 16) + (p[-1] << 8) + *p;
		   }
		switch ( opcode )
		   {
		case cp_rmoveto:
			code = gx_path_add_point(ppath, px += A, py += B);
			break;
		case cp_rlineto:
			code = gx_path_add_line(ppath, px += A, py += B);
			break;
		case cp_hlineto:
			code = gx_path_add_line(ppath, px += A, py);
			break;
		case cp_vlineto:
			code = gx_path_add_line(ppath, px, py += A);
			break;
		case cp_rrcurveto:  /* a b c d => a b a+c b+d a+c+e b+d+f */
			E += (C += A);
			F += (D += B);
curve:			code = gx_path_add_curve(ppath, px + A, py + B,
						 px + C, py + D,
						 px + E, py + F);
			px += E, py += F;
			break;
		case cp_hvcurveto:	/* a b c d => a 0 a+b c a+b c+d */
			F = C + D, D = C, E = C = A + B, B = 0;
			goto curve;
		case cp_vhcurveto:	/* a b c d => 0 a b a+c b+d a+c */
			E = B + D, F = D = A + C, C = B, B = A, A = 0;
			goto curve;
		case cp_ccurveto:	/* a b c d => a b a b a+c b+d */
			F = B + D, E = A + C, D = B, C = A;
			goto curve;
		case cp_closepath:
			code = gx_path_close_subpath(ppath);
			gx_path_current_point(ppath, ppos);
			px = ppos->x, py = ppos->y;
			break;
		default:
			return_error(gs_error_rangecheck);
		   }
		if ( code < 0 )
		  break;
#undef A
#undef B
#undef C
#undef D
#undef E
#undef F
	   }
	pr->ptr = p;
	ppos->x = px, ppos->y = py;
	return code;
}
