/* Copyright (C) 1992, 1995 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gp_mswin.c */
/*
 * Microsoft Windows 3.n platform support for Ghostscript.
 * Original version by Russell Lang and Maurice Castro with help from
 * Programming Windows, 2nd Ed., Charles Petzold, Microsoft Press;
 * initially created from gp_dosfb.c and gp_itbc.c 5th June 1992.
 */

/* Modified for Win32 & Microsoft C/C++ 8.0 32-Bit, 26.Okt.1994 */
/* by Friedrich Nowak                                           */

#include "stdio_.h"
#include "string_.h"
#include "memory_.h"
#include <stdlib.h>
#include <stdarg.h>
#include "ctype_.h"
#include "dos_.h"
#include <io.h>
#include "malloc_.h"
#include <fcntl.h>
#include <signal.h>
#include "gx.h"
#include "gp.h"
#include "gpcheck.h"
#include "gserrors.h"
#include "gsexit.h"
#include "gxdevice.h"

#include "windows_.h"
#if WINVER >= 0x030a
#include <shellapi.h>
#ifdef __WIN32__
#include <winspool.h>
#endif
#endif
#include "gp_mswin.h"
#include "gp_mswtx.h"
#ifdef __DLL__
#include "gsdll.h"
/* use longjmp instead of exit when using DLL */
#include <setjmp.h>
extern jmp_buf gsdll_env;
#endif

/* for imitation pipes */
#include "stream.h"
#include "gxiodev.h"			/* must come after stream.h */

/* Library routines not declared in a standard header */
extern char *getenv(P1(const char *));

/* Imported from gp_msdos.c */
int gp_file_is_console(P1(FILE *));

/* ------ from gnuplot winmain.c plus new stuff ------ */

/* limits */
#define MAXSTR 255

/* public handles */
HWND hwndtext;
HINSTANCE phInstance;

const LPSTR szAppName = "Ghostscript";
const LPSTR szImgName = "Ghostscript Image";
#ifdef __WIN32__
const LPSTR szIniName = "gswin32.ini";
#else
const LPSTR szIniName = "gswin.ini";
#endif
const LPSTR szIniSection = "Text";
BOOL is_win31 = FALSE;
BOOL is_winnt = FALSE;
#define is_win95 is_winnt
char FAR win_prntmp[MAXSTR];	/* filename of PRN temporary file */
private int is_printer(const char *name);
int win_init = 0;		/* flag to know if gp_exit has been called */
int win_exit_status;

/* gsview.exe */
BOOL gsview = FALSE;
HWND gsview_hwnd = NULL;
BOOL gsview_next = FALSE;
LPSTR gsview_option = "-sGSVIEW=";

/* redirected stdio */
TW textwin;	/* text window structure */

/* imitation pipes */
HGLOBAL pipe_hglobal = NULL;
LPBYTE pipe_lpbyte = NULL;
UINT pipe_count = 0;
#ifdef __WIN32__
HANDLE pipe_hmapfile;
LPBYTE pipe_mapptr;
BOOL pipe_eof;
#endif

BOOL CALLBACK _export AbortProc(HDC, int);

#ifdef __DLL__
#ifdef __WIN32__
BOOL
DllInit(HANDLE hInst, DWORD fdwReason, LPVOID lpReserved)
{
	phInstance = hInst;
	return TRUE;
}


#else
int WINAPI _export
LibMain(HINSTANCE hInstance, WORD wDataSeg, WORD wHeapSize, LPSTR lpszCmdLine)
{
	phInstance = hInstance;
	return 1;
}

int WINAPI _export
WEP(int nParam)
{
	return 1;
}
#endif
#else
int main(int argc, char *argv[]);

/* our exit handler */
/* also called from Text Window WM_CLOSE */
void win_exit(void)
{
#if WINVER >= 0x030a
	/* disable Drag Drop */
	if (is_win31)
		DragAcceptFiles(hwndtext, FALSE);
#endif
	/* if we didn't exit through gs_exit() then do so now */
	if (win_init)
		gs_exit(0);

	fcloseall();
	if (win_exit_status) {
		/* display message box so error messages in hwndtext can be read */
		char buf[20];
		if (IsIconic(textwin.hWndText))
		    ShowWindow(textwin.hWndText, SW_SHOWNORMAL);
		BringWindowToTop(textwin.hWndText);  /* make hwndtext visible */
		sprintf(buf, "Exit code %d\nSee text window for details",win_exit_status);
		MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	}
	
	/* tell gsview that we are closing */
	if (gsview)
		SendMessage(gsview_hwnd, WM_GSVIEW, GSWIN_CLOSE, (LPARAM)NULL);

	TextClose(&textwin);
#if !defined(_MSC_VER)
	exit(win_exit_status);
#endif
}

int PASCAL 
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdLine, int cmdShow)
{
	int i;
#if defined(_MSC_VER)    /* MSC doesn't give us _argc and _argv[] so ...   */
#define MAXCMDTOKENS 128
	int	_argc=0;
	LPSTR	_argv[MAXCMDTOKENS];
#ifdef __WIN32__
	_argv[_argc] = "gswin32.exe";
#else
	_argv[_argc] = "gswin.exe";
#endif
	_argv[++_argc] = _fstrtok( lpszCmdLine, " ");
	while (_argv[_argc] != NULL)
		_argv[++_argc] = _fstrtok( NULL, " ");
#endif
	is_win31 = FALSE;
	is_winnt = FALSE;
	{
	DWORD version = GetVersion();
	if (((LOBYTE(LOWORD(version))<<8) | HIBYTE(LOWORD(version))) >= 0x30a)
		is_win31 = TRUE;
#ifdef __WIN32__
	if ((HIWORD(version) & 0x8000) == 0)
		is_winnt = TRUE;
	/* treat Windows 95 (Windows 4.0) the same as Windows NT */
	if (((LOBYTE(LOWORD(version))<<8) | HIBYTE(LOWORD(version))) >= 0x400)
		is_winnt = TRUE;
#endif
	}

	if (hPrevInstance) {
		MessageBox((HWND)NULL,"Can't run twice", szAppName, MB_ICONHAND | MB_OK);
		return FALSE;
	}

	/* copy the hInstance into a variable so it can be used */
	phInstance = hInstance;

	/* start up the text window */
	textwin.hInstance = hInstance;
	textwin.hPrevInstance = hPrevInstance;
	textwin.nCmdShow = cmdShow;
	textwin.Title = szAppName;
	textwin.hIcon = LoadIcon(hInstance, (LPSTR)MAKEINTRESOURCE(TEXT_ICON));
	textwin.DragPre = "(";
	textwin.DragPost = ") run\r";
	textwin.ScreenSize.x = 80;
	textwin.ScreenSize.y = 80;
	textwin.KeyBufSize = 2048;
	textwin.CursorFlag = 1;	/* scroll to cursor after \n & \r */
	textwin.shutdown = win_exit;
	GetPrivateProfileString(szIniSection, "FontName", 
	    "", textwin.fontname, sizeof(textwin.fontname), szIniName);
	textwin.fontsize = GetPrivateProfileInt(szIniSection, "FontSize", 
	    0, szIniName);
	if ( (textwin.fontname[0] == '\0') || (textwin.fontsize < 8) ) {
	  char buf[16];
#ifdef __WIN32__
	  if (is_winnt) {
	    /* NT 3.5 doesn't have Terminal font so set a different default */
	    strcpy(textwin.fontname, "Courier New");
	    textwin.fontsize = 10;
	  }
	  else
#endif
	  {
	    strcpy(textwin.fontname, "Terminal");
	    textwin.fontsize = 9;
	  }
	  /* save text size */
          WritePrivateProfileString(szIniSection, "FontName", textwin.fontname,
	    szIniName);
	  sprintf(buf, "%d", textwin.fontsize);
	  WritePrivateProfileString(szIniSection, "FontSize", buf, szIniName);
	}

	if (TextInit(&textwin))
		exit(1);
	hwndtext = textwin.hWndText;
#ifndef _MSC_VER
	(void) atexit((atexit_t)win_exit); /* setup exit handler */
#else
	(void) atexit(win_exit);
#endif
	/* check if we are to use gsview.exe */
	for (i=0; i<_argc; i++) {
	    if (!strncmp(_argv[i], gsview_option, strlen(gsview_option))) {
		gsview_hwnd = (HWND)atoi(_argv[i]+strlen(gsview_option));
		if (gsview_hwnd != (HWND)NULL) {
			if (!IsWindow(gsview_hwnd)) {
				char buf[80];
				sprintf(buf,"GSVIEW window handle %u is invalid",(int)gsview_hwnd);
				MessageBox(hwndtext, buf, szAppName, MB_OK);
				return 0;
			}
			gsview = TRUE;
			/* give gsview the handle to our text window */
			SendMessage(gsview_hwnd, WM_GSVIEW, HWND_TEXT, (LPARAM)hwndtext);
		}
	    }
	}

	main(_argc, _argv);

	/* never reached */
	win_exit(); 
	return win_exit_status;
}

#endif /* !__DLL__ */

BOOL CALLBACK _export
AbortProc(HDC hdcPrn, int code)
{
    process_interrupts();
#ifdef __DLL__
    if (code == SP_OUTOFDISK)
	return (FALSE);	/* cancel job */
#endif
    return(TRUE);
}
  
/* ------ Process message loop ------ */
/*
 * Check messages and interrupts; return true if interrupted.
 * This is called frequently - it must be quick!
 */
int
gp_check_interrupts(void)
{
#ifndef __DLL__
	TextMessage();
#endif
	return 0;
}

/* ====== Generic platform procedures ====== */

/* ------ Initialization/termination (from gp_itbc.c) ------ */

/* Do platform-dependent initialization. */
void
gp_init(void)
{
	win_init = 1;
}

/* Do platform-dependent cleanup. */
void
gp_exit(int exit_status, int code)
{
	win_init = 0;
	win_exit_status = exit_status;
}

/* Exit the program. */
void
gp_do_exit(int exit_status)
{
#if defined(__DLL__)
	/* Use longjmp since exit would terminate caller */
	/* setjmp code will check gs_exit_status */
	longjmp(gsdll_env, gs_exit_status);
#else
	exit(exit_status);
#endif
}

/* ------ Printer accessing ------ */
  
/* Forward references */
private int gp_printfile(P2(const char *, const char *));

/* Open a connection to a printer.  A null file name means use the */
/* standard printer connected to the machine, if any. */
/* Return NULL if the connection could not be opened. */
FILE *
gp_open_printer(char *fname, int binary_mode)
{
	if (is_printer(fname)) 
	{	FILE *pfile;
		pfile = gp_open_scratch_file(gp_scratch_file_name_prefix, 
			win_prntmp, "wb");
		return pfile;
	}
	else
		return fopen(fname, (binary_mode ? "wb" : "w"));
}

/* Close the connection to the printer. */
void
gp_close_printer(FILE *pfile, const char *fname)
{
char buf[128];
	fclose(pfile);
	if (!is_printer(fname))
	    return;		/* a file, not a printer */

	gp_printfile(win_prntmp, fname);
	unlink(win_prntmp);
}

/* Printer abort procedure and progress/cancel dialog box */
/* Used by Win32 and mswinprn device */

HWND hDlgModeless;

BOOL CALLBACK _export
PrintAbortProc(HDC hdcPrn, int code)
{
    MSG msg;
    while (hDlgModeless && PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
	if (hDlgModeless || !IsDialogMessage(hDlgModeless,&msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
	}
    }
    return(hDlgModeless!=0);
}

/* Modeless dialog box - Cancel printing */
BOOL CALLBACK _export
CancelDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message) {
	case WM_INITDIALOG:
	    SetWindowText(hDlg, szAppName);
	    return TRUE;
	case WM_COMMAND:
	    switch(LOWORD(wParam)) {
		case IDCANCEL:
		    DestroyWindow(hDlg);
		    hDlgModeless = 0;
		    EndDialog(hDlg, 0);
		    return TRUE;
	    }
    }
    return FALSE;
}

#ifndef __WIN32__

/* Windows does not provide API's in the SDK for writing directly to a */
/* printer.  Instead you are supposed to use the Windows printer drivers. */
/* Ghostscript has its own printer drivers, so we need to use some API's */
/* that are documented only in the Device Driver Adaptation Guide */
/* that comes with the DDK.  Prototypes taken from DDK <print.h> */
DECLARE_HANDLE(HPJOB);

HPJOB	WINAPI OpenJob(LPSTR, LPSTR, HPJOB);
int	WINAPI StartSpoolPage(HPJOB);
int	WINAPI EndSpoolPage(HPJOB);
int	WINAPI WriteSpool(HPJOB, LPSTR, int);
int	WINAPI CloseJob(HPJOB);
int	WINAPI DeleteJob(HPJOB, int);
int	WINAPI WriteDialog(HPJOB, LPSTR, int);
int	WINAPI DeleteSpoolPage(HPJOB);

#endif		/* WIN32 */

/* Dialog box to select printer port */
BOOL CALLBACK _export
SpoolDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
LPSTR entry;
    switch(message) {
	case WM_INITDIALOG:
	    entry = (LPSTR)lParam;
	    while (*entry) {
		SendDlgItemMessage(hDlg, SPOOL_PORT, LB_ADDSTRING, 0, (LPARAM)entry);
		entry += lstrlen(entry)+1;
	    }
	    SendDlgItemMessage(hDlg, SPOOL_PORT, LB_SETCURSEL, 0, (LPARAM)0);
	    return TRUE;
	case WM_COMMAND:
	    switch(LOWORD(wParam)) {
		case SPOOL_PORT:
#ifdef WIN32
		    if (HIWORD(wParam)
#else
		    if (HIWORD(lParam)
#endif
			               == LBN_DBLCLK)
			PostMessage(hDlg, WM_COMMAND, IDOK, 0L);
		    return FALSE;
		case IDOK:
		    EndDialog(hDlg, 1+(int)SendDlgItemMessage(hDlg, SPOOL_PORT, LB_GETCURSEL, 0, 0L));
		    return TRUE;
		case IDCANCEL:
		    EndDialog(hDlg, 0);
		    return TRUE;
	    }
    }
    return FALSE;
}

/* return TRUE if queue looks like a valid Win32 printer name */
private int
is_win32_spool(const char *queue)
{
char *prefix = "\\\\spool";  /* 7 characters long */
int i;
    for (i=0; i<7; i++) {
	if (prefix[i] == '\\') {
	    if ((*queue != '\\') && (*queue != '/'))
	        return FALSE;
	}
	else if (tolower(*queue) != prefix[i])
	    return FALSE;
	queue++;
    }
    if (*queue && (*queue != '\\') && (*queue != '/'))
	return FALSE;
    return TRUE;
}


private int 
is_printer(const char *name)
{
char buf[128];
    /* is printer if no name given */
    if (strlen(name) == 0)
	return TRUE;

    /*  is printer if name appears in win.ini [ports] section */
    GetProfileString("ports", name, "XYZ", buf, sizeof(buf));
    if ( strlen(name) == 0 || strcmp(buf,"XYZ"))
	return TRUE;

#ifdef __WIN32__
    /* is printer if name prefixed by \\spool\ */
    if ( (is_winnt || is_win95) && is_win32_spool(name) )
	return TRUE;
#endif

    return FALSE;
}
 
#ifdef __WIN32__		/* ******** WIN32 ******** */

/******************************************************************/
/* Print File to port or queue */
/* port==NULL means prompt for port or queue with dialog box */

/* This is messy because Microsoft changed the spooler interface */
/* between Window 3.1 and Windows 95/NT */
/* and didn't provide the spooler interface in Win32s */

/* Win95, WinNT: Use OpenPrinter, WritePrinter etc. */
private int gp_printfile_win32(const char *filename, char *port);

/* Win32s: write direct to port */ 
private int gp_printfile_port(const char *filename, const char *port);


/* valid values for pmport are:
 *   ""
 *      action: WinNT and Win95 use default queue, Win32s prompts for port
 *   "LPT1:" (or other port that appears in win.ini [ports]
 *      action: fwrite to port, with cancel dialog box
 *   "\\spool\printer name"     WinNT and Win95 only
 *      action:  send to printer using WritePrinter.
 *   "\\spool"                  WinNT and Win95 only
 *      action:  prompt for queue name then send to printer using WritePrinter.
 *   
/* Print File */
private int
gp_printfile(const char *filename, const char *pmport)
{
	/* treat WinNT and Win95 differently */
        if (is_winnt || is_win95) {
	    if (strlen(pmport)==0) { /* get default printer */
		char buf[256];
		char *p;
		/* should really use registry to obtain the default printer */
	        GetProfileString("windows", "device", "", buf, sizeof(buf));
		if ( (p = strchr(buf, ',')) != NULL )
		    *p = '\0';
	        return gp_printfile_win32(filename, buf);
	    }
	    else if (is_win32_spool(pmport))  {
	 	if (strlen(pmport) >= 8)
	            return gp_printfile_win32(filename, (char *)pmport+8);
		else
	            return gp_printfile_win32(filename, (char *)NULL);
	    }
	    else
	        return gp_printfile_port(filename, pmport);
	}
	else
	    return gp_printfile_port(filename, pmport);
}

#define PRINT_BUF_SIZE 16384u
#define PORT_BUF_SIZE 4096

char *
get_queues(void)
{
int i;
DWORD count, needed;
PRINTER_INFO_1 *prinfo;
char *enumbuffer;
char *buffer;
char *p;
    /* enumerate all available printers */
    EnumPrinters(PRINTER_ENUM_CONNECTIONS | PRINTER_ENUM_LOCAL, NULL, 1, NULL, 0, &needed, &count);
    enumbuffer = malloc(needed);
    if (enumbuffer == (char *)NULL)
	return FALSE;
    if (!EnumPrinters(PRINTER_ENUM_CONNECTIONS | PRINTER_ENUM_LOCAL, NULL, 1, (LPBYTE)enumbuffer, needed, &needed, &count)) {
	char buf[256];
	free(enumbuffer);
	sprintf(buf, "EnumPrinters() failed, error code = %d", GetLastError());
	MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	return NULL;
    }
    prinfo = (PRINTER_INFO_1 *)enumbuffer;
    if ((buffer = malloc(PORT_BUF_SIZE)) == (char *)NULL) {
	free(enumbuffer);
	return NULL;
    }
    /* copy printer names to single buffer */
    p = buffer;
    for (i=0; i<count; i++) {
	if (strlen(prinfo[i].pName) + 1 < (PORT_BUF_SIZE- (p-buffer))) {
	    strcpy(p, prinfo[i].pName);
	    p += strlen(p) + 1;
	}
    }
    *p = '\0';	/* double null at end */
    free(enumbuffer);
    return buffer;
}


char *
get_ports(void)
{
char *buffer;
#ifdef __WIN32__
	if (is_win95 || is_winnt)
	    return get_queues();
#endif

	if ((buffer = malloc(PORT_BUF_SIZE)) == (char *)NULL)
	    return NULL;
	GetProfileString("ports", NULL, "", buffer, PORT_BUF_SIZE);
	return buffer;
}

/* return TRUE if queuename available */
/* return FALSE if cancelled or error */
/* if queue non-NULL, use as suggested queue */
BOOL
get_queuename(char *portname, const char *queue)
{
char *buffer;
char *p;
int i, iport;

    buffer = get_queues();
    if ( (queue == (char *)NULL) || (strlen(queue)==0) ) {
	/* select a queue */
	iport = DialogBoxParam(phInstance, "QueueDlgBox", hwndtext, SpoolDlgProc, (LPARAM)buffer);
	if (!iport) {
	    free(buffer);
	    return FALSE;
	}
	p = buffer;
	for (i=1; i<iport && strlen(p)!=0; i++)
	    p += lstrlen(p)+1;
	/* prepend \\spool\ which is used by Ghostscript to distinguish */
	/* real files from queues */
	strcpy(portname, "\\\\spool\\");
	strcat(portname, p);
    }
    else {
	strcpy(portname, "\\\\spool\\");
	strcat(portname, queue);
    }

    free(buffer);
    return TRUE;
}

/* return TRUE if portname available */
/* return FALSE if cancelled or error */
/* if port non-NULL, use as suggested port */
BOOL
get_portname(char *portname, const char *port)
{
char *buffer;
char *p;
int i, iport;
char filename[MAXSTR];
	buffer = get_ports();
	if ( (port == (char *)NULL) || (strlen(port)==0) ) {
	    if (buffer == (char *)NULL)
		return NULL;
	    /* select a port */
	    iport = DialogBoxParam(phInstance, "SpoolDlgBox", hwndtext, SpoolDlgProc, (LPARAM)buffer);
	    if (!iport) {
	        free(buffer);
	        return FALSE;
	    }
	    p = buffer;
	    for (i=1; i<iport && strlen(p)!=0; i++)
	        p += lstrlen(p)+1;
	    strcpy(portname, p);
	}
	else
	    strcpy(portname, port);

	if (strlen(portname) == 0)
	    return FALSE;
	if (strcmp(portname,"FILE:") == 0) {
	    OPENFILENAME ofn;
	    filename[0] = '\0';
	    memset(&ofn, 0, sizeof(OPENFILENAME));
	    ofn.lStructSize = sizeof(OPENFILENAME);
	    ofn.hwndOwner = hwndtext;
	    ofn.lpstrFile = filename;
	    ofn.nMaxFile = sizeof(filename);
	    ofn.Flags = OFN_PATHMUSTEXIST;
	    if (!GetSaveFileName(&ofn)) {
	        free(buffer);
	        return FALSE;
	    }
	    strcpy(portname, filename);
	}
	free(buffer);
	return TRUE;
}


/* True Win32 method, using OpenPrinter, WritePrinter etc. */
private int 
gp_printfile_win32(const char *filename, char *port)
{
DWORD needed;
DWORD count;
char *buffer;
char portname[MAXSTR];
FILE *f;
HANDLE printer;
DOC_INFO_1 di;
DWORD written;

    if (!get_queuename(portname, port))
	return FALSE;
    port = portname + 8;	/* skip over \\spool\ */

    if ((buffer = malloc(PRINT_BUF_SIZE)) == (char *)NULL)
        return FALSE;
	
    if ((f = fopen(filename, "rb")) == (FILE *)NULL) {
	free(buffer);
	return FALSE;
    }

    /* open a printer */
    if (!OpenPrinter(port, &printer, NULL)) {
	char buf[256];
	sprintf(buf, "OpenPrinter() failed for \042%s\042, error code = %d", port, GetLastError());
	MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	free(buffer);
	return FALSE;
    }
    /* from here until ClosePrinter, should AbortPrinter on error */

    di.pDocName = szAppName;
    di.pOutputFile = NULL;
    di.pDatatype = "RAW";  /* for available types see EnumPrintProcessorDatatypes */
    if (!StartDocPrinter(printer, 1, (LPBYTE)&di)) {
	char buf[256];
	sprintf(buf, "StartDocPrinter() failed, error code = %d", GetLastError());
	MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	AbortPrinter(printer);
	free(buffer);
	return FALSE;
    }
   
    /* copy file to printer */
    while ((count = fread(buffer, 1, PRINT_BUF_SIZE, f)) != 0 ) {
	if (!WritePrinter(printer, (LPVOID)buffer, count, &written)) {
	    free(buffer);
	    fclose(f);
	    AbortPrinter(printer);
	    return FALSE;
	}
    }
    fclose(f);
    free(buffer);

    if (!EndDocPrinter(printer)) {
	char buf[256];
	sprintf(buf, "EndDocPrinter() failed, error code = %d", GetLastError());
	MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	AbortPrinter(printer);
	return FALSE;
    }

    if (!ClosePrinter(printer)) {
	char buf[256];
	sprintf(buf, "ClosePrinter() failed, error code = %d", GetLastError());
	MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
	return FALSE;
    }
    return TRUE;
}


/* Open port as a file */
/* Works under Win32 */
/* With minor changes would work under Win16 not needed */
private int
gp_printfile_port(const char *filename, const char *port)
{
/* Get printer port list from win.ini, then copy to \\.\port */
char *buffer;
char portname[MAXSTR];
FILE *f;
long count;
int error = FALSE;
long lsize;
long ldone;
char pcdone[20];
MSG msg;
FILE *outfile;

	if (!get_portname(portname, port))
	    return FALSE;

	if ((buffer = malloc(PRINT_BUF_SIZE)) == (char *)NULL)
	    return FALSE;

	if ((f = fopen(filename, "rb")) == (FILE *)NULL) {
	    free(buffer);
	    return FALSE;
	}
	fseek(f, 0L, SEEK_END);
	lsize = ftell(f);
	if (lsize <= 0)
	    lsize = 1;
	fseek(f, 0L, SEEK_SET);

	
	outfile = fopen(portname, "wb");
	if (outfile == (FILE *)NULL) {
		fclose(f);
		free(buffer);
		return FALSE;
	}

	hDlgModeless = CreateDialog(phInstance, "CancelDlgBox", hwndtext, CancelDlgProc);
	ldone = 0;

	while (!error && hDlgModeless 
	  && (count = fread(buffer, 1, PRINT_BUF_SIZE, f)) != 0 ) {
	    if (fwrite(buffer, 1, count, outfile) < count)
		error = TRUE;
	    ldone += count;
	    sprintf(pcdone, "%d %%done", (int)(ldone * 100 / lsize));
	    SetWindowText(GetDlgItem(hDlgModeless, CANCEL_PCDONE), pcdone);
	    while (PeekMessage(&msg, hDlgModeless, 0, 0, PM_REMOVE)) {
		if ((hDlgModeless == 0) || !IsDialogMessage(hDlgModeless, &msg)) {
		    TranslateMessage(&msg);
		    DispatchMessage(&msg);
  		}
  	    }
  	}
	free(buffer);
	fclose(f);

	if (!hDlgModeless)
	    error=TRUE;
	DestroyWindow(hDlgModeless);
	hDlgModeless = 0;
	fclose(outfile);
	return !error;
}



#else				/* ******** !WIN32 ******** */

/* Print File to port */
private int
gp_printfile(const char *filename, const char *pmport)
{
#define PRINT_BUF_SIZE 16384u
char *buffer;
char *portname;
int i, port;
FILE *f;
DLGPROC lpfnSpoolProc;
WORD count;
DLGPROC lpfnCancelProc;
int error = FALSE;
long lsize;
long ldone;
char pcdone[20];
MSG msg;
HPJOB hJob;

	/* get list of ports */
	if ((buffer = malloc(PRINT_BUF_SIZE)) == (char *)NULL)
	    return FALSE;

	if ( (strlen(pmport)==0) || (strcmp(pmport, "PRN")==0) ) {
	    GetProfileString("ports", NULL, "", buffer, PRINT_BUF_SIZE);
	    /* select a port */
#ifdef __DLL__
	    lpfnSpoolProc = (DLGPROC)GetProcAddress(phInstance, "SpoolDlgProc");
#else
	    lpfnSpoolProc = (DLGPROC)MakeProcInstance((FARPROC)SpoolDlgProc, phInstance);
#endif
	    port = DialogBoxParam(phInstance, "SpoolDlgBox", hwndtext, lpfnSpoolProc, (LPARAM)buffer);
#ifndef __DLL__
	    FreeProcInstance((FARPROC)lpfnSpoolProc);
#endif
	    if (!port) {
	        free(buffer);
	        return FALSE;
	    }
	    portname = buffer;
	    for (i=1; i<port && strlen(portname)!=0; i++)
	        portname += lstrlen(portname)+1;
	}
	else
	    portname = (char *)pmport;	/* Print Manager port name already supplied */
	
	if ((f = fopen(filename, "rb")) == (FILE *)NULL) {
	    free(buffer);
	    return FALSE;
	}
	fseek(f, 0L, SEEK_END);
	lsize = ftell(f);
	if (lsize <= 0)
	    lsize = 1;
	fseek(f, 0L, SEEK_SET);

	hJob = OpenJob(portname, filename, (HDC)NULL);
	switch ((int)hJob) {
	    case SP_APPABORT:
	    case SP_ERROR:
	    case SP_OUTOFDISK:
	    case SP_OUTOFMEMORY:
	    case SP_USERABORT:
		fclose(f);
		free(buffer);
		return FALSE;
	}
	if (StartSpoolPage(hJob) < 0)
	    error = TRUE;

#ifdef __DLL__
	lpfnCancelProc = (DLGPROC)GetProcAddress(phInstance, "CancelDlgProc");
#else
	lpfnCancelProc = (DLGPROC)MakeProcInstance((FARPROC)CancelDlgProc, phInstance);
#endif
	hDlgModeless = CreateDialog(phInstance, "CancelDlgBox", hwndtext, lpfnCancelProc);
	ldone = 0;

	while (!error && hDlgModeless 
	  && (count = fread(buffer, 1, PRINT_BUF_SIZE, f)) != 0 ) {
	    if (WriteSpool(hJob, buffer, count) < 0)
		error = TRUE;
	    ldone += count;
	    sprintf(pcdone, "%d %%done", (int)(ldone * 100 / lsize));
	    SetWindowText(GetDlgItem(hDlgModeless, CANCEL_PCDONE), pcdone);
	    while (PeekMessage(&msg, hDlgModeless, 0, 0, PM_REMOVE)) {
		if ((hDlgModeless == 0) || !IsDialogMessage(hDlgModeless, &msg)) {
		    TranslateMessage(&msg);
		    DispatchMessage(&msg);
  		}
  	    }
  	}
	free(buffer);
	fclose(f);

	if (!hDlgModeless)
	    error=TRUE;
	DestroyWindow(hDlgModeless);
	hDlgModeless = 0;
#ifndef __DLL__
	FreeProcInstance((FARPROC)lpfnCancelProc);
#endif
	EndSpoolPage(hJob);
	if (error)
	    DeleteJob(hJob, 0);
	else
	    CloseJob(hJob);
	return !error;
}

#endif				/* ******** (!)WIN32 ******** */

/* ------ File naming and accessing ------ */

/* Create and open a scratch file with a given name prefix. */
/* Write the actual file name at fname. */
FILE *
gp_open_scratch_file(const char *prefix, char *fname, const char *mode)
{	char *temp;
	if ( (temp = getenv("TEMP")) == NULL )
		*fname = 0;
	else
	{	strcpy(fname, temp);
		/* Prevent X's in path from being converted by mktemp. */
		for ( temp = fname; *temp; temp++ )
			*temp = tolower(*temp);
		if ( strlen(fname) && (fname[strlen(fname)-1] != '\\') )
			strcat(fname, "\\");
	}
	strcat(fname, prefix);
	strcat(fname, "XXXXXX");
	mktemp(fname);
	return fopen(fname, mode);
}

/* Open a file with the given name, as a stream of uninterpreted bytes. */
FILE *
gp_fopen(const char *fname, const char *mode)
{	return fopen(fname, mode);
}

/* ====== Substitute for stdio ====== */

/* Forward references */
private void win_std_init(void);
private stream_proc_process(win_std_read_process);
private stream_proc_process(win_pipe_read_process);
private stream_proc_process(win_std_write_process);

/* Use a pseudo IODevice to get win_stdio_init called at the right time. */
/* This is bad architecture; we'll fix it later. */
private iodev_proc_init(win_stdio_init);
gx_io_device gs_iodev_wstdio = {
	"wstdio", "Special",
	{ win_stdio_init, iodev_no_open_device,
	  iodev_no_open_file, iodev_no_fopen, iodev_no_fclose,
	  iodev_no_delete_file, iodev_no_rename_file,
	  iodev_no_file_status, iodev_no_enumerate_files
	}
};

/* Do one-time initialization */
private int
win_stdio_init(gx_io_device *iodev, gs_memory_t *mem)
{
	win_std_init();		/* redefine stdin/out/err to our window routines */
	return 0;
}

/* Define alternate 'open' routines for our stdin/out/err streams. */

extern int iodev_stdin_open(P4(gx_io_device *, const char *, stream **,
			       gs_memory_t *));
private int
win_stdin_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stdin_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code != 1 )
	  return code;
#ifndef __DLL__
	s->procs.process =
	  (gsview ? win_pipe_read_process : win_std_read_process);
#else
	s->procs.process = win_std_read_process;
#endif
	s->file = NULL;
	return 0;
}

extern int iodev_stdout_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
win_stdout_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stdout_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code != 1 )
	  return code;
	s->procs.process = win_std_write_process;
	s->file = NULL;
	return 0;
}

extern int iodev_stderr_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
win_stderr_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{	int code = iodev_stderr_open(iodev, access, ps, mem);
	stream *s = *ps;
	if ( code != 1 )
	  return code;
	s->procs.process = win_std_write_process;
	s->file = NULL;
	return 0;
}

/* Patch stdin/out/err to use our windows. */
private void
win_std_init(void)
{
	/* If stdxxx is the console, replace the 'open' routines, */
	/* which haven't gotten called yet. */

	if ( gp_file_is_console(gs_stdin) )
	  gs_findiodevice((const byte *)"%stdin", 6)->procs.open_device =
	    win_stdin_open;

	if ( gp_file_is_console(gs_stdout) )
	  gs_findiodevice((const byte *)"%stdout", 7)->procs.open_device =
	    win_stdout_open;

	if ( gp_file_is_console(gs_stderr) )
	  gs_findiodevice((const byte *)"%stderr", 7)->procs.open_device =
	    win_stderr_open;
}

#ifdef __DLL__
private int
win_std_read_process(stream_state *st, stream_cursor_read *ignore_pr,
  stream_cursor_write *pw, bool last)
{
	int count = pw->limit - pw->ptr;

	if ( count == 0 ) 		/* empty buffer */
		return 1;

/* callback to get more input */
	count = (*pgsdll_callback)(GSDLL_STDIN, pw->ptr+1, count);
	if (count == 0) {
	    /* EOF */
	    /* what should we do? */
	    return EOFC;
	}

	pw->ptr += count;
	return 1;
}
#else /* !__DLL__ */
/* We should really use a private buffer for line reading, */
/* because we can't predict the size of the supplied input area.... */
private int
win_std_read_process(stream_state *st, stream_cursor_read *ignore_pr,
  stream_cursor_write *pw, bool last)
{	byte *buf = pw->ptr;
	/* Implement line editing here. */
#define ERASELINE 21		/* ^U */
#define ERASECHAR1 8		/* ^H */
#define ERASECHAR2 127		/* DEL */
	byte *dest = buf;
	byte *limit = pw->limit - 1;	/* always leave room for \n */
	uint ch;
	if ( dest > limit )		/* empty buffer */
		return 1;
	while ( (ch = TextGetCh(&textwin)) != '\n' )
	{	switch ( ch )
		{
		default:
			if ( dest == limit )
				MessageBeep(-1);
			else
			{	*++dest = ch;
				TextPutCh(&textwin, (char) ch);
			}
			break;
		case ERASELINE:
			while ( dest > buf )
			{	TextPutCh(&textwin, '\b');
				TextPutCh(&textwin, ' ');
				TextPutCh(&textwin, '\b');
				dest--;
			}
			break;
		case ERASECHAR1:
		case ERASECHAR2:
			if ( dest > buf )
			{	TextPutCh(&textwin, '\b');
				TextPutCh(&textwin, ' ');
				TextPutCh(&textwin, '\b');
				dest--;
			}
			break;
		}
	}
	*++dest = ch;
	TextPutCh(&textwin, (char) ch);
	pw->ptr = dest;
	return 1;
}
#endif /* !__DLL__ */


private int
win_std_write_process(stream_state *st, stream_cursor_read *pr,
  stream_cursor_write *ignore_pw, bool last)
{	uint count = pr->limit - pr->ptr;
#ifdef __DLL__
	(*pgsdll_callback)(GSDLL_STDOUT, (char *)(pr->ptr +1), count);
	pr->ptr = pr->limit;
#else
	/* The prototype for TextWriteBuf is missing a const.... */
	TextWriteBuf(&textwin, (char *)(pr->ptr + 1), count);
	pr->ptr = pr->limit;
	(void)gp_check_interrupts();
#endif
	return 0;
}

/* This is used instead of the stdio version. */
/* The declaration must be identical to that in <stdio.h>. */
#if defined(_WIN32) && defined(_MSC_VER)
int _CRTAPI2
fprintf(FILE *file, const char *fmt, ...)
#else
int _Cdecl _FARFUNC
fprintf(FILE _FAR *file, const char *fmt, ...)
#endif
{
int count;
va_list args;
	va_start(args,fmt);
	if ( gp_file_is_console(file) ) {
		char buf[1024];
		count = vsprintf(buf,fmt,args);
#ifdef __DLL__
		(*pgsdll_callback)(GSDLL_STDOUT, buf, count);
#else
/*		MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP); */
		TextWriteBuf(&textwin, buf, count);
#endif
	}
	else
		count = vfprintf(file, fmt, args);
	va_end(args);
	return count;
}

/* ------ Imitation pipes (only used with gsview) ------ */

/* Forward references */
private void win_pipe_request(void);
private int win_pipe_read(char *, unsigned int);

private int
win_pipe_read_process(stream_state *st, stream_cursor_read *ignore_pr,
  stream_cursor_write *pw, bool last)
{	if ( pw->ptr < pw->limit )
	{	int nread = win_pipe_read(pw->ptr + 1, 1);
		if ( nread <= 0 )
			return EOFC;
		pw->ptr++;
	}
	return 1;
}

/* get a block from gsview */
private void
win_pipe_request(void)
{
    MSG msg;
	/* request another block */
#ifdef __WIN32__
	if (is_winnt) {
	    pipe_hglobal = (HGLOBAL)NULL;
	    pipe_lpbyte = (LPBYTE)NULL;
	    if (pipe_eof)
		return;
	}
	else
#endif
	{
	    if (pipe_lpbyte != (LPBYTE)NULL)
		    GlobalUnlock(pipe_hglobal);
	    if (pipe_hglobal != (HGLOBAL)NULL)
		    GlobalFree(pipe_hglobal);
	    pipe_hglobal = (HGLOBAL)NULL;
	    pipe_lpbyte = (LPBYTE)NULL;
	}

	SendMessage(gsview_hwnd, WM_GSVIEW, PIPE_REQUEST, 0L);
	/* wait for block */
	/* don't use PeekMessage in a loop - would waste CPU cycles under WinNT */
    	while ((pipe_hglobal == (HGLOBAL)NULL)
	  && GetMessage(&msg, 0, 0, 0)) {
	    TranslateMessage(&msg);
	    DispatchMessage(&msg);
	}
#ifdef __WIN32__
	if (is_winnt) {
	    if (pipe_mapptr == NULL) {
		char buf[64];
		sprintf(buf, "gsview_%d", gsview_hwnd);
		pipe_hmapfile = OpenFileMapping(FILE_MAP_READ, FALSE, buf);
		pipe_mapptr = MapViewOfFile(pipe_hmapfile, FILE_MAP_READ, 0, 0, 0);
		if (pipe_mapptr == NULL) {
	            MessageBox((HWND)NULL, "Imitation pipe handle is zero", szAppName, MB_OK | MB_ICONSTOP);
		    pipe_lpbyte = NULL;
		    pipe_count = 0;
		    return;
		}
	    }
	    pipe_lpbyte = pipe_mapptr+sizeof(WORD);
	    pipe_count = *((WORD *)pipe_mapptr);
	}
	else
#endif
	{
	  if (pipe_hglobal == (HGLOBAL)NULL) {
	    MessageBox((HWND)NULL, "Imitation pipe handle is zero", szAppName, MB_OK | MB_ICONSTOP);
          }
	  else {
	    pipe_lpbyte = GlobalLock(pipe_hglobal);
	    if (pipe_lpbyte == (LPBYTE)NULL)  {
		char buf[256];
		sprintf(buf, "Imitation pipe handle = 0x%x, pointer = NULL", pipe_hglobal);
	        MessageBox((HWND)NULL, buf, szAppName, MB_OK | MB_ICONSTOP);
		pipe_count = 0;
	    }
	    else {
	        pipe_count = *((WORD FAR *)pipe_lpbyte);
	        pipe_lpbyte += sizeof(WORD);
	    }
	  }
	}
}

/* pipe_read - similar to fread */
private int
win_pipe_read(char *ptr, unsigned int count)
{
	int n;
	if (!pipe_count)
	    win_pipe_request();
	if (!pipe_count)
	    return -1;
	n = min(count, pipe_count);
	memcpy(ptr, pipe_lpbyte, n);
	pipe_lpbyte += n;
	pipe_count -= n;
	return n;
}
