/* Copyright (C) 1995 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* imain.h */
/* Interface to gsmain.c */

#ifndef imain_INCLUDED
#  define imain_INCLUDED

#include "gsexit.h"			/* exported by gsmain.c */

/*
 * This file defines the intended API between Ghostscript front ends
 * (such as gs.c, the customary command-line-driven front end)
 * and gsmain.c, which provides top-level control of the interpreter.
 * Someday, it will include reasonable documentation.
 */

/*
 * Define an instance of the interpreter.  This is used almost nowhere
 * as yet, but will eventually replace all (?) static variables.
 */
typedef struct gs_main_instance_s gs_main_instance;

/* ================ Exported procedures from gsmain.c ================ */

/* ---------------- Instance creation ---------------- */

/* Multiple instances are not supported yet: */
/*gs_main_instance *gs_main_alloc_instance(P1(gs_memory_t *));*/
/* Instead, we only provide a default instance: */
gs_main_instance *gs_main_instance_default(P0());

/* ---------------- Initialization ---------------- */

void gs_main_init0(P5(gs_main_instance *minst, FILE *in, FILE *out, FILE *err,
		      int max_lib_paths));
void gs_main_init1(P1(gs_main_instance *minst));
void gs_main_init2(P1(gs_main_instance *minst));

void gs_main_add_lib_path(P2(gs_main_instance *minst, const char *path));

void gs_main_set_lib_paths(P1(gs_main_instance *minst));

int gs_main_lib_open(P3(gs_main_instance *minst, const char *fname,
			ref *pfile));

/* ---------------- Execution ---------------- */

/*
 * The value returned by gs_run_file and gs_run_string[_with_length] is
 * 0 if the interpreter ran to completion, e_Quit for a normal quit,
 * e_Fatal for a non-zero quit or a fatal error.
 * e_Fatal stores the exit code in the third argument.
 * The str argument of gs_run_string[_with_length] must be allocated
 * in non-garbage-collectable space (e.g., by malloc or gs_malloc,
 * or statically).
 */
int gs_main_run_file(P5(gs_main_instance *minst,
			const char *fname,
			int user_errors, int *pexit_code,
			ref *perror_object));
int gs_main_run_string(P5(gs_main_instance *minst,
			  const char *str,
			  int user_errors, int *pexit_code,
			  ref *perror_object));
int gs_main_run_string_with_length(P6(gs_main_instance *minst,
				      const char *str, uint length,
				      int user_errors, int *pexit_code,
				      ref *perror_object));

/*
 * Open the file for gs_main_run_file.  This is an internal routine
 * that is only exported for some special clients.
 */
int gs_main_run_file_open(P3(gs_main_instance *minst,
			     const char *file_name, ref *pfref));

/*
 * The next 3 procedures provide for feeding input to the interpreter
 * in arbitrary chunks, unlike run_string, which requires that each string
 * be a properly formed PostScript program fragment.  To use them:
 *	Call run_string_begin.
 *	Call run_string_continue as many times as desired,
 *	  stopping if it returns anything other than e_NeedInput.
 *	If run_string_continue didn't indicate an error or a quit
 *	  (i.e., a return value other than e_NeedInput), call run_string_end
 *	  to provide an EOF indication.
 * Note that run_string_continue takes a pointer and a length, like
 * run_string_with_length.
 */
int gs_main_run_string_begin(P4(gs_main_instance *minst, int user_errors,
				int *pexit_code, ref *perror_object));
int gs_main_run_string_continue(P6(gs_main_instance *minst,
				   const char *str, uint length,
				   int user_errors, int *pexit_code,
				   ref *perror_object));
int gs_main_run_string_end(P4(gs_main_instance *minst, int user_errors,
			      int *pexit_code, ref *perror_object));


/* ---------------- Operand stack access ---------------- */

/*
 * The following procedures are not used in normal operation;
 * they exist only to allow clients driving the interpreter through
 * the gs_run_xxx procedures to get results back.
 * They return 0, e_stackunderflow, or e_typecheck.
 */
int gs_pop_boolean(P2(gs_main_instance *minst, bool *result));
int gs_pop_integer(P2(gs_main_instance *minst, long *result));
int gs_pop_real(P2(gs_main_instance *minst, float *result));
/* gs_pop_string returns 1 if the string is read-only. */
int gs_pop_string(P2(gs_main_instance *minst, gs_string *result));

/* ---------------- Debugging ---------------- */

void gs_debug_dump_stack(P2(int code, ref *perror_object));

/* ---------------- Termination ---------------- */

void gs_main_finit(P3(gs_main_instance *minst, int exit_status, int code));

/* ================ Other exported procedures ================ */

/* Direct interface to the interpreter. */
/* Clients do not normally use this. */
int gs_interpret(P4(ref *pref, int user_errors, int *pexit_code, ref *perror_object));

#endif					/* imain_INCLUDED */
