/* Copyright (C) 1995 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gxcldev.h */
/* Internal definitions for Ghostscript command lists. */
#include "gxclist.h"
#include "gsropt.h"
#include "gxfixed.h"			/* for gzpath.h */
#include "gzline.h"
#include "gzpath.h"			/* for included paths */
#include "strimpl.h"			/* for compressed bitmaps */
#include "srlx.h"			/* ditto */

/****************************************************************
 * NOTE: the facilities for including paths in command lists    *
 * have never been tested and are not being used yet.  Don't    *
 * try to use them!  This includes all opcodes marked \*?*\.    *
 ****************************************************************/

#define cdev ((gx_device_clist *)dev)

/* ---------------- Commands ---------------- */

/* A command always consists of an operation followed by operands. */
/* The operands are implicit in the procedural code. */
/* Operands marked # are variable-size, encoded with the variable-size */
/* integer encoding.  Operands marked $ are sized according to the */
/* device depth.  Operands in <> are sized according to other state */
/* information. */
typedef enum {
	cmd_op_misc = 0x00,		/* (see below) */
	  cmd_opv_end_run = 0x00,	/* (nothing) */
	  cmd_opv_set_tile_size = 0x01,	/* width#, height#, rep_w#, rep_h# */
	  cmd_opv_set_tile_size_colored = 0x02,  /* width#, height#, rep_w#, */
					/* rep_h# */
	  cmd_opv_set_tile_phase = 0x03, /* x#, y# */
	  cmd_opv_set_tile_index = 0x04, /* index */
	  cmd_opv_set_tile_index_long = 0x05,  /* index# */
	  cmd_opv_set_tile_bits = 0x06,	/* index, <bits> */
	  cmd_opv_set_tile_bits_long = 0x07,  /* index#, <bits> */
	  cmd_opv_set_lop = 0x08,	/* lop# */
	  cmd_opv_set_tile_color0 = 0x09,  /* color$ */
	  cmd_opv_set_tile_color1 = 0x0a,  /* color$ */
	  cmd_opv_enable_lop = 0x0b,	/* (nothing) */
	  cmd_opv_disable_lop = 0x0c,	/* (nothing) */
/*?*/	  cmd_opv_set_line_width = 0x0d,   /* width(float) */
/*?*/	  cmd_opv_set_cap_join = 0x0e,	/* cap*16+join */
/*?*/	  cmd_opv_set_flatness = 0x0f,	/* flatness(float) */
	cmd_op_set_color0 = 0x10,	/* dcolor+8 in op byte | color$ */
	cmd_op_set_color1 = 0x20,	/* dcolor+8 in op byte | color$ */
	cmd_op_delta_tile_bits = 0x30,	/* n-1 in op byte, index#, */
					/* n x <offset, bits> */
	cmd_op_fill_rect = 0x40,	/* dy2dh2 in op byte, x#, w# | rect# */
	cmd_op_fill_rect_short = 0x50,	/* dh in op byte,dx,dw | rect_short */
	cmd_op_fill_rect_tiny = 0x60,	/* dw in op byte, rect_tiny */
	cmd_op_tile_rect = 0x70,	/* dy2dh2 in op byte, x#, w# | rect# */
	cmd_op_tile_rect_short = 0x80,	/* dh in op byte,dx,dw | rect_short */
	cmd_op_tile_rect_tiny = 0x90,	/* dw in op byte, rect_tiny */
	cmd_op_copy_mono = 0xa0,	/* (rect#, data_x#, raster# | */
					/*  d_x+1 in op byte, x#, y#, w, h), */
					/* <bits> */
	cmd_op_copy_mono_rle = 0xb0,	/* rect#, data_x#, raster#, <bits> */
	cmd_op_copy_color = 0xc0,	/* rect#, data_x#, raster#, <bits> */
	cmd_op_delta_tile_index = 0xd0,	/* d_index+8 in op byte */
	cmd_op_copy_alpha = 0xe0,	/* depth in op byte, then */
					/* same as copy_color */
/*?*/	cmd_op_path = 0xf0,		/* nbytes, <path> */
/*?*/	  cmd_opv_append = 0xf0,
/*?*/	  cmd_opv_fill = 0xf1,
/*?*/	  cmd_opv_eofill = 0xf2,
/*?*/	  cmd_opv_stroke = 0xf3,
/*?*/	  cmd_opv_clip = 0xf4,
/*?*/	  cmd_opv_eoclip = 0xf5,
	cmd_op_end
} gx_cmd_op;

#define cmd_op_name_strings\
  "(misc)", "set_color[0]", "set_color[1]", "delta_tile_bits",\
  "fill_rect", "fill_rect_short", "fill_rect_tiny", "tile_rect",\
  "tile_rect_short", "tile_rect_tiny", "copy_mono", "copy_mono_rle",\
  "copy_color", "delta_tile_index", "copy_alpha", "(path)"

#define cmd_misc_op_name_strings\
  "end_run", "set_tile_size", "set_tile_size_colored", "set_tile_phase",\
  "set_tile_index", "set_tile_index_long", "set_tile_bits", "set_tile_bits_long",\
  "set_lop", "set_tile_color[0]", "set_tile_color[1]", "enable_lop",\
  "disable_lop", "set_line_width", "set_cap_join", "set_flatness"

#define cmd_path_op_name_strings\
  "append", "fill", "eofill", "stroke", "clip", "eoclip", "?f6?", "?f7",\
  "?f8?", "?f9?", "?fa?", "?fb?", "?fc?", "?fd?", "?fe?", "?ff?"

#ifdef DEBUG
extern const char *cmd_op_names[16];
extern const char *cmd_misc_op_names[16];
extern const char *cmd_path_op_names[16];
#endif

/* Define the size of the largest command, not counting any bitmap or */
/* variable-length operands. */
#define cmd_tile_d 3			/* size of tile delta */
/* The variable-size integer encoding is little-endian.  The low 7 bits */
/* of each byte contain data; the top bit is 1 for all but the last byte. */
#define cmd_max_intsize(siz)\
  (((siz) * 8 + 6) / 7)
#define cmd_largest_size\
  max(1 + 6 * cmd_max_intsize(sizeof(int)) /* copy_mono */,\
      1 + cmd_max_intsize(sizeof(int)) + 16 * cmd_tile_d /* delta_tile 15 */)

/*
 * We represent paths in a command list in a straightforward way,
 * similar to CharStrings in a Type 1 font.
 */
typedef enum {
	cp_rmoveto,
	cp_rlineto,
	cp_hlineto,
	cp_vlineto,
	cp_rrcurveto,
	cp_hvcurveto,
	cp_vhcurveto,
	cp_ccurveto,		/* conic curveto, p1 = p2 */
	cp_closepath
} clist_path_op;
static const byte clist_path_op_num_operands[] = { 2, 2, 1, 1, 6, 4, 4, 4, 0 };

#define clist_path_op_name_strings\
  "rmoveto", "rlineto", "hlineto", "vlineto",\
  "rrcurveto", "hvcurveto", "vhcurveto", "ccurveto",\
  "closepath"

/* ---------------- Command parameters ---------------- */

/* Rectangle */
typedef struct {
	int x, y, width, height;
} gx_cmd_rect;
/* Short rectangle */
typedef struct {
	byte dx, dwidth, dy, dheight;	/* dy and dheight are optional */
} gx_cmd_rect_short;
#define cmd_min_short (-128)
#define cmd_max_short 127
/* Tiny rectangle */
typedef struct {
	unsigned dx : 4;
	unsigned dy : 4;
} gx_cmd_rect_tiny;
#define cmd_min_tiny (-8)
#define cmd_max_tiny 7

/*
 * We represent path coordinates as 'fixed' values in a variable-length,
 * relative form:
 *	0sxxxxxx xxxxxxxx xxxxxxxx
 *	10sxxxxx xxxxxxxx xxxxxxxx xxxxxxxx
 *	11------ (a full-size `fixed' value)
 */
#define is_diff3(d) !(((d) + 0x400000) & -0x800000)
#define is_diff4(d) !(((d) + 0x20000000) & -0x40000000)

/* ---------------- Block file entries ---------------- */

typedef struct cmd_block_s {
	int band;
#define cmd_band_all (-1)		/* do in all bands */
#define cmd_band_end (-2)		/* end of band file */
	long pos;			/* starting position in cfile */
} cmd_block;

/* ---------------- Band state ---------------- */

/* Remember the current state of one band when writing or reading. */
struct gx_clist_state_s {
	gx_color_index colors[2];	/* most recent colors */
	tile_slot *tile;		/* most recent tile */
	gs_int_point tile_phase;	/* most recent tile phase */
	gx_color_index tile_colors[2];	/* most recent tile colors */
	gx_cmd_rect rect;		/* most recent rectangle */
	gs_logical_operation_t lop;	/* most recent logical op */
	int lop_enabled;		/* 0 = don't use lop, 1 = use lop, */
					/* -1 is used internally */
		/* Following are only used when writing */
	cmd_list list;			/* list of commands for band */
};
static tile_slot no_tile = { ~0L };

/* The initial values for a band state */
static const gx_clist_state cls_initial =
{	 { gx_no_color_index, gx_no_color_index },
	&no_tile, { 0, 0 }, { gx_no_color_index, gx_no_color_index },
	 { 0, 0, 0, 0 }, lop_default, 0,
	 { 0, 0 }
};

/* Define the size of the command buffer used for reading. */
/* This is needed to split up large copy_ and path operations. */
#define cbuf_size 500
